-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
M.type = "auxiliary"
M.relevantDevice = nil

local stateGroups = {}
local electricsMap = {}
local state = "off"

local function objInTable(tbl, obj)
  for k,v in pairs(tbl) do
    if v == obj then return true end
  end
  return false
end

local function addObjToTableIfNotExists(tbl, obj)
  if not objInTable(tbl, obj) then table.insert(tbl, obj) end
end

local function killAllElectrics()
  for k,v in pairs(electricsMap) do
    electrics.values[v] = 0
  end
end

local function activatePhase(phase)
  for k,electric in pairs(phase.electrics) do electrics.values[electric] = 1 end
end

local function deactivatePhase(phase)
  for k,electric in pairs(phase.electrics) do electrics.values[electric] = 0 end
end

local function resetState(stateData) 
  for k2, statePhaseGroup in pairs(stateData) do
    statePhaseGroup.timer = 0.0
    statePhaseGroup.phase = 1
  end
end

local function activateStateFirstPhases(stateData)
  for k2, statePhaseGroup in pairs(stateData) do
    local phases = statePhaseGroup.phases
    if #phases > 0 then
      activatePhase(phases[1])
    end
  end
end

local function getState()
  return state
end

local function setState(newState)
  local oldState = state
  state = newState
  if oldState ~= newState then 
    -- reset all electrics
    killAllElectrics() 
    
    -- reset the timers on this state
    local stateData = stateGroups[state]
    if stateData ~= nil then
      resetState(stateData)
      activateStateFirstPhases(stateData)
    end
  end
  
end

local function updateState(dt)
  local stateData = stateGroups[state]
  if stateData == nil then return end
  
  for k2, statePhaseGroup in pairs(stateData) do
    local curPhaseIndex = statePhaseGroup.phase
    local curPhaseTime = statePhaseGroup.timer
    local phases = statePhaseGroup.phases
    local phaseCount = #phases
    
    if phaseCount > 0 then
      -- get phase and increment group timer
      local curPhase = phases[curPhaseIndex]
      curPhaseTime = curPhaseTime + dt
      statePhaseGroup.timer = curPhaseTime
      
      -- advance state
      if curPhaseTime > curPhase.duration then
        -- deactivate current
        deactivatePhase(curPhase)
        
        -- inc index
        local newPhaseIndex = curPhaseIndex + 1
        if newPhaseIndex > phaseCount then newPhaseIndex = 1 end
        statePhaseGroup.phase = newPhaseIndex
        
        -- activate next phase
        activatePhase(phases[newPhaseIndex])
        
        --reset timer
        statePhaseGroup.timer = curPhaseTime - curPhase.duration
      end
    end
    
  end
end

local function updateGFX(dt)
  if state ~= "off" then updateState(dt) end
end

--remaps electrics to account for the lack of presence of the amber lights
local function remapElectric(electric, remapAmberF, remapAmberR)
    if remapAmberF and electric == "amberlight_fr" then
        return "redlight_fr"
    end
    if remapAmberF and electric == "amberlight_fl" then
        return "redlight_fl"
    end
    if remapAmberR and electric == "amberlight_rr" then
        return "redlight_rr"
    end
    if remapAmberR and electric == "amberlight_rl" then
        return "redlight_rl"
    end
    return electric
end

local function init(jbeamData)
  local mergeGroups = jbeamData.stateGroups or {}
  
  stateGroups = {}
  electricsMap = {}
  state = "off"
  
  -- determine if we should remap ambers
  local remapAmberR = false
  local remapAmberF = false
  local partConfig = v.config.parts
  if partConfig and jbeamData.amberLightSlotNames then
    local lightSlotNames = jbeamData.amberLightSlotNames
    remapAmberF = lightSlotNames.front and (partConfig[lightSlotNames.front] == nil or partConfig[lightSlotNames.front]:len() == 0)
    remapAmberR = lightSlotNames.rear and (partConfig[lightSlotNames.rear] == nil or partConfig[lightSlotNames.rear]:len() == 0)
  end
  
  --d_ = data
  --c_ = constructed
  for k,d_stateData in pairs(mergeGroups) do
    -- "moving": [ etc
    local c_stateData = {} 
    
    for k2, d_statePhaseGroup in pairs(d_stateData) do
      --"main": [ etc
      --set total duration as well as timer
      local c_statePhaseGroup = {}
      local phasesInGroup = {}
      local statePhaseGroupDuration = 0
      local statePhaseGroupNumPhases = 0
      
      -- for each duration pair
      for k3, d_statePhase in pairs(d_statePhaseGroup) do
        local c_statePhase = {}
        
        c_statePhase.electrics = {}
        c_statePhase.duration = d_statePhase.duration or 0.0
        statePhaseGroupDuration = statePhaseGroupDuration + d_statePhase.duration
        statePhaseGroupNumPhases = statePhaseGroupNumPhases + 1
        
        --add electrics
        if d_statePhase.electrics ~= nil and type(d_statePhase.electrics) == "table" then
          for k4, electric in pairs(d_statePhase.electrics) do
            table.insert(c_statePhase.electrics, remapElectric(electric, remapAmberF, remapAmberR))
          end
        end
        
        --accept a single "electric" too
        if d_statePhase.electric ~= nil and type(d_statePhase.electric) == "string" then
          table.insert(c_statePhase.electrics, remapElectric(d_statePhase.electric, remapAmberF, remapAmberR))
        end
        
        --add all electrics to our global map
        for k4, electric in pairs(c_statePhase.electrics) do
          addObjToTableIfNotExists(electricsMap, electric)
        end
        
        -- add to phasesInGroup
        table.insert(phasesInGroup, c_statePhase)
      end
      
      -- 
      c_statePhaseGroup.timer = 0
      c_statePhaseGroup.duration = statePhaseGroupDuration
      c_statePhaseGroup.phase = 1
      c_statePhaseGroup.phases = phasesInGroup
      c_stateData[k2] = c_statePhaseGroup
    end
    
    stateGroups[k] = c_stateData
  end
  
  if jbeamData.debug == true then
    dump(stateGroups)
  end
  killAllElectrics()
end

M.init = init
M.updateGFX = updateGFX
M.getState = getState
M.setState = setState

return M
