local M = {}
M.type = "auxiliary"

local crossingArmInhibitElectricName = "xarm_inhibit"
local doorElectricName = "sidedoor"
local crossingArmElectricName = "crossingarm"
local stopSignElectricName = "stopsign"
local warningLightsMasterElectricName = "wl_master"
local warningLightsEmergencyElectricName = "wl_emergency"

local doorsOpenState = false
local lastDoorsOpenState = false

local xarmInhibitState = false
local warningLightsMaster = true
local warningLightsEmergency = false

local warningLightsOn = false

local doorOpenSoundEvent = nil
local doorCloseSoundEvent = nil
local doorSoundCooldown = 0
local doorSoundCutoffTime = 0

local function updateElectricsSystem()
  local xArmIsOut =  (warningLightsOn and (doorsOpenState and not xarmInhibitState)) or warningLightsEmergency
  local stopSignIsOut = (warningLightsOn and doorsOpenState) or warningLightsEmergency
  electrics.values[crossingArmElectricName] = xArmIsOut and 1 or 0
  electrics.values[stopSignElectricName] = stopSignIsOut and 1 or 0
  
end

local function updatePneumaticsSystem()
  electrics.values[doorElectricName] = doorsOpenState and 1 or 0
end

local function updateElectricsVisual()
  electrics.values[warningLightsMasterElectricName] = warningLightsMaster and 1 or 0
  electrics.values[warningLightsEmergencyElectricName] = warningLightsEmergency and 1 or 0
end

local function updateElectricsFull()
  if warningLightsMaster then
    updateElectricsSystem()
  end
  updatePneumaticsSystem()
  updateElectricsVisual()
end

local function getCurrentLightsState()
  if warningLightsMaster then
    if warningLightsEmergency then return "stopped" end
    if warningLightsOn then
      return doorsOpenState and "stopped" or "moving"
    end
  end
  return "off"
end

local function updateLightsState()
  controller.getControllerSafe("schoolBusLights").setState(getCurrentLightsState())
end

local function setWarningLightsOn(state)
  warningLightsOn = state and warningLightsMaster
  updateLightsState()
  updateElectricsFull()
end

local function toggleWarningLightsMaster()
  warningLightsMaster = not warningLightsMaster
  warningLightsOn = warningLightsOn and warningLightsMaster
  updateLightsState()
  updateElectricsFull()
end

local function setWarningLightsEmergency(state)
  warningLightsEmergency = state
  updateLightsState()
  updateElectricsFull()
end

local function toggleWarningLightsEmergency()
  setWarningLightsEmergency(not warningLightsEmergency)
end

local function toggleWarningLights()
  local lightsController = controller.getControllerSafe("schoolBusLights")
  setWarningLightsOn(not warningLightsOn)
end

local function toggleXArmInhibit()
  xarmInhibitState = not xarmInhibitState
  updateElectricsFull()
end

local function setDoorOpen(state)
  doorsOpenState = state
  updateLightsState()
  updateElectricsFull()
  
  if not state and lastDoorsOpenState and getCurrentLightsState() ~= "off" then
    setWarningLightsOn(false)
    updateLightsState()
  end
  
  if state and doorOpenSoundEvent and doorSoundCooldown == 0 then
    obj:setVolume(doorOpenSoundEvent, 1.0)
    obj:playSFX(doorOpenSoundEvent)
  elseif not state and doorCloseSoundEvent and doorSoundCooldown == 0 then
    obj:setVolume(doorCloseSoundEvent, 1.0)
    obj:playSFX(doorCloseSoundEvent)
  end
  doorSoundCooldown = 1.0
  doorSoundCutoffTime = 1.5
end

local function toggleDoorsOpen()
  setDoorOpen(not doorsOpenState)
end

local function updateGFX(dt)
  doorSoundCooldown = math.max(0, doorSoundCooldown - dt)

  local lastDoorSoundCutoffTime = doorSoundCutoffTime
  doorSoundCutoffTime = math.max(0, doorSoundCutoffTime - dt)
  if lastDoorSoundCutoffTime > doorSoundCutoffTime and doorSoundCutoffTime == 0 then
    obj:stopSFX(doorCloseSoundEvent)
    obj:stopSFX(doorOpenSoundEvent)
  end
  
    -- 
  lastDoorsOpenState = doorsOpenState
end

local function onReset()
  obj:cutSFX(doorCloseSoundEvent)
  obj:cutSFX(doorOpenSoundEvent)
  doorsOpenState = false
  setWarningLightsOn(false)
  setWarningLightsEmergency(false)
  updateElectricsFull()
end

local function initSounds(jbeamData)
  local doorSoundNode = jbeamData.doorSoundNode and beamstate.nodeNameMap[jbeamData.doorSoundNode] or 0
  doorOpenSoundEvent = obj:createSFXSource("event:>Vehicle>Pneumatics>Door_Open", "AudioDefaultLoop3D", "DoorOpen", doorSoundNode)
  doorCloseSoundEvent = obj:createSFXSource("event:>Vehicle>Pneumatics>Door_Close", "AudioDefaultLoop3D", "DoorClose", doorSoundNode)
end

local function init(jbeamData)
  doorElectricName = jbeamData.doorElectricName or doorElectricName
  crossingArmElectricName = jbeamData.crossingArmElectricName or crossingArmElectricName
  stopSignElectricName = jbeamData.stopSignElectricName or stopSignElectricName
  warningLightsMasterElectricName = jbeamData.warningLightsMasterElectricName or warningLightsMasterElectricName
  warningLightsEmergencyElectricName = jbeamData.warningLightsEmergencyElectricName or warningLightsEmergencyElectricName
  updateElectricsFull()
end

-- public interface
M.init = init
M.initSounds = initSounds
M.reset = onReset
M.updateGFX = updateGFX

M.toggleDoorsOpen = toggleDoorsOpen
M.toggleXArmInhibit = toggleXArmInhibit
M.toggleWarningLightsMaster = toggleWarningLightsMaster
M.toggleWarningLights = toggleWarningLights
M.setWarningLightsOn = setWarningLightsOn
M.toggleWarningLightsEmergency = toggleWarningLightsEmergency
M.setWarningLightsEmergency = setWarningLightsEmergency

return M
