-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}

local max = math.max

local targetAcceleration = 3

local isEnabled = false
local targetSpeed = 100 / 3.6
local rampedTargetSpeed = 0
local throttleSmooth = newTemporalSmoothing(200, 200)
local speedPID = newPIDStandard(0.3, 2, 0.0, 0, 1, 1, 1, 0, 2)

local function onReset()
  throttleSmooth:reset()
end


local function computeThrottle(dt)
  if not isEnabled then
    return -1
  end

  --ramp up/down our target speed with our desired target acceleration to avoid integral wind-up
  if rampedTargetSpeed ~= targetSpeed then
    local upperLimit = targetSpeed > rampedTargetSpeed and targetSpeed or rampedTargetSpeed
    local lowerLimit = targetSpeed < rampedTargetSpeed and targetSpeed or rampedTargetSpeed
    rampedTargetSpeed = clamp(rampedTargetSpeed + sign(targetSpeed - rampedTargetSpeed) * targetAcceleration * dt, lowerLimit, upperLimit)
  end

  local currentSpeed = electrics.values.wheelspeed or 0
  local output = speedPID:get(currentSpeed, rampedTargetSpeed, dt)
  return throttleSmooth:getUncapped(output, dt)

end


local function setSpeed(tgtSpeed)
  isEnabled = true
  targetSpeed = tgtSpeed / 3.6
  rampedTargetSpeed = electrics.values.wheelspeed or 0
end


local function setEnabled(enabled)
  isEnabled = enabled
  electrics.values.throttleOverride = nil
  rampedTargetSpeed = electrics.values.wheelspeed or 0
  throttleSmooth:reset()
end

local function setTargetAcceleration(target)
  targetAcceleration = target
end



-- public interface
M.onReset = onReset
M.setSpeed = setSpeed
M.setEnabled = setEnabled
M.computeThrottle = computeThrottle
M.setTargetAcceleration = setTargetAcceleration

return M
