'use strict';
angular.module('beamng.apps')
.directive('sgrRacedash', ['$log', '$timeout', '$interval', function ($log, $timeout, $interval) {
  return {
    template: `
      <div style="width: 100%; height: 100%; position: relative;">
        <object style="width:100%; height:100%;" type="image/svg+xml" data="/ui/modules/apps/SGRRaceDash/SGRRaceDash.svg"></object>
      </div>
    `,
    replace: true,
    restrict: 'EA',
    scope: true,
    link: function (scope, element) {
      let svg;
      let isShowingTime = false;
      let isShowingAirspeed = false;
      let isShowingTrip = false;
      let isShowingFuel = true;
      let isShowingTps = false;
      let isImperial = true;
      let isShowingWaterTemp = true;
      let isShowingOilTemp = false;
      let maxRPM = 9000;
      let cumulativeDistance = 0;
      let lastUpdateTime = Date.now();
      let lastData = null;
      let timeInterval;

      const streamsList = ['engineInfo', 'electrics', 'engineThermalData'];
      StreamsManager.add(streamsList);

      function saveState() {
        const state = {
          isShowingTime,
          isShowingAirspeed,
          isShowingTrip,
          isShowingFuel,
          isShowingTps,
          isImperial,
          isShowingWaterTemp,
          isShowingOilTemp,
          cumulativeDistance
        };
        localStorage.setItem('sgrRacedashState', JSON.stringify(state));
      }

      function loadState() {
        const savedState = localStorage.getItem('sgrRacedashState');
        if (savedState) {
          const state = JSON.parse(savedState);
          isShowingTime = state.isShowingTime;
          isShowingAirspeed = state.isShowingAirspeed;
          isShowingTrip = state.isShowingTrip;
          isShowingFuel = state.isShowingFuel;
          isShowingTps = state.isShowingTps;
          isImperial = state.isImperial;
          isShowingWaterTemp = state.isShowingWaterTemp;
          isShowingOilTemp = state.isShowingOilTemp;
          cumulativeDistance = state.cumulativeDistance;

          updateUnits();
          updateDisplayLabels();
          if (isShowingTrip) {
            updateTripMeter();
          } else if (isShowingTime) {
            updateTimeDisplay();
          }
          updateSpeedLabel();
        }
      }

      function updateDisplayLabels() {
        let tachLabel = svg.getElementById('tachLabel');
        let fuelLabel = svg.getElementById('fuelLabel');
        let tempLabel = svg.getElementById('tempLabel');

        if (tachLabel) {
          if (isShowingTime) {
            tachLabel.textContent = 'TIME';
          } else if (isShowingTrip) {
            tachLabel.textContent = 'TRIP';
          } else {
            tachLabel.textContent = 'TACH';
          }
        }

        if (fuelLabel) {
          fuelLabel.textContent = isShowingFuel ? 'FUELVL' : 'TPS';
        }

        if (tempLabel) {
          tempLabel.textContent = isShowingWaterTemp ? 'WTRTEMP' : 'OILTEMP';
        }
      }

      function updateSpeedLabel() {
        let mphLabel = svg.getElementById('mphLabel');
        let mphUnitLabel = svg.getElementById('mphUnitLabel');

        if (mphLabel) {
          if (isShowingAirspeed) {
            mphLabel.textContent = 'GPS';
          } else {
            mphLabel.textContent = 'SPEED';
          }
        }

        if (mphUnitLabel) {
          mphUnitLabel.textContent = isImperial ? 'MPH' : 'KPH';
          mphUnitLabel.setAttribute('visibility', 'visible');
        }
      }

      function injectFontStyle() {
        const styleElement = document.createElement('style');
        styleElement.textContent = `
          @font-face {
            font-family: 'DS-Digital';
            src: url('/ui/common/DS-DIGII.ttf') format('truetype');
          }
          text {
            font-family: 'DS-Digital', sans-serif;
          }
        `;

        const svgRoot = svg.querySelector('svg');
        if (svgRoot) {
          svgRoot.appendChild(styleElement);
        }
      }

      function updateFuelDigits(value) {
        let valueString = Math.floor(value).toString();
        let totalDigits = 3;
        let startIndex = totalDigits - valueString.length;

        for (let i = 0; i < totalDigits; i++) {
          let digitElement = svg.getElementById(`fuelDigit${i + 1}`);
          if (digitElement) {
            if (i >= startIndex) {
              let currentDigit = valueString[i - startIndex];
              digitElement.textContent = currentDigit;
              digitElement.setAttribute('dx', currentDigit === '1' ? '15' : '0');
              digitElement.setAttribute('visibility', 'visible');
            } else {
              digitElement.textContent = '';
              digitElement.setAttribute('visibility', 'hidden');
            }
          }
        }
      }

      function updateTpsDigits(value) {
        let tpsValue = Math.floor(value * 100);
        updateFuelDigits(tpsValue);
      }

      function updateTwoStepIndicator(isTwoStepActive) {
        let twoStepIndicator = svg.getElementById('twoStepIndicator2');
        if (twoStepIndicator) {
          twoStepIndicator.textContent = isTwoStepActive ? 'ON' : 'OFF';
        }
      }

      function updateRPMDigits(rpm) {
        if (isShowingTime || isShowingTrip) return;

        let rpmString = Math.round(rpm).toString();
        let totalDigits = 5;
        let startIndex = totalDigits - rpmString.length;

        for (let i = 0; i < totalDigits; i++) {
          let digitElement = svg.getElementById(`tachDigit${i + 1}`);
          if (digitElement) {
            if (i >= startIndex) {
              let currentDigit = rpmString[i - startIndex];
              digitElement.textContent = currentDigit;
              digitElement.setAttribute('dx', currentDigit === '1' ? '15' : '0');
              digitElement.setAttribute('visibility', 'visible');
            } else {
              digitElement.textContent = '';
              digitElement.setAttribute('visibility', 'hidden');
            }
          }
        }
      }

      function updatePressureDigits(value) {
        let pressureValue = isImperial ? Math.floor(value) : (value * 0.0689476).toFixed(2);
        let valueString = isImperial
          ? pressureValue.toString()
          : pressureValue.toString().replace('.', '');
        let totalDigits = isImperial ? 3 : 4;

        for (let i = 0; i < totalDigits; i++) {
          let digitElement = svg.getElementById(`pressDigit${i + 1}`);
          if (digitElement) {
            let indexFromRight = totalDigits - valueString.length;
            if (i >= indexFromRight) {
              let currentDigit = valueString[i - indexFromRight];
              digitElement.textContent = currentDigit;
              digitElement.setAttribute('dx', currentDigit === '1' ? '15' : '0');
              digitElement.setAttribute('visibility', 'visible');
            } else {
              digitElement.textContent = '';
              digitElement.setAttribute('visibility', 'hidden');
            }
          }
        }

        let decimalElement = svg.getElementById('pressDigit.');
        if (decimalElement) {
          decimalElement.setAttribute('visibility', isImperial ? 'hidden' : 'visible');
        }

        let pressUnitLabel = svg.getElementById('pressUnitLabel');
        if (pressUnitLabel) {
          pressUnitLabel.textContent = isImperial ? 'PSI' : 'BAR';
          pressUnitLabel.setAttribute('visibility', 'visible');
        }
      }

      function toggleTemperatureDisplay() {
        if (isShowingWaterTemp) {
          isShowingWaterTemp = false;
          isShowingOilTemp = true;
        } else {
          isShowingWaterTemp = true;
          isShowingOilTemp = false;
        }
        saveState();
        updateDisplayLabels();
      }

      function updateTemperatureDigits(value) {
        let valueString = Math.floor(value).toString();
        let totalDigits = 3;
        let startIndex = totalDigits - valueString.length;

        for (let i = 0; i < totalDigits; i++) {
          let digitElement = svg.getElementById(`tempDigit${i + 1}`);
          if (digitElement) {
            if (i >= startIndex) {
              let currentDigit = valueString[i - startIndex];
              digitElement.textContent = currentDigit;
              digitElement.setAttribute('dx', currentDigit === '1' ? '15' : '0');
              digitElement.setAttribute('visibility', 'visible');
            } else {
              digitElement.textContent = '';
              digitElement.setAttribute('visibility', 'hidden');
            }
          }
        }

        let tempUnitLabel = svg.getElementById('tempUnitLabel');
        if (tempUnitLabel) {
          tempUnitLabel.textContent = isImperial ? 'F' : 'C';
          tempUnitLabel.setAttribute('visibility', 'visible');
        }
      }

      function updateTimeDisplay() {
        const now = new Date();
        let hours = now.getHours();
        const minutes = now.getMinutes().toString().padStart(2, '0');
        const isPM = hours >= 12;
        hours = hours % 12 || 12;
        let hourString = hours.toString();
        const timeString = `${hourString.padStart(2, '0')}${minutes}`;
        const totalDigits = 5;

        for (let i = 0; i < totalDigits; i++) {
          let digitElement = svg.getElementById(`tachDigit${i + 1}`);
          if (digitElement) {
            if (i === 2) {
              digitElement.textContent = ':';
              digitElement.setAttribute('visibility', 'visible');
              digitElement.setAttribute('dx', '0');
            } else {
              let currentChar = timeString[i - (i > 2 ? 1 : 0)];
              if (i === 0 && currentChar === '0') {
                digitElement.textContent = '';
                digitElement.setAttribute('visibility', 'hidden');
              } else {
                digitElement.textContent = currentChar;
                digitElement.setAttribute('dx', currentChar === '1' ? '15' : '0');
                digitElement.setAttribute('visibility', 'visible');
              }
            }
          }
        }

        let timeUnitLabel = svg.getElementById('timeUnitLabel');
        if (timeUnitLabel) {
          timeUnitLabel.setAttribute('visibility', 'hidden');
        }
      }

      function updateMPHDigits(speed) {
        let speedValue = isImperial ? Math.round(speed * 2.23694) : Math.round(speed * 3.6);
        let speedString = speedValue.toString();
        let totalDigits = 3;
        let startIndex = totalDigits - speedString.length;

        for (let i = 0; i < totalDigits; i++) {
          let digitElement = svg.getElementById(`mphDigit${i + 1}`);
          if (digitElement) {
            if (i >= startIndex) {
              let currentDigit = speedString[i - startIndex];
              digitElement.textContent = currentDigit;
              digitElement.setAttribute('dx', currentDigit === '1' ? '15' : '0');
              digitElement.setAttribute('visibility', 'visible');
            } else {
              digitElement.textContent = '';
              digitElement.setAttribute('visibility', 'hidden');
            }
          }
        }

        let mphUnitLabel = svg.getElementById('mphUnitLabel');
        if (mphUnitLabel) {
          mphUnitLabel.textContent = isImperial ? 'MPH' : 'KPH';
          mphUnitLabel.setAttribute('visibility', 'visible');
        }
      }

      function updateTripMeterDistance(speed) {
        let currentTime = Date.now();
        if (lastUpdateTime !== null) {
          let deltaTime = (currentTime - lastUpdateTime) / 1000;
          let speedInMilesPerSecond = speed * 0.000621371;
          let distanceTraveled = speedInMilesPerSecond * deltaTime;

          cumulativeDistance += distanceTraveled;

          if (cumulativeDistance > 999.9) {
            cumulativeDistance = 999.9;
          }
        }

        if (isShowingTrip) {
          updateTripMeter();
        }

        lastUpdateTime = currentTime;
        saveState();
      }

      function updateUnits() {
        let mphUnitLabel = svg.getElementById('mphUnitLabel');
        if (mphUnitLabel) {
          mphUnitLabel.textContent = isImperial ? 'MPH' : 'KPH';
          mphUnitLabel.setAttribute('visibility', 'visible');
        }

        let tempUnitLabel = svg.getElementById('tempUnitLabel');
        if (tempUnitLabel) {
          tempUnitLabel.textContent = isImperial ? 'F' : 'C';
          tempUnitLabel.setAttribute('visibility', 'visible');
        }

        let pressUnitLabel = svg.getElementById('pressUnitLabel');
        if (pressUnitLabel) {
          pressUnitLabel.textContent = isImperial ? 'PSI' : 'BAR';
          pressUnitLabel.setAttribute('visibility', 'visible');
        }
      }

      function updateTripMeter() {
        let displayedDistance = isImperial ? cumulativeDistance : cumulativeDistance * 1.60934;

        if (displayedDistance > 999.9) {
          displayedDistance = 999.9;
        }

        let tripString = displayedDistance.toFixed(1).toString().padStart(5, '0');

        for (let i = 0; i < 5; i++) {
          let digitElement = svg.getElementById(`tachDigit${i + 1}`);
          if (digitElement) {
            let currentDigit = tripString[i];
            digitElement.textContent = currentDigit;
            digitElement.setAttribute('dx', currentDigit === '1' ? '15' : '0');
            digitElement.setAttribute('visibility', 'visible');
          }
        }

        let timeUnitLabel = svg.getElementById('timeUnitLabel');
        if (timeUnitLabel) {
          timeUnitLabel.textContent = isImperial ? 'MI' : 'KM';
          timeUnitLabel.setAttribute('visibility', 'visible');
        }
      }

      function updateGearIndicator(gear) {
        let gearIndicator = svg.getElementById('gearNumber');
        if (gearIndicator) {
          gearIndicator.textContent = gear;
        }
      }

      function getGearDisplay(gear, transmissionType) {
        if (transmissionType === 'realistic' || transmissionType === 'manual') {
          if (typeof gear === 'string' && gear.startsWith('m')) {
            return gear.toUpperCase();
          } else {
            switch (gear) {
              case 0:
                return 'N';
              case -1:
                return 'R';
              default:
                return gear.toString();
            }
          }
        } else if (transmissionType === 'arcade') {
          return 'D';
        } else if (transmissionType === 'automatic') {
          if (gear > 0) return 'D';
          else if (gear < 0) return 'R';
          else return 'N';
        }
        return 'N';
      }

      function toggleDisplay() {
        let tachLabel = svg.getElementById('tachLabel');
        let timeUnitLabel = svg.getElementById('timeUnitLabel');

        if (isShowingTime) {
          tachLabel.textContent = 'TACH';
          isShowingTime = false;
          isShowingTrip = false;
          if (timeInterval) $interval.cancel(timeInterval);
          if (timeUnitLabel) {
            timeUnitLabel.setAttribute('visibility', 'hidden');
          }
        } else if (isShowingTrip) {
          tachLabel.textContent = 'TIME';
          isShowingTime = true;
          isShowingTrip = false;
          timeInterval = $interval(updateTimeDisplay, 1000);
          updateTimeDisplay();
          if (timeUnitLabel) {
            timeUnitLabel.setAttribute('visibility', 'hidden');
          }
        } else {
          tachLabel.textContent = 'TRIP';
          isShowingTrip = true;
          isShowingTime = false;
          updateTripMeter();
          if (timeUnitLabel) {
            timeUnitLabel.textContent = isImperial ? 'MI' : 'KM';
            timeUnitLabel.setAttribute('visibility', 'visible');
          }
        }
        saveState();
      }

      function updateShiftLights(rpm, signal_L, signal_R) {
        if (!svg) return;
        if (!maxRPM || isNaN(maxRPM) || maxRPM <= 0) maxRPM = 9000;

        const greenThreshold = maxRPM * 0.80;
        const yellowThreshold = maxRPM * 0.90;
        const redThreshold = maxRPM * 0.95;

        if (!signal_L && !signal_R) {
          setLightVisibility('yellightleftouter', rpm >= yellowThreshold);
          setLightVisibility('yellightleftcenter', rpm >= yellowThreshold + 100);
          setLightVisibility('yellightleftinner', rpm >= yellowThreshold + 200);
          setLightVisibility('yellightrightouter', rpm >= yellowThreshold);
          setLightVisibility('yellightrightcenter', rpm >= yellowThreshold + 100);
          setLightVisibility('yellightrightinner', rpm >= yellowThreshold + 200);
        } else {
          setLightVisibility('yellightleftouter', signal_L);
          setLightVisibility('yellightleftcenter', signal_L);
          setLightVisibility('yellightleftinner', signal_L);
          setLightVisibility('yellightrightouter', signal_R);
          setLightVisibility('yellightrightcenter', signal_R);
          setLightVisibility('yellightrightinner', signal_R);
        }

        setLightVisibility('grenlightleftouter', rpm >= greenThreshold);
        setLightVisibility('grenlightleftinner', rpm >= greenThreshold + 100);
        setLightVisibility('grenlightrightouter', rpm >= greenThreshold);
        setLightVisibility('grenlightrightinner', rpm >= greenThreshold + 100);

        setLightVisibility('redlightleftouter', rpm >= redThreshold);
        setLightVisibility('redlightleftinner', rpm >= redThreshold + 100);
        setLightVisibility('redlightrightouter', rpm >= redThreshold);
        setLightVisibility('redlightrightinner', rpm >= redThreshold + 100);
      }

      function setLightVisibility(lightId, shouldShow) {
        const lightElement = svg.getElementById(lightId);
        if (lightElement) {
          lightElement.setAttribute('visibility', shouldShow ? 'visible' : 'hidden');
        }
      }

      function updateTachLines(rpm) {
        if (!svg) return;
        const clampedRPM = Math.max(0, Math.min(9000, rpm));
        if (clampedRPM < 1000) {
          for (let i = 1; i <= 81; i++) {
            const lineElement = svg.getElementById(`line-${i}`);
            if (lineElement) {
              lineElement.setAttribute('visibility', 'hidden');
            }
          }
          return;
        }

        const lineIndex = Math.ceil((clampedRPM - 1000) / 1000 * 10) + 1;

        for (let i = 1; i <= 81; i++) {
          const lineElement = svg.getElementById(`line-${i}`);
          if (lineElement) {
            lineElement.setAttribute('visibility', i <= lineIndex ? 'visible' : 'hidden');
          }
        }
      }

      function toggleSpeedDisplay() {
        isShowingAirspeed = !isShowingAirspeed;
        updateSpeedLabel();
        saveState();
      }

      function toggleFuelDisplay() {
        let fuelLabel = svg.getElementById('fuelLabel');

        if (isShowingFuel) {
          fuelLabel.textContent = 'TPS';
          isShowingFuel = false;
          if (lastData && lastData.electrics) {
            let tpsValue = lastData.electrics.throttle;
            updateTpsDigits(tpsValue);
          }
        } else {
          fuelLabel.textContent = 'FUELVL';
          isShowingFuel = true;
          if (lastData && lastData.engineInfo) {
            let fuelLevel = lastData.engineInfo[11] / lastData.engineInfo[12];
            updateFuelDigits(fuelLevel * 100);
          }
        }
        saveState();
      }

      function addRightClickListeners() {
        const tachLabel = svg.getElementById('tachLabel');
        const rpmDigits = [...Array(5).keys()].map(i => svg.getElementById(`tachDigit${i + 1}`));
        const mphLabel = svg.getElementById('mphLabel');
        const fuelLabel = svg.getElementById('fuelLabel');
        const pressureLabel = svg.getElementById('pressLabel');
        const tempLabel = svg.getElementById('tempLabel');
        const elementsToWatch = [tachLabel, ...rpmDigits, mphLabel, fuelLabel, pressureLabel, tempLabel];

        elementsToWatch.forEach(el => {
          if (el) {
            el.addEventListener('contextmenu', function (e) {
              e.preventDefault();
              if (el === mphLabel) toggleSpeedDisplay();
              else if (el === fuelLabel) toggleFuelDisplay();
              else if (el === pressureLabel) togglePressureDisplay();
              else if (el === tempLabel) toggleTemperatureDisplay();
              else toggleDisplay();
            });
          }
        });
      }

      element[0].querySelector('object').addEventListener('load', function () {
        $timeout(function () {
          svg = element[0].querySelector('object').contentDocument;
          if (!svg) return;

          injectFontStyle();
          addRightClickListeners();

          const unitButton = svg.getElementById('unitButtonGroup');
          if (unitButton) {
            unitButton.addEventListener('click', function () {
              isImperial = !isImperial;
              updateUnits();
              saveState();
            });
          }

          const resetTripButtonGroup = svg.getElementById('resetTripButtonGroup');
          if (resetTripButtonGroup) {
            resetTripButtonGroup.addEventListener('click', function () {
              if (isShowingTrip) {
                cumulativeDistance = 0;
                updateTripMeter();
                saveState();
              }
            });
          }

          loadState();

          scope.$on('streamsUpdate', function (event, data) {
            if (!svg) return;
            lastData = data;

            if (data.engineInfo && data.engineInfo[1] !== undefined) {
              let newMaxRPM = data.engineInfo[1];
              if (isNaN(newMaxRPM) || newMaxRPM <= 0) newMaxRPM = 9000;
              if (maxRPM !== newMaxRPM) maxRPM = newMaxRPM;
            } else {
              maxRPM = 9000;
            }

            if (data.engineInfo) {
              let currentRPM = data.engineInfo[4];
              if (isNaN(currentRPM) || currentRPM < 0) currentRPM = 0;

              updateRPMDigits(currentRPM);
              updateTachLines(currentRPM);
              updateShiftLights(currentRPM, data.electrics.signal_L, data.electrics.signal_R);
            }

            if (data.electrics && data.electrics.gear !== undefined) {
              const transmissionType = data.electrics.transmissionType || 'manual';
              const gear = getGearDisplay(data.electrics.gear, transmissionType);
              updateGearIndicator(gear);
            }

            if (data.electrics) {
              let speed = 0;
              if (isShowingAirspeed && data.electrics.airspeed !== undefined) {
                speed = data.electrics.airspeed;
              } else if (!isShowingAirspeed && data.electrics.wheelspeed !== undefined) {
                speed = data.electrics.wheelspeed;
              }
              updateMPHDigits(speed);

              if (data.electrics.wheelspeed !== undefined) updateTripMeterDistance(speed);

              if (isShowingFuel && data.engineInfo) {
                let fuelLevel = data.engineInfo[11] / data.engineInfo[12];
                if (!isNaN(fuelLevel)) updateFuelDigits(fuelLevel * 100);
              } else if (!isShowingFuel && data.electrics) {
                let tpsValue = data.electrics.throttle;
                if (!isNaN(tpsValue)) updateTpsDigits(tpsValue);
              }

              if (data.electrics.twoStep !== undefined) {
                const isTwoStepActive = data.electrics.twoStep;
                updateTwoStepIndicator(isTwoStepActive);
              }
            }

            let pressureValue = 0;
            if (data.electrics && data.electrics.boost !== undefined) pressureValue = data.electrics.boost;
            if (!isNaN(pressureValue)) updatePressureDigits(pressureValue);

            let tempValue = 0;
            if (isShowingWaterTemp) {
              if (data.engineThermalData && data.engineThermalData.coolantTemperature !== undefined) {
                tempValue = data.engineThermalData.coolantTemperature;
              } else if (data.electrics && data.electrics.watertemp !== undefined) {
                tempValue = data.electrics.watertemp;
              }
            } else if (isShowingOilTemp) {
              if (data.engineThermalData && data.engineThermalData.oilTemperature !== undefined) {
                tempValue = data.engineThermalData.oilTemperature;
              } else if (data.electrics && data.electrics.oiltemp !== undefined) {
                tempValue = data.electrics.oiltemp;
              }
            }

            if (!isNaN(tempValue)) {
              if (isImperial) tempValue = (tempValue * 9 / 5) + 32;
              updateTemperatureDigits(tempValue);
            }

            lastUpdateTime = Date.now();
          });
        }, 500);
      });

      scope.$on('$destroy', function () {
        if (timeInterval) $interval.cancel(timeInterval);
        if (svg && svg.getStreams) StreamsManager.remove(svg.getStreams());
        saveState();
      });
    }
  };
}]);
