'use strict';
angular.module('beamng.apps')
  .directive('sgrnanotriple3', ['$timeout', function ($timeout) {
    return {
      template: `
        <div style="width:100%;height:100%;position:relative;">
          <object id="nanoSvg"
                  style="width:100%;height:100%;"
                  type="image/svg+xml"
                  data="/ui/modules/apps/sgrnanotriple3/sgrnanotriple3.svg">
          </object>
        </div>
      `,
      replace: true,
      restrict: 'EA',
      scope: true,
      link: function (scope, element) {
        const obj = element[0].querySelector('#nanoSvg');

        const streamsList = ['electrics', 'engineInfo'];
        if (typeof StreamsManager !== 'undefined') {
          StreamsManager.add(streamsList);
          scope.$on('$destroy', function () {
            StreamsManager.remove(streamsList);
          });
        }

        let lastStreams = { electrics: {}, engineInfo: [] };

        const hpBuffer = [];
        const tqBuffer = [];
        const bufferWindow = 500;

        function convertTorqueLbftToNm(lbft) {
          return lbft / 0.737562;
        }

        function calcTorqueFromHP(hp, rpm, useImperial) {
          if (hp == null || rpm == null || rpm <= 0) return null;
          const tqLbft = (hp * 5252) / rpm;
          return useImperial ? tqLbft : convertTorqueLbftToNm(tqLbft);
        }

        function pushToBuffer(buffer, value) {
          if (value == null) return;
          buffer.push({ value, time: performance.now() });
          const cutoff = performance.now() - bufferWindow;
          while (buffer.length && buffer[0].time < cutoff) {
            buffer.shift();
          }
        }

        function getAverage(buffer) {
          if (!buffer.length) return null;
          const sum = buffer.reduce((acc, x) => acc + x.value, 0);
          return sum / buffer.length;
        }

        function updateUI(doc) {
          if (!doc) return;
          const s = lastStreams;

          const rpm   = (typeof s.electrics.rpm === 'number') ? s.electrics.rpm : null;
          const hpRaw = (Array.isArray(s.engineInfo) && typeof s.engineInfo[20] === 'number') ? s.engineInfo[20] : null;

          let useImperial = false;
          try {
            const conv = UiUnits.torque(1);
            if (conv && conv.unit && conv.unit.toLowerCase().includes('lb')) {
              useImperial = true;
            }
          } catch {}

          let hpVal = hpRaw;
          let tqVal = calcTorqueFromHP(hpRaw, rpm, useImperial);

          if (s.electrics.ignitionLevel !== 2) {
            hpVal = 0;
            tqVal = 0;
          }

          if (hpVal != null) pushToBuffer(hpBuffer, hpVal);
          if (tqVal != null) pushToBuffer(tqBuffer, tqVal);

          const hpAvg = getAverage(hpBuffer);
          const tqAvg = getAverage(tqBuffer);

          const label1 = doc.getElementById('datalabel1');
          const data1  = doc.getElementById('data1');
          const alt1   = doc.getElementById('altdata1');
          if (label1) label1.textContent = 'Wheel HP';
          if (data1)  data1.textContent  = hpAvg == null ? '--' : Math.round(hpAvg);
          if (alt1)   alt1.textContent   = hpAvg == null ? '' : 'hp';

          const label2 = doc.getElementById('datalabel2');
          const data2  = doc.getElementById('data2');
          const alt2   = doc.getElementById('altdata2');
          if (label2) label2.textContent = 'Wheel TQ';
          if (data2)  data2.textContent  = tqAvg == null ? '--' : Math.round(tqAvg);
          if (alt2)   alt2.textContent   = tqAvg == null ? '' : (useImperial ? 'lb-ft' : 'Nm');

          const label3 = doc.getElementById('datalabel3');
          const data3  = doc.getElementById('data3');
          if (label3) label3.textContent = 'RPM';
          if (data3)  data3.textContent  = rpm == null ? '--' : Math.round(rpm);
        }

        function initSvg() {
          const doc = obj.contentDocument;
          if (!doc) return;
          updateUI(doc);
        }

        obj.addEventListener('load', () => $timeout(initSvg, 0));
        if (obj.contentDocument) $timeout(initSvg, 0);

        scope.$on('streamsUpdate', function (_evt, streams) {
          if (streams) lastStreams = streams;
          const doc = obj.contentDocument;
          if (doc) updateUI(doc);
        });
      }
    };
  }]);
