'use strict';
angular.module('beamng.apps')
.directive('sgrnanocustom4', ['$timeout', function ($timeout) {
  return {
    template: `
      <div style="width:100%;height:100%;position:relative;">
        <object id="nanoSvg"
                style="width:100%;height:100%;"
                type="image/svg+xml"
                data="/ui/modules/apps/sgrnanocustom4/sgrnanocustom4.svg">
        </object>
      </div>
    `,
    replace: true,
    restrict: 'EA',
    scope: true,
    link: function (scope, element) {
      const obj = element[0].querySelector('#nanoSvg');
      const streamsList = ['electrics', 'sensors', 'engineThermalData'];
      if (typeof StreamsManager !== 'undefined') {
        StreamsManager.add(streamsList);
        scope.$on('$destroy', function () { StreamsManager.remove(streamsList); });
      }

      let lastStreams = { electrics: {}, sensors: {}, engineThermalData: {} };

      const toSpeedString = (mps) => {
        if (mps == null) return '--';
        try { const conv = UiUnits.speed(mps); return `${Math.round(conv.val)} ${conv.unit}`; }
        catch { return `${Math.round(mps*3.6)} km/h`; }
      };
      const percentString = (x) => x==null?'--':`${Math.round(x*100)}%`;
      const feetString    = (m) => m==null?'--':`${Math.round(m*3.28084)} ft`;
      const tempFString   = (x) => x==null?'--':`${Math.round((x*9/5)+32)} °F`;
      const quartsString  = (kg) => kg==null?'--':`${(kg*1.057).toFixed(2)} qt`;
      const pitchString   = (p) => p==null?'--':`${(p*90).toFixed(1)}°`;
      const gearString    = (g) => g===-1?'R':g===0?'N':(g||'--');

      const sources = [
        { name:'Wheel Speed', valueFrom:s=>toSpeedString(s.electrics?.wheelspeed) },
        { name:'Air Speed',   valueFrom:s=>toSpeedString(s.electrics?.airspeed) },
        { name:'Engine Load', valueFrom:s=>percentString(s.electrics?.engineLoad) },
        { name:'Gear',        valueFrom:s=>gearString(s.electrics?.gear) },
        { name:'Throttle',    valueFrom:s=>percentString(s.electrics?.throttle) },
        { name:'Brake',       valueFrom:s=>percentString(s.electrics?.brake) },
        { name:'Altitude',    valueFrom:s=>feetString(s.electrics?.altitude) },
        { name:'Transbrake',  valueFrom:s=>s.electrics?.transbrake?'ON':'OFF' },
        { name:'Two Step',    valueFrom:s=>s.electrics?.twoStep?'ON':'OFF' },
        { name:'Coolant Temp',valueFrom:s=>tempFString(s.engineThermalData?.coolantTemperature) },
        { name:'Oil Temp',    valueFrom:s=>tempFString(s.engineThermalData?.oilTemperature) },
        { name:'Oil Mass',    valueFrom:s=>quartsString(s.engineThermalData?.oilMass) },
        { name:'Pitch',       valueFrom:s=>pitchString(s.sensors?.pitch) },
        { name:'RPM',         valueFrom:s=>s.electrics?.rpm?Math.round(s.electrics.rpm):'--' },
        { name:'Boost',       valueFrom:s=>{
            const b = s.electrics?.boost;
            if (typeof b !== 'number') return '--';
            return `${Math.max(0, Math.round(b))} PSI`;
          } }
      ];

      const STORAGE_KEY = 'sgrnanocustom4_state';
      let idx = loadState(12);

      function saveState() {
        localStorage.setItem(STORAGE_KEY, JSON.stringify({ idx }));
      }
      function loadState(defaultIdx = 0) {
  try {
    const saved = JSON.parse(localStorage.getItem(STORAGE_KEY));
    if (saved && typeof saved.idx === 'number') return saved.idx;
  } catch {}
  return defaultIdx;
}

      const ACTIVE_COLOR = '#39ff14';
      const INACTIVE_STROKE = 'white';
      const wrap = (i, n) => (i % n + n) % n;

      function getTextEl(doc, groupId) {
        const g = doc.getElementById(groupId);
        return g ? g.querySelector('text') : null;
      }
      function applyFilterIfPresent(doc, el, filterId) {
        if (!el) return;
        if (doc.getElementById(filterId)) el.setAttribute('filter', `url(#${filterId})`);
      }
      function clearFilter(el) { if (el) el.removeAttribute('filter'); }
      function collectBoxes(doc) {
        return [
          doc.getElementById('box1'),
          doc.getElementById('box2'),
          doc.getElementById('box3'),
          doc.getElementById('box4'),
          doc.getElementById('box5')
        ];
      }

      function makeUnderlinePathForRect(rectEl) {
        if (!rectEl) return '';
        const x = parseFloat(rectEl.getAttribute('x'));
        const y = parseFloat(rectEl.getAttribute('y'));
        const w = parseFloat(rectEl.getAttribute('width'));
        const h = parseFloat(rectEl.getAttribute('height'));
        const cx = x + w/2;
        const baselineY = y + h + 2;
        const rx = w/2;
        const arcHeight = 10;
        const leftX = cx - rx;
        const rightX = cx + rx;
        const peakY = baselineY + arcHeight;
        return `M ${leftX} ${baselineY} Q ${cx} ${peakY} ${rightX} ${baselineY}`;
      }

      function updateUI(doc) {
        if (!doc) return;
        const secText = getTextEl(doc, 'secondary-display');
        const mainText = getTextEl(doc, 'main-display');
        const current = sources[wrap(idx, sources.length)];
        const boxes = collectBoxes(doc);
        const underline = doc.getElementById('box-underline');

        if (secText) {
          secText.textContent = current.name;
          applyFilterIfPresent(doc, secText, 'neonWhite');
        }
        if (mainText) {
          mainText.textContent = current.valueFrom(lastStreams);
          applyFilterIfPresent(doc, mainText, 'neonWhite');
        }

        let activeRect = null;
        boxes.forEach((b, i) => {
          if (!b) return;
          if (i === wrap(idx, boxes.length)) {
            b.style.fill   = ACTIVE_COLOR;
            b.style.stroke = ACTIVE_COLOR;
            applyFilterIfPresent(doc, b, 'neonGreen');
            activeRect = b;
          } else {
            b.style.fill   = 'none';
            b.style.stroke = INACTIVE_STROKE;
            clearFilter(b);
          }
        });

        if (underline && activeRect) {
          underline.setAttribute('d', makeUnderlinePathForRect(activeRect));
          underline.style.stroke = ACTIVE_COLOR;
          underline.style.fill   = 'none';
          underline.setAttribute('visibility', 'visible');
          applyFilterIfPresent(doc, underline, 'neonGreen');
        } else if (underline) {
          underline.setAttribute('visibility', 'hidden');
          clearFilter(underline);
        }
      }

      function wireButtons(doc) {
        const leftBtn = doc.getElementById('circle-left');
        const rightBtn = doc.getElementById('circle-right');
        if (leftBtn) leftBtn.style.cursor = 'pointer';
        if (rightBtn) rightBtn.style.cursor = 'pointer';
        const onLeft = (e) => { if (e) e.preventDefault(); idx = wrap(idx - 1, sources.length); saveState(); updateUI(doc); };
        const onRight = (e) => { if (e) e.preventDefault(); idx = wrap(idx + 1, sources.length); saveState(); updateUI(doc); };
        if (leftBtn) leftBtn.addEventListener('click', onLeft);
        if (rightBtn) rightBtn.addEventListener('click', onRight);
        scope.$on('$destroy', function () {
          if (leftBtn) leftBtn.removeEventListener('click', onLeft);
          if (rightBtn) rightBtn.removeEventListener('click', onRight);
        });
      }

      function initSvg() {
        const doc = obj.contentDocument;
        if (!doc) return;
        wireButtons(doc);
        updateUI(doc);
      }

      obj.addEventListener('load', function () { $timeout(initSvg, 0); });
      if (obj.contentDocument) { $timeout(initSvg, 0); }

      scope.$on('streamsUpdate', function (_evt, streams) {
        if (streams) {
          lastStreams = streams;
        }
        const doc = obj.contentDocument;
        if (doc) updateUI(doc);
      });
    }
  };
}]);
