'use strict';
angular.module('beamng.apps')
.directive('sgrDragTreeTimer', ['$timeout', function ($timeout) {
  return {
    template: `
      <div style="width: 100%; height: 100%; position: relative;">
        <object id="tachSvg" style="width:100%; height:100%;"
                type="image/svg+xml"
                data="/ui/modules/apps/sgrdragtreetimer/sgrDragTreeTimer.svg">
        </object>
      </div>
    `,
    replace: true,
    restrict: 'EA',
    scope: true,
    link: function (scope, element) {
      let svg;
      let isTimerRunning = false;
      let timerStartTime = null;
      let yellowLightsTimeout = null;
      let greenLightTimeout = null;
      let carExceeds1mphTime = null;
      let greenLightActivationTime = null;
      let yellowLightsLit = false;
      let resetConditionMet = false;
      let scheduledGreenTime = null;
      let isDistanceTimerRunning = false;
      let distanceTimerStart = null;
      let totalDistanceFeet = 0;
      let lastUpdateTime = performance.now();
      let hasPassed60ft = false;
      let hasPassed330ft = false;
      let hasPassedEighth = false;
      let hasPassed1000ft = false;
      let hasPassedQuarter = false;
      let stageTimerStart = null;
      let bestEighthMph = 0;
      let bestQuarterMph = 0;
      let passLockedIn = false;
      const requiredStageTime = 1.5 * 1000;
      let best60ftTime = Infinity;
      let best330ftTime = Infinity;
      let bestEighthTime = Infinity;
      let best1000ftTime = Infinity;
      let bestQuarterTime = Infinity;
      let time60ftElement, bestTime60ftElement,
          time330ftElement, bestTime330ftElement,
          timeEighthElement, bestTimeEighthElement,
          time1000ftElement, bestTime1000ftElement,
          timeQuarterElement, bestTimeQuarterElement;
      let lastSpeedInMph = 0;
      const treeTypes = ['.400 Pro Tree', '.500 Pro Tree', 'Sportsman Tree', 'Instant Green'];
      let currentTreeIndex = 0;
      const SPORTSMAN_INTERVAL = 500;
      const objectElement = element[0].querySelector('object');
      const streamsList = ['electrics'];
      StreamsManager.add(streamsList);
      let carWasStopped = false;
      let stopCheckTimer = 0;
      const stopTimeRequired = 1.0;
      const speedStopThreshold = 0.4;
      const speedStartThreshold = 1.0;

      function checkPrestageConditions(streams) {
        const brake = streams.electrics.brake > 0.8;
        const clutch = streams.electrics.clutch > 0.8;
        const transbrake = streams.electrics.transbrake;
        return brake || clutch || transbrake;
      }

      function checkStageConditions(streams) {
        const throttle = streams.electrics.throttle > 0.8;
        const brake = streams.electrics.brake > 0.8;
        const clutch = streams.electrics.clutch > 0.8;
        const transbrake = streams.electrics.transbrake;
        return throttle && (brake || clutch || transbrake);
      }

      function setGlow(element, doGlow) {
        if (!element) return;
        if (doGlow) {
          element.setAttribute('filter', 'url(#glow)');
        } else {
          element.removeAttribute('filter');
        }
      }

      function updatePrestageLights(conditionMet) {
        if (svg) {
          const prestageLights = svg.getElementById('prestage-lights');
          if (prestageLights) {
            const color = (conditionMet || yellowLightsLit) ? '#FFFF00' : '#8B8000';
            prestageLights.querySelectorAll('circle').forEach((light) => {
              light.setAttribute('fill', color);
              setGlow(light, color === '#FFFF00');
            });
          }
        }
      }

      function updateStageLights(conditionMet) {
        if (svg) {
          const stageLights = svg.getElementById('stage-lights');
          if (stageLights) {
            const color = (conditionMet || yellowLightsLit) ? '#FFFF00' : '#8B8000';
            stageLights.querySelectorAll('circle').forEach((light) => {
              light.setAttribute('fill', color);
              setGlow(light, color === '#FFFF00');
            });
          }
        }
      }

      function activateYellowLightsAll() {
        if (!svg) return;
        const yellowLights = [
          svg.getElementById('yellowlights1'),
          svg.getElementById('yellowlights2'),
          svg.getElementById('yellowlights3')
        ];
        yellowLights.forEach((group) => {
          if (group) {
            group.querySelectorAll('circle').forEach((light) => {
              light.setAttribute('fill', '#FFFF00');
              setGlow(light, true);
            });
          }
        });
        yellowLightsLit = true;
      }

      function activateGreenLights() {
        if (svg) {
          const greenLights = svg.getElementById('greenlights');
          if (greenLights) {
            greenLights.querySelectorAll('circle').forEach((light) => {
              light.setAttribute('fill', '#00FF00');
              setGlow(light, true);
            });
          }
        }
        greenLightActivationTime = performance.now();
      }

      function activateRedLight() {
        if (svg) {
          const redLights = svg.getElementById('redlights');
          if (redLights) {
            redLights.querySelectorAll('circle').forEach((light) => {
              light.setAttribute('fill', '#FF0000');
              setGlow(light, true);
            });
          }
        }
      }

      function activateYellowGroup(groupId) {
        if (!svg) return;
        const group = svg.getElementById(groupId);
        if (!group) return;
        group.querySelectorAll('circle').forEach((light) => {
          light.setAttribute('fill', '#FFFF00');
          setGlow(light, true);
        });
        yellowLightsLit = true;
      }

      function updateReactionTimeDisplay(rtValue) {
        if (!svg) return;
        const rtTextElement = svg.getElementById('rt-time');
        if (!rtTextElement) return;

        const sign = rtValue < 0 ? '-' : '';
        const absValue = Math.abs(rtValue);

        let formattedValue = absValue.toFixed(3);

        if (absValue < 1) {
          formattedValue = formattedValue.substring(1);
        }

        rtTextElement.textContent = sign + formattedValue;
      }

      function scheduleGreenAfter(ms, isInstantGreen = false) {
        scheduledGreenTime = performance.now() + ms;
        greenLightActivationTime = scheduledGreenTime;

        greenLightTimeout = setTimeout(() => {
          activateGreenLights();
        }, ms);
      }

      function startTreeSequence() {
        const randomDelay = (Math.random() * (2.5 - 1.5) + 1.5) * 1000;
        const currentTree = treeTypes[currentTreeIndex];
        scheduledGreenTime = performance.now() + randomDelay + 400;

        if (currentTree === '.400 Pro Tree') {
          yellowLightsTimeout = setTimeout(() => {
            activateYellowLightsAll();
            scheduleGreenAfter(400);
          }, randomDelay);
        } else if (currentTree === '.500 Pro Tree') {
          yellowLightsTimeout = setTimeout(() => {
            activateYellowLightsAll();
            scheduleGreenAfter(500);
          }, randomDelay);
        } else if (currentTree === 'Sportsman Tree') {
          yellowLightsTimeout = setTimeout(() => {
            activateYellowGroup('yellowlights1');
            setTimeout(() => {
              activateYellowGroup('yellowlights2');
              setTimeout(() => {
                activateYellowGroup('yellowlights3');
                scheduleGreenAfter(SPORTSMAN_INTERVAL);
              }, SPORTSMAN_INTERVAL);
            }, SPORTSMAN_INTERVAL);
          }, randomDelay);
        } else if (currentTree === 'Instant Green') {
          scheduleGreenAfter(randomDelay);
        }
      }

      function startTimer() {
        if (!isTimerRunning) {
          isTimerRunning = true;
          timerStartTime = performance.now();
          carExceeds1mphTime = null;
          greenLightActivationTime = null;
          resetConditionMet = false;
          updateReactionTimeDisplay(0.0);
          startTreeSequence();
        }
      }

      function stopTimer() {
        if (isTimerRunning) {
          clearTimeout(yellowLightsTimeout);
          clearTimeout(greenLightTimeout);
          isTimerRunning = false;
        }
      }

      function resetTimerAndLights() {
        stopTimer();
        stopDistanceTimer();
        resetLights();
        yellowLightsLit = false;
        updateReactionTimeDisplay(0.0);
        if (time60ftElement) time60ftElement.textContent = '0.000';
        if (time330ftElement) time330ftElement.textContent = '0.000';
        if (timeEighthElement) timeEighthElement.textContent = '0.000@0';
        if (time1000ftElement) time1000ftElement.textContent = '0.000';
        if (timeQuarterElement) timeQuarterElement.textContent = '0.000@0';
      }

      function resetLights() {
        if (!svg) return;
        const yellowLights = [
          svg.getElementById('yellowlights1'),
          svg.getElementById('yellowlights2'),
          svg.getElementById('yellowlights3')
        ];
        yellowLights.forEach((group) => {
          if (group) {
            group.querySelectorAll('circle').forEach((light) => {
              light.setAttribute('fill', '#8B8000');
              setGlow(light, false);
            });
          }
        });
        const greenLights = svg.getElementById('greenlights');
        if (greenLights) {
          greenLights.querySelectorAll('circle').forEach((light) => {
            light.setAttribute('fill', '#006400');
            setGlow(light, false);
          });
        }
        const redLights = svg.getElementById('redlights');
        if (redLights) {
          redLights.querySelectorAll('circle').forEach((light) => {
            light.setAttribute('fill', '#8B0000');
            setGlow(light, false);
          });
        }
        updateStageLights(false);
      }

      function checkResetConditions(speedInMph) {
        if (speedInMph > 1) {
          resetConditionMet = true;
        } else if (resetConditionMet && speedInMph < 0.8) {
          resetTimerAndLights();
        }
      }

      function startDistanceTimer() {
        if (!isDistanceTimerRunning) {
          isDistanceTimerRunning = true;
          distanceTimerStart = performance.now();
          totalDistanceFeet = 0;
          lastUpdateTime = performance.now();
          hasPassed60ft = false;
          hasPassed330ft = false;
          hasPassedEighth = false;
          hasPassed1000ft = false;
          hasPassedQuarter = false;
        }
      }

      function stopDistanceTimer() {
        if (isDistanceTimerRunning) {
          isDistanceTimerRunning = false;
        }
      }

      function checkDistanceMilestones(speedInMph) {
        const elapsedTime = (performance.now() - distanceTimerStart) / 1000 - 0.08;
        if (!hasPassed60ft && totalDistanceFeet >= 59) {
          hasPassed60ft = true;
          updateMilestoneTime('60ft-time', 'best-60ft-time', elapsedTime, speedInMph, false);
        }
        if (!hasPassed330ft && totalDistanceFeet >= 329) {
          hasPassed330ft = true;
          updateMilestoneTime('330ft-time', 'best-330ft-time', elapsedTime, speedInMph, false);
        }
        if (!hasPassedEighth && totalDistanceFeet >= 659) {
          hasPassedEighth = true;
          updateMilestoneTime('eighth-time', 'best-eighth-time', elapsedTime, speedInMph, true);
        }
        if (!hasPassed1000ft && totalDistanceFeet >= 999) {
          hasPassed1000ft = true;
          updateMilestoneTime('1000ft-time', 'best-1000ft-time', elapsedTime, speedInMph, false);
        }
        if (!hasPassedQuarter && totalDistanceFeet >= 1319) {
          hasPassedQuarter = true;
          updateMilestoneTime('quarter-mile-time', 'best-quarter-mile-time', elapsedTime, speedInMph, true);
        }
      }

      function updateMilestoneTime(currentId, bestId, timeValue, speedInMph, showMph) {
        if (!svg) return;
        const currentText = svg.getElementById(currentId);
        const bestText = svg.getElementById(bestId);
        let displayText = timeValue.toFixed(3);
        if (showMph) {
          displayText += '@' + speedInMph.toFixed(2);
        }
        if (currentText) {
          currentText.textContent = displayText;
        }
        let bestVar;
        switch (currentId) {
          case '60ft-time':
            bestVar = best60ftTime;
            break;
          case '330ft-time':
            bestVar = best330ftTime;
            break;
          case 'eighth-time':
            bestVar = bestEighthTime;
            if (timeValue < bestEighthTime) {
              bestEighthMph = speedInMph;
            }
            break;
          case '1000ft-time':
            bestVar = best1000ftTime;
            break;
          case 'quarter-mile-time':
            bestVar = bestQuarterTime;
            if (timeValue < bestQuarterTime) {
              bestQuarterMph = speedInMph;
            }
            break;
          default:
            bestVar = Infinity;
            break;
        }
        if (timeValue < bestVar) {
          switch (currentId) {
            case '60ft-time':
              best60ftTime = timeValue;
              break;
            case '330ft-time':
              best330ftTime = timeValue;
              break;
            case 'eighth-time':
              bestEighthTime = timeValue;
              break;
            case '1000ft-time':
              best1000ftTime = timeValue;
              break;
            case 'quarter-mile-time':
              bestQuarterTime = timeValue;
              break;
          }
          if (bestText) {
            bestText.textContent = displayText;
          }
          saveBestTimes();
        }
      }

      function updateTreeTypeText() {
        const treeTextElem = svg ? svg.getElementById('treetype') : null;
        if (treeTextElem) {
          treeTextElem.textContent = treeTypes[currentTreeIndex];
        }
      }

      function saveTreeType() {
        try {
          localStorage.setItem('treeTypeIndex', currentTreeIndex.toString());
        } catch (e) {}
      }

      function loadTreeType() {
        try {
          const storedIndex = localStorage.getItem('treeTypeIndex');
          if (storedIndex !== null) {
            const idx = parseInt(storedIndex, 10);
            if (idx >= 0 && idx < treeTypes.length) {
              currentTreeIndex = idx;
            }
          }
        } catch (e) {}
      }

      function cycleTreeLeft() {
        currentTreeIndex--;
        if (currentTreeIndex < 0) {
          currentTreeIndex = treeTypes.length - 1;
        }
        saveTreeType();
        updateTreeTypeText();
        resetTimerAndLights();
      }

      function cycleTreeRight() {
        currentTreeIndex++;
        if (currentTreeIndex >= treeTypes.length) {
          currentTreeIndex = 0;
        }
        saveTreeType();
        updateTreeTypeText();
        resetTimerAndLights();
      }

      function updateStoppedState(dt, currentSpeedMph) {
        if (currentSpeedMph < speedStopThreshold) {
          stopCheckTimer += dt;
          if (stopCheckTimer >= stopTimeRequired) {
            carWasStopped = true;
          }
        } else {
          stopCheckTimer = 0;
        }
      }

      function canStartTimers() {
        return carWasStopped && !isTimerRunning;
      }

      function saveBestTimes() {
        try {
          localStorage.setItem('best60ftTime', best60ftTime.toString());
          localStorage.setItem('best330ftTime', best330ftTime.toString());
          localStorage.setItem('bestEighthTime', bestEighthTime.toString());
          localStorage.setItem('bestEighthMph', bestEighthMph.toString());
          localStorage.setItem('best1000ftTime', best1000ftTime.toString());
          localStorage.setItem('bestQuarterTime', bestQuarterTime.toString());
          localStorage.setItem('bestQuarterMph', bestQuarterMph.toString());
        } catch (e) {}
      }

      function loadBestTimes() {
        try {
          const stored60 = localStorage.getItem('best60ftTime');
          const stored330 = localStorage.getItem('best330ftTime');
          const storedEighth = localStorage.getItem('bestEighthTime');
          const storedEighthMph = localStorage.getItem('bestEighthMph');
          const stored1000 = localStorage.getItem('best1000ftTime');
          const storedQuarter = localStorage.getItem('bestQuarterTime');
          const storedQuarterMph = localStorage.getItem('bestQuarterMph');

          if (stored60 !== null) best60ftTime = parseFloat(stored60);
          if (stored330 !== null) best330ftTime = parseFloat(stored330);
          if (storedEighth !== null) bestEighthTime = parseFloat(storedEighth);
          if (storedEighthMph !== null) bestEighthMph = parseFloat(storedEighthMph);
          if (stored1000 !== null) best1000ftTime = parseFloat(stored1000);
          if (storedQuarter !== null) bestQuarterTime = parseFloat(storedQuarter);
          if (storedQuarterMph !== null) bestQuarterMph = parseFloat(storedQuarterMph);
        } catch (e) {}
      }

      function resetBestTimes() {
        best60ftTime = Infinity;
        best330ftTime = Infinity;
        bestEighthTime = Infinity;
        bestEighthMph = 0;
        best1000ftTime = Infinity;
        bestQuarterTime = Infinity;
        bestQuarterMph = 0;

        if (bestTime60ftElement) bestTime60ftElement.textContent = '0.000';
        if (bestTime330ftElement) bestTime330ftElement.textContent = '0.000';
        if (bestTimeEighthElement) bestTimeEighthElement.textContent = '0.000@0';
        if (bestTime1000ftElement) bestTime1000ftElement.textContent = '0.000';
        if (bestTimeQuarterElement) bestTimeQuarterElement.textContent = '0.000@0';

        saveBestTimes();
      }

      objectElement.addEventListener('load', function () {
        svg = objectElement.contentDocument;
        if (!svg) return;
        resetLights();
        updateReactionTimeDisplay(0.0);
        time60ftElement = svg.getElementById('60ft-time');
        bestTime60ftElement = svg.getElementById('best-60ft-time');
        time330ftElement = svg.getElementById('330ft-time');
        bestTime330ftElement = svg.getElementById('best-330ft-time');
        timeEighthElement = svg.getElementById('eighth-time');
        bestTimeEighthElement = svg.getElementById('best-eighth-time');
        time1000ftElement = svg.getElementById('1000ft-time');
        bestTime1000ftElement = svg.getElementById('best-1000ft-time');
        timeQuarterElement = svg.getElementById('quarter-mile-time');
        bestTimeQuarterElement = svg.getElementById('best-quarter-mile-time');
        if (time60ftElement) time60ftElement.textContent = '0.000';
        if (bestTime60ftElement) bestTime60ftElement.textContent = '0.000';
        loadTreeType();
        updateTreeTypeText();
        loadBestTimes();
        if (bestTime60ftElement && best60ftTime !== Infinity) {
          bestTime60ftElement.textContent = best60ftTime.toFixed(3);
        }
        if (bestTime330ftElement && best330ftTime !== Infinity) {
          bestTime330ftElement.textContent = best330ftTime.toFixed(3);
        }
        if (bestTimeEighthElement && bestEighthTime !== Infinity) {
          bestTimeEighthElement.textContent = bestEighthTime.toFixed(3) + '@0';
        }
        if (bestTime1000ftElement && best1000ftTime !== Infinity) {
          bestTime1000ftElement.textContent = best1000ftTime.toFixed(3);
        }
        if (bestTimeQuarterElement && bestQuarterTime !== Infinity) {
          bestTimeQuarterElement.textContent = bestQuarterTime.toFixed(3) + '@0';
        }
        if (bestTimeEighthElement && bestEighthTime !== Infinity) {
          bestTimeEighthElement.textContent = bestEighthTime.toFixed(3) + '@' + bestEighthMph.toFixed(2);
        }
        if (bestTimeQuarterElement && bestQuarterTime !== Infinity) {
          bestTimeQuarterElement.textContent = bestQuarterTime.toFixed(3) + '@' + bestQuarterMph.toFixed(2);
        }
        const arrowLeft = svg.getElementById('arrow-left');
        const arrowRight = svg.getElementById('arrow-right');
        if (arrowLeft) {
          arrowLeft.addEventListener('click', cycleTreeLeft);
        }
        if (arrowRight) {
          arrowRight.addEventListener('click', cycleTreeRight);
        }
        const resetBest = svg.getElementById('reset');
        if (resetBest) {
          resetBest.addEventListener('click', resetBestTimes);
        }
      });

      scope.$on('streamsUpdate', function (event, streams) {
        if (streams.electrics) {
          const speedInMph = streams.electrics.airspeed * 2.237;
          lastSpeedInMph = speedInMph;
          const speedInFeetPerSecond = speedInMph * 1.4667;
          const currentTime = performance.now();
          const dt = (currentTime - lastUpdateTime) / 1000;
          lastUpdateTime = currentTime;

          updateStoppedState(dt, speedInMph);

          const prestageCondition = checkPrestageConditions(streams);
          const stageCondition = checkStageConditions(streams);

          updatePrestageLights(prestageCondition);
          updateStageLights(stageCondition);

          if (stageCondition && prestageCondition) {
            if (stageTimerStart === null) {
              stageTimerStart = currentTime;
            }
          } else {
            stageTimerStart = null;
          }

          if (
            !yellowLightsLit &&
            prestageCondition &&
            stageCondition &&
            canStartTimers() &&
            stageTimerStart !== null &&
            currentTime - stageTimerStart >= requiredStageTime
          ) {
            startTimer();
          }

          if (speedInMph > speedStartThreshold && !carExceeds1mphTime) {
            carExceeds1mphTime = performance.now();

            if (!greenLightActivationTime || carExceeds1mphTime < scheduledGreenTime) {
              activateRedLight();
              clearTimeout(greenLightTimeout);

              let reactionTime = (carExceeds1mphTime - scheduledGreenTime) / 1000;

              if (reactionTime < -2.5) {
                reactionTime = -2.5;
              }

              updateReactionTimeDisplay(reactionTime);
            } else {
              const reactionTime = (carExceeds1mphTime - greenLightActivationTime) / 1000;
              updateReactionTimeDisplay(reactionTime);
            }
          }

          if (isDistanceTimerRunning) {
            totalDistanceFeet += speedInFeetPerSecond * dt;
            checkDistanceMilestones(lastSpeedInMph);
          }

          if (carWasStopped && !isDistanceTimerRunning && speedInMph > speedStartThreshold) {
            startDistanceTimer();
          }

          checkResetConditions(speedInMph);

          if (!isTimerRunning && speedInMph < 0.4) {
            updateStageLights(false);
          }
        }
      });

      scope.$on('$destroy', function () {
        StreamsManager.remove(streamsList);
        resetTimerAndLights();
      });
    }
  };
}]);
