'use strict';
angular.module('beamng.apps')
.directive('sgrDraggyRoll', ['$timeout', function ($timeout) {
  return {
    template: `
      <div style="width: 100%; height: 100%; position: relative;">
        <object id="tachSvg" style="width:100%; height:100%;" type="image/svg+xml" data="/ui/modules/apps/sgrdraggyroll/sgrdraggyroll.svg"></object>
      </div>
    `,
    replace: true,
    restrict: 'EA',
    scope: true,
    link: function (scope, element) {
      const objectElement = element[0].querySelector('object');
      const streamsList = ['electrics'];
      let lightsStayOnTimeout = null;
      let timerInterval = null;
      let timerStartTime = null;
      let isTimerRunning = false;
      let isRunCompleted = false;
      let totalDistance = 0;
      let lastUpdateTime = performance.now();
      let countdownStarted = false;
      let countdownInProgress = false;
      let countdownTimeouts = [];

      const timerType = 'everything';
      localStorage.setItem('timerType', timerType);

      let timeAt60 = null;
      let timeAt100 = null;

      let hasPassed1_4mile = false;
      let timeTo1_4mile = null;
      let speedAt1_4mile = null;
      let time60To130 = null;
      let time100To150 = null;
      let time100To200 = null;

      const groups = ['group-1', 'group-2', 'group-3', 'group-4', 'group-5', 'group-6', 'group-7', 'group-8'];
      let nextAvailableGroupIndex = 0;
      let metricEnabled = localStorage.getItem('3metricEnabled') !== null ? JSON.parse(localStorage.getItem('3metricEnabled')) : false;
      let autoResetEnabled = localStorage.getItem('3autoResetEnabled') !== null ? JSON.parse(localStorage.getItem('3autoResetEnabled')) : true;

      StreamsManager.add(streamsList);

      objectElement.addEventListener('load', function () {
        $timeout(function () {
          const svgDoc = objectElement.contentDocument;

          function initializeGroups() {
            groups.forEach((groupId) => {
              const group = svgDoc.getElementById(groupId);
              if (group) {
                group.style.display = 'none';
              }
            });
            const startGroup = svgDoc.getElementById('start-group');
            if (startGroup) {
              startGroup.style.display = 'block';
            }
          }

          initializeGroups();

          function updateTimerDisplay(value, showMilliseconds = false) {
            const timerText = svgDoc.getElementById('timer');
            if (timerText) {
              timerText.textContent = showMilliseconds ? `${value.toFixed(3)}s` : `${value.toFixed(2)}s`;
            }
          }

          function displayMetric(label, data) {
            if (nextAvailableGroupIndex >= groups.length) {
              for (let i = 0; i < groups.length - 1; i++) {
                const groupId = groups[i + 1];
                const labelElement = svgDoc.getElementById(`label-${i + 1}`);
                const dataElement = svgDoc.getElementById(`data-${i + 1}`);
                const groupElement = svgDoc.getElementById(groupId);

                if (groupElement) {
                  groupElement.style.display = 'block';
                }
                if (labelElement) {
                  labelElement.textContent = svgDoc.getElementById(`label-${i + 2}`).textContent;
                }
                if (dataElement) {
                  dataElement.textContent = svgDoc.getElementById(`data-${i + 2}`).textContent;
                }
              }
              nextAvailableGroupIndex--;
            }

            if (nextAvailableGroupIndex === 0) {
              const startGroup = svgDoc.getElementById('start-group');
              if (startGroup) {
                startGroup.style.display = 'none';
              }
            }

            const groupId = groups[nextAvailableGroupIndex];
            const labelElement = svgDoc.getElementById(`label-${nextAvailableGroupIndex + 1}`);
            const dataElement = svgDoc.getElementById(`data-${nextAvailableGroupIndex + 1}`);
            const groupElement = svgDoc.getElementById(groupId);

            if (groupElement) {
              groupElement.style.display = 'block';
            }
            if (labelElement) {
              labelElement.textContent = label;
            }
            if (dataElement) {
              dataElement.textContent = data;
            }

            nextAvailableGroupIndex++;
          }

          function startTimer() {
            if (!isTimerRunning && !isRunCompleted) {
              timerStartTime = performance.now();
              isTimerRunning = true;
              timerInterval = setInterval(() => {
                const elapsedTime = (performance.now() - timerStartTime) / 1000;
                updateTimerDisplay(elapsedTime);
              }, 20);
              lastUpdateTime = performance.now();
              totalDistance = 0;
            }
          }

          function stopAndResetTimerDisplay() {
            if (timerInterval) {
              clearInterval(timerInterval);
              timerInterval = null;
            }
            isTimerRunning = false;
            updateTimerDisplay(0.0);
            timerStartTime = null;
          }

          function resetAllStats() {
            stopAndResetTimerDisplay();
            totalDistance = 0;
            isRunCompleted = false;
            time60To130 = null;
            time100To150 = null;
            time100To200 = null;
            timeAt60 = null;
            timeAt100 = null;
            timeTo1_4mile = null;
            speedAt1_4mile = null;
            hasPassed1_4mile = false;
            nextAvailableGroupIndex = 0;
            initializeGroups();
            turnOffAllLightsInstantly();
          }

          function turnOnAllLightsInstantly() {
            clearCountdownTimeouts();
            const lights = [
              svgDoc.getElementById('circle-red-1'),
              svgDoc.getElementById('circle-red-2'),
              svgDoc.getElementById('circle-yellow-1'),
              svgDoc.getElementById('circle-yellow-2'),
              svgDoc.getElementById('circle-yellow-3'),
              svgDoc.getElementById('circle-green-1')
            ];

            lights.forEach((light) => {
              light.style.fill = '#00cc00';
              light.style.opacity = '1';
            });
          }

          function turnOffAllLightsInstantly() {
            clearCountdownTimeouts();
            const lights = [
              { element: svgDoc.getElementById('circle-red-1'), defaultColor: '#ff3333' },
              { element: svgDoc.getElementById('circle-red-2'), defaultColor: '#ff3333' },
              { element: svgDoc.getElementById('circle-yellow-1'), defaultColor: '#ffff00' },
              { element: svgDoc.getElementById('circle-yellow-2'), defaultColor: '#ffff00' },
              { element: svgDoc.getElementById('circle-yellow-3'), defaultColor: '#ffff00' },
              { element: svgDoc.getElementById('circle-green-1'), defaultColor: '#00cc00' }
            ];

            lights.forEach((light) => {
              if (light.element) {
                light.element.style.fill = light.defaultColor;
                light.element.style.opacity = '0.5';
              }
            });
          }

          function lightUpStartGroupSequentially() {
            const lights = [
              svgDoc.getElementById('circle-red-1'),
              svgDoc.getElementById('circle-red-2'),
              svgDoc.getElementById('circle-yellow-1'),
              svgDoc.getElementById('circle-yellow-2'),
              svgDoc.getElementById('circle-yellow-3'),
              svgDoc.getElementById('circle-green-1')
            ];

            lights.forEach((light, index) => {
              countdownTimeouts.push($timeout(() => {
                light.style.opacity = '1';
              }, index * 500));
            });
          }

          function clearCountdownTimeouts() {
            countdownTimeouts.forEach(timeout => $timeout.cancel(timeout));
            countdownTimeouts = [];
          }

          function handleLightCountdown(streams) {
            const speedMph = streams.electrics.airspeed * 2.237;

            if (speedMph > 58) {
              turnOnAllLightsInstantly();
              countdownInProgress = false;
              countdownStarted = false;
              if (!isTimerRunning) startTimer();
            } else if (speedMph >= 45 && speedMph <= 58) {
              if (!countdownStarted && !countdownInProgress) {
                countdownStarted = true;
                countdownInProgress = true;
                lightUpStartGroupSequentially();
              }
            } else if (speedMph < 45) {
              if (countdownStarted || countdownInProgress || isTimerRunning) {
                countdownStarted = false;
                countdownInProgress = false;
                turnOffAllLightsInstantly();
                if (autoResetEnabled) {
                  resetAllStats();
                } else {
                  stopAndResetTimerDisplay();
                }
              }
            }

            if (speedMph >= 60 && timeAt60 === null) {
              timeAt60 = (performance.now() - timerStartTime) / 1000;
            }

            if (speedMph >= 100 && timeAt100 === null) {
              timeAt100 = (performance.now() - timerStartTime) / 1000;
            }

            if (speedMph >= 130 && timeAt60 !== null && time60To130 === null) {
              time60To130 = (performance.now() - timerStartTime) / 1000 - timeAt60;
              displayMetric('60-130 MPH', `${time60To130.toFixed(3)}s`);
            }

            if (speedMph >= 150 && timeAt100 !== null && time100To150 === null) {
              time100To150 = (performance.now() - timerStartTime) / 1000 - timeAt100;
              displayMetric('100-150 MPH', `${time100To150.toFixed(3)}s`);
            }

            if (speedMph >= 200 && timeAt100 !== null && time100To200 === null) {
              time100To200 = (performance.now() - timerStartTime) / 1000 - timeAt100;
              displayMetric('100-200 MPH', `${time100To200.toFixed(3)}s`);
            }

            if (totalDistance >= 1320 && !hasPassed1_4mile) {
              hasPassed1_4mile = true;
              timeTo1_4mile = (performance.now() - timerStartTime) / 1000;
              speedAt1_4mile = speedMph;
              displayMetric('1/4 MILE', `${timeTo1_4mile.toFixed(3)}@${(metricEnabled ? (speedAt1_4mile * 1.60934).toFixed(2) : speedAt1_4mile.toFixed(2))} ${metricEnabled ? 'KMH' : 'MPH'}`);
              isRunCompleted = true;
            }
          }

          const resetButton = svgDoc.getElementById('resetButton');
          if (resetButton) {
            resetButton.addEventListener('click', resetAllStats);
          }

          const autoResetCheckbox = svgDoc.getElementById('auto-reset-checkbox');
          const metricCheckbox = svgDoc.getElementById('metric-checkbox');

          // Update checkbox colors based on saved states
          if (metricCheckbox) {
            metricEnabled = localStorage.getItem('3metricEnabled') === 'true';
            metricCheckbox.setAttribute('fill', metricEnabled ? '#00cc00' : '#ff3333');
          }

          if (autoResetCheckbox) {
            autoResetEnabled = localStorage.getItem('3autoResetEnabled') === 'true';
            autoResetCheckbox.setAttribute('fill', autoResetEnabled ? '#00cc00' : '#ff3333');
          }

          if (metricCheckbox) {
            metricCheckbox.addEventListener('click', function () {
              metricEnabled = !metricEnabled;
              localStorage.setItem('3metricEnabled', metricEnabled);
              metricCheckbox.setAttribute('fill', metricEnabled ? '#00cc00' : '#ff3333');
            });
          }

          if (autoResetCheckbox) {
            autoResetCheckbox.addEventListener('click', function () {
              autoResetEnabled = !autoResetEnabled;
              localStorage.setItem('3autoResetEnabled', autoResetEnabled);
              autoResetCheckbox.setAttribute('fill', autoResetEnabled ? '#00cc00' : '#ff3333');
            });
          }

          scope.$on('streamsUpdate', function (event, streams) {
            if (streams.electrics) {
              const airspeedMps = streams.electrics.airspeed;
              const airspeedMph = airspeedMps * 2.237;
              const airspeedDisplay = metricEnabled ? (airspeedMps * 3.6).toFixed(2) : airspeedMph.toFixed(2);

              const currentTime = performance.now();
              const timeElapsed = (currentTime - lastUpdateTime) / 1000;
              lastUpdateTime = currentTime;

              if (isTimerRunning) {
                totalDistance += airspeedMph * 1.46667 * timeElapsed;
              }

              handleLightCountdown(streams);
            }
          });

          scope.$on('abortRun', function () {
            resetAllStats();
          });

          scope.$on('$destroy', function () {
            StreamsManager.remove(streamsList);
            clearCountdownTimeouts();
            if (timerInterval) {
              clearInterval(timerInterval);
            }
          });
        }, 500);
      });
    }
  };
}]);
