'use strict';
angular.module('beamng.apps')
.directive('sgrDraggyQuarterMile', ['$timeout', function ($timeout) {
  return {
    template: `
      <div style="width: 100%; height: 100%; position: relative;">
        <object id="tachSvg" style="width:100%; height:100%;" type="image/svg+xml" data="/ui/modules/apps/sgrdraggyquartermile/sgrdraggyquartermile.svg"></object>
      </div>
    `,
    replace: true,
    restrict: 'EA',
    scope: true,
    link: function (scope, element) {
      const objectElement = element[0].querySelector('object');
      const streamsList = ['electrics'];
      let lightsStayOnTimeout = null;
      let timerInterval = null;
      let timerStartTime = null;
      let isTimerRunning = false;
      let isRunCompleted = false;
      let totalDistance = 0;
      let lastUpdateTime = performance.now();
      let countdownStarted = false;
      let countdownTimeouts = [];

      const timerType = 'quartermile';
      localStorage.setItem('timerType', timerType);

      let hasPassed60ft = false;
      let hasPassed330ft = false;
      let hasPassed1_8mile = false;
      let hasPassed1000ft = false;
      let hasPassed1_4mile = false;
      let timeTo1_8mile = null;
      let speedAt1_8mile = null;
      let timeTo1000ft = null;
      let timeTo1_4mile = null;
      let speedAt1_4mile = null;

      const groups = ['group-1', 'group-2', 'group-3', 'group-4', 'group-5', 'group-6', 'group-7', 'group-8'];
      let nextAvailableGroupIndex = 0;
      let metricEnabled = localStorage.getItem('2metricEnabled') !== null ? JSON.parse(localStorage.getItem('2metricEnabled')) : false;
      let autoResetEnabled = localStorage.getItem('2autoResetEnabled') !== null ? JSON.parse(localStorage.getItem('2autoResetEnabled')) : true;

      let lightTimeouts = [];

      StreamsManager.add(streamsList);

      objectElement.addEventListener('load', function () {
        $timeout(function () {
          const svgDoc = objectElement.contentDocument;

          function initializeGroups() {
            groups.forEach((groupId) => {
              const group = svgDoc.getElementById(groupId);
              if (group) {
                group.style.display = 'none';
              }
            });
            const startGroup = svgDoc.getElementById('start-group');
            if (startGroup) {
              startGroup.style.display = 'block';
            }
          }

          initializeGroups();

          function updateTimerDisplay(value, showMilliseconds = false) {
            const timerText = svgDoc.getElementById('timer');
            if (timerText) {
              timerText.textContent = showMilliseconds ? `${value.toFixed(3)}s` : `${value.toFixed(2)}s`;
            }
          }

          function displayMetric(label, data) {
            if (nextAvailableGroupIndex >= groups.length) {
              for (let i = 0; i < groups.length - 1; i++) {
                const groupId = groups[i + 1];
                const labelElement = svgDoc.getElementById(`label-${i + 1}`);
                const dataElement = svgDoc.getElementById(`data-${i + 1}`);
                const groupElement = svgDoc.getElementById(groupId);

                if (groupElement) {
                  groupElement.style.display = 'block';
                }
                if (labelElement) {
                  labelElement.textContent = svgDoc.getElementById(`label-${i + 2}`).textContent;
                }
                if (dataElement) {
                  dataElement.textContent = svgDoc.getElementById(`data-${i + 2}`).textContent;
                }
              }
              nextAvailableGroupIndex--;
            }

            if (nextAvailableGroupIndex === 0) {
              const startGroup = svgDoc.getElementById('start-group');
              if (startGroup) {
                startGroup.style.display = 'none';
              }
            }

            const groupId = groups[nextAvailableGroupIndex];
            const labelElement = svgDoc.getElementById(`label-${nextAvailableGroupIndex + 1}`);
            const dataElement = svgDoc.getElementById(`data-${nextAvailableGroupIndex + 1}`);
            const groupElement = svgDoc.getElementById(groupId);

            if (groupElement) {
              groupElement.style.display = 'block';
            }
            if (labelElement) {
              labelElement.textContent = label;
            }
            if (dataElement) {
              dataElement.textContent = data;
            }

            nextAvailableGroupIndex++;
          }

          function startTimer() {
            if (!isTimerRunning && !isRunCompleted) {
              timerStartTime = performance.now();
              isTimerRunning = true;
              timerInterval = setInterval(() => {
                const elapsedTime = (performance.now() - timerStartTime) / 1000;
                updateTimerDisplay(elapsedTime);
              }, 20);
            }
          }

          function stopTimer() {
            if (timerInterval) {
              clearInterval(timerInterval);
              timerInterval = null;
              isTimerRunning = false;
            }
          }

          function resetTimerDisplay() {
            stopTimer();
            updateTimerDisplay(0.0);
          }

          function clearCountdownTimeouts() {
            countdownTimeouts.forEach(timeout => $timeout.cancel(timeout));
            countdownTimeouts = [];
          }

          function lightUpStartGroupSequentially() {
            const lights = [
              svgDoc.getElementById('circle-red-1'),
              svgDoc.getElementById('circle-red-2'),
              svgDoc.getElementById('circle-yellow-1'),
              svgDoc.getElementById('circle-yellow-2'),
              svgDoc.getElementById('circle-yellow-3'),
              svgDoc.getElementById('circle-green-1')
            ];

            lights.forEach((light, index) => {
              countdownTimeouts.push($timeout(() => {
                light.style.opacity = '1';
              }, index * 500));
            });
          }

          function turnOnAllLightsInstantly() {
            clearCountdownTimeouts();
            const lights = [
              svgDoc.getElementById('circle-red-1'),
              svgDoc.getElementById('circle-red-2'),
              svgDoc.getElementById('circle-yellow-1'),
              svgDoc.getElementById('circle-yellow-2'),
              svgDoc.getElementById('circle-yellow-3'),
              svgDoc.getElementById('circle-green-1')
            ];

            lights.forEach((light) => {
              light.style.fill = '#00cc00';
              light.style.opacity = '1';
            });
          }

          function turnOffAllLightsInstantly() {
            clearCountdownTimeouts();
            const lights = [
              { element: svgDoc.getElementById('circle-red-1'), color: '#ff3333' },
              { element: svgDoc.getElementById('circle-red-2'), color: '#ff3333' },
              { element: svgDoc.getElementById('circle-yellow-1'), color: '#ffff00' },
              { element: svgDoc.getElementById('circle-yellow-2'), color: '#ffff00' },
              { element: svgDoc.getElementById('circle-yellow-3'), color: '#ffff00' },
              { element: svgDoc.getElementById('circle-green-1'), color: '#00cc00' }
            ];

            lights.forEach((light) => {
              light.element.style.fill = light.color;
              light.element.style.opacity = '0.5';
            });
          }

          function checkLightCountdownConditions(streams) {
            const throttle = streams.electrics.throttle > 0.8;
            const brake = streams.electrics.brake > 0.8;
            const clutch = streams.electrics.clutch > 0.8;
            const transbrake = streams.electrics.transbrake;

            return (brake && throttle) || (clutch && throttle) || (transbrake && throttle);
          }

          function handleLightCountdown(streams) {
            const speedMph = streams.electrics.airspeed * 2.237;

            if (speedMph > 2) {
              turnOnAllLightsInstantly();
              countdownStarted = true;
              if (lightsStayOnTimeout) {
                $timeout.cancel(lightsStayOnTimeout);
                lightsStayOnTimeout = null;
              }
            } else if (checkLightCountdownConditions(streams)) {
              if (!countdownStarted) {
                countdownStarted = true;
                lightUpStartGroupSequentially();
              }
              if (lightsStayOnTimeout) {
                $timeout.cancel(lightsStayOnTimeout);
                lightsStayOnTimeout = null;
              }
            } else {
              if (!lightsStayOnTimeout) {
                lightsStayOnTimeout = $timeout(() => {
                  turnOffAllLightsInstantly();
                  countdownStarted = false;
                }, 1000);
              }
            }

            if (totalDistance >= 660 && !hasPassed1_8mile) {
              hasPassed1_8mile = true;
              timeTo1_8mile = (performance.now() - timerStartTime) / 1000;
              speedAt1_8mile = speedMph;
              displayMetric('1/8 MILE', `${timeTo1_8mile.toFixed(3)}@${(metricEnabled ? (speedAt1_8mile * 1.60934).toFixed(2) : speedAt1_8mile.toFixed(2))} ${metricEnabled ? 'KMH' : 'MPH'}`);
            }

            if (totalDistance >= 1320 && !hasPassed1_4mile) {
              hasPassed1_4mile = true;
              timeTo1_4mile = (performance.now() - timerStartTime) / 1000;
              speedAt1_4mile = speedMph;
              displayMetric('1/4 MILE', `${timeTo1_4mile.toFixed(3)}@${(metricEnabled ? (speedAt1_4mile * 1.60934).toFixed(2) : speedAt1_4mile.toFixed(2))} ${metricEnabled ? 'KMH' : 'MPH'}`);
            }
          }

          function checkDistanceMarks() {
            const elapsedTime = (performance.now() - timerStartTime) / 1000;

            if (totalDistance >= 60 && !hasPassed60ft) {
              hasPassed60ft = true;
              displayMetric('60 FT', `${elapsedTime.toFixed(3)}s`);
            }

            if (totalDistance >= 330 && !hasPassed330ft) {
              hasPassed330ft = true;
              displayMetric('330 FT', `${elapsedTime.toFixed(3)}s`);
            }
          }

          function resetToDefault() {
            stopTimer();
            updateTimerDisplay(0.0);
            totalDistance = 0;
            hasPassed60ft = false;
            hasPassed330ft = false;
            hasPassed1_8mile = false;
            hasPassed1000ft = false;
            hasPassed1_4mile = false;
            isRunCompleted = false;
            timeTo1_8mile = null;
            speedAt1_8mile = null;
            timeTo1000ft = null;
            timeTo1_4mile = null;
            speedAt1_4mile = null;
            nextAvailableGroupIndex = 0;
            initializeGroups();
            turnOffAllLightsInstantly();
          }

          const resetButton = svgDoc.getElementById('resetButton');
          if (resetButton) {
            resetButton.addEventListener('click', resetToDefault);
          }

          const autoResetCheckbox = svgDoc.getElementById('auto-reset-checkbox');
          const metricCheckbox = svgDoc.getElementById('metric-checkbox');

          // Update checkbox colors based on saved states
          if (metricCheckbox) {
            metricEnabled = localStorage.getItem('2metricEnabled') === 'true';
            metricCheckbox.setAttribute('fill', metricEnabled ? '#00cc00' : '#ff3333');
          }

          if (autoResetCheckbox) {
            autoResetEnabled = localStorage.getItem('2autoResetEnabled') === 'true';
            autoResetCheckbox.setAttribute('fill', autoResetEnabled ? '#00cc00' : '#ff3333');
          }

          if (metricCheckbox) {
            metricCheckbox.addEventListener('click', function () {
              metricEnabled = !metricEnabled;
              localStorage.setItem('2metricEnabled', metricEnabled);
              metricCheckbox.setAttribute('fill', metricEnabled ? '#00cc00' : '#ff3333');
            });
          }

          if (autoResetCheckbox) {
            autoResetCheckbox.addEventListener('click', function () {
              autoResetEnabled = !autoResetEnabled;
              localStorage.setItem('2autoResetEnabled', autoResetEnabled);
              autoResetCheckbox.setAttribute('fill', autoResetEnabled ? '#00cc00' : '#ff3333');
            });
          }

          scope.$on('streamsUpdate', function (event, streams) {
            if (streams.electrics) {
              const airspeedMps = streams.electrics.airspeed;
              const airspeedMph = airspeedMps * 2.237;
              const airspeedDisplay = metricEnabled ? (airspeedMps * 3.6).toFixed(2) : airspeedMph.toFixed(2);

              const currentTime = performance.now();
              const timeElapsed = (currentTime - lastUpdateTime) / 1000;
              lastUpdateTime = currentTime;

              const speedInFeetPerSecond = airspeedMph * 1.46667;

              if (airspeedMph < 0.4) {
                totalDistance = 0;
              } else {
                totalDistance += speedInFeetPerSecond * timeElapsed;
              }

              handleLightCountdown(streams);
              checkDistanceMarks();

              if (airspeedMph > 0.8 && !isTimerRunning && !isRunCompleted) {
                startTimer();
              } else if (airspeedMph < 0.4 && (isTimerRunning || isRunCompleted)) {
                if (autoResetEnabled && isRunCompleted) {
                  resetToDefault();
                } else if (autoResetEnabled) {
                  isRunCompleted = false;
                  resetToDefault();
                } else {
                  resetTimerDisplay();
                }
              }
            }
          });

          scope.$on('abortRun', function () {
            if (lightsStayOnTimeout) {
              $timeout.cancel(lightsStayOnTimeout);
              lightsStayOnTimeout = null;
            }
            stopTimer();
            turnOffAllLightsInstantly();
          });

          scope.$on('$destroy', function () {
            StreamsManager.remove(streamsList);
            clearCountdownTimeouts();
            if (lightsStayOnTimeout) {
              $timeout.cancel(lightsStayOnTimeout);
            }
            if (timerInterval) {
              clearInterval(timerInterval);
            }
          });
        }, 500);
      });
    }
  };
}]);
