local M = {}

-- Initialize timer variables
local left_timer = 0
local right_timer = 0
local timer_duration = 1 -- duration in seconds

-- Variable to store the last state of the rear fog light
local lastRearFogState = 0

-- Variable to store the state of the work lights
local workLightsState = 0  -- 0: Off, 1: On

-- Beacon light variables
local beaconSpeed = 0  -- Speed factor for beacon rotation
local electricsName = nil
local maxRotationAngle = 15  -- Maximum angle the beacon can rotate based on steering input (in degrees)
local deadzoneThreshold = 0.1  -- Deadzone threshold for steering input
local centerOffset = 180  -- Offset to center the rotation around 180 degrees
local maxSteeringInput = 300  -- Maximum steering input (in degrees) that maps to maxRotationAngle
local linearityFactor = 1.0  -- Adjusts linearity of steering input mapping (default is linear)
local currentAngle = centerOffset  -- Current angle of the beacon light

-- Helper function to clamp a value between min and max
local function clamp(value, min, max)
    return math.max(min, math.min(max, value))
end

-- Function to map steering input to an angle range with adjustable linearity
local function mapSteeringToAngle(steering, maxAngle, maxSteeringInput, linearity)
    local normalizedSteering = clamp(steering / maxSteeringInput, -1, 1)
    if linearity ~= 1.0 then
        normalizedSteering = math.sign(normalizedSteering) * math.abs(normalizedSteering) ^ linearity
    end
    return normalizedSteering * maxAngle
end

-- Function to smoothly transition to a target angle
local function smoothToTarget(current, target, smoothFactor, dt)
    return current + (target - current) * (1 - math.exp(-smoothFactor * dt))
end

-- Function to control corner lights in BeamNG
local function controlCornerLights(dt)
    -- Retrieve necessary electric values
    local steering = electrics.values.steering
    local lights_state = electrics.values.lights_state
    local cornerlight_toggle = electrics.values.cornerlight_toggle
    local hazard_enabled = electrics.values.hazard_enabled
    local wheelspeed = electrics.values.wheelspeed
    local signal_left_input = electrics.values.signal_left_input
    local signal_right_input = electrics.values.signal_right_input
    local reverse = electrics.values.reverse

    -- Check if hazard lights are enabled, do not activate corner lights
    if hazard_enabled == 1 then
        electrics.values.cornerlights_left = 0
        electrics.values.cornerlights_right = 0
        left_timer = 0
        right_timer = 0
        return
    end

    -- Update cornerlight_toggle state based on the button press
    if cornerlight_toggle == 0 then
        -- Turn off both lights if toggle is set to 0
        electrics.values.cornerlights_left = 0
        electrics.values.cornerlights_right = 0
        left_timer = 0
        right_timer = 0
    elseif cornerlight_toggle == 1 then
        -- Check conditions to turn on the left corner light
        local left_condition = (lights_state ~= 0) and (wheelspeed <= 7) and ((steering > 500) or (reverse == 1) or (signal_left_input == 1))

        -- Check conditions to turn on the right corner light
        local right_condition = (lights_state ~= 0) and (wheelspeed <= 7) and ((steering < -500) or (reverse == 1) or (signal_right_input == 1))

        -- Update left corner light state
        if left_condition then
            electrics.values.cornerlights_left = 1
            left_timer = timer_duration -- reset the timer
        elseif left_timer > 0 then
            left_timer = left_timer - dt
            if left_timer <= 0 then
                electrics.values.cornerlights_left = 0
            end
        else
            electrics.values.cornerlights_left = 0
        end

        -- Update right corner light state
        if right_condition then
            electrics.values.cornerlights_right = 1
            right_timer = timer_duration -- reset the timer
        elseif right_timer > 0 then
            right_timer = right_timer - dt
            if right_timer <= 0 then
                electrics.values.cornerlights_right = 0
            end
        else
            electrics.values.cornerlights_right = 0
        end
    end
end

-- Function to mimic trailer signals based on the vehicle's signals
local function updateTrailerSignals()
    -- Check if a trailer is attached
    if electrics.values.fifthwheel_attachmentState == 1 or electrics.values.pintle_attachmentState == 1 then
        electrics.values.trailersignal_L = electrics.values.signal_L
        electrics.values.trailersignal_R = electrics.values.signal_R
    else
        -- Reset trailer signals if no trailer is attached
        electrics.values.trailersignal_L = 0
        electrics.values.trailersignal_R = 0
    end
end

-- Function to toggle the rear fog light
local function toggleRearFogLight()
    -- Toggle the rear fog light regardless of the light state
    electrics.values.rearfog_toggle = not electrics.values.rearfog_toggle
    -- Update the last state
    lastRearFogState = electrics.values.rearfog_toggle
end

-- Function to control work lights based on reverse state and toggle
local function controlWorkLights()
    -- Check if reverse is active
    if electrics.values.reverse == 1 then
        -- Turn on work lights if they are not overridden off
        if workLightsState ~= 0 then
            electrics.values.worklights = 1
        end
    else
        -- Turn off work lights if reverse is not active
        electrics.values.worklights = 0
    end
end

-- Function to toggle work lights override
local function toggleWorkLightsOverride()
    workLightsState = 1 - workLightsState  -- Toggle between 0 and 1
    -- Update work lights state immediately
    if workLightsState == 0 then
        electrics.values.worklights = 0
    else
        -- Turn on work lights only if reverse is active
        if electrics.values.reverse == 1 then
            electrics.values.worklights = 1
        end
    end
end

-- Beacon light update function
local function updateBeacon(dt)
    -- Check if lowhighbeam is greater than 0 to turn on
    if electrics.values.lowhighbeam == 0 then
        electrics.values[electricsName] = 0
        return
    end

    -- If the vehicle is in reverse, center the beacon light
    if electrics.values.reverse == 1 then
        currentAngle = smoothToTarget(currentAngle, centerOffset, 5, dt)
        electrics.values[electricsName] = currentAngle % 360
        return
    end

    -- Calculate the steering angle in degrees, mapped with adjustable linearity to maxRotationAngle
    local steeringAngle = mapSteeringToAngle(electrics.values.steering, maxRotationAngle, maxSteeringInput, linearityFactor)

    -- Apply deadzone to steering input
    if math.abs(electrics.values.steering) < deadzoneThreshold then
        -- No significant steering input, smoothly rotate towards idle position
        currentAngle = smoothToTarget(currentAngle, centerOffset, 5, dt)
    else
        -- Calculate target angle based on steering input
        local targetAngle = centerOffset + steeringAngle
        
        -- Smoothly rotate towards the target angle
        currentAngle = smoothToTarget(currentAngle, targetAngle, 5, dt)
    end

    -- Update the electrics value with the current angle, ensuring it stays within 0 to 360 degrees
    electrics.values[electricsName] = currentAngle % 360
end

-- Initialize function (set cornerlight_toggle to 1 by default)
M.init = function(jbeamData)
    -- Initialize corner lights
    electrics.values.cornerlight_toggle = 1
    
    -- Initialize trailer signals
    electrics.values.trailersignal_L = 0
    electrics.values.trailersignal_R = 0

    -- Initialize the rear fog light state
    electrics.values.rearfog_toggle = 0
    lastRearFogState = 0

    -- Initialize work lights state and override toggle
    electrics.values.worklights = 0  -- Off by default
    workLightsState = 0

    -- Initialize beacon light
    beaconSpeed = jbeamData.spinSpeed or 100  -- Adjust speed if specified in jbeamData
    electricsName = jbeamData.electricsName or "turnlowhighbeam"
    electrics.values[electricsName] = centerOffset  -- Start with the light centered at 180 degrees when initialized
    
    if jbeamData.linearityFactor then
        linearityFactor = jbeamData.linearityFactor  -- Use custom linearity factor if provided in jbeamData
    end
end

-- Update function (called every frame)
M.updateGFX = function(dt)
    -- Update corner lights
    controlCornerLights(dt)

    -- Update trailer signals
    updateTrailerSignals()

    -- Ensure the rear fog light is off when lights are off
    if electrics.values.lights_state == 0 then
        electrics.values.rearfog_toggle = 0
    elseif (electrics.values.lights_state == 1 or electrics.values.lights_state == 2) and electrics.values.rearfog_toggle == 0 then
        -- Restore the last state of the rear fog light when lights are turned back on
        electrics.values.rearfog_toggle = lastRearFogState
    end

    -- Control work lights based on reverse state and override toggle
    controlWorkLights()

    -- Update the beacon light
    updateBeacon(dt)
end

-- Function to handle button presses (for rear fog light and work light toggles)
M.onButtonPress = function(button)
    if button == "rearfog_toggle" then
        toggleRearFogLight()
        -- Update the last state
        lastRearFogState = electrics.values.rearfog_toggle
    elseif button == "worklight_toggle" then
        toggleWorkLightsOverride()
    end
end

return M



