local M = {}
M.type = "auxiliary"
M.relevantDevice = nil
M.defaultOrder = 2000

local liftAxleState = 0
local dumpAxleState = 0
local defaultTargetHeight = 0
local defaultTargetHeightSet = false
local offsetApplied = false
local LoadOffsetApplied = false

local enableLiftAxle = true
local enableWeightTransfer = true

local frontAirbags = {}
local mainAirbags = {}
local liftAxleAirbags = {}
local dumpAxleAirbags = {}
local liftAirbags = {}

local liftAxleOffset = nil
local LoadOffset = nil

local suspensionDataName = "mainairSuspensionData"
local beamGroupName = "rearMain"

local mainAirSuspensionController = "mainairSuspension"
local liftAirSuspensionController = "liftairSuspension"

local function applyLoadOffset()
    -- Only apply the LoadOffset if the trailer is connected and it's not already applied
    if electrics.values.fifthwheel_attachmentState == 1 and not LoadOffsetApplied then
        if liftAxleState == 0 then  -- Prevent LoadOffset if the lift axle is active
            local mainAirSuspension = controller.getControllerSafe(mainAirSuspensionController)
            if mainAirSuspension and #mainAirbags > 0 then
                local newHeight = defaultTargetHeight + LoadOffset
                mainAirSuspension.setTargetLength(mainAirbags, newHeight, immediate)
                LoadOffsetApplied = true
            end
        end
    elseif electrics.values.fifthwheel_attachmentState == 0 and LoadOffsetApplied then
        local mainAirSuspension = controller.getControllerSafe(mainAirSuspensionController)
        if mainAirSuspension and #mainAirbags > 0 then
            mainAirSuspension.setTargetLength(mainAirbags, defaultTargetHeight, immediate)
            LoadOffsetApplied = false
        end
    end
end

local function getTargetLengthFromBeamGroup(beamGroup)
    local suspensionData = v.data[suspensionDataName]
    if suspensionData then
        for _, data in ipairs(suspensionData) do
            if data.beamGroup == beamGroup then
                return data.defaultTargetLength or 0
            end
        end
    end
    log("E", "getTargetLengthFromBeamGroup", "Default target length not found for beam group: " .. beamGroup)
    return 0
end

local function initControlFlags(jbeamData)
    enableLiftAxle = jbeamData.enableLiftAxle ~= false
    enableWeightTransfer = jbeamData.enableWeightTransfer ~= false

    mainAirbags = jbeamData.mainAirbags or {}
    liftAxleAirbags = jbeamData.liftAxleAirbags or {}
    dumpAxleAirbags = jbeamData.dumpAxleAirbags or {}
    liftAirbags = jbeamData.liftAirbags or {}

    liftAxleOffset = jbeamData.liftAxleOffset or 0.05
    suspensionDataName = suspensionDataName
    beamGroupName = beamGroupName

    mainAirSuspensionController = jbeamData.mainAirSuspensionController or mainAirSuspensionController
    liftAirSuspensionController = jbeamData.liftAirSuspensionController or liftAirSuspensionController

    frontAirbags = jbeamData.frontAirbags or frontAirbags
    LoadOffset = jbeamData.LoadOffset or LoadOffset
end

local function toggleLiftAxle()
    if not enableLiftAxle then return end

    if dumpAxleState == 1 then
        guihooks.message("Cannot toggle lift axle while axle dump is active.")
        log("W", "toggleLiftAxle", "Cannot toggle lift axle while axle dump is active")
        return
    end

    liftAxleState = (liftAxleState + 1) % 2
    electrics.values.toggleAxleLiftState = liftAxleState

    local isLifted = (liftAxleState == 1)
    local mainAirSuspension = controller.getControllerSafe(mainAirSuspensionController)
    local liftAirSuspension = controller.getControllerSafe(liftAirSuspensionController)

    if isLifted then
        -- Disable LoadOffset when lifting the axle
        if LoadOffsetApplied then
            local mainAirSuspension = controller.getControllerSafe(mainAirSuspensionController)
            if mainAirSuspension and #mainAirbags > 0 then
                mainAirSuspension.setTargetLength(mainAirbags, defaultTargetHeight, immediate)
                LoadOffsetApplied = false
            end
        end
    end

    if mainAirSuspension and #mainAirbags > 0 then
        local adjustedMainAirLength = isLifted and (defaultTargetHeight + liftAxleOffset) or defaultTargetHeight
        mainAirSuspension.setTargetLength(mainAirbags, adjustedMainAirLength, immediate)

        if #liftAxleAirbags > 0 then
            local liftTargetLength = isLifted and 0 or defaultTargetHeight
            mainAirSuspension.setTargetLength(liftAxleAirbags, liftTargetLength, immediate)
        end
    end

    if liftAirSuspension and #liftAirbags > 0 then
        local liftAdjustmentRate = isLifted and 1.0 or -1.0
        liftAirSuspension.setAdjustmentRate(liftAirbags, liftAdjustmentRate, immediate)
    end

    electrics.values.AxleLiftToggle = isLifted and 1 or 0
    guihooks.message("Lift axles " .. (isLifted and "Lifted" or "Lowered"))
end

local function toggleAxleDump()
    if not enableWeightTransfer then return end

    if liftAxleState == 1 then
        guihooks.message("Cannot toggle axle dump while lift axle is active.")
        log("W", "toggleAxleDump", "Cannot toggle axle dump while lift axle is active")
        return
    end

    dumpAxleState = (dumpAxleState + 1) % 2
    electrics.values.toggleAxleDumpState = dumpAxleState

    local isDumped = (dumpAxleState == 1)
    local mainAirSuspension = controller.getControllerSafe(mainAirSuspensionController)

    if mainAirSuspension and #dumpAxleAirbags > 0 then
        local dumpTargetLength = isDumped and 0 or defaultTargetHeight
        mainAirSuspension.setTargetLength(dumpAxleAirbags, dumpTargetLength, immediate)
    end

    electrics.values.AxleDumpToggle = isDumped and 1 or 0
    guihooks.message("Weight transfer " .. (isDumped and "Activated" or "Deactivated"))
end

local function init(jbeamData)
    electrics.values.toggleAxleLiftState = 0
    electrics.values.toggleAxleDumpState = 0
    defaultTargetHeightSet = false

    if not defaultTargetHeightSet then
        defaultTargetHeight = getTargetLengthFromBeamGroup(beamGroupName)
        defaultTargetHeightSet = true
    end

    initControlFlags(jbeamData)
end

local function resetSuspension()
    guihooks.message("Ride height resetting to factory height...")

    local mainAirSuspension = controller.getControllerSafe(mainAirSuspensionController)

    if mainAirSuspension then
        local resetHeight = defaultTargetHeight

        -- Apply LoadOffset when the trailer is connected and LoadOffset is not already applied
        if electrics.values.fifthwheel_attachmentState == 1 and LoadOffsetApplied then
            resetHeight = defaultTargetHeight + LoadOffset
            LoadOffsetApplied = true
        elseif electrics.values.fifthwheel_attachmentState == 0 then
            LoadOffsetApplied = false
        end

        -- If the lift axle is active, include the lift axle offset
        if liftAxleState == 1 then
            resetHeight = resetHeight + liftAxleOffset
            offsetApplied = true
        elseif liftAxleState == 0 and offsetApplied then
            resetHeight = defaultTargetHeight
            offsetApplied = false
        end

        -- Set the main airbags to the new reset height
        if #mainAirbags > 0 then
            mainAirSuspension.setTargetLength(mainAirbags, resetHeight, immediate)
        end

        -- Reset front airbags to default height
        mainAirSuspension.setTargetLength(frontAirbags, defaultTargetHeight, immediate)

        -- Adjust lift axle airbags if present
        if #liftAxleAirbags > 0 then
            local liftResetHeight = (liftAxleState == 1) and 0 or defaultTargetHeight
            mainAirSuspension.setTargetLength(liftAxleAirbags, liftResetHeight, immediate)
        end

        -- Adjust lift airbags if present
        if #liftAirbags > 0 then
            local liftResetHeight = (liftAxleState == 1) and 0 or defaultTargetHeight
            mainAirSuspension.setTargetLength(liftAirbags, liftResetHeight, immediate)
        end

        -- Reset dump axle airbags if active
        if dumpAxleState == 1 and #dumpAxleAirbags > 0 then
            mainAirSuspension.setTargetLength(dumpAxleAirbags, 0, immediate)
        end
    end
end


local function update()
    applyLoadOffset()
end

M.init = init
M.toggleLiftAxle = toggleLiftAxle
M.toggleAxleDump = toggleAxleDump
M.onDown = toggleLiftAxle
M.resetSuspension = resetSuspension
M.update = update

return M
