local M = {}
M.type = "auxiliary"
M.relevantDevice = nil
M.defaultOrder = 1000

-- Sound variables for air conditioning, vent, and new sound sources
local acsound = {sound = nil, volume = nil, pitch = nil, targetVolume = nil}
local ventsound = {sound = nil, volume = nil, targetVolume = nil}
local chimeSound = {sound = nil, volume = nil, pitch = 1, targetVolume = nil}  -- Looping chime for lights/doors
local chimeSoundOnce = {sound = nil, volume = nil, pitch = 1, targetVolume = nil}  -- Single chime
local alertSound = {sound = nil, volume = nil, pitch = 1, targetVolume = nil}  -- Looping alert sound
local sharedNode = nil  -- Shared node for all sounds
local acvolumeSmoother, acpitchSmoother, ventvolumeSmoother, chimeVolumeSmoother, chimeOnceVolumeSmoother, alertVolumeSmoother
local acmodename = {[0] = "Off", [1] = "Low", [2] = "Medium", [3] = "High"}

-- New variables for sound management
local soundVolume = 1  -- Default sound volume
local soundActive = true  -- Sound is initially active
local driverNodeCid = nil  -- Driver node ID

-- Variables to track door state
local lastDoorFLState = 0
local lastDoorFRState = 0

-- Variable to track low fuel chime status
local lowFuelChimePlayed = false

-- Variable to track low fuel chime status
local lowAirPressureChimePlayed = false

-- Initialize air conditioning mode to "off"
local function setacmode(acm)
    electrics.values.aircon = acm
    guihooks.message("Air conditioner fan speed: " .. acmodename[acm])
end

-- Increase air conditioning speed
local function airconup()
    if electrics.values.aircon >= 3 then return end  
    electrics.values.aircon = electrics.values.aircon + 1
    setacmode(electrics.values.aircon)
end

-- Decrease air conditioning speed
local function aircondown()
    if electrics.values.aircon <= 0 then return end  
    electrics.values.aircon = electrics.values.aircon - 1
    setacmode(electrics.values.aircon)
end

-- Function to check if the camera is inside the vehicle cabin
local function isCamInside()
    local camPos = obj:getCameraPosition()
    if driverNodeCid == nil then return false end  -- Safety check
    local combinedPos = obj:getPosition() + obj:getNodePosition(driverNodeCid)
    return camPos:distance(combinedPos) <= 0.6
end

-- Variables to track low air pressure chime status
local lowAirPressureChimePlayed = false  -- Track low air pressure chime status

-- Update GFX and sound depending on state
local function updateGFX(dt)
    local camInside = isCamInside()

    -- Update air conditioning sound
    if acsound.sound and soundActive then
        local acpower = electrics.values.aircon > 0 and not isStarting  -- Disable AC while starting
        acsound.targetVolume = acpower and (camInside and soundVolume or 0) or 0
        
        -- Set pitch for AC sound based on AC mode
        if electrics.values.aircon == 1 then
            acsound.pitch = 0.7  -- Lower pitch for low mode
        elseif electrics.values.aircon == 2 then
            acsound.pitch = 0.9  -- Normal pitch for medium mode
        elseif electrics.values.aircon == 3 then
            acsound.pitch = 1.0  -- Higher pitch for high mode
        else
            acsound.pitch = 0  -- Off mode
        end
        
        acsound.volume = acvolumeSmoother:get(acsound.targetVolume, dt)
        obj:setVolumePitch(acsound.sound, acsound.volume * 0.1, acsound.pitch)
    end

    -- Update vent sound
    if ventsound.sound and soundActive then
        ventsound.targetVolume = acsound.targetVolume * electrics.values.aircon * (camInside and soundVolume or 0)
        ventsound.volume = ventvolumeSmoother:get(ventsound.targetVolume, dt)

        -- Set pitch for vent sound based on AC mode
        if electrics.values.aircon == 1 then
            ventsound.pitch = 0.8  -- Lower pitch for low mode
        elseif electrics.values.aircon == 2 then
            ventsound.pitch = 1.0  -- Normal pitch for medium mode
        elseif electrics.values.aircon == 3 then
            ventsound.pitch = 1.2  -- Higher pitch for high mode
        else
            ventsound.pitch = 0  -- Off mode
        end

        obj:setVolumePitch(ventsound.sound, ventsound.volume * 0.1, ventsound.pitch)
    end

    -- Control for looping chime sound
    if chimeSound.sound and soundActive then
        chimeSound.targetVolume = camInside and 
            (electrics.values.lights_state == 1 or electrics.values.lights_state == 2) and 
            (electrics.values.doorFLCoupler_notAttached == 1 or electrics.values.doorFRCoupler_notAttached == 1) and 
            (electrics.values.ignitionLevel == 0) and 
            soundVolume or 0
        
        chimeSound.volume = chimeVolumeSmoother:get(chimeSound.targetVolume, dt)
        if chimeSound.volume > 0 then
            obj:setVolumePitch(chimeSound.sound, chimeSound.volume * 0.5, 1)  -- Use fixed pitch
        end
    end

    -- Check for door state change to play single chime sound
    local currentDoorFLState = electrics.values.doorFLCoupler_notAttached
    local currentDoorFRState = electrics.values.doorFRCoupler_notAttached
    
    if (currentDoorFLState == 1 and lastDoorFLState == 0) or (currentDoorFRState == 1 and lastDoorFRState == 0) then
        -- Play the chime once when a door is opened (and only if camera is inside)
        if chimeSoundOnce.sound and camInside and electrics.values.ignitionLevel ~= 0 and
           (electrics.values.lights_state == 0 or electrics.values.lights_state == 1 or electrics.values.lights_state == 2) then
            obj:setVolumePitch(chimeSoundOnce.sound, 0.5, 1)  -- Set volume and pitch
            obj:playSFX(chimeSoundOnce.sound)  -- Play the sound
        end
    end

    -- Independent check for low fuel chime, only plays inside the cab
    if electrics.values.lowfuel == 1 then
        if not lowFuelChimePlayed and chimeSoundOnce.sound and camInside then  -- Check if camera is inside
            obj:setVolumePitch(chimeSoundOnce.sound, 0.5, 1)  -- Set volume and pitch
            obj:playSFX(chimeSoundOnce.sound)  -- Play the sound for low fuel
            lowFuelChimePlayed = true  -- Mark the chime as played
        end
    else
        lowFuelChimePlayed = false  -- Reset flag when fuel is no longer low
    end

    -- Independent check for low air pressure chime, only plays inside the cab
    if electrics.values.lowAirPressure == 1 and electrics.values.ignitionLevel == 1 then
        if not lowAirPressureChimePlayed and chimeSoundOnce.sound and camInside then
            obj:setVolumePitch(chimeSoundOnce.sound, 0.5, 1)  -- Set volume and pitch
            obj:playSFX(chimeSoundOnce.sound)  -- Play the sound for low air pressure
            lowAirPressureChimePlayed = true  -- Mark the chime as played
        end
    else
        lowAirPressureChimePlayed = false  -- Reset flag when air pressure is no longer low
    end

    -- Control for looping alert sound
    if alertSound.sound and soundActive then
        alertSound.targetVolume = camInside and 
            electrics.values.lowAirPressure  == 1 and electrics.values.engineRunning == 1 and
            soundVolume or 0
        
        alertSound.volume = alertVolumeSmoother:get(alertSound.targetVolume, dt)
        if alertSound.volume > 0 then
            obj:setVolumePitch(alertSound.sound, alertSound.volume * 0.5, 1)  -- Use fixed pitch
        end
    end

    -- Update the last door states
    lastDoorFLState = currentDoorFLState
    lastDoorFRState = currentDoorFRState
end

-- Function to check if the air conditioning system is on (low, medium, or high)
local function isAirConditioningOn()
    return (electrics.values.aircon > 0) and 1 or 0
end

-- Initialize function
local function init(jbeamData)
    driverNodeCid = beamstate.nodeNameMap["driver"]  -- Set driver's node
    electrics.values.aircon = 0  -- Initial AC state

    sharedNode = beamstate.nodeNameMap[jbeamData.sharedNode or "dsh2"]  -- Assign shared node

    acvolumeSmoother = newTemporalSmoothingNonLinear(2.5)
    acpitchSmoother = newTemporalSmoothingNonLinear(2.5)
    ventvolumeSmoother = newTemporalSmoothingNonLinear(2.5)
	
    chimeVolumeSmoother = newTemporalSmoothingNonLinear(10)  
    chimeOnceVolumeSmoother = newTemporalSmoothingNonLinear(10)  
	alertVolumeSmoother = newTemporalSmoothingNonLinear(10)
end

-- Initialize all sounds
local function initSounds()
    if sharedNode then
        -- Use the shared node for all sounds
        ventsound.sound = obj:createSFXSource("vehicles/segra/sounds/vent.ogg", "AudioCloseLoop3D", "vent", sharedNode)
        acsound.sound = obj:createSFXSource("vehicles/segra/sounds/ac.ogg", "AudioCloseLoop3D", "aircon", sharedNode)
        chimeSound.sound = obj:createSFXSource("vehicles/segra/sounds/beltchime.ogg", "AudioCloseLoop3D", "chime", sharedNode)
        chimeSoundOnce.sound = obj:createSFXSource("vehicles/segra/sounds/beltchime.ogg", "Audio2D", "chimeOnce", sharedNode)
		alertSound.sound = obj:createSFXSource("vehicles/segra/sounds/alertchime.ogg", "AudioCloseLoop3D", "alert", sharedNode)
    end
end

-- Expose public functions and variables
M.init = init
M.initSounds = initSounds
M.updateGFX = updateGFX
M.setacmode = setacmode
M.airconup = airconup
M.aircondown = aircondown
M.isAirConditioningOn = isAirConditioningOn

return M