-- written by DaddelZeit
-- DO NOT USE WITHOUT PERMISSION

local M = {}
M.type = "auxiliary"

local min = math.min
local abs = math.abs

local hasBuiltPie
local engine

local onOff = 1
local saveFilePath = "/temp/vehicles/fullsuv/startstop.json"
local lastGear
local state = { "idle", "stopping", "wait", "starting" }
local currentState = 1
local processRunning = false
local frequencyCalled
local tempDisabled
local timer = 0
local timeSinceStop = 0
local hasCoupler = false
local origStarterValues = {}

local origThresholds = {}
local speedThreshold
local pitchThreshold
local brakeThreshold
local minEngineTemp
local steerThreshold
local idleTimeNeeded
local needThrottle
local useSmartAdaption

-- smart stuff
local brakeTimer = 0 -- from 1 (~5kmh) to 0 / pedal input to allow premature braking
local brakeIterateAmount = 0

local function nextState()
  currentState = math.max((currentState + 1) % 5, 1) -- because lua tables start at 1
end

local function handleStarterSpeed(call)
  if call == "spedup" then
    engine.starterTorque = origStarterValues.starterTorque * 2.5
    engine.starterMaxAV = engine.idleAV / 2
    --engine.starterThrottleKillTime = origStarterValues.starterThrottleKillTime / 2.5
    --engine.idleStartCoef = 0.8
  elseif call == "orig" then
    engine.starterTorque = origStarterValues.starterTorque
    engine.starterMaxAV = origStarterValues.starterMaxAV
    --engine.starterThrottleKillTime = origStarterValues.starterThrottleKillTime
    --engine.idleStartCoef = origStarterValues.idleStartCoef
  end
end

local function saveJson()
  jsonWriteFile(
    saveFilePath,
    {
    onOff = onOff

    },
    true
  )
end

local brakeMeasurementTaken = false
local function applyData()
  idleTimeNeeded = 1.5 - brakeTimer * electrics.values.brake * 3

  speedThreshold = 5 + brakeTimer * 3 + electrics.values.brake
end

local function dataHandling(dt)
  if useSmartAdaption == false then return end

  local evals = electrics.values
  if evals.wheelspeed <= 1.5 then
    if evals.brake ~= 0 and not brakeMeasurementTaken then
      brakeTimer = brakeTimer + dt
    end

    if evals.wheelspeed < 0.05 and not brakeMeasurementTaken then
      brakeIterateAmount = brakeIterateAmount + 1

      applyData()
      brakeMeasurementTaken = true
    end
  else
    brakeMeasurementTaken = false
    brakeTimer = 0
  end
end

local function updateGFXAuto(dt)
  if onOff == 1 or processRunning then -- system is enabled
    dataHandling(dt)

    local evals = electrics.values

    timer = timer + dt
    if timer >= 5 then
      frequencyCalled = 0
      timer = 0
    end

    if frequencyCalled >= 2 then
      tempDisabled = true
    end

    if tempDisabled then
      if evals.wheelspeed > 5 then
        tempDisabled = false
      end
    end

    if evals.watertemp == nil then return end
    local roll, pitch, yaw = obj:getRollPitchYaw()
    if (evals.watertemp <= minEngineTemp) or (pitchThreshold <= pitch) and not processRunning then return end
    if (evals.parkingbrake ~= 0) and not processRunning then return end
    if (hasCoupler) and not processRunning then return end
    if (evals.steering > steerThreshold) and not processRunning then return end
    if (evals.steering < -steerThreshold) and not processRunning then return end
    if ((evals.gear == "N" and lastGear == "N" and evals.gearboxMode == "realistic") or (evals.gear == "P" and lastGear == "P") or (evals.gear == "R" and lastGear == "R") or (evals.gear == "S"..evals.gearIndex and lastGear == "S"..evals.gearIndex) or (evals.gear == "M"..evals.gearIndex and lastGear == "M"..evals.gearIndex)) and not processRunning then lastGear = evals.gear return end
    if tempDisabled and not processRunning then return end

    if state[currentState] == "idle" then
      if evals.wheelspeed <= speedThreshold and evals.brake > brakeThreshold then
        timeSinceStop = timeSinceStop + dt
        if timeSinceStop >= idleTimeNeeded then
          frequencyCalled = frequencyCalled + 1
          nextState()
          electrics.values.startStopActive = 1
          engine:activateStarter()
          processRunning = true
          timeSinceStop = 0
          handleStarterSpeed("spedup")
        end
      else
        handleStarterSpeed("orig")
      end
    elseif state[currentState] == "stopping" then
      if evals.rpm < 50 or evals.throttle ~= 0 then
        engine:disable()
        nextState()
      end
      electrics.values.running = 1
      electrics.values.ignition = 1
      electrics.values.checkengine = 0
      damageTracker.setDamage("engine", "engineDisabled", false)
    elseif state[currentState] == "wait" then
      local isNextState = false
      if needThrottle then
        if evals.brake <= brakeThreshold and evals.throttle ~= 0 then
          engine:enable()
          engine:activateStarter()
          engine:setIgnition(1)
          nextState()
          isNextState = true
        end
      else
        if evals.brake <= brakeThreshold then
          engine:enable()
          engine:activateStarter()
          engine:setIgnition(1)
          nextState()
          isNextState = true
        end
      end
      if (((evals.gear == "N" and evals.gearboxMode == "realistic") or evals.gear == "R" or evals.gear == "P" or evals.gear == "S"..evals.gearIndex or evals.gear == "M"..evals.gearIndex) or tempDisabled) and not isNextState then
        engine:enable()
        engine:activateStarter()
        engine:setIgnition(1)
        nextState()
      end

      electrics.values.running = 1
      electrics.values.ignition = 1
      electrics.values.checkengine = 0
      damageTracker.setDamage("engine", "engineDisabled", false)
    elseif state[currentState] == "starting" then
      if evals.rpm >= (engine.idleAV * 9.549296596425384) - 200 then -- engine is on
        nextState()
        processRunning = false
      end
      electrics.values.startStopActive = 0
      electrics.values.running = 1
      electrics.values.ignition = 1
    end

    lastGear = evals.gear
  end
end

local function updateGFXManual(dt)
  if onOff == 1 or processRunning then -- system is enabled
    local evals = electrics.values
    dataHandling(dt)

    timer = timer + dt
    if timer >= 6 then
      frequencyCalled = 0
      timer = 0
    end

    if frequencyCalled >= 2 then
      tempDisabled = true
    end

    if tempDisabled then
      if evals.wheelspeed > 5 then
        tempDisabled = false
      end
    end

    if evals.watertemp == nil then return end
    local roll, pitch, yaw = obj:getRollPitchYaw()
    if (evals.watertemp <= minEngineTemp) or (pitchThreshold <= pitch) and not processRunning then return end
    if (evals.parkingbrake ~= 0) and not processRunning then return end
    if (hasCoupler) and not processRunning then return end
    if (evals.steering > steerThreshold) and not processRunning then return end
    if (evals.steering < -steerThreshold) and not processRunning then return end
    if (((evals.gear > 0 and lastGear > 0) and evals.gearboxMode == "realistic") or (evals.gear < 0 and lastGear < 0)) and not processRunning then lastGear = evals.gear return end
    if tempDisabled and not processRunning then return end

    if state[currentState] == "idle" then
      if needThrottle then
        if evals.wheelspeed <= speedThreshold and evals.brake > brakeThreshold and evals.clutch == 0 then
          timeSinceStop = timeSinceStop + dt

          if evals.rpm > engine.idleAV then -- engine is on
            handleStarterSpeed("orig")
          end

          if timeSinceStop >= idleTimeNeeded then
            frequencyCalled = frequencyCalled + 1
            nextState()
            engine:activateStarter()
            electrics.values.startStopActive = 1
            processRunning = true
            timeSinceStop = 0
            handleStarterSpeed("spedup")
          end
        end
      else
        if evals.wheelspeed <= speedThreshold and evals.brake > brakeThreshold and evals.rpm >= 500 then
          timeSinceStop = timeSinceStop + dt

          if timeSinceStop >= 3 then
            frequencyCalled = frequencyCalled + 1
            nextState()
            engine:activateStarter()
            processRunning = true
            timeSinceStop = 0
          end
        end
      end
    elseif state[currentState] == "stopping" then
      if evals.rpm < 50 then
        engine:disable()
        nextState()
      end

      electrics.values.running = 1
      electrics.values.ignition = 1
      electrics.values.checkengine = 0
      damageTracker.setDamage("engine", "engineDisabled", false)
    elseif state[currentState] == "wait" then
      local isNextState = false
      if needThrottle then
        if evals.clutch ~= 0 then
          engine:enable()
          engine:activateStarter()
          nextState()
          isNextState = true
        end
        if tempDisabled and not isNextState then
          engine:enable()
          engine:activateStarter()
          nextState()
          isNextState = true
        end
      else
        if evals.brake <= brakeThreshold then
          engine:enable()
          engine:activateStarter()
          nextState()
          isNextState = true
        end
        if ((evals.gear > 0 and evals.gearboxMode == "realistic") or evals.gear < 0 or tempDisabled) and not isNextState then
          engine:enable()
          engine:activateStarter()
          nextState()
          isNextState = true
        end
      end

      electrics.values.running = 1
      electrics.values.ignition = 1
      electrics.values.checkengine = 0
      damageTracker.setDamage("engine", "engineDisabled", false)
    elseif state[currentState] == "starting" then
      if evals.rpm >= (engine.idleAV * 9.549296596425384) - 200 then -- engine is on
        handleStarterSpeed("orig")
        nextState()
        processRunning = false
      end
      electrics.values.running = 1
      electrics.values.ignition = 1
      electrics.values.startStopActive = 0
    end

    lastGear = evals.gear
  end
end

local modeTable = { ["0"] = "off", ["1"] = "on" }
local function setMode(newVal)
  local name = modeTable[tostring(newVal)]
  tempDisabled = false
  frequencyCalled = 0
  onOff = newVal
  saveJson()
  guihooks.message("Start-Stop Automatic: " .. string.sentenceCase(name), 5, "vehicle.startstopauto")
end

local function toggleMode()
  local newVal = 1-(onOff or 0)
  setMode(newVal)
end

local function onCouplerAttached(arg)
  --hasCoupler = true
end

local function onCouplerDetached()
  --hasCoupler = false
end

local function reset()
  electrics.values.startStopActive = 0
end

local function init(jbeamData)
  engine = powertrain.getDevice("mainEngine") or powertrain.getDevicesByType("combustionEngine")[1]
  if not engine then M.toggleMode = nop M.updateGFX = nop M.setMode = nop return end

  local gearbox = powertrain.getDevice("gearbox")
  if gearbox.type == "automaticGearbox" or gearbox.type == "cvtGearbox" or gearbox.type == "dctGearbox" then M.updateGFX = updateGFXAuto lastGear = "N"
  elseif gearbox.type == "manualGearbox" then M.updateGFX = updateGFXManual lastGear = 0
  else M.updateGFX = nil end -- not supported

  speedThreshold = jbeamData.speedThreshold or 2
  pitchThreshold = jbeamData.pitchThreshold or 0.1
  brakeThreshold = jbeamData.brakeThreshold or 0
  minEngineTemp = jbeamData.minEngineTemp or 70
  steerThreshold = jbeamData.speedThreshold or 110
  idleTimeNeeded = jbeamData.idleTimeNeeded or 3
  needThrottle = jbeamData.needThrottle or false
  useSmartAdaption = jbeamData.useSmartAdaption or false

  origThresholds = {
    speedThreshold = jbeamData.speedThreshold or 2,
    pitchThreshold = jbeamData.pitchThreshold or 0.1,
    brakeThreshold = jbeamData.brakeThreshold or 0,
    steerThreshold = jbeamData.speedThreshold or 110,
    idleTimeNeeded = jbeamData.idleTimeNeeded or 3
  }

  origStarterValues = {
    starterTorque = engine.starterTorque,
    starterMaxAV = engine.starterMaxAV,
  }

  local jsonData = jsonReadFile(saveFilePath)
  if jsonData then onOff = jsonData.onOff end

  setMode(onOff or 1)
  currentState = 1

  electrics.values.startStopActive = 0

  if not hasBuiltPie then
    core_quickAccess.addEntry(
      {
        level = "/powertrain/",
        generator = function(entries)
          local entry = {
            title = "Start-Stop Automatic",
            priority = 40,
            icon = "powertrain_engine", -- systems_exhaust-valve
            onSelect = function()
              controller.getControllerSafe("startStopAutomatic").toggleMode()
              return { "reload" }
            end
          }
          if onOff == 1 then entry.color = '#ff6600' end
          table.insert(entries, entry)
        end
      }
    )
  end
  hasBuiltPie = true
end

M.reset = reset
M.onCouplerAttached = onCouplerAttached
M.onCouplerDetached = onCouplerDetached
M.init = init
M.updateGFX = nop -- disable on first start, defined in init()
M.toggleMode = toggleMode
M.setMode = setMode

return M
