-- Panamera 2025 – спойлер из двух частей (FSM + driveModes)
-- В Sport и Sport+ держим спойлер всегда открытым.
-- Работает без inputSource: значения пишутся в electrics.

local M = {}

-- Настройки
local cfg = {
  autoBySpeed      = true,
  speedOpenKmh     = 90,
  speedCloseKmh    = 70,

  baseOpenTime     = 0.9,
  lidOpenTime      = 0.6,

  lidPreOpen       = 0.65, -- сколько открыть крышки прежде чем пойдёт база (0..1)
  basePreClose     = 0.25, -- до какого уровня опустить базу перед сведением крышек (0..1)

  -- Ключи режимов из контроллера driveModes (как в твоём примере)
  sportKey         = "ttSport",
  sportPlusKey     = "ttSport+",

  startOpened      = false
}

-- Состояния автомата
local S = { CLOSED=1, OPENING_LIDS=2, OPENING_BASE=3, OPEN=4, CLOSING_BASE=5, CLOSING_LIDS=6 }

-- Текущее состояние
local st = {
  phase = S.CLOSED,
  wantOpen = false,
  base = 0.0,
  lid  = 0.0
}

local driveModesCtrl -- кэш контроллера режимов

local function ease(x) if x<=0 then return 0 elseif x>=1 then return 1 else return x*x*(3-2*x) end end
local function writeOut()
  local baseOut, lidOut = ease(st.base), ease(st.lid)
  electrics.values.spoiler_base   = baseOut
  electrics.values.spoiler_lid_L  = lidOut
  electrics.values.spoiler_lid_R  = lidOut
  -- дубли на всякий случай
  electrics.values.spoiler        = baseOut
  electrics.values.active_aero    = baseOut
  electrics.values.spoiler_left   = lidOut
  electrics.values.spoiler_right  = lidOut
end
local function setTarget(open) st.wantOpen = open and true or false end

local function onInit()
  if electrics.create then electrics.create("wheelspeed") end
  -- найдём контроллер режимов (если есть)
  if controller and controller.getController then
    driveModesCtrl = controller.getController("driveModes")
  end
  if cfg.startOpened then
    st.phase = S.OPEN; st.base, st.lid = 1,1
  else
    st.phase = S.CLOSED; st.base, st.lid = 0,0
  end
  writeOut()
end

local function onReset()
  st.base, st.lid = 0, 0
  st.phase = S.CLOSED
  setTarget(cfg.startOpened)
  if st.wantOpen then st.phase = S.OPEN; st.base, st.lid = 1,1 end
  writeOut()
end

-- приоритет: режимы → скорость
local function updateTargetByModeAndSpeed()
  -- 1) режимы
  local forceOpen = false
  if not driveModesCtrl and controller and controller.getController then
    driveModesCtrl = controller.getController("driveModes")
  end
  if driveModesCtrl and driveModesCtrl.getCurrentDriveModeKey then
    local mode = driveModesCtrl.getCurrentDriveModeKey()
    if mode == cfg.sportKey or mode == cfg.sportPlusKey then
      forceOpen = true
    end
  end
  if forceOpen then
    st.wantOpen = true
    return
  end

  -- 2) скорость (если разрешено)
  if cfg.autoBySpeed then
    local kmh = (electrics.values.wheelspeed or 0) * 3.6
    if kmh >= cfg.speedOpenKmh then
      st.wantOpen = true
    elseif kmh <= cfg.speedCloseKmh then
      st.wantOpen = false
    end
  end
end

local function advanceVal(cur, target, dt, timeToFull)
  local rate = dt / math.max(0.001, timeToFull)
  cur = cur + (target - cur) * rate * 2
  if cur < 0 then cur = 0 elseif cur > 1 then cur = 1 end
  return cur
end

local function updateGFX(dt)
  updateTargetByModeAndSpeed()

  -- переходы автомата
  if st.phase == S.CLOSED and st.wantOpen then
    st.phase = S.OPENING_LIDS
  elseif st.phase == S.OPEN and not st.wantOpen then
    st.phase = S.CLOSING_BASE
  end

  if st.phase == S.OPENING_LIDS then
    st.lid  = advanceVal(st.lid , 1, dt, cfg.lidOpenTime)
    st.base = advanceVal(st.base, 0, dt, cfg.baseOpenTime)
    if st.lid >= cfg.lidPreOpen - 1e-3 then st.phase = S.OPENING_BASE end

  elseif st.phase == S.OPENING_BASE then
    st.lid  = advanceVal(st.lid , 1, dt, cfg.lidOpenTime)
    st.base = advanceVal(st.base, 1, dt, cfg.baseOpenTime)
    if st.base >= 0.999 and st.lid >= 0.999 then st.phase = S.OPEN end

  elseif st.phase == S.OPEN then
    st.base = advanceVal(st.base, 1, dt, cfg.baseOpenTime)
    st.lid  = advanceVal(st.lid , 1, dt, cfg.lidOpenTime)

  elseif st.phase == S.CLOSING_BASE then
    st.base = advanceVal(st.base, 0, dt, cfg.baseOpenTime)
    st.lid  = advanceVal(st.lid , 1, dt, cfg.lidOpenTime)
    if st.base <= cfg.basePreClose + 1e-3 then st.phase = S.CLOSING_LIDS end

  elseif st.phase == S.CLOSING_LIDS then
    st.lid  = advanceVal(st.lid , 0, dt, cfg.lidOpenTime)
    st.base = advanceVal(st.base, 0, dt, cfg.baseOpenTime)
    if st.base <= 0.001 and st.lid <= 0.001 then st.phase = S.CLOSED end
  end

  writeOut()
end

-- Внешние вызовы по желанию
local function open()  setTarget(true)  end
local function close() setTarget(false) end
local function toggle() setTarget(not st.wantOpen) end

M.onInit    = onInit
M.onReset   = onReset
M.updateGFX = updateGFX
M.open      = open
M.close     = close
M.toggle    = toggle

return M
