-- lua/ge/extensions/lowranceGPS.lua
-- Lowrance-style GPS: live trail + waypoints + per-map named saves (single JSON)

local M = {}

-- ---------- map / relative paths ----------
local REL_ROOT = "lowranceGPS"   -- relative to user path

local function currentLevelName()
  local mf = getMissionFilename() or ""
  return (mf:match("levels/([^/]+)/")) or "unknown"
end

local function mapDir()
  return ("%s/%s/"):format(REL_ROOT, currentLevelName())
end

local function routeFile(name)
  name = name or "default"
  return mapDir() .. name .. ".json"
end

local function ensureDirs()
  if not FS:directoryExists(REL_ROOT) then pcall(function() FS:directoryCreate(REL_ROOT) end) end
  local md = mapDir()
  if not FS:directoryExists(md) then pcall(function() FS:directoryCreate(md) end) end
end

local function saveJson(relPath, tbl)
  ensureDirs()
  local ok, err = pcall(jsonWriteFile, relPath, tbl, true)
  if not ok then log('E','lowranceGPS','jsonWriteFile failed for '..tostring(relPath)..': '..tostring(err)) end
end

local function loadJson(relPath)
  local ok, t = pcall(jsonReadFile, relPath)
  return (ok and type(t) == 'table') and t or {}
end

-- ---------- state ----------
local routeFlat  = {}   -- [x1,y1,x2,y2,...]
local wptsList   = {}   -- { {x=.., y=.., label='..'}, ... }
local currentName = nil -- currently recording route name (for autosave)
local recording  = false
local lastX, lastY
local step2      = 4    -- 2m^2
local maxKts     = 0
local poiIds     = {}
local kbKey      = nil -- currently bound keyboard key (engine-side)
local vkBind     = nil -- Windows virtual-key code expected via onKeyEvent
local lastMapName = nil -- track current map to detect changes
local mapCheckAccum = 0 -- throttle periodic map-change checks

local function flatToObj(flat)
  local out = {}
  for i = 1, #flat, 2 do out[#out+1] = { x = flat[i], y = flat[i+1] } end
  return out
end

-- ---------- UI pushes ----------
local function pushRoute()
  -- Only emit our app-specific event to avoid the engine's thick white trail.
  -- The UI app draws its own colored overlay and handles visibility.
  guihooks.trigger("lowranceGPS.route", { markers = routeFlat })
end

local function pushWpts()
  -- IMPORTANT: use "markers", not "points"
  local payload = { markers = wptsList, color = "#FFD200" }
  -- send only app-specific events to avoid the engine drawing thick white paths
  guihooks.trigger("lowranceGPS.markers",     payload)
  -- also send a route-scoped event so the UI can isolate waypoints per route
  local name = M._currentName or currentName
  if name then
    guihooks.trigger("lowranceGPS.wptsNamed", { name = name, markers = wptsList })
  end
  -- additionally, ask stock navigator to render icon markers (if supported)
  local static = { markers = {} }
  for _, p in ipairs(wptsList) do
    if type(p) == 'table' and p.icon and p.icon ~= '' then
      local iconPath = "/ui/modules/apps/lowranceNav/mikeslowranceicons/"..tostring(p.icon)
      table.insert(static.markers, {
        x = p.x, y = p.y, size = 18,
        image = iconPath, texture = iconPath, sprite = iconPath, icon = iconPath, url = iconPath, path = iconPath,
        w = 18, h = 18
      })
    end
  end
  -- This is safe no-op if the listener ignores missing fields
  pcall(function() guihooks.trigger("NavigationStaticMarkers", static) end)
  -- and try POI provider path for builds that draw POIs but ignore static markers
  local function addPoi(id, x, y, icon, title)
    -- try a few known call patterns; all wrapped in pcall so unknown calls are harmless
    local ok = false
    pcall(function()
      if freeroam_bigMapPoiProvider and freeroam_bigMapPoiProvider.addCustomPoi then
        freeroam_bigMapPoiProvider.addCustomPoi({ id = id, pos = vec3(x or 0, y or 0, 0), icon = icon, title = title or 'WPT' })
        ok = true
      end
    end)
    if not ok then pcall(function()
      if freeroam_bigMapPoiProvider and freeroam_bigMapPoiProvider.addPoi then
        freeroam_bigMapPoiProvider.addPoi({ id = id, x = x, y = y, icon = icon, title = title or 'WPT' })
        ok = true
      end
    end) end
    if not ok then pcall(function()
      if gameplay_markerInteraction and gameplay_markerInteraction.addPoi then
        gameplay_markerInteraction.addPoi({ id = id, x = x, y = y, icon = icon, title = title or 'WPT' })
        ok = true
      end
    end) end
  end
  -- clear previous POIs if we have some
  if #poiIds > 0 then
    pcall(function()
      if freeroam_bigMapPoiProvider and freeroam_bigMapPoiProvider.removePoi then
        for _, id in ipairs(poiIds) do freeroam_bigMapPoiProvider.removePoi(id) end
      elseif gameplay_markerInteraction and gameplay_markerInteraction.removePoi then
        for _, id in ipairs(poiIds) do gameplay_markerInteraction.removePoi(id) end
      end
    end)
    poiIds = {}
  end
  -- add POIs for current icon waypoints
  local idx = 1
  for _, p in ipairs(wptsList) do
    if type(p) == 'table' and p.icon and p.icon ~= '' then
      local id = string.format('lnv_wpt_%s_%s_%d', currentLevelName(), tostring(name or 'route'), idx)
      local iconPath = "/ui/modules/apps/lowranceNav/mikeslowranceicons/"..tostring(p.icon)
      addPoi(id, p.x, p.y, iconPath, p.label or 'WPT')
      poiIds[#poiIds+1] = id
      idx = idx + 1
    end
  end
end

local function clearStaticWpts()
  -- one-shot clear for any prior generic static markers drawn by other sessions
  pcall(function() guihooks.trigger("NavigationStaticMarkers", { markers = {} }) end)
  -- also clear any engine ground-marker route so no thick white line is drawn
  pcall(function() if core_groundMarkers and core_groundMarkers.setPath then core_groundMarkers.setPath(nil) end end)
end

local function pushHud(kts, pos, rotDeg)
  guihooks.trigger("lowranceGPS.hud", {
    airspeed = kts or 0,
    maxAirspeed = maxKts,
    x = pos and pos.x or nil,
    y = pos and pos.y or nil,
    rot = rotDeg
  })
end

-- ---------- API ----------
function M.addWaypointAtPlayer(arg)
  local v = be:getPlayerVehicle(0); if not v then return end
  local p = v:getPosition()
  local label, icon
  if type(arg) == 'table' then
    label = arg.label or arg.name or 'WPT'
    icon  = arg.icon
  else
    label = arg or 'WPT'
  end
  wptsList[#wptsList+1] = { x = p.x, y = p.y, label = label or 'WPT', icon = icon }
  pushWpts()
  if recording and currentName then
    saveJson(routeFile(currentName), {
      route     = routeFlat,
      waypoints = wptsList,
      meta      = { map = currentLevelName(), name = currentName, time = os.time() }
    })
  end
end

function M.startRouteRecord()
  recording = true
  routeFlat = {}; lastX, lastY = nil, nil
  -- start fresh local waypoint list for this (unnamed) run
  wptsList = {}
  M._currentName = nil
  clearStaticWpts()
  -- ensure engine nav path is cleared
  pcall(function() if core_groundMarkers and core_groundMarkers.setPath then core_groundMarkers.setPath(nil) end end)
  pushRoute()
end

-- Start recording and immediately create/update a named save. Autosaves as we add points/WPTs.
function M.startRouteRecordNamed(name)
  name = name or os.date('route_%Y%m%d_%H%M%S')
  currentName = name
  recording = true
  routeFlat = {}; lastX, lastY = nil, nil
  -- start fresh waypoints for this named route
  wptsList = {}
  M._currentName = name
  clearStaticWpts()
  -- ensure engine nav path is cleared
  pcall(function() if core_groundMarkers and core_groundMarkers.setPath then core_groundMarkers.setPath(nil) end end)
  saveJson(routeFile(name), {
    route     = routeFlat,
    waypoints = wptsList,
    meta      = { map = currentLevelName(), name = name, time = os.time() }
  })
  M.list()
  pushRoute()
end

function M.stopRouteRecord()
  recording = false
  -- final autosave if named
  if currentName then
    saveJson(routeFile(currentName), {
      route     = routeFlat,
      waypoints = wptsList,
      meta      = { map = currentLevelName(), name = currentName, time = os.time() }
    })
  end
  pushRoute()
end

-- Reset max airspeed (called from UI)
function M.resetMax()
  maxKts = 0
  pushHud(0)
end

-- Engine-side keyboard binding (global, not UI focus dependent)
local function unbindKeyboard()
  pcall(function()
    if not kbKey then return end
    if core_input_actionMap and core_input_actionMap.bindCmd then
      core_input_actionMap.bindCmd('keyboard', kbKey, '', '')
    elseif actionMap and actionMap.bindCmd then
      actionMap.bindCmd('keyboard', kbKey, '', '')
    end
  end)
  kbKey = nil
end

function M.bindKeyboardKey(key)
  -- key: string like 'b', 'space', '1', etc.
  if type(key) ~= 'string' or key == '' then return end
  unbindKeyboard()
  kbKey = key
  -- Bind to add waypoint on press
  pcall(function()
    local make = 'be:queueLuaCommand("extensions.lowranceGPS._hotkey()"); lua extensions.lowranceGPS._hotkey(); Lua:extensions.lowranceGPS._hotkey();'
    if core_input_actionMap and core_input_actionMap.bindCmd then
      core_input_actionMap.bindCmd('keyboard', kbKey, make, '')
    elseif actionMap and actionMap.bindCmd then
      actionMap.bindCmd('keyboard', kbKey, make, '')
    end
  end)
end

function M._hotkey()
  -- Drop WPT only while recording, else ignore
  if recording then M.addWaypointAtPlayer('WPT') end
end

-- Direct VK binding from UI; avoids input-map complexity
function M.setVk(code)
  if type(code) == 'number' then vkBind = code end
end

-- Engine keyboard callback (global). Only used if vkBind is set.
local function onKeyEvent(virtualKey, modifiers, eventType)
  -- eventType: 1 down, 2 up
  if vkBind and eventType == 1 and virtualKey == vkBind then
    M._hotkey()
    return true
  end
end

-- single JSON (route + waypoints) per name
function M.saveNamed(name)
  name = name or "default"
  saveJson(routeFile(name), {
    route     = routeFlat,
    waypoints = wptsList,
    meta      = { map = currentLevelName(), name = name, time = os.time() }
  })
  M.list()
end

function M.loadNamed(name)
  name = name or "default"
  ensureDirs()
  local t = loadJson(routeFile(name))
  routeFlat = t.route or {}
  do
    local w = t.waypoints or {}
    if #w > 0 and type(w[1]) == 'table' then
      wptsList = w
    else
      wptsList = flatToObj(w)
    end
  end
  lastX, lastY = nil, nil
  M._currentName = name
  pushRoute(); pushWpts()
end

function M.save() M.saveNamed("default") end
function M.load() M.loadNamed("default") end
function M.deleteNamed(name)
  name = name or "default"
  ensureDirs()
  local p = routeFile(name)
  if FS:fileExists(p) then pcall(function() FS:removeFile(p) end) end
  -- if the deleted route is currently loaded, clear current buffers
  if name == (M._currentName or "") then
    routeFlat = {}
    wptsList  = {}
    lastX, lastY = nil, nil
    pushRoute(); pushWpts()
  end
  M.list()
end

-- list routes for dropdown
function M.list()
  ensureDirs()
  local files = FS:findFiles(mapDir(), "*.json", 0, false) or {}
  local names = {}
  for _, p in ipairs(files) do
    local n = p:match("([^/\\]+)%.json$") or p
    names[#names+1] = n
  end
  table.sort(names)
  guihooks.trigger("lowranceGPS.routesList", { map = currentLevelName(), routes = names })
end

-- Reads a named save and sends route only (for preview/overlay); does not mutate state
function M.peekNamed(name)
  name = name or "default"
  ensureDirs()
  local t = loadJson(routeFile(name))
  local rf = (t and t.route) or {}
  guihooks.trigger("lowranceGPS.route", { markers = rf })
end

-- Reads a named save and sends its waypoints only; does not mutate state
function M.peekNamedWpts(name)
  name = name or "default"
  ensureDirs()
  local t = loadJson(routeFile(name)) or {}
  local w = t.waypoints or {}
  if #w > 0 and type(w[1]) ~= 'table' then w = flatToObj(w) end
  guihooks.trigger("lowranceGPS.wptsNamed", { name = name, markers = w })
end

-- Alias to fetch only the route (kept for UI expectations)
function M.peekNamedRoute(name)
  M.peekNamed(name)
end

-- Helper: refresh UI/state when the active map changes
local function refreshForMapChange()
  -- Clear any static engine markers and path
  pcall(clearStaticWpts)
  -- Reset recording/session state so UI doesn't keep old overlays
  recording   = false
  currentName = nil
  M._currentName = nil
  routeFlat = {}
  wptsList  = {}
  lastX, lastY = nil, nil
  -- Push empty payloads so UI resets and then list routes for the new map
  pushRoute()
  pushWpts()
  M.list()
  M.listWptIcons()
  -- Ask stock UI navigator to (re)send the dashboard map to UI
  pcall(function()
    if extensions and extensions.ui_uiNavi and extensions.ui_uiNavi.requestUIDashboardMap then
      extensions.ui_uiNavi.requestUIDashboardMap()
    end
  end)
  -- Also prompt POI/minimap markers for builds that require explicit resend
  pcall(function()
    if freeroam_bigMapPoiProvider and freeroam_bigMapPoiProvider.sendMissionLocationsToMinimap then
      freeroam_bigMapPoiProvider.sendMissionLocationsToMinimap()
    end
  end)
end

-- ---------- tick ----------
local lastRotDeg = 0
local function onUpdate(dt)
  local v = be:getPlayerVehicle(0); if not v then return end
  local vel = v:getVelocity()
  local kts = (vel and vel:length() or 0) * 1.94384
  if kts > maxKts then maxKts = kts end
  local p = v:getPosition() or { x = 0, y = 0 }
  -- Heading from vehicle forward vector (independent of camera and speed)
  local dir = v.getDirectionVector and v:getDirectionVector() or nil
  if dir then
    lastRotDeg = math.deg(math.atan2(dir.y or 0, dir.x or 0))
  else
    -- as a fallback, approximate from velocity when moving
    if vel and (math.abs(vel.x) + math.abs(vel.y)) > 0.05 then
      lastRotDeg = math.deg(math.atan2(vel.y or 0, vel.x or 0))
    end
  end
  pushHud(kts, p, lastRotDeg)

  -- Ensure engine nav path stays cleared so no thick white route is drawn by stock UI
  pcall(function()
    if core_groundMarkers and core_groundMarkers.currentlyHasTarget and core_groundMarkers.currentlyHasTarget() then
      if core_groundMarkers.setPath then core_groundMarkers.setPath(nil) end
    end
  end)

  if recording then
    local p = v:getPosition()
    local x, y = p.x, p.y
    if not lastX then
      routeFlat[#routeFlat+1] = x; routeFlat[#routeFlat+1] = y
      lastX, lastY = x, y
      pushRoute()
      if currentName then saveJson(routeFile(currentName), { route = routeFlat, waypoints = wptsList, meta = { map = currentLevelName(), name = currentName, time = os.time() } }) end
    else
      local dx, dy = x-lastX, y-lastY
      if (dx*dx + dy*dy) >= step2 then
        routeFlat[#routeFlat+1] = x; routeFlat[#routeFlat+1] = y
        lastX, lastY = x, y
        pushRoute()
        if currentName then saveJson(routeFile(currentName), { route = routeFlat, waypoints = wptsList, meta = { map = currentLevelName(), name = currentName, time = os.time() } }) end
      end
    end
  end

  -- Fallback: periodically detect map change and refresh UI
  mapCheckAccum = (mapCheckAccum or 0) + (dt or 0)
  if mapCheckAccum > 1.0 then
    mapCheckAccum = 0
    local now = currentLevelName()
    if not lastMapName then lastMapName = now end
    if now ~= lastMapName then
      lastMapName = now
      refreshForMapChange()
    end
  end
end

local function onExtensionLoaded()
  ensureDirs()
  M.list()
  M.listWptIcons()
  lastMapName = currentLevelName()
end
-- also clear any old static markers when loading the extension UI
pcall(clearStaticWpts)
-- and make sure any engine route is cleared on load
pcall(function() if core_groundMarkers and core_groundMarkers.setPath then core_groundMarkers.setPath(nil) end end)

-- Register a Controls-menu action so users can bind a key the stock way
local function registerControlsAction()
  pcall(function()
    local action = 'lowranceGPS_add_wpt'
    local title  = 'GPS: Add Waypoint'
    local group  = 'Lowrance GPS'
    if core_input_actionMap and core_input_actionMap.registerAction then
      core_input_actionMap.registerAction(action, title, group)
    elseif actionMap and actionMap.registerAction then
      actionMap.registerAction(action, title, group)
    end
  end)
end

registerControlsAction()

-- Handle input action events from the engine (Controls binding)
local function onAction(action, value, actionData)
  if action == 'lowranceGPS_add_wpt' and (value == 1 or value == true) then
    M._hotkey()
  end
end

-- Discover waypoint icon svgs in the UI folder and send to UI
function M.listWptIcons()
  local base = 'ui/modules/apps/lowranceNav/mikeslowranceicons/'
  local files = {}
  pcall(function()
    local found = FS:findFiles(base, '*.svg', 0, false) or {}
    for _, p in ipairs(found) do
      local name = p:match('([^/\\]+)$') or p
      files[#files+1] = name
    end
    table.sort(files)
  end)
  guihooks.trigger('lowranceGPS.iconsList', { base = '/'..base, icons = files })
end

-- Clear current UI-visible route and waypoints and any static markers.
function M.clearUiWpts()
  recording   = false
  currentName = nil
  M._currentName = nil
  routeFlat = {}
  wptsList  = {}
  lastX, lastY = nil, nil
  -- remove any POIs added previously
  pcall(function()
    if poiIds and #poiIds > 0 then
      if freeroam_bigMapPoiProvider and freeroam_bigMapPoiProvider.removePoi then
        for _, id in ipairs(poiIds) do freeroam_bigMapPoiProvider.removePoi(id) end
      elseif gameplay_markerInteraction and gameplay_markerInteraction.removePoi then
        for _, id in ipairs(poiIds) do gameplay_markerInteraction.removePoi(id) end
      end
    end
  end)
  poiIds = {}
  -- Clear engine static markers and any engine path
  pcall(function() guihooks.trigger("NavigationStaticMarkers", { markers = {} }) end)
  pcall(function() if core_groundMarkers and core_groundMarkers.setPath then core_groundMarkers.setPath(nil) end end)
  -- Push empty payloads so UI resets its caches/overlays
  pushRoute()
  pushWpts()
end

-- Add waypoint at explicit world coordinates (from UI crosshair)
function M.addWaypointAt(arg)
  if type(arg) ~= 'table' then return end
  local x = tonumber(arg.x) or 0
  local y = tonumber(arg.y) or 0
  local label = arg.label or arg.name or 'WPT'
  local icon  = arg.icon
  wptsList[#wptsList+1] = { x = x, y = y, label = label, icon = icon }
  pushWpts()
  if recording and currentName then
    saveJson(routeFile(currentName), {
      route     = routeFlat,
      waypoints = wptsList,
      meta      = { map = currentLevelName(), name = currentName, time = os.time() }
    })
  end
end

-- Mission lifecycle hooks: refresh UI/routes when switching maps
local function onClientStartMission(mission)
  local now = currentLevelName()
  if now ~= lastMapName then
    lastMapName = now
    refreshForMapChange()
  end
end

local function onClientEndMission()
  -- Clear UI overlays when leaving a map
  pcall(clearStaticWpts)
  pushRoute()
  pushWpts()
  lastMapName = nil
end

M.onUpdate              = onUpdate
M.onExtensionLoaded     = onExtensionLoaded
M.onClientStartMission = onClientStartMission
M.onClientEndMission    = onClientEndMission
M.addWaypointAtPlayer   = M.addWaypointAtPlayer
M.addWaypointAt         = M.addWaypointAt
M.startRouteRecord      = M.startRouteRecord
M.startRouteRecordNamed = M.startRouteRecordNamed
M.stopRouteRecord       = M.stopRouteRecord
M.save                  = M.save
M.load                  = M.load
M.saveNamed             = M.saveNamed
M.loadNamed             = M.loadNamed
M.peekNamed             = M.peekNamed
M.peekNamedWpts         = M.peekNamedWpts
M.peekNamedRoute        = M.peekNamedRoute
M.deleteNamed           = M.deleteNamed
M.resetMax              = M.resetMax
M.bindKeyboardKey       = M.bindKeyboardKey
M.unbindKeyboard        = M.unbindKeyboard
M.setVk                 = M.setVk
M.listWptIcons          = M.listWptIcons
M.clearUiWpts          = M.clearUiWpts
M.list                  = M.list

return M



