--[[
This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
If a copy of the bCDDL was not distributed with this
file, You can obtain one at http://beamng.com/bCDDL-1.1.txt
This module contains a set of functions which manipulate behaviours of vehicles.
]]

-- this function provides various construction functions for wheels

-- modified "addPressureWheels" to "addPressureWheelDiv" by fillman86. If the "high polly wheels" part isn't on, then it'll go to default construction
-- Otherwise it splits to the new version, with sidewalls, and treat subdivision

local M = {}

local jbeamUtils = require("jbeam/utils")

local jbeamWheels = require('jbeam/sections/wheels')
local defaultProcessWheels = jbeamWheels.processWheels
local wheelsToRotators = {}

-- these are defined in C, do not change the values
local NORMALTYPE = 0
local NONCOLLIDABLE = 2
local BEAM_ANISOTROPIC = 1
local BEAM_PRESSURED = 3
local BEAM_LBEAM = 4
local BEAM_SUPPORT = 7

local function cleanupWheelOptions(options)
  options.hubBeamSpring = nil
  options.hubBeamDamp = nil
  options.hubBeamDampCutoffHz = nil
  options.hubBeamDeform = nil
  options.hubBeamStrength = nil
  options.hubNodeWeight = nil
  options.hubCollision = nil
  options.hubNodeMaterial = nil
  options.hubFrictionCoef = nil
  options.hubGroup = nil
  options.disableHubMeshBreaking = nil
  options.hubSideBeamSpring = nil
  options.hubSideBeamDamp = nil
  options.hubSideBeamDeform = nil
  options.hubSideBeamStrength = nil
  options.hubSideBeamDampCutoffHz = nil
  options.hubReinfBeamSpring = nil
  options.hubReinfBeamDamp = nil
  options.hubReinfBeamDeform = nil
  options.hubReinfBeamStrength = nil
  options.hubReinfBeamDampCutoffHz = nil
  options.hubTreadBeamSpring = nil
  options.hubTreadBeamDamp = nil
  options.hubTreadBeamDeform = nil
  options.hubTreadBeamStrength = nil
  options.hubTreadBeamDampCutoffHz = nil
  options.hubPeripheryBeamSpring = nil
  options.hubPeripheryBeamDamp = nil
  options.hubPeripheryBeamDeform = nil
  options.hubPeripheryBeamStrength = nil
  options.hubPeripheryBeamDampCutoffHz = nil
  options.hubStabilizerBeamSpring = nil
  options.hubStabilizerBeamDamp = nil
  options.hubStabilizerBeamDeform = nil
  options.hubStabilizerBeamStrength = nil
  options.hubcapBeamSpring = nil
  options.hubcapBeamDamp = nil
  options.hubcapBeamDeform = nil
  options.hubcapBeamStrength = nil
  options.hubcapNodeWeight = nil
  options.hubcapCollision = nil
  options.hubcapNodeMaterial = nil
  options.hubcapFrictionCoef = nil
  options.hubcapGroup = nil
  options.disableHubcapMeshBreaking = nil
  options.hubcapAttachBeamSpring = nil
  options.hubcapAttachBeamDamp = nil
  options.hubcapAttachBeamDeform = nil
  options.hubcapAttachBeamStrength = nil
  options.hubcapBreakGroup = nil
  options.wheelSideBeamSpring = nil
  options.wheelSideBeamDamp = nil
  options.wheelSideBeamDeform = nil
  options.wheelSideBeamStrength = nil
  options.wheelSideBeamSpringExpansion = nil
  options.wheelSideBeamDampExpansion = nil
  options.wheelSideTransitionZone = nil
  options.wheelSideBeamPrecompression = nil
  options.wheelSideReinfBeamSpring = nil
  options.wheelSideReinfBeamDamp = nil
  options.wheelSideReinfBeamDeform = nil
  options.wheelSideReinfBeamStrength = nil
  options.wheelSideReinfBeamSpringExpansion = nil
  options.wheelSideReinfBeamDampExpansion = nil
  options.wheelSideReinfTransitionZone = nil
  options.wheelSideReinfBeamPrecompression = nil
  options.wheelReinfBeamSpring = nil
  options.wheelReinfBeamDamp = nil
  options.wheelReinfBeamSpringExpansion = nil
  options.wheelReinfBeamDampExpansion = nil
  options.wheelReinfBeamDeform = nil
  options.wheelReinfBeamStrength = nil
  options.wheelReinfBeamPrecompression = nil
  options.wheelReinfBeamDampCutoffHz = nil
  options.wheelTreadBeamSpring = nil
  options.wheelTreadBeamDamp = nil
  options.wheelTreadBeamDeform = nil
  options.wheelTreadBeamStrength = nil
  options.wheelTreadBeamPrecompression = nil
  options.wheelTreadBeamDampCutoffHz = nil
  options.wheelTreadReinfBeamSpring = nil
  options.wheelTreadReinfBeamDamp = nil
  options.wheelTreadReinfBeamDeform = nil
  options.wheelTreadReinfBeamStrength = nil
  options.wheelTreadReinfBeamPrecompression = nil
  options.wheelTreadReinfBeamDampCutoffHz = nil
  options.wheelPeripheryBeamSpring = nil
  options.wheelPeripheryBeamDamp = nil
  options.wheelPeripheryBeamDeform = nil
  options.wheelPeripheryBeamStrength = nil
  options.wheelPeripheryBeamPrecompression = nil
  options.wheelPeripheryBeamDampCutoffHz = nil
  options.wheelPeripheryReinfBeamSpring = nil
  options.wheelPeripheryReinfBeamDamp = nil
  options.wheelPeripheryReinfBeamDeform = nil
  options.wheelPeripheryReinfBeamStrength = nil
  options.wheelPeripheryReinfBeamPrecompression = nil
  options.wheelPeripheryReinfBeamDampCutoffHz = nil
  options.hubRadius = nil
  options.hubWidth = nil
  options.tireWidth = nil
  options.name = nil
  options.wheelOffset = nil
  options.radius = nil
  options.rotorMaterial = nil
  options.numRays = nil
  options.torqueArm = nil
  options.torqueArm2 = nil
  options.wheelDir = nil
  options.padMaterial = nil
  options.pressurePSI = nil
  options.parkingTorque = nil
  options.steerAxisDown = nil

  return options
end

local function cleanupBeamOptions(options)
  options.selfCollision = nil
  options.frictionCoef = nil
  options.loadSensitivitySlope = nil
  options.treadCoef = nil
  options.fullLoadCoef = nil
  options.collision = nil
  options.slidingFrictionCoef = nil

  return options
end

local function addPressTri(tris, presGroup, presPSI, n1, n2, n3, dCoef, tType, sDragCoef)
  tableInsertC(tris, {
      id1 = n1, id2 = n2, id3 = n3,
      dragCoef = dCoef, triangleType = tType,
      pressureGroup = presGroup, pressurePSI = presPSI, skinDragCoef = sDragCoef
    })
end

local function addTri(tris, n1, n2, n3, dCoef, tType)
  tableInsertC(tris, {
      id1 = n1, id2 = n2, id3 = n3,
      dragCoef = dCoef, triangleType = tType
    })
end

local allBeams = {}
local function dupeCheck(beamGroup,b)
  local checker = false
  for i=1, #allBeams do
    if allBeams[i] == b then
      print("duplicate found")
      checker = true
    end
  end
  if checker then
    print("duplicate= " .. b.node1 .. " == " .. b.node2)
  else
    table.insert(allBeams, b)
    table.insert(beamGroup,b)
  end
end

--MARK:add wheels
local function addPressureWheelDiv(vehicle, wheelKey, wheel)  
  local node1   = vehicle.nodes[wheel.node1]
  local node2   = vehicle.nodes[wheel.node2]
  local node1_pos = vec3(node1.pos)
  local node2_pos = vec3(node2.pos)
  
  local treadDiv      = wheel.treadDiv      or (node1_pos.y > 0 and vehicle.variables['$raasubDiv'].val           or  vehicle.variables['$faasubDiv'].val         ) or 2       
  local bulgeOffset   = wheel.bulgeOffset   or (node1_pos.y > 0 and vehicle.variables['$rbulgeOffset'].val        or  vehicle.variables['$fbulgeOffset'].val      ) or 0       
  local sideWallReinf = wheel.sideWallReinf or (node1_pos.y > 0 and vehicle.variables['$rsideWallReinf'].val      or  vehicle.variables['$fsideWallReinf'].val    ) or 0.25    
  local antiBulge     = wheel.antiBulge     or (node1_pos.y > 0 and vehicle.variables['$rantiBulge'].val          or  vehicle.variables['$fantiBulge'].val        ) or 0       
  local antiSquish    = wheel.antiSquish    or (node1_pos.y > 0 and vehicle.variables['$rantiSquish'].val         or  vehicle.variables['$fantiSquish'].val       ) or 0       
  local xPat          = wheel.xPat          or (node1_pos.y > 0 and vehicle.variables['$rxPat'].val               or  vehicle.variables['$fxPat'].val             ) or 0.02    
  local fullStif      = wheel.fullStif      or (node1_pos.y > 0 and vehicle.variables['$rafullStif'].val          or  vehicle.variables['$fafullStif'].val        ) or 1       
  local pAntiRein     = wheel.pAntiRein     or (node1_pos.y > 0 and vehicle.variables['$rpAntiRein'].val          or  vehicle.variables['$fpAntiRein'].val        ) or 0.8     
  local pCornerL      = wheel.pCornerL      or (node1_pos.y > 0 and vehicle.variables['$rpCornerL'].val * 5       or  vehicle.variables['$fpCornerL'].val * 5     ) or 0.1     
  local newBulge      = wheel.newBulge      or (node1_pos.y > 0 and vehicle.variables['$rbulge'].val              or  vehicle.variables['$fbulge'].val            ) or 0       
  local xxFricComp    = wheel.xxFricComp    or (node1_pos.y > 0 and vehicle.variables['$rxxFricComp'].val/100     or  vehicle.variables['$fxxFricComp'].val/100   ) or 1       
  local xxSFricComp   = wheel.xxSFricComp   or (node1_pos.y > 0 and vehicle.variables['$rxxSFricComp'].val/100    or  vehicle.variables['$fxxSFricComp'].val/100  ) or 1       
  local newNumRay     = vehicle.variables['$rnumRays'] and (node1_pos.y > 0 and vehicle.variables['$rnumRays'].val or  vehicle.variables['$fnumRays'].val         ) or wheel.numRays
  local tyreWeight    = node1_pos.y > 0 and vehicle.variables['$raatreadWeight'].val/100  or  vehicle.variables['$faatreadWeight'].val/100
  local tyreWall      =  wheel.radius - wheel.hubRadius
  local tyreWallMiddle= (wheel.radius + wheel.hubRadius) / 2
  local tyreRatio     = (wheel.radius - wheel.hubRadius) / wheel.tireWidth
  
  -- Stabilizer
  wheel.nodeStabilizer = wheel.nodeStabilizer or wheel.nodeS
  wheel.treadCoef = wheel.treadCoef or 1
  wheel.nodeS = nil
  local nodeStabilizerExists = false
  local wheelAngleRad = math.rad(wheel.wheelAngle or 0)

  if wheel.nodeStabilizer and wheel.nodeStabilizer ~= 9999 and vehicle.nodes[wheel.nodeStabilizer] then
    nodeStabilizerExists = true
  else
    wheel.nodeStabilizer = nil
  end

  if node1 == nil or node2 == nil then
    log('W', "jbeam.addPressureWheel","invalid pressureWheel")
    return
  end

  local nodebase = vehicle.maxIDs.nodes
  --print("testing out vehicle.maxIDs.nodes- ".. tostring(vehicle.maxIDs.nodes) .. " -555555555555555555555555555")

  local tireExists = true
  if wheel.radius == nil then
    tireExists = false
    wheel.radius = 0.5
  end

  if wheel.pressurePSI == nil then
    tireExists = false
  end

  if wheel.hasTire ~= nil and wheel.hasTire == false then
    tireExists = false
  end

  if wheel.hubRadius == nil then wheel.hubRadius = 0.65 * wheel.radius end
  if newNumRay == nil then newNumRay = 10 end
  --newNumRay = 24

  -- add collision to the wheels nodes ;)
  wheel.collision = true

  -- calculate surface
  -- if tireExists and wheel.tireWidth and wheel.hubWidth and wheel.radius and wheel.hubRadius then
  --   wheel._surface = math.pi * (2 * (wheel.tireWidth * wheel.radius + wheel.hubWidth * wheel.hubRadius) +
  --     (wheel.radius + wheel.hubRadius) * math.sqrt(square(wheel.radius - wheel.hubRadius) + square(wheel.tireWidth - wheel.hubRadius)))
  -- end

  local middleNodes = {}

  local node4_pos = vec3(node2.pos)
  local node5_pos = vec3(node2.pos)

  local tyreBulge = newBulge or 0

  -- calculate axis
  local axis = node2_pos - node1_pos
  axis:normalize()

  local midpoint = (node2_pos + node1_pos) * 0.5
  if wheel.wheelOffset ~= nil then
    local offset = wheel.wheelOffset
    midpoint = midpoint + axis * offset
  end

  --MARK: node pos
  --this is lateral placement only, 3D is in the marked tyre section
  if wheel.tireWidth ~= nil then
    local halfWidth = 0.5 * wheel.tireWidth
    node1_pos = midpoint - axis * halfWidth
    node2_pos = midpoint + axis * halfWidth
    node4_pos = midpoint - axis * (wheel.tireWidth * ((tyreWall * 0.5) + 0.5 + tyreBulge))--sidewalls
    node5_pos = midpoint + axis * (wheel.tireWidth * ((tyreWall * 0.5) + 0.5 + tyreBulge))--sidewalls

    --calcualting band separation
    local postions = {}
    if treadDiv > 0 then
      local dividedWidth = wheel.tireWidth/(treadDiv+1)
      for i=1, (treadDiv/2) do --the "for" loop is to increasingly add more offset divisions
        local step = i/(treadDiv+1)
        local curMaths = ((1-step)^ tyreRatio) * wheel.tireWidth
        local evenMaths = wheel.tireWidth - (dividedWidth*i)
        local averageMaths = (curMaths + evenMaths*i)/((1+i))
        local currentPos = node2_pos - axis * averageMaths --position incrementally calculation, like normal nodes, except from opposite edge
        table.insert(middleNodes, currentPos)--table used for placing nodes
        table.insert(postions, averageMaths)--table for mirror purposes
      end
      local curLen = #postions --this is to order the new nodes into the right place
      if treadDiv % 2 ~= 0 then --if it's odd, then add a centre line
        local currentPos = midpoint - axis * 0
        table.insert(middleNodes, currentPos)--table used for placing nodes
        curLen = curLen + 1 --we need to make sure that the following nodes are after this centre node
      end
      for i = 1, (treadDiv/2) do --now we copy our homework from the first half
        local currentPos = node1_pos + axis * postions[i]
        table.insert(middleNodes, curLen+1, currentPos) --we add the nodes, but due to the order of the maths, we insert each new one from the centre(ish)
      end
    end    
  end

  local cleanwheel = deepcopy(wheel)
  cleanwheel.axleBeams = nil
  cleanwheel.childParts = nil
  cleanwheel.slotType = nil
  cleanwheel.enableABS = nil
  cleanwheel.enableBrakeThermals = nil
  cleanwheel.enableHubcaps = nil
  cleanwheel.enableTireLbeams = nil
  cleanwheel.enableTirePeripheryReinfBeams = nil
  cleanwheel.enableTireReinfBeams = nil
  cleanwheel.enableTireSideReinfBeams = nil
  cleanwheel.enableTreadReinfBeams = nil
  cleanwheel.hasTire = nil
  cleanwheel.brakeDiameter = nil
  cleanwheel.brakeInputSplit = nil
  cleanwheel.brakeMass = nil
  cleanwheel.brakeSplitCoef = nil
  cleanwheel.brakeSpring = nil
  cleanwheel.brakeTorque = nil
  cleanwheel.brakeType = nil
  cleanwheel.brakeVentingCoef = nil
  cleanwheel.heatCoefNodeToEnv = nil
  cleanwheel.heatCoefEnvMultStationary = nil
  cleanwheel.heatCoefEnvTerminalSpeed = nil
  cleanwheel.heatCoefNodeToCore = nil
  cleanwheel.heatCoefCoreToNodes = nil
  cleanwheel.heatCoefNodeToSurface = nil
  cleanwheel.heatCoefFriction = nil
  cleanwheel.heatCoefFlashFriction = nil
  cleanwheel.heatCoefStrain = nil
  cleanwheel.heatAffectsPressure = nil
  cleanwheel.smokingTemp = nil
  cleanwheel.meltingTemp = nil
  cleanwheel.frictionLowTemp = nil
  cleanwheel.frictionHighTemp = nil
  cleanwheel.frictionLowSlope = nil
  cleanwheel.frictionHighSlope = nil
  cleanwheel.frictionSlopeSmoothCoef = nil
  cleanwheel.frictionCoefLow = nil
  cleanwheel.frictionCoefMiddle = nil
  cleanwheel.frictionCoefHigh = nil

  local wheelNodes = cleanupWheelOptions(deepcopy(cleanwheel))
  --edited in by fillman86 ************************************************************** V
  --modify friction coefficients
  --rough napkin maths, on how many nodes probably contact the ground, at any given moment, above the standard 3 nodes
  local rayNodeCompensator  = math.max(1,math.floor(newNumRay/8))--more numRays means more nodes contacting the ground
  local divNodeCompensator  = ((treadDiv * rayNodeCompensator) + (rayNodeCompensator-1))/4--how many times more nodes contacting, vs default 3 nodes
  print("divNodeCompensator- "..divNodeCompensator)
  wheelNodes.frictionCoef   = (wheelNodes.frictionCoef / divNodeCompensator) * xxFricComp
  --set sidewall friction value
  local defaultFricCo       = wheelNodes.frictionCoef
  local sideWallFricCo      = wheelNodes.frictionCoef * 0.7
  wheelNodes.slidingFrictionCoef = wheelNodes.slidingFrictionCoef * xxSFricComp
  
  --if it's a wide tyre, we'll at least 'try' to decrease the weight a little, but not too much, it'll become unstable
  if wheel.tireWidth > 0.130 and treadDiv > 0 then
    wheelNodes.nodeWeight = (((wheelNodes.nodeWeight*treadDiv*2) + (wheelNodes.nodeWeight / treadDiv))/((treadDiv*2)+1))*tyreWeight
  end

  --this is used to mirror the node pattern for each side
  local inverse = 1
  if axis.x >= 0 then
    inverse = -1
  end

  local rayRotSWO   = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  local rayRotSWI   = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2.6666667 * inverse))
  local rayRotSWI1st= quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 8 * -inverse))
  local rayRot
  if treadDiv > 0 then
    rayRot      = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * ((treadDiv + 2) * inverse)))
  else
    rayRot      = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  end

  local rayVec
  local rayVecSWO
  local rayVecSWI
  local treadNodes = {}
  local treadBands = {}
  local treadNodePos = {}
  if tireExists then
    local oppCount
    if treadDiv > 0 then
      oppCount = newNumRay * ((treadDiv + 4)/2)
   else
      oppCount = newNumRay * 2
    end
    rayVec = axis:perpendicularN() * wheel.radius
    rayVec = quatFromAxisAngle(axis, -wheelAngleRad) * rayVec
    rayVecSWO = axis:perpendicularN() * math.min(wheel.radius, math.max(wheel.hubRadius, tyreWallMiddle + bulgeOffset))
    rayVecSWO = quatFromAxisAngle(axis, -wheelAngleRad) * rayVecSWO

    rayVecSWI = rayRotSWI1st * rayVecSWO
    local hasEvenRayCount = newNumRay % 2 == 0
    --MARK: Tyres
    -- add nodes first
    for i = 0, newNumRay - 1, 1 do
      wheelNodes.frictionCoef = sideWallFricCo
      local currentBand = {}
      local currentNodePos = {}
      local rayPoint = node1_pos + rayVec
      -- outer sidewall
      rayVecSWO = rayRotSWO * rayVecSWO
      rayPoint = node4_pos + rayVecSWO
      local n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos

      -- outer corner
      wheelNodes.frictionCoef = defaultFricCo
      rayPoint = node1_pos + rayVec
      rayVec = rayRot * rayVec
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
        vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
      end

      --add middle nodes from table
      if treadDiv > 0 then
        for NodeCount,nodePos in ipairs(middleNodes) do -- start putting in the middle nodes
          rayPoint = nodePos + rayVec
          rayVec = rayRot * rayVec
          n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
          table.insert(treadNodes, vehicle.nodes[n])
          table.insert(currentBand,vehicle.nodes[n])
          currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
          --[[ if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
            vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
          end ]]
        end
      end

      -- inner corner
      rayPoint = node2_pos + rayVec
      rayVec = rayRot * rayVec
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
        vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
      end

      -- inner sidewall
      wheelNodes.frictionCoef = sideWallFricCo
      rayPoint = node5_pos + rayRotSWI * rayVecSWI
      rayVecSWI = rayRotSWI * rayVecSWO
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes,  vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      rayVecSWO = rayRotSWO * rayVecSWO
      table.insert(treadBands, currentBand)
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      table.insert(treadNodePos, currentNodePos)
      currentNodePos = nil
      currentBand = nil
    end --end of: edited by  added by fillman86 *************************************************** ^
  end
  wheelNodes.frictionCoef = defaultFricCo

  -- add Hub nodes
  local hubNodes = {}
  local hubnodebase = vehicle.maxIDs.nodes

  local hubOptions = deepcopy(cleanwheel)
  hubOptions.beamSpring = hubOptions.hubBeamSpring or hubOptions.beamSpring
  hubOptions.beamDamp = hubOptions.hubBeamDamp or hubOptions.beamDamp
  hubOptions.beamDeform = hubOptions.hubBeamDeform or hubOptions.beamDeform
  hubOptions.beamStrength = hubOptions.hubBeamStrength or hubOptions.beamStrength
  hubOptions.dampCutoffHz = hubOptions.hubBeamDampCutoffHz or nil
  hubOptions.nodeWeight = hubOptions.hubNodeWeight or hubOptions.nodeWeight
  hubOptions.collision = hubOptions.hubCollision or hubOptions.collision
  hubOptions.nodeMaterial = hubOptions.hubNodeMaterial or hubOptions.nodeMaterial
  hubOptions.frictionCoef = hubOptions.hubFrictionCoef or hubOptions.frictionCoef
  hubOptions.group = hubOptions.hubGroup or hubOptions.group
  hubOptions.disableMeshBreaking = hubOptions.disableHubMeshBreaking or hubOptions.disableMeshBreaking

  local hubSideOptions = deepcopy(hubOptions)
  hubSideOptions.beamSpring = hubSideOptions.hubSideBeamSpring or hubSideOptions.beamSpring
  hubSideOptions.beamDamp = hubSideOptions.hubSideBeamDamp or hubSideOptions.beamDamp
  hubSideOptions.beamDeform = hubSideOptions.hubSideBeamDeform or hubSideOptions.beamDeform
  hubSideOptions.beamStrength = hubSideOptions.hubSideBeamStrength or hubSideOptions.beamStrength
  hubSideOptions.dampCutoffHz = hubSideOptions.hubSideBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubReinfOptions = deepcopy(hubSideOptions)
  hubReinfOptions.beamSpring = hubReinfOptions.hubReinfBeamSpring or hubReinfOptions.beamSpring
  hubReinfOptions.beamDamp = hubReinfOptions.hubReinfBeamDamp or hubReinfOptions.beamDamp
  hubReinfOptions.beamDeform = hubReinfOptions.hubReinfBeamDeform or hubReinfOptions.beamDeform
  hubReinfOptions.beamStrength = hubReinfOptions.hubReinfBeamStrength or hubReinfOptions.beamStrength
  hubReinfOptions.dampCutoffHz = hubReinfOptions.hubReinfBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubTreadOptions = deepcopy(hubOptions)
  hubTreadOptions.beamSpring    = hubTreadOptions.hubTreadBeamSpring       or hubTreadOptions.beamSpring
  hubTreadOptions.beamDamp      = hubTreadOptions.hubTreadBeamDamp         or hubTreadOptions.beamDamp  
  hubTreadOptions.beamDeform    = hubTreadOptions.hubTreadBeamDeform       or hubTreadOptions.beamDeform  
  hubTreadOptions.beamStrength  = hubTreadOptions.hubTreadBeamStrength     or hubTreadOptions.beamStrength
  hubTreadOptions.dampCutoffHz  = hubTreadOptions.hubTreadBeamDampCutoffHz or hubOptions.dampCutoffHz     

  local hubPeripheryOptions = deepcopy(hubOptions)
  hubPeripheryOptions.beamSpring = hubPeripheryOptions.hubPeripheryBeamSpring or hubPeripheryOptions.beamSpring
  hubPeripheryOptions.beamDamp = hubPeripheryOptions.hubPeripheryBeamDamp or hubPeripheryOptions.beamDamp
  hubPeripheryOptions.beamDeform = hubPeripheryOptions.hubPeripheryBeamDeform or hubPeripheryOptions.beamDeform
  hubPeripheryOptions.beamStrength = hubPeripheryOptions.hubPeripheryBeamStrength or hubPeripheryOptions.beamStrength
  hubPeripheryOptions.dampCutoffHz = hubPeripheryOptions.hubPeripheryBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubStabilizerOptions = deepcopy(hubSideOptions)
  hubStabilizerOptions.beamSpring = hubStabilizerOptions.hubStabilizerBeamSpring or hubStabilizerOptions.beamSpring
  hubStabilizerOptions.beamDamp = hubStabilizerOptions.hubStabilizerBeamDamp or hubStabilizerOptions.beamDamp
  hubStabilizerOptions.beamDeform = hubStabilizerOptions.hubStabilizerBeamDeform or hubStabilizerOptions.beamDeform
  hubStabilizerOptions.beamStrength = hubStabilizerOptions.hubStabilizerBeamStrength or hubStabilizerOptions.beamStrength
  hubStabilizerOptions.dampCutoffHz = hubStabilizerOptions.hubStabilizerBeamDampCutoffHz or hubOptions.dampCutoffHz

  cleanupWheelOptions(hubOptions) -- used for nodes
  cleanupBeamOptions(cleanupWheelOptions(hubSideOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubTreadOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubPeripheryOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubStabilizerOptions))

  if wheel.hubWidth ~= nil then
    local halfWidth = 0.5 * wheel.hubWidth
    node1_pos = midpoint - axis * halfWidth
    node2_pos = midpoint + axis * halfWidth
  end

  --MARK: wheels
  rayVec = axis:perpendicularN() * wheel.hubRadius
  rayVec = quatFromAxisAngle(axis, -wheelAngleRad) * rayVec
  local rayRotSWII = rayRotSWI
  rayRotSWII.x, rayRotSWI.y, rayRotSWI.z = rayRotSWI.x * -1, rayRotSWI.y * -1, rayRotSWI.z * -1
  local rayVecI = rayRotSWII / rayRotSWI1st * rayVec
  
  for i = 0, newNumRay - 1, 1 do
    -- inner
    local rayPoint = node2_pos + rayVecI
    rayVecI = rayRotSWO * rayRotSWO * rayVecI
    --rayVec = rayRotSWO * rayVec
    local n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubOptions)
    table.insert(hubNodes, vehicle.nodes[n])
    --local testLength = node2_pos - rayPoint
    --print("inner lenghts- " .. tostring(testLength:length()))

    -- outer
    rayPoint = node1_pos + rayVec
    n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubOptions)
    table.insert(hubNodes, vehicle.nodes[n])
    rayVec = rayRotSWO * rayRotSWO * rayVec
  end

  --Hub Cap
  local hubcapOptions = deepcopy(cleanwheel)
  hubcapOptions.beamSpring = hubcapOptions.hubcapBeamSpring or hubcapOptions.beamSpring
  hubcapOptions.beamDamp = hubcapOptions.hubcapBeamDamp or hubcapOptions.beamDamp
  hubcapOptions.beamDeform = hubcapOptions.hubcapBeamDeform or hubcapOptions.beamDeform
  hubcapOptions.beamStrength = hubcapOptions.hubcapBeamStrength or hubcapOptions.beamStrength
  hubcapOptions.nodeWeight = hubcapOptions.hubcapNodeWeight or hubcapOptions.nodeWeight
  hubcapOptions.collision = hubcapOptions.hubcapCollision or hubcapOptions.collision
  hubcapOptions.nodeMaterial = hubcapOptions.hubcapNodeMaterial or hubcapOptions.nodeMaterial
  hubcapOptions.frictionCoef = hubcapOptions.hubcapFrictionCoef or hubcapOptions.frictionCoef
  hubcapOptions.hubcapRadius = hubcapOptions.hubcapRadius or hubcapOptions.hubRadius
  hubcapOptions.group = hubcapOptions.hubcapGroup or hubcapOptions.group
  hubcapOptions.disableMeshBreaking = hubcapOptions.disableHubcapMeshBreaking or hubOptions.disableMeshBreaking
  hubcapOptions.wheelID = nil

  cleanupWheelOptions(hubcapOptions) -- used for nodes

  local hubcapnodebase
  if wheel.enableHubcaps ~= nil and wheel.enableHubcaps == true and newNumRay%2 ~= 1 then
    local hubcapOffset
    if wheel.hubcapOffset ~= nil then
      hubcapOffset = wheel.hubcapOffset
      hubcapOffset = axis * hubcapOffset
    end

    hubcapnodebase = vehicle.maxIDs.nodes
    local hubCapNumRays = newNumRay/2
    rayVec = axis:perpendicularN() * hubcapOptions.hubcapRadius
    --rayVec = quatFromAxisAngle(axis, - wheelAngleRad + 2 * math.pi / (hubCapNumRays * 4)) * rayVec
    rayRot = quatFromAxisAngle(axis, 2 * math.pi / hubCapNumRays)

    for i = 0, hubCapNumRays -1, 1 do
      local rayPoint = node1_pos + rayVec - hubcapOffset
      rayVec = rayRot * rayVec
      jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubcapOptions)
    end

    --hubcapOptions.collision = false
    --hubcapOptions.selfCollision = false
    hubcapOptions.nodeWeight = wheel.hubcapCenterNodeWeight
    --make the center rigidifying node
    local hubcapAxis = node1_pos + axis * wheel.hubcapWidth
    jbeamUtils.addNodeWithOptions(vehicle, hubcapAxis, NORMALTYPE, hubcapOptions)

    --hubcapOptions.collision = nil
    --hubcapOptions.selfCollision = nil
    hubcapOptions.nodeWeight = nil
  end

  local weightComp = math.min(1, tyreWeight)

  local hubcapAttachOptions = deepcopy(cleanwheel)
  hubcapAttachOptions.beamSpring  = hubcapAttachOptions.hubcapAttachBeamSpring    or hubcapAttachOptions.beamSpring
  hubcapAttachOptions.beamDamp    = hubcapAttachOptions.hubcapAttachBeamDamp      or hubcapAttachOptions.beamDamp
  hubcapAttachOptions.beamDeform  = hubcapAttachOptions.hubcapAttachBeamDeform    or hubcapAttachOptions.beamDeform
  hubcapAttachOptions.beamStrength= hubcapAttachOptions.hubcapAttachBeamStrength  or hubcapAttachOptions.beamStrength
  hubcapAttachOptions.breakGroup  = hubcapAttachOptions.hubcapBreakGroup          or hubcapAttachOptions.breakGroup
  hubcapAttachOptions.wheelID     = nil
  hubcapAttachOptions.disableMeshBreaking = true

  local hubcapSupportOptions = deepcopy(hubcapAttachOptions)
  hubcapSupportOptions.beamSpring   = hubcapSupportOptions.hubcapSupportBeamSpring    or  hubcapSupportOptions.beamSpring
  hubcapSupportOptions.beamDamp     = hubcapSupportOptions.hubcapSupportBeamDamp      or  hubcapSupportOptions.beamDamp
  hubcapSupportOptions.beamDeform   = hubcapSupportOptions.hubcapSupportBeamDeform    or  hubcapSupportOptions.beamDeform
  hubcapSupportOptions.beamStrength = hubcapSupportOptions.hubcapSupportBeamStrength  or  hubcapSupportOptions.beamStrength
  hubcapSupportOptions.breakGroup   = nil
  hubcapSupportOptions.wheelID      = nil
  hubcapSupportOptions.disableMeshBreaking = true

  local sideOptions = deepcopy(cleanwheel)
  sideOptions.beamSpring        = sideOptions.wheelSideBeamSpring          and sideOptions.wheelSideBeamSpring          * fullStif or 0
  sideOptions.beamDamp          = sideOptions.wheelSideBeamDamp            and sideOptions.wheelSideBeamDamp            * fullStif or 0
  sideOptions.beamDeform        = sideOptions.wheelSideBeamDeform          and sideOptions.wheelSideBeamDeform          * fullStif or (sideOptions.beamDeform      and sideOptions.beamDeform      * fullStif or sideOptions.beamDeform     )
  sideOptions.beamStrength      = sideOptions.wheelSideBeamStrength        and sideOptions.wheelSideBeamStrength        * fullStif or (sideOptions.beamStrength    and sideOptions.beamStrength    * fullStif or sideOptions.beamStrength   )
  sideOptions.springExpansion   = sideOptions.wheelSideBeamSpringExpansion and sideOptions.wheelSideBeamSpringExpansion * fullStif or (sideOptions.springExpansion and sideOptions.springExpansion * fullStif or sideOptions.springExpansion)
  sideOptions.dampExpansion     = sideOptions.wheelSideBeamDampExpansion   and sideOptions.wheelSideBeamDampExpansion   * fullStif or (sideOptions.dampExpansion   and sideOptions.dampExpansion   * fullStif or sideOptions.dampExpansion  )
  sideOptions.transitionZone    = sideOptions.wheelSideTransitionZone       or sideOptions.transitionZone
  sideOptions.beamPrecompression= sideOptions.wheelSideBeamPrecompression   or 1

  local pyramidOptions = deepcopy(cleanwheel)
  pyramidOptions.beamSpring        = sideOptions.wheelSideBeamSpring          and sideOptions.wheelSideBeamSpring          * antiSquish  * fullStif or 0
  pyramidOptions.beamDamp          = sideOptions.wheelSideBeamDamp            and sideOptions.wheelSideBeamDamp            * antiSquish  * fullStif or 0
  pyramidOptions.beamDeform        = sideOptions.wheelSideBeamDeform          and sideOptions.wheelSideBeamDeform          * antiSquish  * fullStif or (sideOptions.beamDeform      and sideOptions.beamDeform      * fullStif * antiSquish or sideOptions.beamDeform     )
  pyramidOptions.beamStrength      = sideOptions.wheelSideBeamStrength        and sideOptions.wheelSideBeamStrength        * antiSquish  * fullStif or (sideOptions.beamStrength    and sideOptions.beamStrength    * fullStif * antiSquish or sideOptions.beamStrength   )
  pyramidOptions.springExpansion   = sideOptions.wheelSideBeamSpringExpansion and sideOptions.wheelSideBeamSpringExpansion * antiBulge   * fullStif or (sideOptions.springExpansion and sideOptions.springExpansion * fullStif * antiBulge  or sideOptions.springExpansion)
  pyramidOptions.dampExpansion     = sideOptions.wheelSideBeamDampExpansion   and sideOptions.wheelSideBeamDampExpansion   * antiBulge   * fullStif or (sideOptions.dampExpansion   and sideOptions.dampExpansion   * fullStif * antiBulge  or sideOptions.dampExpansion  )
  pyramidOptions.transitionZone    = sideOptions.wheelSideTransitionZone       or sideOptions.transitionZone
  pyramidOptions.beamPrecompression= sideOptions.wheelSideBeamPrecompression   or 1

  local pAntiRein50 = pAntiRein * 0.25
  local sideReinfOptions = deepcopy(sideOptions)
  sideReinfOptions.beamSpring           = sideOptions.wheelSideReinfBeamSpring          and sideOptions.wheelSideReinfBeamSpring          * pAntiRein   * fullStif * weightComp or (sideOptions.wheelSideBeamSpring and sideOptions.wheelSideBeamSpring * pAntiRein   * fullStif * weightComp or sideOptions.wheelSideBeamSpring)
  sideReinfOptions.beamDamp             = sideOptions.wheelSideReinfBeamDamp            and sideOptions.wheelSideReinfBeamDamp            * pAntiRein   * fullStif * weightComp or (sideOptions.wheelSideBeamDamp   and sideOptions.wheelSideBeamDamp   * pAntiRein   * fullStif * weightComp or sideOptions.wheelSideBeamDamp  )
  sideReinfOptions.beamDeform           = sideOptions.wheelSideReinfBeamDeform          and sideOptions.wheelSideReinfBeamDeform          * pAntiRein   * fullStif * weightComp or (sideOptions.beamDeform          and sideOptions.beamDeform          * pAntiRein   * fullStif * weightComp or sideOptions.beamDeform     )
  sideReinfOptions.beamStrength         = sideOptions.wheelSideReinfBeamStrength        and sideOptions.wheelSideReinfBeamStrength        * pAntiRein   * fullStif * weightComp or (sideOptions.beamStrength        and sideOptions.beamStrength        * pAntiRein   * fullStif * weightComp or sideOptions.beamStrength   )
  sideReinfOptions.springExpansion      = sideOptions.wheelSideReinfBeamSpringExpansion and sideOptions.wheelSideReinfBeamSpringExpansion * pAntiRein50 * fullStif * weightComp or (sideOptions.springExpansion     and sideOptions.springExpansion     * pAntiRein50 * fullStif * weightComp or sideOptions.springExpansion)
  sideReinfOptions.dampExpansion        = sideOptions.wheelSideReinfBeamDampExpansion   and sideOptions.wheelSideReinfBeamDampExpansion   * pAntiRein50 * fullStif * weightComp or (sideOptions.dampExpansion       and sideOptions.dampExpansion       * pAntiRein50 * fullStif * weightComp or sideOptions.dampExpansion  )
  sideReinfOptions.transitionZone       = sideOptions.wheelSideReinfTransitionZone      or sideOptions.transitionZone
  sideReinfOptions.beamPrecompression   = sideOptions.wheelSideReinfBeamPrecompression  or 1
  sideReinfOptions.disableMeshBreaking  = true

  local reinfOptions = deepcopy(cleanwheel)
  reinfOptions.beamSpring         = reinfOptions.wheelReinfBeamSpring           and reinfOptions.wheelReinfBeamSpring          * xPat*fullStif or 0
  reinfOptions.beamDamp           = reinfOptions.wheelReinfBeamDamp             and reinfOptions.wheelReinfBeamDamp            * xPat*fullStif or 0
  reinfOptions.springExpansion    = reinfOptions.wheelReinfBeamSpringExpansion  and reinfOptions.wheelReinfBeamSpringExpansion * xPat*fullStif or (reinfOptions.wheelReinfBeamSpringExpansion and reinfOptions.wheelReinfBeamSpringExpansion * xPat * fullStif  or reinfOptions.wheelReinfBeamSpringExpansion)
  reinfOptions.dampExpansion      = reinfOptions.wheelReinfBeamDampExpansion    and reinfOptions.wheelReinfBeamDampExpansion   * xPat*fullStif or (reinfOptions.wheelReinfBeamDampExpansion   and reinfOptions.wheelReinfBeamDampExpansion   * xPat * fullStif  or reinfOptions.wheelReinfBeamDampExpansion  )
  reinfOptions.beamDeform         = reinfOptions.wheelReinfBeamDeform           and reinfOptions.wheelReinfBeamDeform          * xPat*fullStif or (reinfOptions.beamDeform                    and reinfOptions.beamDeform                    * xPat * fullStif  or reinfOptions.beamDeform)
  reinfOptions.beamStrength       = reinfOptions.wheelReinfBeamStrength         and reinfOptions.wheelReinfBeamStrength        * xPat*fullStif or (reinfOptions.beamStrength                  and reinfOptions.beamStrength                  * xPat * fullStif  or reinfOptions.beamStrength)
  reinfOptions.beamPrecompression = reinfOptions.wheelReinfBeamPrecompression   or 1
  reinfOptions.dampCutoffHz       = reinfOptions.wheelReinfBeamDampCutoffHz     or nil
  reinfOptions.disableMeshBreaking= true

  local cornerReinfOptions = deepcopy(cleanwheel)
  cornerReinfOptions.beamSpring         = reinfOptions.wheelReinfBeamSpring           and reinfOptions.wheelReinfBeamSpring          * pCornerL*fullStif or 0
  cornerReinfOptions.beamDamp           = reinfOptions.wheelReinfBeamDamp             and reinfOptions.wheelReinfBeamDamp            * pCornerL*fullStif or 0
  cornerReinfOptions.springExpansion    = reinfOptions.wheelReinfBeamSpringExpansion  and reinfOptions.wheelReinfBeamSpringExpansion * pCornerL*fullStif or (reinfOptions.wheelReinfBeamSpringExpansion and reinfOptions.wheelReinfBeamSpringExpansion * pCornerL * fullStif  or reinfOptions.wheelReinfBeamSpringExpansion)
  cornerReinfOptions.dampExpansion      = reinfOptions.wheelReinfBeamDampExpansion    and reinfOptions.wheelReinfBeamDampExpansion   * pCornerL*fullStif or (reinfOptions.wheelReinfBeamDampExpansion   and reinfOptions.wheelReinfBeamDampExpansion   * pCornerL * fullStif  or reinfOptions.wheelReinfBeamDampExpansion  )
  cornerReinfOptions.beamDeform         = reinfOptions.wheelReinfBeamDeform           and reinfOptions.wheelReinfBeamDeform          * pCornerL*fullStif or (reinfOptions.beamDeform                    and reinfOptions.beamDeform                    * pCornerL * fullStif  or reinfOptions.beamDeform)
  cornerReinfOptions.beamStrength       = reinfOptions.wheelReinfBeamStrength         and reinfOptions.wheelReinfBeamStrength        * pCornerL*fullStif or (reinfOptions.beamStrength                  and reinfOptions.beamStrength                  * pCornerL * fullStif  or reinfOptions.beamStrength)
  cornerReinfOptions.beamPrecompression = reinfOptions.wheelReinfBeamPrecompression   or 1
  cornerReinfOptions.dampCutoffHz       = reinfOptions.wheelReinfBeamDampCutoffHz     or nil
  cornerReinfOptions.disableMeshBreaking= true

  local treadOptions = deepcopy(cleanwheel)
  treadOptions.beamSpring         = treadOptions.wheelTreadBeamSpring         and treadOptions.wheelTreadBeamSpring  * weightComp * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring  * weightComp * fullStif or treadOptions.beamSpring  )
  treadOptions.beamDamp           = treadOptions.wheelTreadBeamDamp           and treadOptions.wheelTreadBeamDamp    * weightComp * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp    * weightComp * fullStif or treadOptions.beamDamp    )
  treadOptions.beamDeform         = treadOptions.wheelTreadBeamDeform         and treadOptions.wheelTreadBeamDeform   * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform   * fullStif or treadOptions.beamDeform  )
  treadOptions.beamStrength       = treadOptions.wheelTreadBeamStrength       and treadOptions.wheelTreadBeamStrength * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength * fullStif or treadOptions.beamStrength)
  treadOptions.beamPrecompression = treadOptions.wheelTreadBeamPrecompression or  1
  treadOptions.dampCutoffHz       = treadOptions.wheelTreadBeamDampCutoffHz   or  nil
  treadOptions.disableMeshBreaking= true

  local treadReinfOptions = deepcopy(treadOptions)
  treadReinfOptions.beamSpring          = treadOptions.wheelTreadReinfBeamSpring          and treadOptions.wheelTreadReinfBeamSpring   * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring  * fullStif or treadOptions.beamSpring  )
  treadReinfOptions.beamDamp            = treadOptions.wheelTreadReinfBeamDamp            and treadOptions.wheelTreadReinfBeamDamp     * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp    * fullStif or treadOptions.beamDamp    )
  treadReinfOptions.beamDeform          = treadOptions.wheelTreadReinfBeamDeform          and treadOptions.wheelTreadReinfBeamDeform   * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform  * fullStif or treadOptions.beamDeform  )
  treadReinfOptions.beamStrength        = treadOptions.wheelTreadReinfBeamStrength        and treadOptions.wheelTreadReinfBeamStrength * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength* fullStif or treadOptions.beamStrength)
  treadReinfOptions.beamPrecompression  = treadOptions.wheelTreadReinfBeamPrecompression  or 1
  treadReinfOptions.dampCutoffHz        = treadOptions.wheelTreadReinfBeamDampCutoffHz    or nil
  treadReinfOptions.disableMeshBreaking = true
  if sideWallReinf >= 0.05 then
    treadReinfOptions.beamSpring  = treadOptions.wheelTreadReinfBeamSpring         and treadOptions.wheelTreadReinfBeamSpring   * sideWallReinf * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring   * sideWallReinf * fullStif or treadOptions.beamSpring  )
    treadReinfOptions.beamDamp    = treadOptions.wheelTreadReinfBeamDamp           and treadOptions.wheelTreadReinfBeamDamp     * sideWallReinf * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp     * sideWallReinf * fullStif or treadOptions.beamDamp    )
    treadReinfOptions.beamDeform  = treadOptions.wheelTreadReinfBeamDeform         and treadOptions.wheelTreadReinfBeamDeform   * sideWallReinf * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform   * sideWallReinf * fullStif or treadOptions.beamDeform  )
    treadReinfOptions.beamStrength= treadOptions.wheelTreadReinfBeamStrength       and treadOptions.wheelTreadReinfBeamStrength * sideWallReinf * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength * sideWallReinf * fullStif or treadOptions.beamStrength)
  end

  local peripheryOptions = deepcopy(treadOptions)
  peripheryOptions.beamSpring         = peripheryOptions.wheelPeripheryBeamSpring   and peripheryOptions.wheelPeripheryBeamSpring   * fullStif  or (peripheryOptions.beamSpring   and peripheryOptions.beamSpring   * fullStif or peripheryOptions.beamSpring  )
  peripheryOptions.beamDamp           = peripheryOptions.wheelPeripheryBeamDamp     and peripheryOptions.wheelPeripheryBeamDamp     * fullStif  or (peripheryOptions.beamDamp     and peripheryOptions.beamDamp     * fullStif or peripheryOptions.beamDamp    )
  peripheryOptions.beamDeform         = peripheryOptions.wheelPeripheryBeamDeform   and peripheryOptions.wheelPeripheryBeamDeform   * fullStif  or (peripheryOptions.beamDeform   and peripheryOptions.beamDeform   * fullStif or peripheryOptions.beamDeform  )
  peripheryOptions.beamStrength       = peripheryOptions.wheelPeripheryBeamStrength and peripheryOptions.wheelPeripheryBeamStrength * fullStif  or (peripheryOptions.beamStrength and peripheryOptions.beamStrength * fullStif or peripheryOptions.beamStrength)
  peripheryOptions.beamPrecompression = peripheryOptions.wheelPeripheryBeamPrecompression or 1
  peripheryOptions.dampCutoffHz       = peripheryOptions.wheelPeripheryBeamDampCutoffHz   or nil

  local peripheryReinfOptions = deepcopy(peripheryOptions)
  peripheryReinfOptions.beamSpring        = peripheryReinfOptions.wheelPeripheryReinfBeamSpring   and peripheryReinfOptions.wheelPeripheryReinfBeamSpring   * fullStif  or (peripheryOptions.beamSpring   and peripheryOptions.beamSpring   * fullStif or peripheryOptions.beamSpring  )
  peripheryReinfOptions.beamDamp          = peripheryReinfOptions.wheelPeripheryReinfBeamDamp     and peripheryReinfOptions.wheelPeripheryReinfBeamDamp     * fullStif  or (peripheryOptions.beamDamp     and peripheryOptions.beamDamp     * fullStif or peripheryOptions.beamDamp    )
  peripheryReinfOptions.beamDeform        = peripheryReinfOptions.wheelPeripheryReinfBeamDeform   and peripheryReinfOptions.wheelPeripheryReinfBeamDeform   * fullStif  or (peripheryOptions.beamDeform   and peripheryOptions.beamDeform   * fullStif or peripheryOptions.beamDeform  )
  peripheryReinfOptions.beamStrength      = peripheryReinfOptions.wheelPeripheryReinfBeamStrength and peripheryReinfOptions.wheelPeripheryReinfBeamStrength * fullStif  or (peripheryOptions.beamStrength and peripheryOptions.beamStrength * fullStif or peripheryOptions.beamStrength)
  peripheryReinfOptions.beamPrecompression= peripheryReinfOptions.wheelPeripheryReinfBeamPrecompression or 1
  peripheryReinfOptions.dampCutoffHz      = peripheryOptions.wheelPeripheryReinfBeamDampCutoffHz        or nil

  --treadOptions.beamSpring = treadOptions.beamSpring * (weightComp - weightComp*weightComp)
  --treadOptions.beamDamp   = treadOptions.beamDamp   * (weightComp - weightComp*weightComp)


  cleanupBeamOptions(cleanupWheelOptions(hubcapAttachOptions))
  cleanupBeamOptions(cleanupWheelOptions(sideOptions))
  cleanupBeamOptions(cleanupWheelOptions(sideReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(reinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(treadOptions))
  cleanupBeamOptions(cleanupWheelOptions(treadReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(peripheryOptions))
  cleanupBeamOptions(cleanupWheelOptions(peripheryReinfOptions))

  vehicle.triangles = vehicle.triangles or {}
  local pressureGroupName = '_wheelPressureGroup' .. wheel.wheelID
  local wheelPressure = wheel.pressurePSI or 10
  local wheelDragCoef = wheel.dragCoef or 100
  local wheelSkinDragCoef = wheel.skinDragCoef
  local wheelTreadTriangleType = NORMALTYPE
  local wheelSide1TriangleType = NORMALTYPE
  local wheelSide2TriangleType = NORMALTYPE

  if (wheel.triangleCollision or false) == false then
    wheelTreadTriangleType = NONCOLLIDABLE
    wheelSide1TriangleType = NONCOLLIDABLE
    wheelSide2TriangleType = NONCOLLIDABLE
  end

  if wheel.treadTriangleCollision == false then
    wheelTreadTriangleType = NONCOLLIDABLE
  end

  if wheel.side1TriangleCollision == false then
    wheelSide1TriangleType = NONCOLLIDABLE
  end

  if wheel.side2TriangleCollision == false then
    wheelSide2TriangleType = NONCOLLIDABLE
  end

  local hubTriangleCollision = wheel.hubTriangleCollision == true and true or false
  local hubSide1TriangleCollision = wheel.hubSide1TriangleCollision == true and true or false
  local hubSide2TriangleCollision = wheel.hubSide2TriangleCollision == true and true or false

  local rimBeams = {}
  local sideBeams = {}
  local peripheryBeams = {}
  local treadBeams = {}
  local reinfBeams = {}

  local inaxisnode = wheel.node1
  local outaxisnode = wheel.node2
  local vTris = vehicle.triangles

  --modified by fillman86 ******************************************************************* V
  local timesdone = 0

  for i = 0, newNumRay - 1 do
    local i2 = 2*i --keep
    local nextdelta = 2*((i+1)%newNumRay) --keep
    local outhubnode = hubnodebase + i2 --keep
    local inhubnode = outhubnode + 1 --keep
    local nextouthubnode = hubnodebase + nextdelta --keep
    local nextinhubnode = nextouthubnode + 1 --keep

    local curBand     = treadBands[i+1]
    local nexBand     = treadBands[i+2] and treadBands[i+2] or treadBands[(i+2)-(newNumRay)]
    local nexNexBand  = treadBands[i+3] and treadBands[i+3] or treadBands[(i+3)-(newNumRay)]
    local nex3Band    = treadBands[i+4] and treadBands[i+4] or treadBands[(i+4)-(newNumRay)]
    local preBand     = treadBands[i] and treadBands[i] or treadBands[newNumRay]
    local prePreBand  = treadBands[i-1] and treadBands[i-1] or treadBands[newNumRay-1]

    -- Hub caps
    if wheel.enableHubcaps and newNumRay%2 ~= 1 and i < ((newNumRay)/2) then
      local hubcapnode = hubcapnodebase + i
      local nexthubcapnode = hubcapnodebase + ((i+1)%(newNumRay/2))
      local nextnexthubcapnode = hubcapnodebase + ((i+2)%(newNumRay/2))
      local hubcapaxisnode = hubcapnode + (newNumRay/2) - i
      local hubcapinhubnode = inhubnode + i2
      local prevhubcaphubnode = hubnodebase + 2 *((i2 + newNumRay - 1)%newNumRay)+1
      local nexthubcapinhubnode = hubcapinhubnode + 2
      local hubcapouthubnode = hubcapinhubnode - 1

      --hubcap periphery
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, nexthubcapnode,    NORMALTYPE, hubcapOptions)
      --attach to center node
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapaxisnode,    NORMALTYPE, hubcapOptions)
      --attach to axis
      if wheel.enableExtraHubcapBeams == true then
        --jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node1, NORMALTYPE, hubcapAttachOptions)
        --jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node2, NORMALTYPE, hubcapAttachOptions)
        if i == 1 then
          jbeamUtils.addBeamWithOptions(vehicle, hubcapaxisnode, wheel.node1, NORMALTYPE, hubcapAttachOptions)
          jbeamUtils.addBeamWithOptions(vehicle, hubcapaxisnode, wheel.node2, NORMALTYPE, hubcapAttachOptions)
        end
      end

      --span beams
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, nextnexthubcapnode,    NORMALTYPE, hubcapOptions)

      --attach it
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapinhubnode,   NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, prevhubcaphubnode, NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapouthubnode,  NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapouthubnode,  BEAM_SUPPORT, hubcapSupportOptions)

      --self:jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node1,    NORMALTYPE, hubcapAttachOptions)
      --self:jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node2,    NORMALTYPE, hubcapAttachOptions)
    end

    --hub tread
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, inhubnode,      NORMALTYPE, hubTreadOptions)
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  nextouthubnode, NORMALTYPE, hubTreadOptions)

    --hub periphery beams
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, nextouthubnode, NORMALTYPE, hubPeripheryOptions)
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  nextinhubnode,  NORMALTYPE, hubPeripheryOptions)

    --hub axis beams
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, inaxisnode, NORMALTYPE, hubReinfOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, outaxisnode, NORMALTYPE, hubSideOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  inaxisnode, NORMALTYPE, hubSideOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  outaxisnode, NORMALTYPE, hubReinfOptions))

    --Beams to stability node
    if nodeStabilizerExists then
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,  wheel.nodeStabilizer, NORMALTYPE, hubStabilizerOptions)
    end

    if hubSide1TriangleCollision then
      addTri(vTris, nextouthubnode, outhubnode, outaxisnode, wheelDragCoef * 0.5, NORMALTYPE)
    end

    if hubSide2TriangleCollision then
      addTri(vTris, inhubnode, nextinhubnode, inaxisnode, wheelDragCoef * 0.5, NORMALTYPE)
    end      

    --MARK: beams
    if tireExists then
      local tbnc = 4+treadDiv --tbnc is short for "tread band node count"
      --to bring back dupeCheck, replace all following "table.insert" with "dupeCheck", and it'll check all beams
      --tire tread
      for cNode = 1, tbnc-1 do --connecting the band transversely, and to the next band
        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, curBand[cNode+1].cid, NORMALTYPE, treadOptions))

        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexBand[cNode+1].cid, NORMALTYPE, treadOptions))
        if cNode >= 2 and cNode <= tbnc-2 then --this would create a wierd beam on the first node
          table.insert(treadBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode+1].cid, nexBand[cNode].cid, NORMALTYPE, treadOptions))
        end
      end

      --lateral sidewall reinforcement
      if sideWallReinf >= 0.05 then
        table.insert(treadBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[tbnc].cid, NORMALTYPE, treadReinfOptions))
      end

      --paired treadnodes. To do with hidden elastic floor stuff, for rolling resistance
      if vehicle.variables['$xxpaired'].val == 0 then
        for cNode = 2, tbnc/2 do
          --[[ vehicle.nodes[curBand[cNode].cid].pairedNode = nexBand[(tbnc+1)-cNode].cid
          vehicle.nodes[nexBand[(tbnc+1)-cNode].cid].pairedNode = curBand[cNode].cid

          vehicle.nodes[curBand[cNode].cid].pairedNode2 = preBand[(tbnc+1)-cNode].cid
          vehicle.nodes[preBand[(tbnc+1)-cNode].cid].pairedNode2 = curBand[cNode].cid ]]
          vehicle.nodes[curBand[cNode].cid].pairedNode = curBand[(tbnc+1)-cNode].cid
          vehicle.nodes[curBand[(tbnc+1)-cNode].cid].pairedNode = curBand[cNode].cid

          vehicle.nodes[curBand[cNode].cid].pairedNode2 = preBand[(tbnc+1)-cNode].cid
          vehicle.nodes[preBand[(tbnc+1)-cNode].cid].pairedNode2 = curBand[cNode].cid
        end
        --[[ for cNode = 2, tbnc-1 do
          vehicle.nodes[curBand[cNode].cid].pairedNode  = nexBand[cNode].cid
          vehicle.nodes[curBand[cNode].cid].pairedNode2 = preBand[cNode].cid
        end ]]

        if not (tbnc%2==0) then
          local middleNode = math.floor(tbnc/2)+1
          vehicle.nodes[curBand[middleNode].cid].pairedNode = nexBand[middleNode].cid
          vehicle.nodes[curBand[middleNode].cid].pairedNode2 = preBand[middleNode].cid
        end
      end


      -- All periphery beams
      for cNode = 1, tbnc do
        table.insert(peripheryBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexBand[cNode].cid,    NORMALTYPE, peripheryOptions))
      end

      -- Periphery reinforcement beams
      for cNode = 2, tbnc-1 do
        table.insert(peripheryBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexNexBand[cNode].cid,    BEAM_ANISOTROPIC, sideReinfOptions))

        table.insert(peripheryBeams, --skip 3 bands ahead
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nex3Band[cNode].cid,    BEAM_ANISOTROPIC, sideReinfOptions))
      end

      --wheel->tire wall beams. Connect wheelNode to either tireNode
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,curBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, curBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,preBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, preBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))--default is BEAM_ANISOTROPIC, sideOptions <^ alternative is NORMALTYPE, treadOptions

      --adding triangles wheel <-to-> tread strands, to stop bulging
      if antiBulge >=0.05 or antiSquish >=0.05 then
        if (tbnc%2==0) then
          for CurrentNode = 3, tbnc - 2 do
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, inhubnode,     curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, outhubnode,    curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextinhubnode, curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode,curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
          end
        else  
          for CurrentNode = 3, tbnc - 2 do
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, inhubnode,     curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, outhubnode,    curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextinhubnode, curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode,curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
          end
        end
      end

      --note: I've left in a bunch of alternative beam options if people want to experiment with re-enabling them |WARNING: may be incompatible now

      --wheel->tire's corner: forward "L-Beams"
      --[[ dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  outhubnode,     curBand[1].cid,         BEAM_LBEAM, reinfOptions, inhubnode))
      dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  inhubnode,      curBand[tbnc].cid,      BEAM_LBEAM, reinfOptions, outhubnode))
      if i>= 1 then--wheel->tire's corner: backwards "L-Beams
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,outhubnode,     preBand[1].cid,             BEAM_LBEAM, reinfOptions, inhubnode))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,inhubnode,      preBand[tbnc].cid,          BEAM_LBEAM, reinfOptions, outhubnode))
      else
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,outhubnode,lastBand[1].cid,    BEAM_LBEAM, reinfOptions, inhubnode))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,inhubnode, lastBand[tbnc].cid, BEAM_LBEAM, reinfOptions, outhubnode))
      end ]]

      --first sidewall L-Beams
      --[[ dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  inhubnode, curBand[2].cid,         BEAM_LBEAM, reinfOptions, curBand[1].cid))
      dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  outhubnode,curBand[tbnc-1].cid,BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))
       ]]
      --[[ if xPat >= 0.05 then
        for eachWidthX = 2, tbnc-3 do 
          if i<=newNumRay-3 then --tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   nexNexBand[eachWidthX+2].cid,BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, nexNexBand[eachWidthX].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
          end
          if i==newNumRay-2 then -- second to last tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   firstBand[eachWidthX+2].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, firstBand[eachWidthX].cid,    BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
          end 
          if i==newNumRay-1 then -- last tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   treadBands[2][eachWidthX+2].cid,  BEAM_LBEAM, reinfOptions, firstBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, treadBands[2][eachWidthX].cid,    BEAM_LBEAM, reinfOptions, firstBand[eachWidthX+1].cid))
          end  ]]
          --[[ if i>2 and (eachWidthX>=3 or eachWidthX==tbnc-2) then --better corner L-Beams
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[eachWidthX].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))          
          end
        end
      end ]]

      --tyre corner L-Beams
      if pCornerL ~= 0 then
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[3].cid, BEAM_LBEAM, cornerReinfOptions, curBand[2].cid))
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[tbnc].cid, curBand[tbnc-2].cid, BEAM_LBEAM, cornerReinfOptions, curBand[tbnc-1].cid))
      end
  
      -- Wheel -> corner beams to rigidify sidewalls (same side, not crossing sides), specifically to help rolling resistance, like irl tyres
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  curBand[2].cid,          BEAM_LBEAM, reinfOptions, curBand[1].cid))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode, curBand[tbnc-1].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))
      if treadDiv > 1 then --this is to give extra l beams inwards
        --[[ dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  curBand[3].cid,          BEAM_LBEAM, reinfOptions, curBand[2].cid))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode, curBand[tbnc-2].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc-1].cid)) ]]
      end

      --trying periphery L-Beams |WARNING: probably not compatible anymore
      --[[ for eachWidthX = 2, tbnc-1 do 
        if i<=newNumRay-3 then
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, nexNexBand[eachWidthX].cid,BEAM_LBEAM, reinfOptions, nexBand[eachWidthX].cid))
        end
        if i==newNumRay-2 then -- second to last tread L-Beams
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, firstBand[eachWidthX].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX].cid))
        end 
        if i==newNumRay-1 then -- last tread L-Beams
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, treadBands[2][eachWidthX].cid,  BEAM_LBEAM, reinfOptions, firstBand[eachWidthX].cid))
        end 
      end ]]

      --opposite node trick
      for nods = 2, tbnc-1 do 
        --timesdone = timesdone + 1
        --print("timesdone -"..timesdone)
        if i<=(newNumRay*0.5)-1 then
          --dupeCheck(sideBeams,
          --jbeamUtils.addBeamWithOptions(vehicle, curBand[nods].cid,curBand[nods].oppositeTreadNodeCid, NORMALTYPE, treadOptions))
        end
      end
  



      --[[ for eachStrapNode = 1, tbnc -2 do --this is L-Beams for each perfect band
        dupeCheck( reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachStrapNode].cid, curBand[eachStrapNode+2].cid, BEAM_LBEAM, reinfOptions,  curBand[eachStrapNode+1].cid))
      end ]]

      --checking if inverse tyires are needed
      if inverse >=0 then 
        -- hub pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextouthubnode, outhubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)
        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextinhubnode, nextouthubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)

        -- wheel-tire pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, curBand[tbnc].cid, outhubnode, nextouthubnode, wheelDragCoef * 0.5, wheelSide1TriangleType)
        addPressTri(vTris, pressureGroupName, wheelPressure, curBand[tbnc].cid, nextouthubnode, nexBand[tbnc].cid, wheelDragCoef * 0.5, wheelSide1TriangleType)

        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, preBand[1].cid, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
        addPressTri(vTris, pressureGroupName, wheelPressure, nextinhubnode, inhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)

        for CurrentNode=1,tbnc do
          if (CurrentNode%2~=0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, nexBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, curBand[CurrentNode+1].cid, nexBand[CurrentNode+1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          elseif (CurrentNode%2==0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, curBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, nexBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          end
        end
      else
        -- hub pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, nextouthubnode, inhubnode, outhubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)
        addPressTri(vTris, pressureGroupName, wheelPressure, nextinhubnode, inhubnode, nextouthubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)

        -- wheel-tire pressure tris
          addPressTri(vTris, pressureGroupName, wheelPressure, outhubnode, curBand[tbnc].cid, nextouthubnode, wheelDragCoef * 0.5, wheelSide1TriangleType)
          addPressTri(vTris, pressureGroupName, wheelPressure, nextouthubnode, curBand[tbnc].cid, nexBand[tbnc].cid, wheelDragCoef * 0.5, wheelSide1TriangleType)

        -- inside pressure tris
          addPressTri(vTris, pressureGroupName, wheelPressure,     preBand[1].cid, inhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextinhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)

        for CurrentNode=1,tbnc do
          if (CurrentNode%2~=0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, nexBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode+1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          elseif (CurrentNode%2==0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, nexBand[CurrentNode+1].cid, curBand[CurrentNode+1].cid,nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          end
        end
      end
    else
      if hubTriangleCollision then
        -- hub tris
        addTri(vTris, nextouthubnode, inhubnode, outhubnode, wheelDragCoef * 0.1, NORMALTYPE)
        addTri(vTris, nextinhubnode, inhubnode, nextouthubnode, wheelDragCoef * 0.1, NORMALTYPE)
      end
    end --end of modified by fillman86 ************************************************* ^
    wheelsToRotators = deepcopy(vehicle)
  end

  --[[ if wheelKey == 0 then
    for a,b in ipairs(vehicle.nodes) do
      if b.pairedNode then
        if b.pairedNode2 then
          print(wheelKey .. "|tyre node- ".. b.cid .. ", to node- ".. b.pairedNode .. " and ".. b.pairedNode2)
        else
          print(wheelKey .. "|tyre node- ".. b.cid ..", only 1 pairednode- ".. b.pairedNode)
        end
      end
      if b.pairedNode2 and not b.pairedNode then
        print(wheelKey .. "|tyre node- ".. b.cid ..", only  pairednode2- ".. b.pairedNode2)
      end
    end
  end ]]

  wheel.nodes = hubNodes
  wheel.treadNodes = treadNodes
  wheel.rimBeams = rimBeams
  wheel.sideBeams = sideBeams
  wheel.peripheryBeams = peripheryBeams
  wheel.reinfBeams = reinfBeams
  wheel.treadBeams = treadBeams
  wheel.pressureGroup = pressureGroupName
end

--MARK:add square wall
local function addPressureWheelDivSquareWall(vehicle, wheelKey, wheel)  
  local node1   = vehicle.nodes[wheel.node1]
  local node2   = vehicle.nodes[wheel.node2]
  local node1_pos = vec3(node1.pos)
  local node2_pos = vec3(node2.pos)
  
  local treadDiv      = wheel.treadDiv      or (node1_pos.y > 0 and vehicle.variables['$raasubDiv'].val            or  vehicle.variables['$faasubDiv'].val         ) or 2       
  local bulgeOffset   = wheel.bulgeOffset   or (node1_pos.y > 0 and vehicle.variables['$rbulgeOffset'].val         or  vehicle.variables['$fbulgeOffset'].val      ) or 0       
  local sideWallReinf = wheel.sideWallReinf or (node1_pos.y > 0 and vehicle.variables['$rsideWallReinf'].val       or  vehicle.variables['$fsideWallReinf'].val    ) or 0.25    
  local antiBulge     = wheel.antiBulge     or (node1_pos.y > 0 and vehicle.variables['$rantiBulge'].val           or  vehicle.variables['$fantiBulge'].val        ) or 0       
  local antiSquish    = wheel.antiSquish    or (node1_pos.y > 0 and vehicle.variables['$rantiSquish'].val          or  vehicle.variables['$fantiSquish'].val       ) or 0       
  local xPat          = wheel.xPat          or (node1_pos.y > 0 and vehicle.variables['$rxPat'].val                or  vehicle.variables['$fxPat'].val             ) or 0.02    
  local fullStif      = wheel.fullStif      or (node1_pos.y > 0 and vehicle.variables['$rafullStif'].val           or  vehicle.variables['$fafullStif'].val        ) or 1       
  local pAntiRein     = wheel.pAntiRein     or (node1_pos.y > 0 and vehicle.variables['$rpAntiRein'].val           or  vehicle.variables['$fpAntiRein'].val        ) or 0.8     
  local pCornerL      = wheel.pCornerL      or (node1_pos.y > 0 and vehicle.variables['$rpCornerL'].val * 5        or  vehicle.variables['$fpCornerL'].val * 5     ) or 0.1     
  local newBulge      = wheel.newBulge      or (node1_pos.y > 0 and vehicle.variables['$rbulge'].val               or  vehicle.variables['$fbulge'].val            ) or 0       
  local xxFricComp    = wheel.xxFricComp    or (node1_pos.y > 0 and vehicle.variables['$rxxFricComp'].val/100      or  vehicle.variables['$fxxFricComp'].val/100   ) or 1       
  local xxSFricComp   = wheel.xxSFricComp   or (node1_pos.y > 0 and vehicle.variables['$rxxSFricComp'].val/100     or  vehicle.variables['$fxxSFricComp'].val/100  ) or 1       
  local newNumRay     = vehicle.variables['$rnumRays'] and (node1_pos.y > 0 and vehicle.variables['$rnumRays'].val or  vehicle.variables['$fnumRays'].val          ) or wheel.numRays
  local tyreWeight    = node1_pos.y > 0 and vehicle.variables['$raatreadWeight'].val/100  or  vehicle.variables['$faatreadWeight'].val/100
  local tyreWall      =  wheel.radius - wheel.hubRadius
  local tyreWallMiddle= (wheel.radius + wheel.hubRadius) / 2
  local tyreRatio     = (wheel.radius - wheel.hubRadius) / wheel.tireWidth
  
  -- Stabilizer
  wheel.nodeStabilizer = wheel.nodeStabilizer or wheel.nodeS
  wheel.treadCoef = wheel.treadCoef or 1
  wheel.nodeS = nil
  local nodeStabilizerExists = false
  local wheelAngleRad = math.rad(wheel.wheelAngle or 0)

  if wheel.nodeStabilizer and wheel.nodeStabilizer ~= 9999 and vehicle.nodes[wheel.nodeStabilizer] then
    nodeStabilizerExists = true
  else
    wheel.nodeStabilizer = nil
  end

  if node1 == nil or node2 == nil then
    log('W', "jbeam.addPressureWheel","invalid pressureWheel")
    return
  end

  local nodebase = vehicle.maxIDs.nodes
  --print("testing out vehicle.maxIDs.nodes- ".. tostring(vehicle.maxIDs.nodes) .. " -555555555555555555555555555")

  local tireExists = true
  if wheel.radius == nil then
    tireExists = false
    wheel.radius = 0.5
  end

  if wheel.pressurePSI == nil then
    tireExists = false
  end

  if wheel.hasTire ~= nil and wheel.hasTire == false then
    tireExists = false
  end

  if wheel.hubRadius == nil then wheel.hubRadius = 0.65 * wheel.radius end
  if newNumRay == nil then newNumRay = 10 end
  --newNumRay = 24

  -- add collision to the wheels nodes ;)
  wheel.collision = true

  -- calculate surface
  -- if tireExists and wheel.tireWidth and wheel.hubWidth and wheel.radius and wheel.hubRadius then
  --   wheel._surface = math.pi * (2 * (wheel.tireWidth * wheel.radius + wheel.hubWidth * wheel.hubRadius) +
  --     (wheel.radius + wheel.hubRadius) * math.sqrt(square(wheel.radius - wheel.hubRadius) + square(wheel.tireWidth - wheel.hubRadius)))
  -- end

  local middleNodes = {}

  local node4_pos = vec3(node2.pos)
  local node5_pos = vec3(node2.pos)

  local tyreBulge = newBulge or 0

  -- calculate axis
  local axis = node2_pos - node1_pos
  axis:normalize()

  local midpoint = (node2_pos + node1_pos) * 0.5
  if wheel.wheelOffset ~= nil then
    local offset = wheel.wheelOffset
    midpoint = midpoint + axis * offset
  end

  --MARK: node pos
  --this is lateral placement only, 3D is in the marked tyre section
  if wheel.tireWidth ~= nil then
    local halfWidth = 0.5 * wheel.tireWidth
    node1_pos = midpoint - axis * halfWidth
    node2_pos = midpoint + axis * halfWidth
    node4_pos = midpoint - axis * (wheel.tireWidth * ((tyreWall * 0.5) + 0.5 + tyreBulge))--sidewalls
    node5_pos = midpoint + axis * (wheel.tireWidth * ((tyreWall * 0.5) + 0.5 + tyreBulge))--sidewalls

    --calcualting band separation
    local postions = {}
    if treadDiv > 0 then
      local dividedWidth = wheel.tireWidth/(treadDiv+1)
      for i=1, (treadDiv/2) do --the "for" loop is to increasingly add more offset divisions
        local step = i/(treadDiv+1)
        local curMaths = ((1-step)^ tyreRatio) * wheel.tireWidth
        local evenMaths = wheel.tireWidth - (dividedWidth*i)
        local averageMaths = (curMaths + evenMaths*i)/((1+i))
        local currentPos = node2_pos - axis * averageMaths --position incrementally calculation, like normal nodes, except from opposite edge
        table.insert(middleNodes, currentPos)--table used for placing nodes
        table.insert(postions, averageMaths)--table for mirror purposes
      end
      local curLen = #postions --this is to order the new nodes into the right place
      if treadDiv % 2 ~= 0 then --if it's odd, then add a centre line
        local currentPos = midpoint - axis * 0
        table.insert(middleNodes, currentPos)--table used for placing nodes
        curLen = curLen + 1 --we need to make sure that the following nodes are after this centre node
      end
      for i = 1, (treadDiv/2) do --now we copy our homework from the first half
        local currentPos = node1_pos + axis * postions[i]
        table.insert(middleNodes, curLen+1, currentPos) --we add the nodes, but due to the order of the maths, we insert each new one from the centre(ish)
      end
    end    
  end

  local cleanwheel = deepcopy(wheel)
  cleanwheel.axleBeams = nil
  cleanwheel.childParts = nil
  cleanwheel.slotType = nil
  cleanwheel.enableABS = nil
  cleanwheel.enableBrakeThermals = nil
  cleanwheel.enableHubcaps = nil
  cleanwheel.enableTireLbeams = nil
  cleanwheel.enableTirePeripheryReinfBeams = nil
  cleanwheel.enableTireReinfBeams = nil
  cleanwheel.enableTireSideReinfBeams = nil
  cleanwheel.enableTreadReinfBeams = nil
  cleanwheel.hasTire = nil
  cleanwheel.brakeDiameter = nil
  cleanwheel.brakeInputSplit = nil
  cleanwheel.brakeMass = nil
  cleanwheel.brakeSplitCoef = nil
  cleanwheel.brakeSpring = nil
  cleanwheel.brakeTorque = nil
  cleanwheel.brakeType = nil
  cleanwheel.brakeVentingCoef = nil
  cleanwheel.heatCoefNodeToEnv = nil
  cleanwheel.heatCoefEnvMultStationary = nil
  cleanwheel.heatCoefEnvTerminalSpeed = nil
  cleanwheel.heatCoefNodeToCore = nil
  cleanwheel.heatCoefCoreToNodes = nil
  cleanwheel.heatCoefNodeToSurface = nil
  cleanwheel.heatCoefFriction = nil
  cleanwheel.heatCoefFlashFriction = nil
  cleanwheel.heatCoefStrain = nil
  cleanwheel.heatAffectsPressure = nil
  cleanwheel.smokingTemp = nil
  cleanwheel.meltingTemp = nil
  cleanwheel.frictionLowTemp = nil
  cleanwheel.frictionHighTemp = nil
  cleanwheel.frictionLowSlope = nil
  cleanwheel.frictionHighSlope = nil
  cleanwheel.frictionSlopeSmoothCoef = nil
  cleanwheel.frictionCoefLow = nil
  cleanwheel.frictionCoefMiddle = nil
  cleanwheel.frictionCoefHigh = nil

  local wheelNodes = cleanupWheelOptions(deepcopy(cleanwheel))
  --edited in by fillman86 ************************************************************** V
  --modify friction coefficients
  --rough napkin maths, on how many nodes probably contact the ground, at any given moment, above the standard 3 nodes
  local rayNodeCompensator  = math.max(1,math.floor(newNumRay/8))--more numRays means more nodes contacting the ground
  local divNodeCompensator  = ((treadDiv * rayNodeCompensator) + (rayNodeCompensator-1))/4--how many times more nodes contacting, vs default 3 nodes
  print("divNodeCompensator- "..divNodeCompensator)
  wheelNodes.frictionCoef   = (wheelNodes.frictionCoef / divNodeCompensator) * xxFricComp
  --set sidewall friction value
  local defaultFricCo       = wheelNodes.frictionCoef
  local sideWallFricCo      = wheelNodes.frictionCoef * 0.7
  wheelNodes.slidingFrictionCoef = wheelNodes.slidingFrictionCoef * xxSFricComp
  
  --if it's a wide tyre, we'll at least 'try' to decrease the weight a little, but not too much, it'll become unstable
  if wheel.tireWidth > 0.130 and treadDiv > 0 then
    wheelNodes.nodeWeight = (((wheelNodes.nodeWeight*treadDiv*2) + (wheelNodes.nodeWeight / treadDiv))/((treadDiv*2)+1))*tyreWeight
  end

  --this is used to mirror the node pattern for each side
  local inverse = 1
  if axis.x >= 0 then
    inverse = -1
  end

  local rayRotSWO   = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  local rayRotSWI   = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * (8/3) * inverse))
  local rayRotSWI1st= quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 8 * -inverse))
  local rayRot
  if treadDiv > 0 then
    rayRot      = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * ((treadDiv + 2) * inverse)))
  else
    rayRot      = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  end

  local rayVec
  local rayVecSWO
  local rayVecSWI
  local treadNodes = {}
  local treadBands = {}
  local treadNodePos = {}
  if tireExists then
    local oppCount
    if treadDiv > 0 then
      oppCount = newNumRay * ((treadDiv + 4)/2)
   else
      oppCount = newNumRay * 2
    end
    rayVec = axis:perpendicularN() * wheel.radius --right angle, then giving dimension
    rayVec = quatFromAxisAngle(axis, -wheelAngleRad) * rayVec --this is adjusting any hard coded initial rotation
    rayVecSWO = axis:perpendicularN() * math.min(wheel.radius, math.max(wheel.hubRadius, tyreWallMiddle + bulgeOffset))
    rayVecSWO = quatFromAxisAngle(axis, -wheelAngleRad) * rayVecSWO

    rayVecSWI = axis:perpendicularN() * math.min(wheel.radius, math.max(wheel.hubRadius, tyreWallMiddle + bulgeOffset))
    rayVecSWI = quatFromAxisAngle(axis, -wheelAngleRad) * rayVecSWI

    rayVecSWI = rayRotSWI1st * rayVecSWO
    for i = 1, treadDiv+2, 2 do
      print("doing extra rotation")
      rayVecSWI = rayRot * rayVecSWI
    end
    local hasEvenRayCount = newNumRay % 2 == 0
    --MARK: Tyres
    -- add nodes first
    for i = 0, newNumRay - 1, 1 do
      wheelNodes.frictionCoef = sideWallFricCo
      local currentBand = {}
      local currentNodePos = {}
      local rayPoint = node1_pos + rayVec
      -- outer sidewall
      rayPoint = node4_pos + rayVecSWO
      rayVecSWO = rayRotSWO * rayVecSWO
      local n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos

      -- outer corner
      wheelNodes.frictionCoef = defaultFricCo
      rayPoint = node1_pos + rayVec
      rayVec = rayRot * rayVec
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
        vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
      end

      --add middle nodes from table
      if treadDiv > 0 then
        for NodeCount,nodePos in ipairs(middleNodes) do -- start putting in the middle nodes
          rayPoint = nodePos + rayVec
          rayVec = rayRot * rayVec
          n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
          table.insert(treadNodes, vehicle.nodes[n])
          table.insert(currentBand,vehicle.nodes[n])
          currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
          --[[ if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
            vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
          end ]]
        end
      end

      -- inner corner
      rayPoint = node2_pos + rayVec
      rayVec = rayRot * rayVec
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
        vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
      end

      -- inner sidewall
      wheelNodes.frictionCoef = sideWallFricCo
      rayPoint = node5_pos + rayRotSWI * rayVecSWI
      rayVecSWI = rayRotSWI * rayVecSWO
      for extraRot = 1, (treadDiv+2)/2 do
        rayVecSWI = rayRot * rayVecSWI
      end
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes,  vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      rayVecSWO = rayRotSWO * rayVecSWO
      table.insert(treadBands, currentBand)
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      table.insert(treadNodePos, currentNodePos)
      currentNodePos = nil
      currentBand = nil
    end --end of: edited by  added by fillman86 *************************************************** ^
  end
  wheelNodes.frictionCoef = defaultFricCo

  -- add Hub nodes
  local hubNodes = {}
  local hubnodebase = vehicle.maxIDs.nodes

  local hubOptions = deepcopy(cleanwheel)
  hubOptions.beamSpring = hubOptions.hubBeamSpring or hubOptions.beamSpring
  hubOptions.beamDamp = hubOptions.hubBeamDamp or hubOptions.beamDamp
  hubOptions.beamDeform = hubOptions.hubBeamDeform or hubOptions.beamDeform
  hubOptions.beamStrength = hubOptions.hubBeamStrength or hubOptions.beamStrength
  hubOptions.dampCutoffHz = hubOptions.hubBeamDampCutoffHz or nil
  hubOptions.nodeWeight = hubOptions.hubNodeWeight or hubOptions.nodeWeight
  hubOptions.collision = hubOptions.hubCollision or hubOptions.collision
  hubOptions.nodeMaterial = hubOptions.hubNodeMaterial or hubOptions.nodeMaterial
  hubOptions.frictionCoef = hubOptions.hubFrictionCoef or hubOptions.frictionCoef
  hubOptions.group = hubOptions.hubGroup or hubOptions.group
  hubOptions.disableMeshBreaking = hubOptions.disableHubMeshBreaking or hubOptions.disableMeshBreaking

  local hubSideOptions = deepcopy(hubOptions)
  hubSideOptions.beamSpring = hubSideOptions.hubSideBeamSpring or hubSideOptions.beamSpring
  hubSideOptions.beamDamp = hubSideOptions.hubSideBeamDamp or hubSideOptions.beamDamp
  hubSideOptions.beamDeform = hubSideOptions.hubSideBeamDeform or hubSideOptions.beamDeform
  hubSideOptions.beamStrength = hubSideOptions.hubSideBeamStrength or hubSideOptions.beamStrength
  hubSideOptions.dampCutoffHz = hubSideOptions.hubSideBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubReinfOptions = deepcopy(hubSideOptions)
  hubReinfOptions.beamSpring = hubReinfOptions.hubReinfBeamSpring or hubReinfOptions.beamSpring
  hubReinfOptions.beamDamp = hubReinfOptions.hubReinfBeamDamp or hubReinfOptions.beamDamp
  hubReinfOptions.beamDeform = hubReinfOptions.hubReinfBeamDeform or hubReinfOptions.beamDeform
  hubReinfOptions.beamStrength = hubReinfOptions.hubReinfBeamStrength or hubReinfOptions.beamStrength
  hubReinfOptions.dampCutoffHz = hubReinfOptions.hubReinfBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubTreadOptions = deepcopy(hubOptions)
  hubTreadOptions.beamSpring = hubTreadOptions.hubTreadBeamSpring or hubTreadOptions.beamSpring
  hubTreadOptions.beamDamp = hubTreadOptions.hubTreadBeamDamp or hubTreadOptions.beamDamp
  hubTreadOptions.beamDeform = hubTreadOptions.hubTreadBeamDeform or hubTreadOptions.beamDeform
  hubTreadOptions.beamStrength = hubTreadOptions.hubTreadBeamStrength or hubTreadOptions.beamStrength
  hubTreadOptions.dampCutoffHz = hubTreadOptions.hubTreadBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubPeripheryOptions = deepcopy(hubOptions)
  hubPeripheryOptions.beamSpring = hubPeripheryOptions.hubPeripheryBeamSpring or hubPeripheryOptions.beamSpring
  hubPeripheryOptions.beamDamp = hubPeripheryOptions.hubPeripheryBeamDamp or hubPeripheryOptions.beamDamp
  hubPeripheryOptions.beamDeform = hubPeripheryOptions.hubPeripheryBeamDeform or hubPeripheryOptions.beamDeform
  hubPeripheryOptions.beamStrength = hubPeripheryOptions.hubPeripheryBeamStrength or hubPeripheryOptions.beamStrength
  hubPeripheryOptions.dampCutoffHz = hubPeripheryOptions.hubPeripheryBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubStabilizerOptions = deepcopy(hubSideOptions)
  hubStabilizerOptions.beamSpring = hubStabilizerOptions.hubStabilizerBeamSpring or hubStabilizerOptions.beamSpring
  hubStabilizerOptions.beamDamp = hubStabilizerOptions.hubStabilizerBeamDamp or hubStabilizerOptions.beamDamp
  hubStabilizerOptions.beamDeform = hubStabilizerOptions.hubStabilizerBeamDeform or hubStabilizerOptions.beamDeform
  hubStabilizerOptions.beamStrength = hubStabilizerOptions.hubStabilizerBeamStrength or hubStabilizerOptions.beamStrength
  hubStabilizerOptions.dampCutoffHz = hubStabilizerOptions.hubStabilizerBeamDampCutoffHz or hubOptions.dampCutoffHz

  cleanupWheelOptions(hubOptions) -- used for nodes
  cleanupBeamOptions(cleanupWheelOptions(hubSideOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubTreadOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubPeripheryOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubStabilizerOptions))

  if wheel.hubWidth ~= nil then
    local halfWidth = 0.5 * wheel.hubWidth
    node1_pos = midpoint - axis * halfWidth
    node2_pos = midpoint + axis * halfWidth
  end

  --MARK: wheels
  rayVec = axis:perpendicularN() * wheel.hubRadius
  rayVec = quatFromAxisAngle(axis, -wheelAngleRad) * rayVec
  local rayRotSWII = rayRotSWI
  rayRotSWII.x, rayRotSWI.y, rayRotSWI.z = rayRotSWI.x * -1, rayRotSWI.y * -1, rayRotSWI.z * -1
  local rayVecI = rayRotSWII / rayRotSWI1st * rayVec
  
  for i = 0, newNumRay - 1, 1 do
    -- inner
    local rayPoint = node2_pos + rayVecI
    rayVecI = rayRotSWO * rayRotSWO * rayVecI
    --rayVec = rayRotSWO * rayVec
    local n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubOptions)
    table.insert(hubNodes, vehicle.nodes[n])
    --local testLength = node2_pos - rayPoint
    --print("inner lenghts- " .. tostring(testLength:length()))

    -- outer
    rayPoint = node1_pos + rayVec
    n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubOptions)
    table.insert(hubNodes, vehicle.nodes[n])
    rayVec = rayRotSWO * rayRotSWO * rayVec
  end

  --Hub Cap
  local hubcapOptions = deepcopy(cleanwheel)
  hubcapOptions.beamSpring = hubcapOptions.hubcapBeamSpring or hubcapOptions.beamSpring
  hubcapOptions.beamDamp = hubcapOptions.hubcapBeamDamp or hubcapOptions.beamDamp
  hubcapOptions.beamDeform = hubcapOptions.hubcapBeamDeform or hubcapOptions.beamDeform
  hubcapOptions.beamStrength = hubcapOptions.hubcapBeamStrength or hubcapOptions.beamStrength
  hubcapOptions.nodeWeight = hubcapOptions.hubcapNodeWeight or hubcapOptions.nodeWeight
  hubcapOptions.collision = hubcapOptions.hubcapCollision or hubcapOptions.collision
  hubcapOptions.nodeMaterial = hubcapOptions.hubcapNodeMaterial or hubcapOptions.nodeMaterial
  hubcapOptions.frictionCoef = hubcapOptions.hubcapFrictionCoef or hubcapOptions.frictionCoef
  hubcapOptions.hubcapRadius = hubcapOptions.hubcapRadius or hubcapOptions.hubRadius
  hubcapOptions.group = hubcapOptions.hubcapGroup or hubcapOptions.group
  hubcapOptions.disableMeshBreaking = hubcapOptions.disableHubcapMeshBreaking or hubOptions.disableMeshBreaking
  hubcapOptions.wheelID = nil

  cleanupWheelOptions(hubcapOptions) -- used for nodes

  local hubcapnodebase
  if wheel.enableHubcaps ~= nil and wheel.enableHubcaps == true and newNumRay%2 ~= 1 then
    local hubcapOffset
    if wheel.hubcapOffset ~= nil then
      hubcapOffset = wheel.hubcapOffset
      hubcapOffset = axis * hubcapOffset
    end

    hubcapnodebase = vehicle.maxIDs.nodes
    local hubCapNumRays = newNumRay/2
    rayVec = axis:perpendicularN() * hubcapOptions.hubcapRadius
    --rayVec = quatFromAxisAngle(axis, - wheelAngleRad + 2 * math.pi / (hubCapNumRays * 4)) * rayVec
    rayRot = quatFromAxisAngle(axis, 2 * math.pi / hubCapNumRays)

    for i = 0, hubCapNumRays -1, 1 do
      local rayPoint = node1_pos + rayVec - hubcapOffset
      rayVec = rayRot * rayVec
      jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubcapOptions)
    end

    --hubcapOptions.collision = false
    --hubcapOptions.selfCollision = false
    hubcapOptions.nodeWeight = wheel.hubcapCenterNodeWeight
    --make the center rigidifying node
    local hubcapAxis = node1_pos + axis * wheel.hubcapWidth
    jbeamUtils.addNodeWithOptions(vehicle, hubcapAxis, NORMALTYPE, hubcapOptions)

    --hubcapOptions.collision = nil
    --hubcapOptions.selfCollision = nil
    hubcapOptions.nodeWeight = nil
  end

  local weightComp = math.min(1, tyreWeight)

  local hubcapAttachOptions = deepcopy(cleanwheel)
  hubcapAttachOptions.beamSpring  = hubcapAttachOptions.hubcapAttachBeamSpring    or hubcapAttachOptions.beamSpring
  hubcapAttachOptions.beamDamp    = hubcapAttachOptions.hubcapAttachBeamDamp      or hubcapAttachOptions.beamDamp
  hubcapAttachOptions.beamDeform  = hubcapAttachOptions.hubcapAttachBeamDeform    or hubcapAttachOptions.beamDeform
  hubcapAttachOptions.beamStrength= hubcapAttachOptions.hubcapAttachBeamStrength  or hubcapAttachOptions.beamStrength
  hubcapAttachOptions.breakGroup  = hubcapAttachOptions.hubcapBreakGroup          or hubcapAttachOptions.breakGroup
  hubcapAttachOptions.wheelID     = nil
  hubcapAttachOptions.disableMeshBreaking = true

  local hubcapSupportOptions = deepcopy(hubcapAttachOptions)
  hubcapSupportOptions.beamSpring   = hubcapSupportOptions.hubcapSupportBeamSpring    or  hubcapSupportOptions.beamSpring
  hubcapSupportOptions.beamDamp     = hubcapSupportOptions.hubcapSupportBeamDamp      or  hubcapSupportOptions.beamDamp
  hubcapSupportOptions.beamDeform   = hubcapSupportOptions.hubcapSupportBeamDeform    or  hubcapSupportOptions.beamDeform
  hubcapSupportOptions.beamStrength = hubcapSupportOptions.hubcapSupportBeamStrength  or  hubcapSupportOptions.beamStrength
  hubcapSupportOptions.breakGroup   = nil
  hubcapSupportOptions.wheelID      = nil
  hubcapSupportOptions.disableMeshBreaking = true

  local sideOptions = deepcopy(cleanwheel)
  sideOptions.beamSpring        = sideOptions.wheelSideBeamSpring          and sideOptions.wheelSideBeamSpring          * fullStif or 0
  sideOptions.beamDamp          = sideOptions.wheelSideBeamDamp            and sideOptions.wheelSideBeamDamp            * fullStif or 0
  sideOptions.beamDeform        = sideOptions.wheelSideBeamDeform          and sideOptions.wheelSideBeamDeform          * fullStif or (sideOptions.beamDeform      and sideOptions.beamDeform      * fullStif or sideOptions.beamDeform     )
  sideOptions.beamStrength      = sideOptions.wheelSideBeamStrength        and sideOptions.wheelSideBeamStrength        * fullStif or (sideOptions.beamStrength    and sideOptions.beamStrength    * fullStif or sideOptions.beamStrength   )
  sideOptions.springExpansion   = sideOptions.wheelSideBeamSpringExpansion and sideOptions.wheelSideBeamSpringExpansion * fullStif or (sideOptions.springExpansion and sideOptions.springExpansion * fullStif or sideOptions.springExpansion)
  sideOptions.dampExpansion     = sideOptions.wheelSideBeamDampExpansion   and sideOptions.wheelSideBeamDampExpansion   * fullStif or (sideOptions.dampExpansion   and sideOptions.dampExpansion   * fullStif or sideOptions.dampExpansion  )
  sideOptions.transitionZone    = sideOptions.wheelSideTransitionZone       or sideOptions.transitionZone
  sideOptions.beamPrecompression= sideOptions.wheelSideBeamPrecompression   or 1

  local pyramidOptions = deepcopy(cleanwheel)
  pyramidOptions.beamSpring        = sideOptions.wheelSideBeamSpring          and sideOptions.wheelSideBeamSpring          * antiSquish  * fullStif or 0
  pyramidOptions.beamDamp          = sideOptions.wheelSideBeamDamp            and sideOptions.wheelSideBeamDamp            * antiSquish  * fullStif or 0
  pyramidOptions.beamDeform        = sideOptions.wheelSideBeamDeform          and sideOptions.wheelSideBeamDeform          * antiSquish  * fullStif or (sideOptions.beamDeform      and sideOptions.beamDeform      * fullStif * antiSquish or sideOptions.beamDeform     )
  pyramidOptions.beamStrength      = sideOptions.wheelSideBeamStrength        and sideOptions.wheelSideBeamStrength        * antiSquish  * fullStif or (sideOptions.beamStrength    and sideOptions.beamStrength    * fullStif * antiSquish or sideOptions.beamStrength   )
  pyramidOptions.springExpansion   = sideOptions.wheelSideBeamSpringExpansion and sideOptions.wheelSideBeamSpringExpansion * antiBulge   * fullStif or (sideOptions.springExpansion and sideOptions.springExpansion * fullStif * antiBulge  or sideOptions.springExpansion)
  pyramidOptions.dampExpansion     = sideOptions.wheelSideBeamDampExpansion   and sideOptions.wheelSideBeamDampExpansion   * antiBulge   * fullStif or (sideOptions.dampExpansion   and sideOptions.dampExpansion   * fullStif * antiBulge  or sideOptions.dampExpansion  )
  pyramidOptions.transitionZone    = sideOptions.wheelSideTransitionZone       or sideOptions.transitionZone
  pyramidOptions.beamPrecompression= sideOptions.wheelSideBeamPrecompression   or 1

  local pAntiRein50 = pAntiRein * 0.25
  local sideReinfOptions = deepcopy(sideOptions)
  sideReinfOptions.beamSpring           = sideOptions.wheelSideReinfBeamSpring          and sideOptions.wheelSideReinfBeamSpring          * pAntiRein   * fullStif * weightComp or (sideOptions.wheelSideBeamSpring and sideOptions.wheelSideBeamSpring * pAntiRein   * fullStif * weightComp or sideOptions.wheelSideBeamSpring)
  sideReinfOptions.beamDamp             = sideOptions.wheelSideReinfBeamDamp            and sideOptions.wheelSideReinfBeamDamp            * pAntiRein   * fullStif * weightComp or (sideOptions.wheelSideBeamDamp   and sideOptions.wheelSideBeamDamp   * pAntiRein   * fullStif * weightComp or sideOptions.wheelSideBeamDamp  )
  sideReinfOptions.beamDeform           = sideOptions.wheelSideReinfBeamDeform          and sideOptions.wheelSideReinfBeamDeform          * pAntiRein   * fullStif * weightComp or (sideOptions.beamDeform          and sideOptions.beamDeform          * pAntiRein   * fullStif * weightComp or sideOptions.beamDeform     )
  sideReinfOptions.beamStrength         = sideOptions.wheelSideReinfBeamStrength        and sideOptions.wheelSideReinfBeamStrength        * pAntiRein   * fullStif * weightComp or (sideOptions.beamStrength        and sideOptions.beamStrength        * pAntiRein   * fullStif * weightComp or sideOptions.beamStrength   )
  sideReinfOptions.springExpansion      = sideOptions.wheelSideReinfBeamSpringExpansion and sideOptions.wheelSideReinfBeamSpringExpansion * pAntiRein50 * fullStif * weightComp or (sideOptions.springExpansion     and sideOptions.springExpansion     * pAntiRein50 * fullStif * weightComp or sideOptions.springExpansion)
  sideReinfOptions.dampExpansion        = sideOptions.wheelSideReinfBeamDampExpansion   and sideOptions.wheelSideReinfBeamDampExpansion   * pAntiRein50 * fullStif * weightComp or (sideOptions.dampExpansion       and sideOptions.dampExpansion       * pAntiRein50 * fullStif * weightComp or sideOptions.dampExpansion  )
  sideReinfOptions.transitionZone       = sideOptions.wheelSideReinfTransitionZone      or sideOptions.transitionZone
  sideReinfOptions.beamPrecompression   = sideOptions.wheelSideReinfBeamPrecompression  or 1
  sideReinfOptions.disableMeshBreaking  = true

  local reinfOptions = deepcopy(cleanwheel)
  reinfOptions.beamSpring         = reinfOptions.wheelReinfBeamSpring           and reinfOptions.wheelReinfBeamSpring          * xPat*fullStif or 0
  reinfOptions.beamDamp           = reinfOptions.wheelReinfBeamDamp             and reinfOptions.wheelReinfBeamDamp            * xPat*fullStif or 0
  reinfOptions.springExpansion    = reinfOptions.wheelReinfBeamSpringExpansion  and reinfOptions.wheelReinfBeamSpringExpansion * xPat*fullStif or (reinfOptions.wheelReinfBeamSpringExpansion and reinfOptions.wheelReinfBeamSpringExpansion * xPat * fullStif  or reinfOptions.wheelReinfBeamSpringExpansion)
  reinfOptions.dampExpansion      = reinfOptions.wheelReinfBeamDampExpansion    and reinfOptions.wheelReinfBeamDampExpansion   * xPat*fullStif or (reinfOptions.wheelReinfBeamDampExpansion   and reinfOptions.wheelReinfBeamDampExpansion   * xPat * fullStif  or reinfOptions.wheelReinfBeamDampExpansion  )
  reinfOptions.beamDeform         = reinfOptions.wheelReinfBeamDeform           and reinfOptions.wheelReinfBeamDeform          * xPat*fullStif or (reinfOptions.beamDeform                    and reinfOptions.beamDeform                    * xPat * fullStif  or reinfOptions.beamDeform)
  reinfOptions.beamStrength       = reinfOptions.wheelReinfBeamStrength         and reinfOptions.wheelReinfBeamStrength        * xPat*fullStif or (reinfOptions.beamStrength                  and reinfOptions.beamStrength                  * xPat * fullStif  or reinfOptions.beamStrength)
  reinfOptions.beamPrecompression = reinfOptions.wheelReinfBeamPrecompression   or 1
  reinfOptions.dampCutoffHz       = reinfOptions.wheelReinfBeamDampCutoffHz     or nil
  reinfOptions.disableMeshBreaking= true

  local cornerReinfOptions = deepcopy(cleanwheel)
  cornerReinfOptions.beamSpring         = reinfOptions.wheelReinfBeamSpring           and reinfOptions.wheelReinfBeamSpring          * pCornerL*fullStif or 0
  cornerReinfOptions.beamDamp           = reinfOptions.wheelReinfBeamDamp             and reinfOptions.wheelReinfBeamDamp            * pCornerL*fullStif or 0
  cornerReinfOptions.springExpansion    = reinfOptions.wheelReinfBeamSpringExpansion  and reinfOptions.wheelReinfBeamSpringExpansion * pCornerL*fullStif or (reinfOptions.wheelReinfBeamSpringExpansion and reinfOptions.wheelReinfBeamSpringExpansion * pCornerL * fullStif  or reinfOptions.wheelReinfBeamSpringExpansion)
  cornerReinfOptions.dampExpansion      = reinfOptions.wheelReinfBeamDampExpansion    and reinfOptions.wheelReinfBeamDampExpansion   * pCornerL*fullStif or (reinfOptions.wheelReinfBeamDampExpansion   and reinfOptions.wheelReinfBeamDampExpansion   * pCornerL * fullStif  or reinfOptions.wheelReinfBeamDampExpansion  )
  cornerReinfOptions.beamDeform         = reinfOptions.wheelReinfBeamDeform           and reinfOptions.wheelReinfBeamDeform          * pCornerL*fullStif or (reinfOptions.beamDeform                    and reinfOptions.beamDeform                    * pCornerL * fullStif  or reinfOptions.beamDeform)
  cornerReinfOptions.beamStrength       = reinfOptions.wheelReinfBeamStrength         and reinfOptions.wheelReinfBeamStrength        * pCornerL*fullStif or (reinfOptions.beamStrength                  and reinfOptions.beamStrength                  * pCornerL * fullStif  or reinfOptions.beamStrength)
  cornerReinfOptions.beamPrecompression = reinfOptions.wheelReinfBeamPrecompression   or 1
  cornerReinfOptions.dampCutoffHz       = reinfOptions.wheelReinfBeamDampCutoffHz     or nil
  cornerReinfOptions.disableMeshBreaking= true

  local treadOptions = deepcopy(cleanwheel)
  treadOptions.beamSpring         = treadOptions.wheelTreadBeamSpring         and treadOptions.wheelTreadBeamSpring  * weightComp * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring  * weightComp * fullStif or treadOptions.beamSpring  )
  treadOptions.beamDamp           = treadOptions.wheelTreadBeamDamp           and treadOptions.wheelTreadBeamDamp    * weightComp * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp    * weightComp * fullStif or treadOptions.beamDamp    )
  treadOptions.beamDeform         = treadOptions.wheelTreadBeamDeform         and treadOptions.wheelTreadBeamDeform   * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform   * fullStif or treadOptions.beamDeform  )
  treadOptions.beamStrength       = treadOptions.wheelTreadBeamStrength       and treadOptions.wheelTreadBeamStrength * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength * fullStif or treadOptions.beamStrength)
  treadOptions.beamPrecompression = treadOptions.wheelTreadBeamPrecompression or  1
  treadOptions.dampCutoffHz       = treadOptions.wheelTreadBeamDampCutoffHz   or  nil
  treadOptions.disableMeshBreaking= true

  local treadReinfOptions = deepcopy(treadOptions)
  treadReinfOptions.beamSpring          = treadOptions.wheelTreadReinfBeamSpring          and treadOptions.wheelTreadReinfBeamSpring   * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring  * fullStif or treadOptions.beamSpring  )
  treadReinfOptions.beamDamp            = treadOptions.wheelTreadReinfBeamDamp            and treadOptions.wheelTreadReinfBeamDamp     * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp    * fullStif or treadOptions.beamDamp    )
  treadReinfOptions.beamDeform          = treadOptions.wheelTreadReinfBeamDeform          and treadOptions.wheelTreadReinfBeamDeform   * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform  * fullStif or treadOptions.beamDeform  )
  treadReinfOptions.beamStrength        = treadOptions.wheelTreadReinfBeamStrength        and treadOptions.wheelTreadReinfBeamStrength * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength* fullStif or treadOptions.beamStrength)
  treadReinfOptions.beamPrecompression  = treadOptions.wheelTreadReinfBeamPrecompression  or 1
  treadReinfOptions.dampCutoffHz        = treadOptions.wheelTreadReinfBeamDampCutoffHz    or nil
  treadReinfOptions.disableMeshBreaking = true
  if sideWallReinf >= 0.05 then
    treadReinfOptions.beamSpring  = treadOptions.wheelTreadReinfBeamSpring         and treadOptions.wheelTreadReinfBeamSpring   * sideWallReinf * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring   * sideWallReinf * fullStif or treadOptions.beamSpring  )
    treadReinfOptions.beamDamp    = treadOptions.wheelTreadReinfBeamDamp           and treadOptions.wheelTreadReinfBeamDamp     * sideWallReinf * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp     * sideWallReinf * fullStif or treadOptions.beamDamp    )
    treadReinfOptions.beamDeform  = treadOptions.wheelTreadReinfBeamDeform         and treadOptions.wheelTreadReinfBeamDeform   * sideWallReinf * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform   * sideWallReinf * fullStif or treadOptions.beamDeform  )
    treadReinfOptions.beamStrength= treadOptions.wheelTreadReinfBeamStrength       and treadOptions.wheelTreadReinfBeamStrength * sideWallReinf * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength * sideWallReinf * fullStif or treadOptions.beamStrength)
  end

  local peripheryOptions = deepcopy(treadOptions)
  peripheryOptions.beamSpring         = peripheryOptions.wheelPeripheryBeamSpring   and peripheryOptions.wheelPeripheryBeamSpring   * fullStif  or (peripheryOptions.beamSpring   and peripheryOptions.beamSpring   * fullStif or peripheryOptions.beamSpring  )
  peripheryOptions.beamDamp           = peripheryOptions.wheelPeripheryBeamDamp     and peripheryOptions.wheelPeripheryBeamDamp     * fullStif  or (peripheryOptions.beamDamp     and peripheryOptions.beamDamp     * fullStif or peripheryOptions.beamDamp    )
  peripheryOptions.beamDeform         = peripheryOptions.wheelPeripheryBeamDeform   and peripheryOptions.wheelPeripheryBeamDeform   * fullStif  or (peripheryOptions.beamDeform   and peripheryOptions.beamDeform   * fullStif or peripheryOptions.beamDeform  )
  peripheryOptions.beamStrength       = peripheryOptions.wheelPeripheryBeamStrength and peripheryOptions.wheelPeripheryBeamStrength * fullStif  or (peripheryOptions.beamStrength and peripheryOptions.beamStrength * fullStif or peripheryOptions.beamStrength)
  peripheryOptions.beamPrecompression = peripheryOptions.wheelPeripheryBeamPrecompression or 1
  peripheryOptions.dampCutoffHz       = peripheryOptions.wheelPeripheryBeamDampCutoffHz   or nil

  local peripheryReinfOptions = deepcopy(peripheryOptions)
  peripheryReinfOptions.beamSpring        = peripheryReinfOptions.wheelPeripheryReinfBeamSpring   and peripheryReinfOptions.wheelPeripheryReinfBeamSpring   * fullStif  or (peripheryOptions.beamSpring   and peripheryOptions.beamSpring   * fullStif or peripheryOptions.beamSpring  )
  peripheryReinfOptions.beamDamp          = peripheryReinfOptions.wheelPeripheryReinfBeamDamp     and peripheryReinfOptions.wheelPeripheryReinfBeamDamp     * fullStif  or (peripheryOptions.beamDamp     and peripheryOptions.beamDamp     * fullStif or peripheryOptions.beamDamp    )
  peripheryReinfOptions.beamDeform        = peripheryReinfOptions.wheelPeripheryReinfBeamDeform   and peripheryReinfOptions.wheelPeripheryReinfBeamDeform   * fullStif  or (peripheryOptions.beamDeform   and peripheryOptions.beamDeform   * fullStif or peripheryOptions.beamDeform  )
  peripheryReinfOptions.beamStrength      = peripheryReinfOptions.wheelPeripheryReinfBeamStrength and peripheryReinfOptions.wheelPeripheryReinfBeamStrength * fullStif  or (peripheryOptions.beamStrength and peripheryOptions.beamStrength * fullStif or peripheryOptions.beamStrength)
  peripheryReinfOptions.beamPrecompression= peripheryReinfOptions.wheelPeripheryReinfBeamPrecompression or 1
  peripheryReinfOptions.dampCutoffHz      = peripheryOptions.wheelPeripheryReinfBeamDampCutoffHz        or nil

  --treadOptions.beamSpring = treadOptions.beamSpring * (weightComp - weightComp*weightComp)
  --treadOptions.beamDamp   = treadOptions.beamDamp   * (weightComp - weightComp*weightComp)


  cleanupBeamOptions(cleanupWheelOptions(hubcapAttachOptions))
  cleanupBeamOptions(cleanupWheelOptions(sideOptions))
  cleanupBeamOptions(cleanupWheelOptions(sideReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(reinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(treadOptions))
  cleanupBeamOptions(cleanupWheelOptions(treadReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(peripheryOptions))
  cleanupBeamOptions(cleanupWheelOptions(peripheryReinfOptions))

  vehicle.triangles = vehicle.triangles or {}
  local pressureGroupName = '_wheelPressureGroup' .. wheel.wheelID
  local wheelPressure = wheel.pressurePSI or 10
  local wheelDragCoef = wheel.dragCoef or 100
  local wheelSkinDragCoef = wheel.skinDragCoef
  local wheelTreadTriangleType = NORMALTYPE
  local wheelSide1TriangleType = NORMALTYPE
  local wheelSide2TriangleType = NORMALTYPE

  if (wheel.triangleCollision or false) == false then
    wheelTreadTriangleType = NONCOLLIDABLE
    wheelSide1TriangleType = NONCOLLIDABLE
    wheelSide2TriangleType = NONCOLLIDABLE
  end

  if wheel.treadTriangleCollision == false then
    wheelTreadTriangleType = NONCOLLIDABLE
  end

  if wheel.side1TriangleCollision == false then
    wheelSide1TriangleType = NONCOLLIDABLE
  end

  if wheel.side2TriangleCollision == false then
    wheelSide2TriangleType = NONCOLLIDABLE
  end

  local hubTriangleCollision = wheel.hubTriangleCollision == true and true or false
  local hubSide1TriangleCollision = wheel.hubSide1TriangleCollision == true and true or false
  local hubSide2TriangleCollision = wheel.hubSide2TriangleCollision == true and true or false

  local rimBeams = {}
  local sideBeams = {}
  local peripheryBeams = {}
  local treadBeams = {}
  local reinfBeams = {}

  local inaxisnode = wheel.node1
  local outaxisnode = wheel.node2
  local vTris = vehicle.triangles

  --modified by fillman86 ******************************************************************* V
  local timesdone = 0

  for i = 0, newNumRay - 1 do
    local i2 = 2*i --keep
    local nextdelta = 2*((i+1)%newNumRay) --keep
    local outhubnode = hubnodebase + i2 --keep
    local inhubnode = outhubnode + 1 --keep
    local nextouthubnode = hubnodebase + nextdelta --keep
    local nextinhubnode = nextouthubnode + 1 --keep

    local curBand     = treadBands[i+1]
    local nexBand     = treadBands[i+2] and treadBands[i+2] or treadBands[(i+2)-(newNumRay)]
    local nexNexBand  = treadBands[i+3] and treadBands[i+3] or treadBands[(i+3)-(newNumRay)]
    local nex3Band    = treadBands[i+4] and treadBands[i+4] or treadBands[(i+4)-(newNumRay)]
    local preBand     = treadBands[i] and treadBands[i] or treadBands[newNumRay]
    local prePreBand  = treadBands[i-1] and treadBands[i-1] or treadBands[newNumRay-1]

    -- Hub caps
    if wheel.enableHubcaps and newNumRay%2 ~= 1 and i < ((newNumRay)/2) then
      local hubcapnode = hubcapnodebase + i
      local nexthubcapnode = hubcapnodebase + ((i+1)%(newNumRay/2))
      local nextnexthubcapnode = hubcapnodebase + ((i+2)%(newNumRay/2))
      local hubcapaxisnode = hubcapnode + (newNumRay/2) - i
      local hubcapinhubnode = inhubnode + i2
      local prevhubcaphubnode = hubnodebase + 2 *((i2 + newNumRay - 1)%newNumRay)+1
      local nexthubcapinhubnode = hubcapinhubnode + 2
      local hubcapouthubnode = hubcapinhubnode - 1

      --hubcap periphery
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, nexthubcapnode,    NORMALTYPE, hubcapOptions)
      --attach to center node
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapaxisnode,    NORMALTYPE, hubcapOptions)
      --attach to axis
      if wheel.enableExtraHubcapBeams == true then
        --jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node1, NORMALTYPE, hubcapAttachOptions)
        --jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node2, NORMALTYPE, hubcapAttachOptions)
        if i == 1 then
          jbeamUtils.addBeamWithOptions(vehicle, hubcapaxisnode, wheel.node1, NORMALTYPE, hubcapAttachOptions)
          jbeamUtils.addBeamWithOptions(vehicle, hubcapaxisnode, wheel.node2, NORMALTYPE, hubcapAttachOptions)
        end
      end

      --span beams
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, nextnexthubcapnode,    NORMALTYPE, hubcapOptions)

      --attach it
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapinhubnode,   NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, prevhubcaphubnode, NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapouthubnode,  NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapouthubnode,  BEAM_SUPPORT, hubcapSupportOptions)

      --self:jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node1,    NORMALTYPE, hubcapAttachOptions)
      --self:jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node2,    NORMALTYPE, hubcapAttachOptions)
    end

    --hub tread
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, inhubnode,      NORMALTYPE, hubTreadOptions)
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  nextouthubnode, NORMALTYPE, hubTreadOptions)

    --hub periphery beams
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, nextouthubnode, NORMALTYPE, hubPeripheryOptions)
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  nextinhubnode,  NORMALTYPE, hubPeripheryOptions)

    --hub axis beams
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, inaxisnode, NORMALTYPE, hubReinfOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, outaxisnode, NORMALTYPE, hubSideOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  inaxisnode, NORMALTYPE, hubSideOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  outaxisnode, NORMALTYPE, hubReinfOptions))

    --Beams to stability node
    if nodeStabilizerExists then
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,  wheel.nodeStabilizer, NORMALTYPE, hubStabilizerOptions)
    end

    if hubSide1TriangleCollision then
      addTri(vTris, nextouthubnode, outhubnode, outaxisnode, wheelDragCoef * 0.5, NORMALTYPE)
    end

    if hubSide2TriangleCollision then
      addTri(vTris, inhubnode, nextinhubnode, inaxisnode, wheelDragCoef * 0.5, NORMALTYPE)
    end      

    --MARK: beams
    if tireExists then
      local tbnc = 4+treadDiv --tbnc is short for "tread band's node count"
      --to bring back dupeCheck, replace all following "table.insert" with "dupeCheck", and it'll check all beams
      --tire tread
      for cNode = 1, tbnc-1 do
        --connecting the band transversely
        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, curBand[cNode+1].cid, NORMALTYPE, treadOptions))
        --connecting forward, and +1 node along
        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexBand[cNode+1].cid, NORMALTYPE, treadOptions))
        if cNode >= 2 and cNode <= tbnc-2 then --this would create a wierd beam on the first node
          --connects the next node, to the next band's current node
          table.insert(treadBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode+1].cid, nexBand[cNode].cid, NORMALTYPE, treadOptions))
        end
      end

      --testing square walls, these create the x pattern, inside the squares
      table.insert(treadBeams,
      jbeamUtils.addBeamWithOptions(vehicle, curBand[2].cid, nexBand[1].cid, NORMALTYPE, treadOptions))
      table.insert(treadBeams,
      jbeamUtils.addBeamWithOptions(vehicle, curBand[tbnc].cid, nexBand[tbnc-1].cid, NORMALTYPE, treadOptions))

      --lateral sidewall reinforcement. This is a bit of a cheat
      if sideWallReinf >= 0.05 then
        table.insert(treadBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[tbnc].cid, NORMALTYPE, treadReinfOptions))
      end

      --paired treadnodes. To do with hidden elastic floor stuff, for rolling resistance
      if vehicle.variables['$xxpaired'].val == 0 then
        for cNode = 2, tbnc/2 do
          --[[ vehicle.nodes[curBand[cNode].cid].pairedNode = nexBand[(tbnc+1)-cNode].cid
          vehicle.nodes[nexBand[(tbnc+1)-cNode].cid].pairedNode = curBand[cNode].cid

          vehicle.nodes[curBand[cNode].cid].pairedNode2 = preBand[(tbnc+1)-cNode].cid
          vehicle.nodes[preBand[(tbnc+1)-cNode].cid].pairedNode2 = curBand[cNode].cid ]]
          vehicle.nodes[curBand[cNode].cid].pairedNode = curBand[(tbnc+1)-cNode].cid
          vehicle.nodes[curBand[(tbnc+1)-cNode].cid].pairedNode = curBand[cNode].cid

          vehicle.nodes[curBand[cNode].cid].pairedNode2 = preBand[(tbnc+1)-cNode].cid
          vehicle.nodes[preBand[(tbnc+1)-cNode].cid].pairedNode2 = curBand[cNode].cid
        end
        --[[ for cNode = 2, tbnc-1 do
          vehicle.nodes[curBand[cNode].cid].pairedNode  = nexBand[cNode].cid
          vehicle.nodes[curBand[cNode].cid].pairedNode2 = preBand[cNode].cid
        end ]]

        if not (tbnc%2==0) then
          local middleNode = math.floor(tbnc/2)+1
          vehicle.nodes[curBand[middleNode].cid].pairedNode = nexBand[middleNode].cid
          vehicle.nodes[curBand[middleNode].cid].pairedNode2 = preBand[middleNode].cid
        end
      end

      -- All periphery beams
      for cNode = 1, tbnc do
        table.insert(peripheryBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexBand[cNode].cid,    NORMALTYPE, peripheryOptions))
      end

      -- Periphery reinforcement beams
      for cNode = 2, tbnc-1 do
        table.insert(peripheryBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexNexBand[cNode].cid,    BEAM_ANISOTROPIC, sideReinfOptions))

        table.insert(peripheryBeams, --skip 3 bands ahead
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nex3Band[cNode].cid,    BEAM_ANISOTROPIC, sideReinfOptions))
      end

      --wheel->tire wall beams. Connect wheelNode to either tireNode
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,curBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, curBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,preBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, preBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))--default is BEAM_ANISOTROPIC, sideOptions <^ alternative is NORMALTYPE, treadOptions
      --hubNode square walls, x-pattern fix
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,prePreBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, nexBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))--default is BEAM_ANISOTROPIC, sideOptions <^ alternative is NORMALTYPE, treadOptions

      --adding triangles wheel <-to-> tread strands, to stop bulging
      if antiBulge >=0.05 or antiSquish >=0.05 then
        if (tbnc%2==0) then
          for CurrentNode = 3, tbnc - 2 do
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, inhubnode,     curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, outhubnode,    curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextinhubnode, curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode,curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
          end
        else  
          for CurrentNode = 3, tbnc - 2 do
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, inhubnode,     curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, outhubnode,    curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextinhubnode, curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode,curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
          end
        end
      end

      --note: I've left in a bunch of alternative beam options if people want to experiment with re-enabling them |WARNING: may be incompatible now

      --wheel->tire's corner: forward "L-Beams"
      --[[ dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  outhubnode,     curBand[1].cid,         BEAM_LBEAM, reinfOptions, inhubnode))
      dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  inhubnode,      curBand[tbnc].cid,      BEAM_LBEAM, reinfOptions, outhubnode))
      if i>= 1 then--wheel->tire's corner: backwards "L-Beams
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,outhubnode,     preBand[1].cid,             BEAM_LBEAM, reinfOptions, inhubnode))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,inhubnode,      preBand[tbnc].cid,          BEAM_LBEAM, reinfOptions, outhubnode))
      else
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,outhubnode,lastBand[1].cid,    BEAM_LBEAM, reinfOptions, inhubnode))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,inhubnode, lastBand[tbnc].cid, BEAM_LBEAM, reinfOptions, outhubnode))
      end ]]

      --first sidewall L-Beams
      --[[ dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  inhubnode, curBand[2].cid,         BEAM_LBEAM, reinfOptions, curBand[1].cid))
      dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  outhubnode,curBand[tbnc-1].cid,BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))
       ]]
      --[[ if xPat >= 0.05 then
        for eachWidthX = 2, tbnc-3 do 
          if i<=newNumRay-3 then --tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   nexNexBand[eachWidthX+2].cid,BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, nexNexBand[eachWidthX].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
          end
          if i==newNumRay-2 then -- second to last tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   firstBand[eachWidthX+2].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, firstBand[eachWidthX].cid,    BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
          end 
          if i==newNumRay-1 then -- last tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   treadBands[2][eachWidthX+2].cid,  BEAM_LBEAM, reinfOptions, firstBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, treadBands[2][eachWidthX].cid,    BEAM_LBEAM, reinfOptions, firstBand[eachWidthX+1].cid))
          end  ]]
          --[[ if i>2 and (eachWidthX>=3 or eachWidthX==tbnc-2) then --better corner L-Beams
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[eachWidthX].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))          
          end
        end
      end ]]

      --tyre corner L-Beams
      if pCornerL ~= 0 then
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[3].cid, BEAM_LBEAM, cornerReinfOptions, curBand[2].cid))
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[tbnc].cid, curBand[tbnc-2].cid, BEAM_LBEAM, cornerReinfOptions, curBand[tbnc-1].cid))
      end
  
      -- Wheel -> corner beams to rigidify sidewalls (same side, not crossing sides), specifically to help rolling resistance, like irl tyres
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  curBand[2].cid,          BEAM_LBEAM, reinfOptions, curBand[1].cid))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode, curBand[tbnc-1].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))
      if treadDiv > 1 then --this is to give extra l beams inwards
        --[[ dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  curBand[3].cid,          BEAM_LBEAM, reinfOptions, curBand[2].cid))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode, curBand[tbnc-2].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc-1].cid)) ]]
      end

      --trying periphery L-Beams |WARNING: probably not compatible anymore
      --[[ for eachWidthX = 2, tbnc-1 do 
        if i<=newNumRay-3 then
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, nexNexBand[eachWidthX].cid,BEAM_LBEAM, reinfOptions, nexBand[eachWidthX].cid))
        end
        if i==newNumRay-2 then -- second to last tread L-Beams
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, firstBand[eachWidthX].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX].cid))
        end 
        if i==newNumRay-1 then -- last tread L-Beams
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, treadBands[2][eachWidthX].cid,  BEAM_LBEAM, reinfOptions, firstBand[eachWidthX].cid))
        end 
      end ]]

      --opposite node trick
      for nods = 2, tbnc-1 do 
        --timesdone = timesdone + 1
        --print("timesdone -"..timesdone)
        if i<=(newNumRay*0.5)-1 then
          --dupeCheck(sideBeams,
          --jbeamUtils.addBeamWithOptions(vehicle, curBand[nods].cid,curBand[nods].oppositeTreadNodeCid, NORMALTYPE, treadOptions))
        end
      end
  



      --[[ for eachStrapNode = 1, tbnc -2 do --this is L-Beams for each perfect band
        dupeCheck( reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachStrapNode].cid, curBand[eachStrapNode+2].cid, BEAM_LBEAM, reinfOptions,  curBand[eachStrapNode+1].cid))
      end ]]

      --checking if inverse tyires are needed
      if inverse >=0 then 
        -- hub pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextouthubnode, outhubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)
        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextinhubnode, nextouthubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)

        -- wheel-tire pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, curBand[tbnc].cid, outhubnode, nextouthubnode, wheelDragCoef * 0.5, wheelSide1TriangleType)
        addPressTri(vTris, pressureGroupName, wheelPressure, curBand[tbnc].cid, nextouthubnode, nexBand[tbnc].cid, wheelDragCoef * 0.5, wheelSide1TriangleType)

        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, preBand[1].cid, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
        addPressTri(vTris, pressureGroupName, wheelPressure, nextinhubnode, inhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)

        for CurrentNode=1,tbnc do
          if (CurrentNode%2~=0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, nexBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, curBand[CurrentNode+1].cid, nexBand[CurrentNode+1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          elseif (CurrentNode%2==0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, curBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, nexBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          end
        end
      else
        -- hub pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, nextouthubnode, inhubnode, outhubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)
        addPressTri(vTris, pressureGroupName, wheelPressure, nextinhubnode, inhubnode, nextouthubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)

        -- wheel-tire pressure tris
          addPressTri(vTris, pressureGroupName, wheelPressure, outhubnode, curBand[tbnc].cid, nextouthubnode, wheelDragCoef * 0.5, wheelSide1TriangleType)
          addPressTri(vTris, pressureGroupName, wheelPressure, nextouthubnode, curBand[tbnc].cid, nexBand[tbnc].cid, wheelDragCoef * 0.5, wheelSide1TriangleType)

        -- inside pressure tris
          addPressTri(vTris, pressureGroupName, wheelPressure,     preBand[1].cid, inhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextinhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)

        for CurrentNode=1,tbnc do
          if (CurrentNode%2~=0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, nexBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode+1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          elseif (CurrentNode%2==0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, nexBand[CurrentNode+1].cid, curBand[CurrentNode+1].cid,nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          end
        end
      end
    else
      if hubTriangleCollision then
        -- hub tris
        addTri(vTris, nextouthubnode, inhubnode, outhubnode, wheelDragCoef * 0.1, NORMALTYPE)
        addTri(vTris, nextinhubnode, inhubnode, nextouthubnode, wheelDragCoef * 0.1, NORMALTYPE)
      end
    end --end of modified by fillman86 ************************************************* ^
    wheelsToRotators = deepcopy(vehicle)
  end

  --[[ if wheelKey == 0 then
    for a,b in ipairs(vehicle.nodes) do
      if b.pairedNode then
        if b.pairedNode2 then
          print(wheelKey .. "|tyre node- ".. b.cid .. ", to node- ".. b.pairedNode .. " and ".. b.pairedNode2)
        else
          print(wheelKey .. "|tyre node- ".. b.cid ..", only 1 pairednode- ".. b.pairedNode)
        end
      end
      if b.pairedNode2 and not b.pairedNode then
        print(wheelKey .. "|tyre node- ".. b.cid ..", only  pairednode2- ".. b.pairedNode2)
      end
    end
  end ]]

  wheel.nodes = hubNodes
  wheel.treadNodes = treadNodes
  wheel.rimBeams = rimBeams
  wheel.sideBeams = sideBeams
  wheel.peripheryBeams = peripheryBeams
  wheel.reinfBeams = reinfBeams
  wheel.treadBeams = treadBeams
  wheel.pressureGroup = pressureGroupName
end

--MARK:add quilt
local function addPressureWheelDivQuilted(vehicle, wheelKey, wheel)  
  local node1   = vehicle.nodes[wheel.node1]
  local node2   = vehicle.nodes[wheel.node2]
  local node1_pos = vec3(node1.pos)
  local node2_pos = vec3(node2.pos)
  
  local treadDiv      = 3
  local bulgeOffset   = wheel.bulgeOffset   or (node1_pos.y > 0 and vehicle.variables['$rbulgeOffset'].val         or  vehicle.variables['$fbulgeOffset'].val      ) or 0       
  local sideWallReinf = wheel.sideWallReinf or (node1_pos.y > 0 and vehicle.variables['$rsideWallReinf'].val       or  vehicle.variables['$fsideWallReinf'].val    ) or 0.25    
  local antiBulge     = wheel.antiBulge     or (node1_pos.y > 0 and vehicle.variables['$rantiBulge'].val           or  vehicle.variables['$fantiBulge'].val        ) or 0       
  local antiSquish    = wheel.antiSquish    or (node1_pos.y > 0 and vehicle.variables['$rantiSquish'].val          or  vehicle.variables['$fantiSquish'].val       ) or 0       
  local xPat          = wheel.xPat          or (node1_pos.y > 0 and vehicle.variables['$rxPat'].val                or  vehicle.variables['$fxPat'].val             ) or 0.02    
  local fullStif      = wheel.fullStif      or (node1_pos.y > 0 and vehicle.variables['$rafullStif'].val           or  vehicle.variables['$fafullStif'].val        ) or 1       
  local pAntiRein     = wheel.pAntiRein     or (node1_pos.y > 0 and vehicle.variables['$rpAntiRein'].val           or  vehicle.variables['$fpAntiRein'].val        ) or 0.8     
  local pCornerL      = wheel.pCornerL      or (node1_pos.y > 0 and vehicle.variables['$rpCornerL'].val * 5        or  vehicle.variables['$fpCornerL'].val * 5     ) or 0.1     
  local newBulge      = wheel.newBulge      or (node1_pos.y > 0 and vehicle.variables['$rbulge'].val               or  vehicle.variables['$fbulge'].val            ) or 0       
  local xxFricComp    = wheel.xxFricComp    or (node1_pos.y > 0 and vehicle.variables['$rxxFricComp'].val/100      or  vehicle.variables['$fxxFricComp'].val/100   ) or 1       
  local xxSFricComp   = wheel.xxSFricComp   or (node1_pos.y > 0 and vehicle.variables['$rxxSFricComp'].val/100     or  vehicle.variables['$fxxSFricComp'].val/100  ) or 1       
  local newNumRay     = vehicle.variables['$rnumRays'] and (node1_pos.y > 0 and vehicle.variables['$rnumRays'].val or  vehicle.variables['$fnumRays'].val          ) or wheel.numRays
  local tyreWeight    = node1_pos.y > 0 and vehicle.variables['$raatreadWeight'].val/100  or  vehicle.variables['$faatreadWeight'].val/100
  local tyreWall      =  wheel.radius - wheel.hubRadius
  local tyreWallMiddle= (wheel.radius + wheel.hubRadius) / 2
  local tyreRatio     = (wheel.radius - wheel.hubRadius) / wheel.tireWidth
  
  -- Stabilizer
  wheel.nodeStabilizer = wheel.nodeStabilizer or wheel.nodeS
  wheel.treadCoef = wheel.treadCoef or 1
  wheel.nodeS = nil
  local nodeStabilizerExists = false
  local wheelAngleRad = math.rad(wheel.wheelAngle or 0)

  if wheel.nodeStabilizer and wheel.nodeStabilizer ~= 9999 and vehicle.nodes[wheel.nodeStabilizer] then
    nodeStabilizerExists = true
  else
    wheel.nodeStabilizer = nil
  end

  if node1 == nil or node2 == nil then
    log('W', "jbeam.addPressureWheel","invalid pressureWheel")
    return
  end

  local nodebase = vehicle.maxIDs.nodes
  --print("testing out vehicle.maxIDs.nodes- ".. tostring(vehicle.maxIDs.nodes) .. " -555555555555555555555555555")

  local tireExists = true
  if wheel.radius == nil then
    tireExists = false
    wheel.radius = 0.5
  end

  if wheel.pressurePSI == nil then
    tireExists = false
  end

  if wheel.hasTire ~= nil and wheel.hasTire == false then
    tireExists = false
  end

  if wheel.hubRadius == nil then wheel.hubRadius = 0.65 * wheel.radius end
  if newNumRay == nil then newNumRay = 10 end
  --newNumRay = 24

  -- add collision to the wheels nodes ;)
  wheel.collision = true

  -- calculate surface
  -- if tireExists and wheel.tireWidth and wheel.hubWidth and wheel.radius and wheel.hubRadius then
  --   wheel._surface = math.pi * (2 * (wheel.tireWidth * wheel.radius + wheel.hubWidth * wheel.hubRadius) +
  --     (wheel.radius + wheel.hubRadius) * math.sqrt(square(wheel.radius - wheel.hubRadius) + square(wheel.tireWidth - wheel.hubRadius)))
  -- end

  local middleNodes = {}

  local node4_pos = vec3(node2.pos)
  local node5_pos = vec3(node2.pos)

  local tyreBulge = newBulge or 0

  -- calculate axis
  local axis = node2_pos - node1_pos
  axis:normalize()

  local midpoint = (node2_pos + node1_pos) * 0.5
  if wheel.wheelOffset ~= nil then
    local offset = wheel.wheelOffset
    midpoint = midpoint + axis * offset
  end

  --MARK: node pos
  --this is lateral placement only, 3D is in the marked tyre section
  if wheel.tireWidth ~= nil then
    local halfWidth = 0.5 * wheel.tireWidth
    node1_pos = midpoint - axis * halfWidth
    node2_pos = midpoint + axis * halfWidth
    node4_pos = midpoint - axis * (wheel.tireWidth * ((tyreWall * 0.5) + 0.5 + tyreBulge))--sidewalls
    node5_pos = midpoint + axis * (wheel.tireWidth * ((tyreWall * 0.5) + 0.5 + tyreBulge))--sidewalls

    --calcualting band separation
    local postions = {}
    if treadDiv > 0 then
      local dividedWidth = wheel.tireWidth/(treadDiv+1)
      for i=1, (treadDiv/2) do --the "for" loop is to increasingly add more offset divisions
        local step = i/(treadDiv+1)
        local curMaths = ((1-step)^ tyreRatio) * wheel.tireWidth
        local evenMaths = wheel.tireWidth - (dividedWidth*i)
        local averageMaths = (curMaths + evenMaths*i)/((1+i))
        local currentPos = node2_pos - axis * averageMaths --position incrementally calculation, like normal nodes, except from opposite edge
        table.insert(middleNodes, currentPos)--table used for placing nodes
        table.insert(postions, averageMaths)--table for mirror purposes
      end
      local curLen = #postions --this is to order the new nodes into the right place
      if treadDiv % 2 ~= 0 then --if it's odd, then add a centre line
        local currentPos = midpoint - axis * 0
        table.insert(middleNodes, currentPos)--table used for placing nodes
        curLen = curLen + 1 --we need to make sure that the following nodes are after this centre node
      end
      for i = 1, (treadDiv/2) do --now we copy our homework from the first half
        local currentPos = node1_pos + axis * postions[i]
        table.insert(middleNodes, curLen+1, currentPos) --we add the nodes, but due to the order of the maths, we insert each new one from the centre(ish)
      end
    end    
  end

  local cleanwheel = deepcopy(wheel)
  cleanwheel.axleBeams = nil
  cleanwheel.childParts = nil
  cleanwheel.slotType = nil
  cleanwheel.enableABS = nil
  cleanwheel.enableBrakeThermals = nil
  cleanwheel.enableHubcaps = nil
  cleanwheel.enableTireLbeams = nil
  cleanwheel.enableTirePeripheryReinfBeams = nil
  cleanwheel.enableTireReinfBeams = nil
  cleanwheel.enableTireSideReinfBeams = nil
  cleanwheel.enableTreadReinfBeams = nil
  cleanwheel.hasTire = nil
  cleanwheel.brakeDiameter = nil
  cleanwheel.brakeInputSplit = nil
  cleanwheel.brakeMass = nil
  cleanwheel.brakeSplitCoef = nil
  cleanwheel.brakeSpring = nil
  cleanwheel.brakeTorque = nil
  cleanwheel.brakeType = nil
  cleanwheel.brakeVentingCoef = nil
  cleanwheel.heatCoefNodeToEnv = nil
  cleanwheel.heatCoefEnvMultStationary = nil
  cleanwheel.heatCoefEnvTerminalSpeed = nil
  cleanwheel.heatCoefNodeToCore = nil
  cleanwheel.heatCoefCoreToNodes = nil
  cleanwheel.heatCoefNodeToSurface = nil
  cleanwheel.heatCoefFriction = nil
  cleanwheel.heatCoefFlashFriction = nil
  cleanwheel.heatCoefStrain = nil
  cleanwheel.heatAffectsPressure = nil
  cleanwheel.smokingTemp = nil
  cleanwheel.meltingTemp = nil
  cleanwheel.frictionLowTemp = nil
  cleanwheel.frictionHighTemp = nil
  cleanwheel.frictionLowSlope = nil
  cleanwheel.frictionHighSlope = nil
  cleanwheel.frictionSlopeSmoothCoef = nil
  cleanwheel.frictionCoefLow = nil
  cleanwheel.frictionCoefMiddle = nil
  cleanwheel.frictionCoefHigh = nil

  local wheelNodes = cleanupWheelOptions(deepcopy(cleanwheel))
  --edited in by fillman86 ************************************************************** V
  --modify friction coefficients
  --rough napkin maths, on how many nodes probably contact the ground, at any given moment, above the standard 3 nodes
  local rayNodeCompensator  = math.max(1,math.floor(newNumRay/8))--more numRays means more nodes contacting the ground
  local divNodeCompensator  = ((treadDiv * rayNodeCompensator) + (rayNodeCompensator-1))/4--how many times more nodes contacting, vs default 3 nodes
  print("divNodeCompensator- "..divNodeCompensator)
  wheelNodes.frictionCoef   = (wheelNodes.frictionCoef / divNodeCompensator) * xxFricComp
  --set sidewall friction value
  local defaultFricCo       = wheelNodes.frictionCoef
  local sideWallFricCo      = wheelNodes.frictionCoef * 0.7
  wheelNodes.slidingFrictionCoef = wheelNodes.slidingFrictionCoef * xxSFricComp
  
  --if it's a wide tyre, we'll at least 'try' to decrease the weight a little, but not too much, it'll become unstable
  if wheel.tireWidth > 0.130 and treadDiv > 0 then
    wheelNodes.nodeWeight = (((wheelNodes.nodeWeight*treadDiv*2) + (wheelNodes.nodeWeight / treadDiv))/((treadDiv*2)+1))*tyreWeight
  end

  --this is used to mirror the node pattern for each side
  local inverse = 1
  if axis.x >= 0 then
    inverse = -1
  end

  local rayRotSWO   = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  local rayRotSWI   = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * (8/3) * inverse))
  local rayRotSWI1st= quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 8 * -inverse))
  local rayRot      = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  local rayRot2     = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 2 * inverse))
  rayRot2           = quatFromAxisAngle(axis, 2 * math.pi / (newNumRay * 4 * inverse)) * rayRot2

  local rayVec
  local rayVecSWO
  local rayVecSWI
  local treadNodes  = {}
  local treadBands  = {}
  local treadNodePos= {}
  if tireExists then
    local oppCount
    if treadDiv > 0 then
      oppCount = newNumRay * ((treadDiv + 4)/2)
   else
      oppCount = newNumRay * 2
    end
    rayVec = axis:perpendicularN() * wheel.radius --right angle, then giving dimension
    rayVec = quatFromAxisAngle(axis, -wheelAngleRad) * rayVec --this is adjusting any hard coded initial rotation
    rayVecSWO = axis:perpendicularN() * math.min(wheel.radius, math.max(wheel.hubRadius, tyreWallMiddle + bulgeOffset))
    rayVecSWO = quatFromAxisAngle(axis, -wheelAngleRad) * rayVecSWO

    rayVecSWI = axis:perpendicularN() * math.min(wheel.radius, math.max(wheel.hubRadius, tyreWallMiddle + bulgeOffset))
    rayVecSWI = quatFromAxisAngle(axis, -wheelAngleRad) * rayVecSWI

    --[[ rayVecSWI = rayRotSWI1st * rayVecSWO
    for i = 1, treadDiv+2, 2 do
      print("doing extra rotation")
      rayVecSWI = rayRot * rayVecSWI
    end ]]
    local hasEvenRayCount = newNumRay % 2 == 0
    --MARK: Tyres
    -- add nodes first
    for i = 0, newNumRay - 1, 1 do
      wheelNodes.frictionCoef = sideWallFricCo
      local currentBand = {}
      local currentNodePos = {}
      local rayPoint = node1_pos + rayVec
      -- outer sidewall
      rayPoint = node4_pos + rayVecSWO
      rayVecSWO = rayRot * rayVecSWO
      local n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos

      -- outer corner
      wheelNodes.frictionCoef = defaultFricCo
      rayPoint = node1_pos + rayVec
      --rayVec = rayRot * rayVec
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
        vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
      end

      --add middle nodes from table
      local flipper = false
      if treadDiv > 0 then
        for NodeCount,nodePos in ipairs(middleNodes) do -- start putting in the middle nodes
          flipper = not flipper
          local rayVecTemp = rayVec
          if flipper then
            print("forwards")
            rayVecTemp = rayRot * rayVec
          end
          rayPoint = nodePos + rayVecTemp
          n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
          table.insert(treadNodes, vehicle.nodes[n])
          table.insert(currentBand,vehicle.nodes[n])
          currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
          --[[ if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
            vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
          end ]]
        end
      end

      -- inner corner
      rayPoint = node2_pos + rayVec
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes, vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      if hasEvenRayCount then --safe opposite tread node for dynamic tire radius measurement
        vehicle.nodes[n].oppositeTreadNodeCid = (i <= (newNumRay - 1) / 2) and (n + oppCount) or (n - oppCount)
      end

      -- inner sidewall
      wheelNodes.frictionCoef = sideWallFricCo
      rayPoint = node5_pos + rayVecSWI
      rayVecSWI = rayRot * rayVecSWO
      n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, wheelNodes)
      table.insert(treadNodes,  vehicle.nodes[n])
      table.insert(currentBand, vehicle.nodes[n])
      rayVecSWO = rayRotSWO * rayVecSWO
      table.insert(treadBands, currentBand)
      currentNodePos[vehicle.nodes[n]] = vehicle.nodes[n].pos
      table.insert(treadNodePos, currentNodePos)
      currentNodePos = nil
      currentBand = nil
      rayVec = rayRot * rayRot * rayVec
    end --end of: edited by  added by fillman86 *************************************************** ^
  end
  wheelNodes.frictionCoef = defaultFricCo

  -- add Hub nodes
  local hubNodes = {}
  local hubnodebase = vehicle.maxIDs.nodes

  local hubOptions = deepcopy(cleanwheel)
  hubOptions.beamSpring = hubOptions.hubBeamSpring or hubOptions.beamSpring
  hubOptions.beamDamp = hubOptions.hubBeamDamp or hubOptions.beamDamp
  hubOptions.beamDeform = hubOptions.hubBeamDeform or hubOptions.beamDeform
  hubOptions.beamStrength = hubOptions.hubBeamStrength or hubOptions.beamStrength
  hubOptions.dampCutoffHz = hubOptions.hubBeamDampCutoffHz or nil
  hubOptions.nodeWeight = hubOptions.hubNodeWeight or hubOptions.nodeWeight
  hubOptions.collision = hubOptions.hubCollision or hubOptions.collision
  hubOptions.nodeMaterial = hubOptions.hubNodeMaterial or hubOptions.nodeMaterial
  hubOptions.frictionCoef = hubOptions.hubFrictionCoef or hubOptions.frictionCoef
  hubOptions.group = hubOptions.hubGroup or hubOptions.group
  hubOptions.disableMeshBreaking = hubOptions.disableHubMeshBreaking or hubOptions.disableMeshBreaking

  local hubSideOptions = deepcopy(hubOptions)
  hubSideOptions.beamSpring = hubSideOptions.hubSideBeamSpring or hubSideOptions.beamSpring
  hubSideOptions.beamDamp = hubSideOptions.hubSideBeamDamp or hubSideOptions.beamDamp
  hubSideOptions.beamDeform = hubSideOptions.hubSideBeamDeform or hubSideOptions.beamDeform
  hubSideOptions.beamStrength = hubSideOptions.hubSideBeamStrength or hubSideOptions.beamStrength
  hubSideOptions.dampCutoffHz = hubSideOptions.hubSideBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubReinfOptions = deepcopy(hubSideOptions)
  hubReinfOptions.beamSpring = hubReinfOptions.hubReinfBeamSpring or hubReinfOptions.beamSpring
  hubReinfOptions.beamDamp = hubReinfOptions.hubReinfBeamDamp or hubReinfOptions.beamDamp
  hubReinfOptions.beamDeform = hubReinfOptions.hubReinfBeamDeform or hubReinfOptions.beamDeform
  hubReinfOptions.beamStrength = hubReinfOptions.hubReinfBeamStrength or hubReinfOptions.beamStrength
  hubReinfOptions.dampCutoffHz = hubReinfOptions.hubReinfBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubTreadOptions = deepcopy(hubOptions)
  hubTreadOptions.beamSpring    = hubTreadOptions.hubTreadBeamSpring       and hubTreadOptions.hubTreadBeamSpring       * fullStif or hubTreadOptions.beamSpring   * fullStif
  hubTreadOptions.beamDamp      = hubTreadOptions.hubTreadBeamDamp         and hubTreadOptions.hubTreadBeamDamp         * fullStif or hubTreadOptions.beamDamp     * fullStif
  hubTreadOptions.beamDeform    = hubTreadOptions.hubTreadBeamDeform       and hubTreadOptions.hubTreadBeamDeform       or hubTreadOptions.beamDeform  
  hubTreadOptions.beamStrength  = hubTreadOptions.hubTreadBeamStrength     and hubTreadOptions.hubTreadBeamStrength     or hubTreadOptions.beamStrength
  hubTreadOptions.dampCutoffHz  = hubTreadOptions.hubTreadBeamDampCutoffHz and hubTreadOptions.hubTreadBeamDampCutoffHz or hubOptions.dampCutoffHz     

  local hubPeripheryOptions = deepcopy(hubOptions)
  hubPeripheryOptions.beamSpring = hubPeripheryOptions.hubPeripheryBeamSpring or hubPeripheryOptions.beamSpring
  hubPeripheryOptions.beamDamp = hubPeripheryOptions.hubPeripheryBeamDamp or hubPeripheryOptions.beamDamp
  hubPeripheryOptions.beamDeform = hubPeripheryOptions.hubPeripheryBeamDeform or hubPeripheryOptions.beamDeform
  hubPeripheryOptions.beamStrength = hubPeripheryOptions.hubPeripheryBeamStrength or hubPeripheryOptions.beamStrength
  hubPeripheryOptions.dampCutoffHz = hubPeripheryOptions.hubPeripheryBeamDampCutoffHz or hubOptions.dampCutoffHz

  local hubStabilizerOptions = deepcopy(hubSideOptions)
  hubStabilizerOptions.beamSpring = hubStabilizerOptions.hubStabilizerBeamSpring or hubStabilizerOptions.beamSpring
  hubStabilizerOptions.beamDamp = hubStabilizerOptions.hubStabilizerBeamDamp or hubStabilizerOptions.beamDamp
  hubStabilizerOptions.beamDeform = hubStabilizerOptions.hubStabilizerBeamDeform or hubStabilizerOptions.beamDeform
  hubStabilizerOptions.beamStrength = hubStabilizerOptions.hubStabilizerBeamStrength or hubStabilizerOptions.beamStrength
  hubStabilizerOptions.dampCutoffHz = hubStabilizerOptions.hubStabilizerBeamDampCutoffHz or hubOptions.dampCutoffHz

  cleanupWheelOptions(hubOptions) -- used for nodes
  cleanupBeamOptions(cleanupWheelOptions(hubSideOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubTreadOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubPeripheryOptions))
  cleanupBeamOptions(cleanupWheelOptions(hubStabilizerOptions))

  if wheel.hubWidth ~= nil then
    local halfWidth = 0.5 * wheel.hubWidth
    node1_pos = midpoint - axis * halfWidth
    node2_pos = midpoint + axis * halfWidth
  end

  --MARK: wheels
  rayVec = axis:perpendicularN() * wheel.hubRadius
  rayVec = quatFromAxisAngle(axis, -wheelAngleRad) * rayVec
  
  for i = 0, newNumRay - 1, 1 do
    -- inner
    local rayPoint = node2_pos + rayVec
    --rayVecI = rayRot * rayRot * rayVecI
    --rayVec = rayRotSWO * rayVec
    local n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubOptions)
    table.insert(hubNodes, vehicle.nodes[n])
    --local testLength = node2_pos - rayPoint
    --print("inner lenghts- " .. tostring(testLength:length()))

    -- outer
    rayPoint = node1_pos + rayVec
    n = jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubOptions)
    table.insert(hubNodes, vehicle.nodes[n])
    rayVec = rayRot * rayRot * rayVec
  end

  --Hub Cap
  local hubcapOptions = deepcopy(cleanwheel)
  hubcapOptions.beamSpring = hubcapOptions.hubcapBeamSpring or hubcapOptions.beamSpring
  hubcapOptions.beamDamp = hubcapOptions.hubcapBeamDamp or hubcapOptions.beamDamp
  hubcapOptions.beamDeform = hubcapOptions.hubcapBeamDeform or hubcapOptions.beamDeform
  hubcapOptions.beamStrength = hubcapOptions.hubcapBeamStrength or hubcapOptions.beamStrength
  hubcapOptions.nodeWeight = hubcapOptions.hubcapNodeWeight or hubcapOptions.nodeWeight
  hubcapOptions.collision = hubcapOptions.hubcapCollision or hubcapOptions.collision
  hubcapOptions.nodeMaterial = hubcapOptions.hubcapNodeMaterial or hubcapOptions.nodeMaterial
  hubcapOptions.frictionCoef = hubcapOptions.hubcapFrictionCoef or hubcapOptions.frictionCoef
  hubcapOptions.hubcapRadius = hubcapOptions.hubcapRadius or hubcapOptions.hubRadius
  hubcapOptions.group = hubcapOptions.hubcapGroup or hubcapOptions.group
  hubcapOptions.disableMeshBreaking = hubcapOptions.disableHubcapMeshBreaking or hubOptions.disableMeshBreaking
  hubcapOptions.wheelID = nil

  cleanupWheelOptions(hubcapOptions) -- used for nodes

  local hubcapnodebase
  if wheel.enableHubcaps ~= nil and wheel.enableHubcaps == true and newNumRay%2 ~= 1 then
    local hubcapOffset
    if wheel.hubcapOffset ~= nil then
      hubcapOffset = wheel.hubcapOffset
      hubcapOffset = axis * hubcapOffset
    end

    hubcapnodebase = vehicle.maxIDs.nodes
    local hubCapNumRays = newNumRay/2
    rayVec = axis:perpendicularN() * hubcapOptions.hubcapRadius
    --rayVec = quatFromAxisAngle(axis, - wheelAngleRad + 2 * math.pi / (hubCapNumRays * 4)) * rayVec
    rayRot = quatFromAxisAngle(axis, 2 * math.pi / hubCapNumRays)

    for i = 0, hubCapNumRays -1, 1 do
      local rayPoint = node1_pos + rayVec - hubcapOffset
      rayVec = rayRot * rayVec
      jbeamUtils.addNodeWithOptions(vehicle, rayPoint, NORMALTYPE, hubcapOptions)
    end

    --hubcapOptions.collision = false
    --hubcapOptions.selfCollision = false
    hubcapOptions.nodeWeight = wheel.hubcapCenterNodeWeight
    --make the center rigidifying node
    local hubcapAxis = node1_pos + axis * wheel.hubcapWidth
    jbeamUtils.addNodeWithOptions(vehicle, hubcapAxis, NORMALTYPE, hubcapOptions)

    --hubcapOptions.collision = nil
    --hubcapOptions.selfCollision = nil
    hubcapOptions.nodeWeight = nil
  end

  local weightComp = math.min(1, tyreWeight)

  local hubcapAttachOptions = deepcopy(cleanwheel)
  hubcapAttachOptions.beamSpring  = hubcapAttachOptions.hubcapAttachBeamSpring    or hubcapAttachOptions.beamSpring
  hubcapAttachOptions.beamDamp    = hubcapAttachOptions.hubcapAttachBeamDamp      or hubcapAttachOptions.beamDamp
  hubcapAttachOptions.beamDeform  = hubcapAttachOptions.hubcapAttachBeamDeform    or hubcapAttachOptions.beamDeform
  hubcapAttachOptions.beamStrength= hubcapAttachOptions.hubcapAttachBeamStrength  or hubcapAttachOptions.beamStrength
  hubcapAttachOptions.breakGroup  = hubcapAttachOptions.hubcapBreakGroup          or hubcapAttachOptions.breakGroup
  hubcapAttachOptions.wheelID     = nil
  hubcapAttachOptions.disableMeshBreaking = true

  local hubcapSupportOptions = deepcopy(hubcapAttachOptions)
  hubcapSupportOptions.beamSpring   = hubcapSupportOptions.hubcapSupportBeamSpring    or  hubcapSupportOptions.beamSpring
  hubcapSupportOptions.beamDamp     = hubcapSupportOptions.hubcapSupportBeamDamp      or  hubcapSupportOptions.beamDamp
  hubcapSupportOptions.beamDeform   = hubcapSupportOptions.hubcapSupportBeamDeform    or  hubcapSupportOptions.beamDeform
  hubcapSupportOptions.beamStrength = hubcapSupportOptions.hubcapSupportBeamStrength  or  hubcapSupportOptions.beamStrength
  hubcapSupportOptions.breakGroup   = nil
  hubcapSupportOptions.wheelID      = nil
  hubcapSupportOptions.disableMeshBreaking = true

  local sideOptions = deepcopy(cleanwheel)
  sideOptions.beamSpring        = sideOptions.wheelSideBeamSpring          and sideOptions.wheelSideBeamSpring          * fullStif or 0
  sideOptions.beamDamp          = sideOptions.wheelSideBeamDamp            and sideOptions.wheelSideBeamDamp            * fullStif or 0
  sideOptions.beamDeform        = sideOptions.wheelSideBeamDeform          and sideOptions.wheelSideBeamDeform          * fullStif or (sideOptions.beamDeform      and sideOptions.beamDeform      * fullStif or sideOptions.beamDeform     )
  sideOptions.beamStrength      = sideOptions.wheelSideBeamStrength        and sideOptions.wheelSideBeamStrength        * fullStif or (sideOptions.beamStrength    and sideOptions.beamStrength    * fullStif or sideOptions.beamStrength   )
  sideOptions.springExpansion   = sideOptions.wheelSideBeamSpringExpansion and sideOptions.wheelSideBeamSpringExpansion * fullStif or (sideOptions.springExpansion and sideOptions.springExpansion * fullStif or sideOptions.springExpansion)
  sideOptions.dampExpansion     = sideOptions.wheelSideBeamDampExpansion   and sideOptions.wheelSideBeamDampExpansion   * fullStif or (sideOptions.dampExpansion   and sideOptions.dampExpansion   * fullStif or sideOptions.dampExpansion  )
  sideOptions.transitionZone    = sideOptions.wheelSideTransitionZone       or sideOptions.transitionZone
  sideOptions.beamPrecompression= sideOptions.wheelSideBeamPrecompression   or 1

  local pyramidOptions = deepcopy(cleanwheel)
  pyramidOptions.beamSpring        = sideOptions.wheelSideBeamSpring          and sideOptions.wheelSideBeamSpring          * antiSquish  * fullStif or 0
  pyramidOptions.beamDamp          = sideOptions.wheelSideBeamDamp            and sideOptions.wheelSideBeamDamp            * antiSquish  * fullStif or 0
  pyramidOptions.beamDeform        = sideOptions.wheelSideBeamDeform          and sideOptions.wheelSideBeamDeform          * antiSquish  * fullStif or (sideOptions.beamDeform      and sideOptions.beamDeform      * fullStif * antiSquish or sideOptions.beamDeform     )
  pyramidOptions.beamStrength      = sideOptions.wheelSideBeamStrength        and sideOptions.wheelSideBeamStrength        * antiSquish  * fullStif or (sideOptions.beamStrength    and sideOptions.beamStrength    * fullStif * antiSquish or sideOptions.beamStrength   )
  pyramidOptions.springExpansion   = sideOptions.wheelSideBeamSpringExpansion and sideOptions.wheelSideBeamSpringExpansion * antiBulge   * fullStif or (sideOptions.springExpansion and sideOptions.springExpansion * fullStif * antiBulge  or sideOptions.springExpansion)
  pyramidOptions.dampExpansion     = sideOptions.wheelSideBeamDampExpansion   and sideOptions.wheelSideBeamDampExpansion   * antiBulge   * fullStif or (sideOptions.dampExpansion   and sideOptions.dampExpansion   * fullStif * antiBulge  or sideOptions.dampExpansion  )
  pyramidOptions.transitionZone    = sideOptions.wheelSideTransitionZone       or sideOptions.transitionZone
  pyramidOptions.beamPrecompression= sideOptions.wheelSideBeamPrecompression   or 1

  local pAntiRein50 = pAntiRein * 0.25
  local sideReinfOptions = deepcopy(sideOptions)
  sideReinfOptions.beamSpring           = sideOptions.wheelSideReinfBeamSpring          and sideOptions.wheelSideReinfBeamSpring          * pAntiRein   * fullStif * weightComp or (sideOptions.wheelSideBeamSpring and sideOptions.wheelSideBeamSpring * pAntiRein   * fullStif * weightComp or sideOptions.wheelSideBeamSpring)
  sideReinfOptions.beamDamp             = sideOptions.wheelSideReinfBeamDamp            and sideOptions.wheelSideReinfBeamDamp            * pAntiRein   * fullStif * weightComp or (sideOptions.wheelSideBeamDamp   and sideOptions.wheelSideBeamDamp   * pAntiRein   * fullStif * weightComp or sideOptions.wheelSideBeamDamp  )
  sideReinfOptions.beamDeform           = sideOptions.wheelSideReinfBeamDeform          and sideOptions.wheelSideReinfBeamDeform          * pAntiRein   * fullStif * weightComp or (sideOptions.beamDeform          and sideOptions.beamDeform          * pAntiRein   * fullStif * weightComp or sideOptions.beamDeform     )
  sideReinfOptions.beamStrength         = sideOptions.wheelSideReinfBeamStrength        and sideOptions.wheelSideReinfBeamStrength        * pAntiRein   * fullStif * weightComp or (sideOptions.beamStrength        and sideOptions.beamStrength        * pAntiRein   * fullStif * weightComp or sideOptions.beamStrength   )
  sideReinfOptions.springExpansion      = sideOptions.wheelSideReinfBeamSpringExpansion and sideOptions.wheelSideReinfBeamSpringExpansion * pAntiRein50 * fullStif * weightComp or (sideOptions.springExpansion     and sideOptions.springExpansion     * pAntiRein50 * fullStif * weightComp or sideOptions.springExpansion)
  sideReinfOptions.dampExpansion        = sideOptions.wheelSideReinfBeamDampExpansion   and sideOptions.wheelSideReinfBeamDampExpansion   * pAntiRein50 * fullStif * weightComp or (sideOptions.dampExpansion       and sideOptions.dampExpansion       * pAntiRein50 * fullStif * weightComp or sideOptions.dampExpansion  )
  sideReinfOptions.transitionZone       = sideOptions.wheelSideReinfTransitionZone      or sideOptions.transitionZone
  sideReinfOptions.beamPrecompression   = sideOptions.wheelSideReinfBeamPrecompression  or 1
  sideReinfOptions.disableMeshBreaking  = true

  local reinfOptions = deepcopy(cleanwheel)
  reinfOptions.beamSpring         = reinfOptions.wheelReinfBeamSpring           and reinfOptions.wheelReinfBeamSpring          * xPat*fullStif or 0
  reinfOptions.beamDamp           = reinfOptions.wheelReinfBeamDamp             and reinfOptions.wheelReinfBeamDamp            * xPat*fullStif or 0
  reinfOptions.springExpansion    = reinfOptions.wheelReinfBeamSpringExpansion  and reinfOptions.wheelReinfBeamSpringExpansion * xPat*fullStif or (reinfOptions.wheelReinfBeamSpringExpansion and reinfOptions.wheelReinfBeamSpringExpansion * xPat * fullStif  or reinfOptions.wheelReinfBeamSpringExpansion)
  reinfOptions.dampExpansion      = reinfOptions.wheelReinfBeamDampExpansion    and reinfOptions.wheelReinfBeamDampExpansion   * xPat*fullStif or (reinfOptions.wheelReinfBeamDampExpansion   and reinfOptions.wheelReinfBeamDampExpansion   * xPat * fullStif  or reinfOptions.wheelReinfBeamDampExpansion  )
  reinfOptions.beamDeform         = reinfOptions.wheelReinfBeamDeform           and reinfOptions.wheelReinfBeamDeform          * xPat*fullStif or (reinfOptions.beamDeform                    and reinfOptions.beamDeform                    * xPat * fullStif  or reinfOptions.beamDeform)
  reinfOptions.beamStrength       = reinfOptions.wheelReinfBeamStrength         and reinfOptions.wheelReinfBeamStrength        * xPat*fullStif or (reinfOptions.beamStrength                  and reinfOptions.beamStrength                  * xPat * fullStif  or reinfOptions.beamStrength)
  reinfOptions.beamPrecompression = reinfOptions.wheelReinfBeamPrecompression   or 1
  reinfOptions.dampCutoffHz       = reinfOptions.wheelReinfBeamDampCutoffHz     or nil
  reinfOptions.disableMeshBreaking= true

  local cornerReinfOptions = deepcopy(cleanwheel)
  cornerReinfOptions.beamSpring         = reinfOptions.wheelReinfBeamSpring           and reinfOptions.wheelReinfBeamSpring          * pCornerL*fullStif or 0
  cornerReinfOptions.beamDamp           = reinfOptions.wheelReinfBeamDamp             and reinfOptions.wheelReinfBeamDamp            * pCornerL*fullStif or 0
  cornerReinfOptions.springExpansion    = reinfOptions.wheelReinfBeamSpringExpansion  and reinfOptions.wheelReinfBeamSpringExpansion * pCornerL*fullStif or (reinfOptions.wheelReinfBeamSpringExpansion and reinfOptions.wheelReinfBeamSpringExpansion * pCornerL * fullStif  or reinfOptions.wheelReinfBeamSpringExpansion)
  cornerReinfOptions.dampExpansion      = reinfOptions.wheelReinfBeamDampExpansion    and reinfOptions.wheelReinfBeamDampExpansion   * pCornerL*fullStif or (reinfOptions.wheelReinfBeamDampExpansion   and reinfOptions.wheelReinfBeamDampExpansion   * pCornerL * fullStif  or reinfOptions.wheelReinfBeamDampExpansion  )
  cornerReinfOptions.beamDeform         = reinfOptions.wheelReinfBeamDeform           and reinfOptions.wheelReinfBeamDeform          * pCornerL*fullStif or (reinfOptions.beamDeform                    and reinfOptions.beamDeform                    * pCornerL * fullStif  or reinfOptions.beamDeform)
  cornerReinfOptions.beamStrength       = reinfOptions.wheelReinfBeamStrength         and reinfOptions.wheelReinfBeamStrength        * pCornerL*fullStif or (reinfOptions.beamStrength                  and reinfOptions.beamStrength                  * pCornerL * fullStif  or reinfOptions.beamStrength)
  cornerReinfOptions.beamPrecompression = reinfOptions.wheelReinfBeamPrecompression   or 1
  cornerReinfOptions.dampCutoffHz       = reinfOptions.wheelReinfBeamDampCutoffHz     or nil
  cornerReinfOptions.disableMeshBreaking= true

  local treadOptions = deepcopy(cleanwheel)
  treadOptions.beamSpring         = treadOptions.wheelTreadBeamSpring         and treadOptions.wheelTreadBeamSpring  * weightComp * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring  * weightComp * fullStif or treadOptions.beamSpring  )
  treadOptions.beamDamp           = treadOptions.wheelTreadBeamDamp           and treadOptions.wheelTreadBeamDamp    * weightComp * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp    * weightComp * fullStif or treadOptions.beamDamp    )
  treadOptions.beamDeform         = treadOptions.wheelTreadBeamDeform         and treadOptions.wheelTreadBeamDeform   * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform   * fullStif or treadOptions.beamDeform  )
  treadOptions.beamStrength       = treadOptions.wheelTreadBeamStrength       and treadOptions.wheelTreadBeamStrength * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength * fullStif or treadOptions.beamStrength)
  treadOptions.beamPrecompression = treadOptions.wheelTreadBeamPrecompression or  1
  treadOptions.dampCutoffHz       = treadOptions.wheelTreadBeamDampCutoffHz   or  nil
  treadOptions.disableMeshBreaking= true

  local treadReinfOptions = deepcopy(treadOptions)
  treadReinfOptions.beamSpring          = treadOptions.wheelTreadReinfBeamSpring          and treadOptions.wheelTreadReinfBeamSpring   * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring  * fullStif or treadOptions.beamSpring  )
  treadReinfOptions.beamDamp            = treadOptions.wheelTreadReinfBeamDamp            and treadOptions.wheelTreadReinfBeamDamp     * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp    * fullStif or treadOptions.beamDamp    )
  treadReinfOptions.beamDeform          = treadOptions.wheelTreadReinfBeamDeform          and treadOptions.wheelTreadReinfBeamDeform   * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform  * fullStif or treadOptions.beamDeform  )
  treadReinfOptions.beamStrength        = treadOptions.wheelTreadReinfBeamStrength        and treadOptions.wheelTreadReinfBeamStrength * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength* fullStif or treadOptions.beamStrength)
  treadReinfOptions.beamPrecompression  = treadOptions.wheelTreadReinfBeamPrecompression  or 1
  treadReinfOptions.dampCutoffHz        = treadOptions.wheelTreadReinfBeamDampCutoffHz    or nil
  treadReinfOptions.disableMeshBreaking = true
  if sideWallReinf >= 0.05 then
    treadReinfOptions.beamSpring  = treadOptions.wheelTreadReinfBeamSpring         and treadOptions.wheelTreadReinfBeamSpring   * sideWallReinf * fullStif or (treadOptions.beamSpring   and treadOptions.beamSpring   * sideWallReinf * fullStif or treadOptions.beamSpring  )
    treadReinfOptions.beamDamp    = treadOptions.wheelTreadReinfBeamDamp           and treadOptions.wheelTreadReinfBeamDamp     * sideWallReinf * fullStif or (treadOptions.beamDamp     and treadOptions.beamDamp     * sideWallReinf * fullStif or treadOptions.beamDamp    )
    treadReinfOptions.beamDeform  = treadOptions.wheelTreadReinfBeamDeform         and treadOptions.wheelTreadReinfBeamDeform   * sideWallReinf * fullStif or (treadOptions.beamDeform   and treadOptions.beamDeform   * sideWallReinf * fullStif or treadOptions.beamDeform  )
    treadReinfOptions.beamStrength= treadOptions.wheelTreadReinfBeamStrength       and treadOptions.wheelTreadReinfBeamStrength * sideWallReinf * fullStif or (treadOptions.beamStrength and treadOptions.beamStrength * sideWallReinf * fullStif or treadOptions.beamStrength)
  end

  local peripheryOptions = deepcopy(treadOptions)
  peripheryOptions.beamSpring         = peripheryOptions.wheelPeripheryBeamSpring   and peripheryOptions.wheelPeripheryBeamSpring   * fullStif  or (peripheryOptions.beamSpring   and peripheryOptions.beamSpring   * fullStif or peripheryOptions.beamSpring  )
  peripheryOptions.beamDamp           = peripheryOptions.wheelPeripheryBeamDamp     and peripheryOptions.wheelPeripheryBeamDamp     * fullStif  or (peripheryOptions.beamDamp     and peripheryOptions.beamDamp     * fullStif or peripheryOptions.beamDamp    )
  peripheryOptions.beamDeform         = peripheryOptions.wheelPeripheryBeamDeform   and peripheryOptions.wheelPeripheryBeamDeform   * fullStif  or (peripheryOptions.beamDeform   and peripheryOptions.beamDeform   * fullStif or peripheryOptions.beamDeform  )
  peripheryOptions.beamStrength       = peripheryOptions.wheelPeripheryBeamStrength and peripheryOptions.wheelPeripheryBeamStrength * fullStif  or (peripheryOptions.beamStrength and peripheryOptions.beamStrength * fullStif or peripheryOptions.beamStrength)
  peripheryOptions.beamPrecompression = peripheryOptions.wheelPeripheryBeamPrecompression or 1
  peripheryOptions.dampCutoffHz       = peripheryOptions.wheelPeripheryBeamDampCutoffHz   or nil

  local peripheryReinfOptions = deepcopy(peripheryOptions)
  peripheryReinfOptions.beamSpring        = peripheryReinfOptions.wheelPeripheryReinfBeamSpring   and peripheryReinfOptions.wheelPeripheryReinfBeamSpring   * fullStif  or (peripheryOptions.beamSpring   and peripheryOptions.beamSpring   * fullStif or peripheryOptions.beamSpring  )
  peripheryReinfOptions.beamDamp          = peripheryReinfOptions.wheelPeripheryReinfBeamDamp     and peripheryReinfOptions.wheelPeripheryReinfBeamDamp     * fullStif  or (peripheryOptions.beamDamp     and peripheryOptions.beamDamp     * fullStif or peripheryOptions.beamDamp    )
  peripheryReinfOptions.beamDeform        = peripheryReinfOptions.wheelPeripheryReinfBeamDeform   and peripheryReinfOptions.wheelPeripheryReinfBeamDeform   * fullStif  or (peripheryOptions.beamDeform   and peripheryOptions.beamDeform   * fullStif or peripheryOptions.beamDeform  )
  peripheryReinfOptions.beamStrength      = peripheryReinfOptions.wheelPeripheryReinfBeamStrength and peripheryReinfOptions.wheelPeripheryReinfBeamStrength * fullStif  or (peripheryOptions.beamStrength and peripheryOptions.beamStrength * fullStif or peripheryOptions.beamStrength)
  peripheryReinfOptions.beamPrecompression= peripheryReinfOptions.wheelPeripheryReinfBeamPrecompression or 1
  peripheryReinfOptions.dampCutoffHz      = peripheryOptions.wheelPeripheryReinfBeamDampCutoffHz        or nil

  --treadOptions.beamSpring = treadOptions.beamSpring * (weightComp - weightComp*weightComp)
  --treadOptions.beamDamp   = treadOptions.beamDamp   * (weightComp - weightComp*weightComp)


  cleanupBeamOptions(cleanupWheelOptions(hubcapAttachOptions))
  cleanupBeamOptions(cleanupWheelOptions(sideOptions))
  cleanupBeamOptions(cleanupWheelOptions(sideReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(reinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(treadOptions))
  cleanupBeamOptions(cleanupWheelOptions(treadReinfOptions))
  cleanupBeamOptions(cleanupWheelOptions(peripheryOptions))
  cleanupBeamOptions(cleanupWheelOptions(peripheryReinfOptions))

  vehicle.triangles = vehicle.triangles or {}
  local pressureGroupName = '_wheelPressureGroup' .. wheel.wheelID
  local wheelPressure = wheel.pressurePSI or 10
  local wheelDragCoef = wheel.dragCoef or 100
  local wheelSkinDragCoef = wheel.skinDragCoef
  local wheelTreadTriangleType = NORMALTYPE
  local wheelSide1TriangleType = NORMALTYPE
  local wheelSide2TriangleType = NORMALTYPE

  if (wheel.triangleCollision or false) == false then
    wheelTreadTriangleType = NONCOLLIDABLE
    wheelSide1TriangleType = NONCOLLIDABLE
    wheelSide2TriangleType = NONCOLLIDABLE
  end

  if wheel.treadTriangleCollision == false then
    wheelTreadTriangleType = NONCOLLIDABLE
  end

  if wheel.side1TriangleCollision == false then
    wheelSide1TriangleType = NONCOLLIDABLE
  end

  if wheel.side2TriangleCollision == false then
    wheelSide2TriangleType = NONCOLLIDABLE
  end

  local hubTriangleCollision = wheel.hubTriangleCollision == true and true or false
  local hubSide1TriangleCollision = wheel.hubSide1TriangleCollision == true and true or false
  local hubSide2TriangleCollision = wheel.hubSide2TriangleCollision == true and true or false

  local rimBeams = {}
  local sideBeams = {}
  local peripheryBeams = {}
  local treadBeams = {}
  local reinfBeams = {}

  local inaxisnode = wheel.node1
  local outaxisnode = wheel.node2
  local vTris = vehicle.triangles

  --modified by fillman86 ******************************************************************* V
  local timesdone = 0

  for i = 0, newNumRay - 1 do
    local i2 = 2*i --keep
    local nextdelta = 2*((i+1)%newNumRay) --keep
    local outhubnode = hubnodebase + i2 --keep
    local inhubnode = outhubnode + 1 --keep
    local nextouthubnode = hubnodebase + nextdelta --keep
    local nextinhubnode = nextouthubnode + 1 --keep

    local curBand     = treadBands[i+1]
    local nexBand     = treadBands[i+2] and treadBands[i+2] or treadBands[(i+2)-(newNumRay)]
    local nexNexBand  = treadBands[i+3] and treadBands[i+3] or treadBands[(i+3)-(newNumRay)]
    local nex3Band    = treadBands[i+4] and treadBands[i+4] or treadBands[(i+4)-(newNumRay)]
    local preBand     = treadBands[i] and treadBands[i] or treadBands[newNumRay]
    local prePreBand  = treadBands[i-1] and treadBands[i-1] or treadBands[newNumRay-1]

    -- Hub caps
    if wheel.enableHubcaps and newNumRay%2 ~= 1 and i < ((newNumRay)/2) then
      local hubcapnode = hubcapnodebase + i
      local nexthubcapnode = hubcapnodebase + ((i+1)%(newNumRay/2))
      local nextnexthubcapnode = hubcapnodebase + ((i+2)%(newNumRay/2))
      local hubcapaxisnode = hubcapnode + (newNumRay/2) - i
      local hubcapinhubnode = inhubnode + i2
      local prevhubcaphubnode = hubnodebase + 2 *((i2 + newNumRay - 1)%newNumRay)+1
      local nexthubcapinhubnode = hubcapinhubnode + 2
      local hubcapouthubnode = hubcapinhubnode - 1

      --hubcap periphery
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, nexthubcapnode,    NORMALTYPE, hubcapOptions)
      --attach to center node
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapaxisnode,    NORMALTYPE, hubcapOptions)
      --attach to axis
      if wheel.enableExtraHubcapBeams == true then
        --jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node1, NORMALTYPE, hubcapAttachOptions)
        --jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node2, NORMALTYPE, hubcapAttachOptions)
        if i == 1 then
          jbeamUtils.addBeamWithOptions(vehicle, hubcapaxisnode, wheel.node1, NORMALTYPE, hubcapAttachOptions)
          jbeamUtils.addBeamWithOptions(vehicle, hubcapaxisnode, wheel.node2, NORMALTYPE, hubcapAttachOptions)
        end
      end

      --span beams
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, nextnexthubcapnode,    NORMALTYPE, hubcapOptions)

      --attach it
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapinhubnode,   NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, prevhubcaphubnode, NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapouthubnode,  NORMALTYPE,   hubcapAttachOptions)
      jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, hubcapouthubnode,  BEAM_SUPPORT, hubcapSupportOptions)

      --self:jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node1,    NORMALTYPE, hubcapAttachOptions)
      --self:jbeamUtils.addBeamWithOptions(vehicle, hubcapnode, wheel.node2,    NORMALTYPE, hubcapAttachOptions)
    end

    --hub tread
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, inhubnode,      NORMALTYPE, hubTreadOptions)
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  nextouthubnode, NORMALTYPE, hubTreadOptions)
    --modified by fillman86
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, nextinhubnode,  NORMALTYPE, hubTreadOptions)

    --hub periphery beams
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, nextouthubnode, NORMALTYPE, hubPeripheryOptions)
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  nextinhubnode,  NORMALTYPE, hubPeripheryOptions)

    --hub axis beams
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, inaxisnode, NORMALTYPE, hubReinfOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, outhubnode, outaxisnode, NORMALTYPE, hubSideOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  inaxisnode, NORMALTYPE, hubSideOptions))
    table.insert(rimBeams,
    jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  outaxisnode, NORMALTYPE, hubReinfOptions))

    --Beams to stability node
    if nodeStabilizerExists then
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,  wheel.nodeStabilizer, NORMALTYPE, hubStabilizerOptions)
    end

    if hubSide1TriangleCollision then
      addTri(vTris, nextouthubnode, outhubnode, outaxisnode, wheelDragCoef * 0.5, NORMALTYPE)
    end

    if hubSide2TriangleCollision then
      addTri(vTris, inhubnode, nextinhubnode, inaxisnode, wheelDragCoef * 0.5, NORMALTYPE)
    end      

    --MARK: beams
    if tireExists then
      local tbnc = 4+treadDiv --tbnc is short for "tread band's node count"
      --to bring back dupeCheck, replace all following "table.insert" with "dupeCheck", and it'll check all beams
      --tire tread
      for cNode = 1, tbnc-1 do
        --connecting the band transversely
        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, curBand[cNode+1].cid, NORMALTYPE, treadOptions))
        --connecting forward, and +1 node along
        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexBand[cNode+1].cid, NORMALTYPE, treadOptions))
        if cNode >= 2 and cNode <= tbnc-2 then --this would create a wierd beam on the first node
          --connects the next node, to the next band's current node
          table.insert(treadBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode+1].cid, nexBand[cNode].cid, NORMALTYPE, treadOptions))
        end
      end

      --transverse beams
      for cNode = 2, tbnc-3 do
        table.insert(treadBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, curBand[cNode+2].cid, NORMALTYPE, treadOptions))
      end

      --testing square walls, these create the x pattern, inside the squares
      table.insert(treadBeams,
      jbeamUtils.addBeamWithOptions(vehicle, curBand[2].cid, nexBand[1].cid, NORMALTYPE, treadOptions))
      table.insert(treadBeams,
      jbeamUtils.addBeamWithOptions(vehicle, curBand[tbnc].cid, nexBand[tbnc-1].cid, NORMALTYPE, treadOptions))

      --lateral sidewall reinforcement. This is a bit of a cheat
      if sideWallReinf >= 0.05 then
        table.insert(treadBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[tbnc].cid, NORMALTYPE, treadReinfOptions))
      end

      --paired treadnodes. To do with hidden elastic floor stuff, for rolling resistance
      if vehicle.variables['$xxpaired'].val == 0 then
        for middleNode = 2, tbnc-1 do
          vehicle.nodes[curBand[middleNode].cid].pairedNode = nexBand[middleNode].cid
          vehicle.nodes[curBand[middleNode].cid].pairedNode2 = preBand[middleNode].cid
        end
      end

      -- All periphery beams
      for cNode = 1, tbnc do
        table.insert(peripheryBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexBand[cNode].cid,    NORMALTYPE, peripheryOptions))
      end

      -- Periphery reinforcement beams
      for cNode = 2, tbnc-1 do
        table.insert(peripheryBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nexNexBand[cNode].cid,    BEAM_ANISOTROPIC, sideReinfOptions))

        table.insert(peripheryBeams, --skip 3 bands ahead
        jbeamUtils.addBeamWithOptions(vehicle, curBand[cNode].cid, nex3Band[cNode].cid,    BEAM_ANISOTROPIC, sideReinfOptions))
      end

      --wheel->tire wall beams. Connect wheelNode to either tireNode
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,curBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, curBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,preBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, preBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))--default is BEAM_ANISOTROPIC, sideOptions <^ alternative is NORMALTYPE, treadOptions
      --hubNode square walls, x-pattern fix
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode,nexBand[tbnc].cid,BEAM_ANISOTROPIC, sideOptions))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, nexBand[1].cid,   BEAM_ANISOTROPIC, sideOptions))--default is BEAM_ANISOTROPIC, sideOptions <^ alternative is NORMALTYPE, treadOptions

      --adding triangles wheel <-to-> tread strands, to stop bulging
      if antiBulge >=0.05 or antiSquish >=0.05 then
        if (tbnc%2==0) then
          for CurrentNode = 3, tbnc - 2 do
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, inhubnode,     curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, outhubnode,    curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextinhubnode, curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode,curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
          end
        else  
          for CurrentNode = 3, tbnc - 2 do
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, inhubnode,     curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, outhubnode,    curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextinhubnode, curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
            table.insert(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode,curBand[CurrentNode].cid, BEAM_ANISOTROPIC, pyramidOptions))
          end
        end
      end

      --note: I've left in a bunch of alternative beam options if people want to experiment with re-enabling them |WARNING: may be incompatible now

      --wheel->tire's corner: forward "L-Beams"
      --[[ dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  outhubnode,     curBand[1].cid,         BEAM_LBEAM, reinfOptions, inhubnode))
      dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  inhubnode,      curBand[tbnc].cid,      BEAM_LBEAM, reinfOptions, outhubnode))
      if i>= 1 then--wheel->tire's corner: backwards "L-Beams
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,outhubnode,     preBand[1].cid,             BEAM_LBEAM, reinfOptions, inhubnode))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,inhubnode,      preBand[tbnc].cid,          BEAM_LBEAM, reinfOptions, outhubnode))
      else
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,outhubnode,lastBand[1].cid,    BEAM_LBEAM, reinfOptions, inhubnode))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle,inhubnode, lastBand[tbnc].cid, BEAM_LBEAM, reinfOptions, outhubnode))
      end ]]

      --first sidewall L-Beams
      --[[ dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  inhubnode, curBand[2].cid,         BEAM_LBEAM, reinfOptions, curBand[1].cid))
      dupeCheck(reinfBeams,
      jbeamUtils.addBeamWithOptions(vehicle,  outhubnode,curBand[tbnc-1].cid,BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))
       ]]
      --[[ if xPat >= 0.05 then
        for eachWidthX = 2, tbnc-3 do 
          if i<=newNumRay-3 then --tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   nexNexBand[eachWidthX+2].cid,BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, nexNexBand[eachWidthX].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
          end
          if i==newNumRay-2 then -- second to last tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   firstBand[eachWidthX+2].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, firstBand[eachWidthX].cid,    BEAM_LBEAM, reinfOptions, nexBand[eachWidthX+1].cid))
          end 
          if i==newNumRay-1 then -- last tread L-Beams in an x pattern
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid,   treadBands[2][eachWidthX+2].cid,  BEAM_LBEAM, reinfOptions, firstBand[eachWidthX+1].cid))
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX+2].cid, treadBands[2][eachWidthX].cid,    BEAM_LBEAM, reinfOptions, firstBand[eachWidthX+1].cid))
          end  ]]
          --[[ if i>2 and (eachWidthX>=3 or eachWidthX==tbnc-2) then --better corner L-Beams
            dupeCheck(reinfBeams,
            jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[eachWidthX].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))          
          end
        end
      end ]]

      --tyre corner L-Beams
      if pCornerL ~= 0 then
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, curBand[3].cid, BEAM_LBEAM, cornerReinfOptions, curBand[2].cid))
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[tbnc].cid, curBand[tbnc-2].cid, BEAM_LBEAM, cornerReinfOptions, curBand[tbnc-1].cid))

        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[1].cid, preBand[3].cid, BEAM_LBEAM, cornerReinfOptions, curBand[2].cid))
        table.insert(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, curBand[tbnc].cid, preBand[tbnc-2].cid, BEAM_LBEAM, cornerReinfOptions, curBand[tbnc-1].cid))
      end
  
      -- Wheel -> corner beams to rigidify sidewalls (same side, not crossing sides), specifically to help rolling resistance, like irl tyres
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  curBand[2].cid,          BEAM_LBEAM, reinfOptions, curBand[1].cid))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode, curBand[tbnc-1].cid,     BEAM_LBEAM, reinfOptions, curBand[tbnc].cid))

      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, outhubnode, curBand[1].cid,     BEAM_LBEAM, reinfOptions, inhubnode))
      table.insert(sideBeams,
      jbeamUtils.addBeamWithOptions(vehicle, inhubnode, curBand[tbnc].cid,     BEAM_LBEAM, reinfOptions, outhubnode))

      if treadDiv > 1 then --this is to give extra l beams inwards
        --[[ dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, inhubnode,  curBand[3].cid,          BEAM_LBEAM, reinfOptions, curBand[2].cid))
        dupeCheck(reinfBeams,
        jbeamUtils.addBeamWithOptions(vehicle, nextouthubnode, curBand[tbnc-2].cid, BEAM_LBEAM, reinfOptions, curBand[tbnc-1].cid)) ]]
      end

      --trying periphery L-Beams |WARNING: probably not compatible anymore
      --[[ for eachWidthX = 2, tbnc-1 do 
        if i<=newNumRay-3 then
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, nexNexBand[eachWidthX].cid,BEAM_LBEAM, reinfOptions, nexBand[eachWidthX].cid))
        end
        if i==newNumRay-2 then -- second to last tread L-Beams
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, firstBand[eachWidthX].cid,  BEAM_LBEAM, reinfOptions, nexBand[eachWidthX].cid))
        end 
        if i==newNumRay-1 then -- last tread L-Beams
          dupeCheck(reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachWidthX].cid, treadBands[2][eachWidthX].cid,  BEAM_LBEAM, reinfOptions, firstBand[eachWidthX].cid))
        end 
      end ]]

      --opposite node trick
      for nods = 2, tbnc-1 do 
        --timesdone = timesdone + 1
        --print("timesdone -"..timesdone)
        if i<=(newNumRay*0.5)-1 then
          --dupeCheck(sideBeams,
          --jbeamUtils.addBeamWithOptions(vehicle, curBand[nods].cid,curBand[nods].oppositeTreadNodeCid, NORMALTYPE, treadOptions))
        end
      end
  



      --[[ for eachStrapNode = 1, tbnc -2 do --this is L-Beams for each perfect band
        dupeCheck( reinfBeams,
          jbeamUtils.addBeamWithOptions(vehicle, curBand[eachStrapNode].cid, curBand[eachStrapNode+2].cid, BEAM_LBEAM, reinfOptions,  curBand[eachStrapNode+1].cid))
      end ]]

      --checking if inverse tyires are needed
      if inverse >=0 then 
        -- hub pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextouthubnode, outhubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)
        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextinhubnode, nextouthubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)

        -- wheel-tire pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, curBand[tbnc].cid, outhubnode, nextouthubnode, wheelDragCoef * 0.5, wheelSide1TriangleType)
        addPressTri(vTris, pressureGroupName, wheelPressure, curBand[tbnc].cid, nextouthubnode, nexBand[tbnc].cid, wheelDragCoef * 0.5, wheelSide1TriangleType)

        addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, preBand[1].cid, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
        addPressTri(vTris, pressureGroupName, wheelPressure, nextinhubnode, inhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)

        for CurrentNode=1,tbnc do
          if (CurrentNode%2~=0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, nexBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, curBand[CurrentNode+1].cid, nexBand[CurrentNode+1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          elseif (CurrentNode%2==0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode].cid, curBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, nexBand[CurrentNode+1].cid, nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          end
        end
      else
        -- hub pressure tris
        addPressTri(vTris, pressureGroupName, wheelPressure, nextouthubnode, inhubnode, outhubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)
        addPressTri(vTris, pressureGroupName, wheelPressure, nextinhubnode, inhubnode, nextouthubnode, wheelDragCoef * 0.1, NONCOLLIDABLE)

        -- wheel-tire pressure tris
          addPressTri(vTris, pressureGroupName, wheelPressure, outhubnode, curBand[tbnc].cid, nextouthubnode, wheelDragCoef * 0.5, wheelSide1TriangleType)
          addPressTri(vTris, pressureGroupName, wheelPressure, nextouthubnode, curBand[tbnc].cid, nexBand[tbnc].cid, wheelDragCoef * 0.5, wheelSide1TriangleType)

        -- inside pressure tris
          addPressTri(vTris, pressureGroupName, wheelPressure,     preBand[1].cid, inhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          addPressTri(vTris, pressureGroupName, wheelPressure, inhubnode, nextinhubnode, curBand[1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)

        for CurrentNode=1,tbnc do
          if (CurrentNode%2~=0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, nexBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode+1].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          elseif (CurrentNode%2==0) and CurrentNode<=tbnc-1 then
            addPressTri(vTris, pressureGroupName, wheelPressure, curBand[CurrentNode+1].cid, curBand[CurrentNode].cid,  nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
            addPressTri(vTris, pressureGroupName, wheelPressure, nexBand[CurrentNode+1].cid, curBand[CurrentNode+1].cid,nexBand[CurrentNode].cid, wheelDragCoef * 0.5, wheelSide2TriangleType)
          end
        end
      end
    else
      if hubTriangleCollision then
        -- hub tris
        addTri(vTris, nextouthubnode, inhubnode, outhubnode, wheelDragCoef * 0.1, NORMALTYPE)
        addTri(vTris, nextinhubnode, inhubnode, nextouthubnode, wheelDragCoef * 0.1, NORMALTYPE)
      end
    end --end of modified by fillman86 ************************************************* ^
    wheelsToRotators = deepcopy(vehicle)
  end

  --[[ if wheelKey == 0 then
    for a,b in ipairs(vehicle.nodes) do
      if b.pairedNode then
        if b.pairedNode2 then
          print(wheelKey .. "|tyre node- ".. b.cid .. ", to node- ".. b.pairedNode .. " and ".. b.pairedNode2)
        else
          print(wheelKey .. "|tyre node- ".. b.cid ..", only 1 pairednode- ".. b.pairedNode)
        end
      end
      if b.pairedNode2 and not b.pairedNode then
        print(wheelKey .. "|tyre node- ".. b.cid ..", only  pairednode2- ".. b.pairedNode2)
      end
    end
  end ]]

  wheel.nodes = hubNodes
  wheel.treadNodes = treadNodes
  wheel.rimBeams = rimBeams
  wheel.sideBeams = sideBeams
  wheel.peripheryBeams = peripheryBeams
  wheel.reinfBeams = reinfBeams
  wheel.treadBeams = treadBeams
  wheel.pressureGroup = pressureGroupName
end

local function processWheel2(vehicle, wheelSection, wheelCreationFunction) --modified processWheel because the other isn't a public module
  if vehicle[wheelSection] ~= nil then
    for k = 0, tableSizeC(vehicle[wheelSection]) - 1 do -- why is this not "for k, v in pairs"? Investigate later
      local v = vehicle[wheelSection][k]
      if v.numRays == nil or v.numRays > 0 then --why does this exist? is it to just make sure it's either nothing or a number, and not something like a string?
        local wheelID = jbeamUtils.increaseMax(vehicle, 'wheels')
        v.wheelID = wheelID
        wheelCreationFunction(vehicle, k, v)
        vehicle.wheels[wheelID] = v
        vehicle.wheels[wheelID].wheelSection = wheelSection
        vehicle[wheelSection][k] = nil -- everything is in wheels now
      end
    end
  end
end

jbeamWheels.processWheels = function(vehicle) --monkey hooking into the original processWheels to check if our tyres are being made, otherwise it reverts to the "defaultProcessWheels"
  if vehicle.wheels ~= nil  then
    vehicle.maxIDs.wheels = nil
  end
  vehicle.wheels = {}

  vehicle.pressureWheels2 = {}
  if vehicle.variables['$version'] or vehicle.variables['$version'] or vehicle.pressureWheels.isHighPolly then --if not the tyres we want, then do the default wheels
    local tyreCheck = true
    for wheel, state in pairs(vehicle.pressureWheels) do
      if not state.hasTire then --this is for things like the pigeon with stabiliser wheels
        tyreCheck = false
      end
    end
    if tyreCheck then --addPressureWheelDivSquareWall xxSquareWall
        print(type(_G[vehicle.variables['$version'].title]))
        vehicle.pressureWheels2 = deepcopy(vehicle.pressureWheels)
        if vehicle.variables['$version'].title == "addPressureWheelDiv" then
          processWheel2(vehicle, "pressureWheels2", addPressureWheelDiv)
        end
        if vehicle.variables['$version'].title == "addPressureWheelDivSquareWall" then
          processWheel2(vehicle, "pressureWheels2", addPressureWheelDivSquareWall)
        end
        if vehicle.variables['$version'].title == "addPressureWheelDivQuilted" then
          processWheel2(vehicle, "pressureWheels2", addPressureWheelDivQuilted)
        end
    else
      guihooks.trigger('toastrMsg', {type = "error", title = "I'm sorry, Dave:", msg = "I'm affraid I can't do those wheels", config = {timeOut = 5000}, icon = "mood_bad"})
      guihooks.trigger('Message', {msg = "I'm sorry, Dave, I'm affraid I can't do those wheels", ttl = 10, category = "error", icon = "mood_bad",config={closeButton=true, timeOut=0, extendedTimeOut=0}})
      Engine.Audio.playOnce('AudioGui', 'event:>UI>Career>Drift_Canceled')
      defaultProcessWheels(vehicle)
    end
  else
    defaultProcessWheels(vehicle)
  end

  return true
end

M.addPressureWheelDiv2 = addPressureWheelDiv2
M.processWheel2 = processWheel2
M.processWheels = processWheels

return M


--[[ 
things to note, in your tyre files, you must put:
{"isHighPolly":true},

suggested is:
{"treadDiv":1.00},
{"bulgeOffset":1.00},

the rest can just be set to 1 (but they must be there):
{"sideWallReinf":1.00},
{"antiBulge":1.00},
{"antiSquish":1.00},
{"xPat":1.00},
{"fullStif":1.00},
{"pAntiRein":1.00},
{"pCornerL":1.00},
{"newBulge":1.00},
Because all these are doing, is modifying the existing vales, you can just set the values directly.
Though some are derrivitives of other values, so you may want to change them
]]