-- Edited by DaddelZeit
-- DO NOT USE WITHOUT PERMISSION

local M = {}

local max = math.max
local min = math.min
local abs = math.abs

local constants = {rpmToAV = 0.104719755, avToRPM = 9.549296596425384}

local motors = nil

local sharedFunctions = nil
local gearboxAvailableLogic = nil
local gearboxLogic = nil

local hasBuiltPie = false

M.gearboxHandling = nil
M.timer = nil
M.timerConstants = nil
M.inputValues = nil
M.shiftPreventionData = nil
M.shiftBehavior = nil
M.smoothedValues = nil

M.currentGearIndex = 0
M.throttle = 0
M.brake = 0
M.clutchRatio = 1
M.throttleInput = 0
M.isArcadeSwitched = false
M.isSportModeActive = false

M.smoothedAvgAVInput = 0
M.rpm = 0
M.idleRPM = 0
M.maxRPM = 0

M.engineThrottle = 0
M.engineLoad = 0
M.engineTorque = 0
M.flywheelTorque = 0
M.gearboxTorque = 0

M.ignition = true
M.isEngineRunning = 0

M.oilTemp = 0
M.waterTemp = 0
M.checkEngine = false

M.energyStorages = {}

local automaticHandling = {
  availableModes = {"P", "R", "N", "D"},
  hShifterModeLookup = {[-1] = "R", [0] = "N", "P", "D", "S"},
  gearIndexLookup = {P = -2, R = -1, N = 0, D = 1, S = 2, ["2"] = 3, ["1"] = 4, M1 = 5},
  availableModeLookup = {},
  existingModeLookup = {},
  modeIndexLookup = {},
  modes = {},
  mode = nil,
  modeIndex = 0,
  maxAllowedGearIndex = 0,
  minAllowedGearIndex = 0
}

local regenHandling = {
  defaultRegen = 0.2,
  brakeRegenCoef = 0.3,
  regenThrottlePedal = 0.1
}

local throttleSmoother = newExponentialSmoothingT(15)

local function getGearName()
  return automaticHandling.mode
end

local function getGearPosition()
  return (automaticHandling.modeIndex - 1) / (#automaticHandling.modes - 1), automaticHandling.modeIndex
end

local function gearboxBehaviorChanged(behavior)
  gearboxLogic = gearboxAvailableLogic[behavior]
  M.updateGearboxGFX = gearboxLogic.inGear
  M.shiftUp = gearboxLogic.shiftUp
  M.shiftDown = gearboxLogic.shiftDown
  M.shiftToGearIndex = gearboxLogic.shiftToGearIndex
end

local function applyGearboxMode()
  local autoIndex = automaticHandling.modeIndexLookup[automaticHandling.mode]
  if autoIndex then
    automaticHandling.modeIndex = min(max(autoIndex, 1), #automaticHandling.modes)
    automaticHandling.mode = automaticHandling.modes[automaticHandling.modeIndex]
  end

  input.event("parkingbrake", 0, FILTER_DIRECT)

  local motorDirection = 1 --D
  if automaticHandling.mode == "P" then
    motorDirection = 0
    input.event("parkingbrake", 1, FILTER_DIRECT)
  elseif automaticHandling.mode == "N" then
    motorDirection = 0
  elseif automaticHandling.mode == "R" then
    motorDirection = -1
  end

  for _, v in ipairs(motors) do
    v.motorDirection = motorDirection
  end

  M.isSportModeActive = automaticHandling.mode == "S"
end

local function shiftUp()
  if automaticHandling.mode == "N" then
    M.timer.gearChangeDelayTimer = M.timerConstants.gearChangeDelay
  end

  automaticHandling.modeIndex = min(automaticHandling.modeIndex + 1, #automaticHandling.modes)
  automaticHandling.mode = automaticHandling.modes[automaticHandling.modeIndex]

  applyGearboxMode()
end

local function shiftDown()
  if automaticHandling.mode == "N" then
    M.timer.gearChangeDelayTimer = M.timerConstants.gearChangeDelay
  end

  automaticHandling.modeIndex = max(automaticHandling.modeIndex - 1, 1)
  automaticHandling.mode = automaticHandling.modes[automaticHandling.modeIndex]

  applyGearboxMode()
end

local function shiftToGearIndex(index)
  local desiredMode = automaticHandling.hShifterModeLookup[index]
  if not desiredMode or not automaticHandling.existingModeLookup[desiredMode] then
    if desiredMode and not automaticHandling.existingModeLookup[desiredMode] then
      guihooks.message({txt = "vehicle.drivetrain.cannotShiftAuto", context = {mode = desiredMode}}, 2, "vehicle.shiftLogic.cannotShift")
    end
    desiredMode = "N"
  end
  automaticHandling.mode = desiredMode

  applyGearboxMode()
end

local function updateExposedData()
  local motorCount = 0
  M.rpm = 0
  local load = 0
  local motorTorque = 0
  for _, v in ipairs(motors) do
    M.rpm = max(M.rpm, abs(v.outputAV1) * constants.avToRPM)
    load = load + (v.engineLoad or 0)
    motorTorque = motorTorque + (v.outputTorque1 or 0)
    motorCount = motorCount + 1
  end
  load = load / motorCount

  M.smoothedAvgAVInput = sharedFunctions.updateAvgAVDeviceCategory("engine")
  M.waterTemp = 0
  M.oilTemp = 0
  M.checkEngine = 0
  M.ignition = electrics.values.ignitionLevel > 1
  M.engineThrottle = M.throttle
  M.engineLoad = load
  M.running = electrics.values.ignitionLevel > 1
  M.engineTorque = motorTorque
  M.flywheelTorque = motorTorque
  M.gearboxTorque = motorTorque
  M.isEngineRunning = 1
end

local lastSpeed = 0
local decelerationSmoother = newTemporalSmoothingNonLinear(35,0.5)
local decelerationRate = 0

local function updateRegen(dt)
  local regenThrottlePedal = regenHandling.regenThrottlePedal
  local smoothedinputValues = {
    throttle = 0
  }

  local driveModeCoef = 1
  if controller.getController('driveModes') then
    local data = controller.getControllerSafe('driveModes').serialize()
    if data.activeDriveModeKey:lower() == "comfort" or data.activeDriveModeKey:lower() == "caged" then
      driveModeCoef = 1
      regenThrottlePedal = regenHandling.regenThrottlePedal
      smoothedinputValues.throttle = throttleSmoother:get(M.isArcadeSwitched and M.inputValues.brake or M.inputValues.throttle)
    elseif data.activeDriveModeKey:lower() == "savage" or data.activeDriveModeKey:lower() == "drift" then
      driveModeCoef = 0
      regenThrottlePedal = 0
      smoothedinputValues.throttle = M.isArcadeSwitched and M.inputValues.brake or M.inputValues.throttle
    else
      driveModeCoef = 0.5
      regenThrottlePedal = regenHandling.regenThrottlePedal / 2
      smoothedinputValues.throttle = M.isArcadeSwitched and M.inputValues.brake or M.inputValues.throttle
    end
  end

  local negativeThrottle = smoothedinputValues.throttle - math.max(regenThrottlePedal or 0, 0) -- throttle with negatives (doesnt work as main throttle with ai and causes issues with ui)

  electrics.values.throttleWithRegen = negativeThrottle
  electrics.values.regenThrottlePedal = regenThrottlePedal

  local tireContactCoef = (M.shiftPreventionData.wheelSlipShiftUp and 1 or 0) or (M.shiftPreventionData.wheelSlipShiftDown and 1 or 0)
  local regenReduceCoef = electrics.values.reducedRegen or 1
  local emergencyBrakeCoef = M.brake >= 1 and 0.3 or 1
  local escCoef = electrics.values.escActive and 0 or 1
  local steeringCoef = 1-(clamp(math.abs(sensors.gx2), 3, 8)-3)/5

  local stopCoef = 1
  if electrics.values.stopMode == 0 then
    stopCoef = electrics.values.wheelspeed > 1.25 and 1 or 0
  end

  local coefs = escCoef * steeringCoef * emergencyBrakeCoef * tireContactCoef * driveModeCoef * regenReduceCoef * stopCoef
  if M.gearboxHandling.autoThrottle == false then
    M.throttle = negativeThrottle < 0 and 0 or math.max((smoothedinputValues.throttle - regenThrottlePedal) * 1/(1-regenThrottlePedal), 0)

    local regenThrottle = (min(regenHandling.defaultRegen + M.brake * regenHandling.brakeRegenCoef, 1) * (max(-negativeThrottle, 0) / regenThrottlePedal)) or 0
    electrics.values.regenThrottle = (clamp((regenThrottle - smoothedinputValues.throttle), 0, regenHandling.defaultRegen + M.brake * regenHandling.brakeRegenCoef)) * coefs
  else
    M.throttle = smoothedinputValues.throttle

    local regenThrottle = smoothedinputValues.throttle <= 0 and min(regenHandling.defaultRegen + M.brake * regenHandling.brakeRegenCoef, 1) or 0
    electrics.values.regenThrottle = (clamp((regenThrottle - clamp(smoothedinputValues.throttle*5, 0.0, 0.9)), 0, regenHandling.defaultRegen + M.brake * regenHandling.brakeRegenCoef) + clamp(1 - electrics.values.wheelspeed, 0, 1)) * coefs
  end

  -- do friction brake blend when stopMode is assisted
  local smoothStopProgress = 0
  if electrics.values.stopMode == 0.5 and (M.isArcadeSwitched and M.inputValues.throttle or M.inputValues.brake) == 0 then
    smoothStopProgress = clamp((1 - min(1, electrics.values.wheelspeed/4))^2 - smoothedinputValues.throttle*4, 0, 1)
    M.brake = smoothStopProgress*0.15
  end

  -- brake lights control
  -- m/s^2
  local decelerationRateRaw = (lastSpeed-electrics.values.wheelspeed)/dt
  decelerationRate = decelerationSmoother:get(decelerationRateRaw, dt)

  if decelerationRate > 1.3 and M.inputValues.brake == 0 and not electrics.values.escActive then
    electrics.values.brakelights = 1
  end

  if smoothStopProgress > 0.95 and electrics.values.wheelspeed < 0.05 then
    electrics.values.brakelights = 0
  end
  lastSpeed = electrics.values.wheelspeed
end

local function updateInGearArcade(dt)
  M.throttle = M.inputValues.throttle
  M.brake = M.inputValues.brake
  M.isArcadeSwitched = false
  M.clutchRatio = 1

  local gearIndex = automaticHandling.gearIndexLookup[automaticHandling.mode]
  -- driving backwards? - only with automatic shift - for obvious reasons ;)
  if (gearIndex < 0 and M.smoothedValues.avgAV <= 0.8) or (gearIndex <= 0 and M.smoothedValues.avgAV < -1) then
    M.throttle, M.brake = M.brake, M.throttle
    M.isArcadeSwitched = true
  end

  -- neutral gear handling
  if M.timer.neutralSelectionDelayTimer <= 0 then
    if automaticHandling.mode ~= "P" and abs(M.smoothedValues.avgAV) < M.gearboxHandling.arcadeAutoBrakeAVThreshold and M.throttle <= 0 then
      M.brake = max(M.brake, M.gearboxHandling.arcadeAutoBrakeAmount)
    end

    if automaticHandling.mode ~= "N" and abs(M.smoothedValues.avgAV) < M.gearboxHandling.arcadeAutoBrakeAVThreshold and M.smoothedValues.throttle <= 0 then
      gearIndex = 0
      automaticHandling.mode = "N"
      applyGearboxMode()
    else
      if M.smoothedValues.throttleInput > 0 and M.inputValues.throttle > 0 and M.smoothedValues.brakeInput <= 0 and M.smoothedValues.avgAV > -1 and gearIndex < 1 then
        gearIndex = 1
        M.timer.neutralSelectionDelayTimer = M.timerConstants.neutralSelectionDelay
        automaticHandling.mode = "D"
        applyGearboxMode()
      end

      if M.smoothedValues.brakeInput > 0.1 and M.inputValues.brake > 0 and M.smoothedValues.throttleInput <= 0 and M.smoothedValues.avgAV <= 0.5 and gearIndex > -1 then
        gearIndex = -1
        M.timer.neutralSelectionDelayTimer = M.timerConstants.neutralSelectionDelay
        automaticHandling.mode = "R"
        applyGearboxMode()
      end
    end

    if electrics.values.ignitionLevel <= 1 and automaticHandling.mode ~= "P" then
      gearIndex = 0
      M.timer.neutralSelectionDelayTimer = M.timerConstants.neutralSelectionDelay
      automaticHandling.mode = "P"
      applyGearboxMode()
    end
  end

  M.currentGearIndex = (automaticHandling.mode == "N" or automaticHandling.mode == "P") and 0 or gearIndex
  updateExposedData()
  updateRegen(dt)
end

local function updateInGear(dt)
  M.brake = M.inputValues.brake
  M.isArcadeSwitched = false
  M.clutchRatio = 1

  if electrics.values.ignitionLevel <= 1 and automaticHandling.mode ~= "P" then
    M.timer.neutralSelectionDelayTimer = M.timerConstants.neutralSelectionDelay
    automaticHandling.mode = "P"
    applyGearboxMode()
  end

  local gearIndex = automaticHandling.gearIndexLookup[automaticHandling.mode]
  M.currentGearIndex = (automaticHandling.mode == "N" or automaticHandling.mode == "P") and 0 or gearIndex

  if automaticHandling.mode == "P" then
    input.event("parkingbrake", 1, FILTER_DIRECT)
  end
  updateExposedData()
  updateRegen(dt)
end

local function applyUIChanges(type)
  if type == "regen" then
    electrics.values.reducedRegen = electrics.values.reducedRegen - 0.5
    if electrics.values.reducedRegen == -0.5 then
      electrics.values.reducedRegen = 1
    end
  elseif type == "stop" then
    electrics.values.stopMode = electrics.values.stopMode - 0.5
    if electrics.values.stopMode == -0.5 then
      electrics.values.stopMode = 1
    end
  end

  local jsonData = {
    reducedRegen = electrics.values.reducedRegen,
    stopMode = electrics.values.stopMode
  }

  jsonWriteFile("/temp/vehicles/bastion/ebastion/settings.json", jsonData, true)
end

local function sendTorqueData()
  for _, v in ipairs(motors) do
    v:sendTorqueData()
  end
end

local function setIgnition(enabled)
  for _, motor in ipairs(motors) do
    motor:setIgnition(enabled and 1 or 0)
  end
end

local function init(jbeamData, sharedFunctionTable)
  sharedFunctions = sharedFunctionTable

  M.currentGearIndex = 0
  M.throttle = 0
  M.brake = 0
  M.clutchRatio = 1

  gearboxAvailableLogic = {
    arcade = {
      inGear = updateInGearArcade,
      shiftUp = sharedFunctions.warnCannotShiftSequential,
      shiftDown = sharedFunctions.warnCannotShiftSequential,
      shiftToGearIndex = sharedFunctions.switchToRealisticBehavior
    },
    realistic = {
      inGear = updateInGear,
      shiftUp = shiftUp,
      shiftDown = shiftDown,
      shiftToGearIndex = shiftToGearIndex
    }
  }

  motors = {}
  local motorNames = jbeamData.motorNames or {"mainMotor"}
  for _, v in ipairs(motorNames) do
    local motor = powertrain.getDevice(v)
    if motor then
      M.maxRPM = max(M.maxRPM, motor.maxAV * constants.avToRPM)
      table.insert(motors, motor)
    end
  end

  if #motors <= 0 then
    log("E", "shiftLogic-electricMotor", "No motors have been specified, functionality will be limited!")
  end

  automaticHandling.availableModeLookup = {}
  for _, v in pairs(automaticHandling.availableModes) do
    automaticHandling.availableModeLookup[v] = true
  end

  automaticHandling.modes = {}
  automaticHandling.modeIndexLookup = {}
  local modes = jbeamData.automaticModes or "PRND"
  local modeCount = #modes
  local modeOffset = 0
  for i = 1, modeCount do
    local mode = modes:sub(i, i)
    if automaticHandling.availableModeLookup[mode] then
      automaticHandling.modes[i + modeOffset] = mode
      automaticHandling.modeIndexLookup[mode] = i + modeOffset
      automaticHandling.existingModeLookup[mode] = true
    else
      print("unknown auto mode: " .. mode)
    end
  end

  local defaultMode = jbeamData.defaultAutomaticMode or "N"
  automaticHandling.modeIndex = string.find(modes, defaultMode)
  automaticHandling.mode = automaticHandling.modes[automaticHandling.modeIndex]
  automaticHandling.maxGearIndex = 1
  automaticHandling.minGearIndex = -1

  M.idleRPM = 0
  M.maxGearIndex = automaticHandling.maxGearIndex
  M.minGearIndex = abs(automaticHandling.minGearIndex)
  M.energyStorages = sharedFunctions.getEnergyStorages(motors)

  regenHandling.defaultRegen = jbeamData.defaultRegen or 0.2
  regenHandling.brakeRegenCoef = jbeamData.brakeRegenCoef or 0.3
  regenHandling.regenThrottlePedal = jbeamData.regenThrottlePedal or 1

  local combinedRegen = regenHandling.defaultRegen + 1*regenHandling.brakeRegenCoef
  if regenHandling.defaultRegen > combinedRegen then
    electrics.values.fullRegenAmount = regenHandling.defaultRegen
  else
    electrics.values.fullRegenAmount = combinedRegen
  end

  if jbeamData.motorTorqueScale then
    for _,motor in pairs(motors) do
      motor.outputTorqueState = jbeamData.motorTorqueScale
      --motor:scaleOutputTorque(jbeamData.motorTorqueScale)
    end
  end

  if not hasBuiltPie then
    electrics.values.reducedRegen = 1
    electrics.values.stopMode = 1

    local jsonData = jsonReadFile("/temp/vehicles/bastion/ebastion/settings.json")
    if jsonData then
      electrics.values.reducedRegen = jsonData.reducedRegen or 1
      electrics.values.stopMode = jsonData.stopMode or 1
    end

    -- regen mode
    core_quickAccess.addEntry(
      {
          level = "/powertrain/",
          generator = function(entries)
              local entryTable = {
                  title = "Regeneration Mode",
                  priority = 10,
                  icon = "material_not_interested",
                  onSelect = function()
                    require("controller/shiftLogic-electricMotorEBastion").applyUIChanges("regen")
                    return { "reload" }
                  end
              }
              if electrics.values.reducedRegen == 1 then
                entryTable.title = "Regeneration: Normal"
                entryTable.icon = "editor_number_2"
              elseif electrics.values.reducedRegen == 0.5 then
                entryTable.title = "Regeneration: Reduced"
                entryTable.icon = "editor_number_1"
              elseif electrics.values.reducedRegen == 0 then
                entryTable.title = "Regeneration: Off"
              end
              table.insert(entries, entryTable)
          end
      }
    )

    -- stop mode
    core_quickAccess.addEntry(
      {
          level = "/powertrain/",
          generator = function(entries)
              local entryTable = {
                  title = "Stop Mode",
                  priority = 10,
                  icon = "garage_brakes",
                  onSelect = function()
                    require("controller/shiftLogic-electricMotorEBastion").applyUIChanges("stop")
                    return { "reload" }
                  end
              }
              if electrics.values.stopMode == 1 then
                  entryTable.title = "Stop Mode: Normal"
              elseif electrics.values.stopMode == 0.5 then
                entryTable.title = "Stop Mode: Assisted"
              elseif electrics.values.stopMode == 0 then
                entryTable.title = "Stop Mode: Crawling"
              end
              table.insert(entries, entryTable)
          end
      }
    )

    hasBuiltPie = true
  end

  applyGearboxMode()
end

local function getState()
  local data = {grb_mde = automaticHandling.mode}

  return tableIsEmpty(data) and nil or data
end

local function setState(data)
  if data.grb_mde then
    automaticHandling.mode = data.grb_mde
    automaticHandling.modeIndex = automaticHandling.modeIndexLookup[automaticHandling.mode]
    applyGearboxMode()
  end
end

M.init = init

M.applyUIChanges = applyUIChanges
M.gearboxBehaviorChanged = gearboxBehaviorChanged
M.shiftUp = shiftUp
M.shiftDown = shiftDown
M.shiftToGearIndex = shiftToGearIndex
M.updateGearboxGFX = nop
M.setIgnition = setIgnition
M.getGearName = getGearName
M.getGearPosition = getGearPosition
M.sendTorqueData = sendTorqueData

M.getState = getState
M.setState = setState

return M
