-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}

M.outputPorts = {[1] = true}
M.deviceCategories = {engine = true}

local clamp = clamp
local abs = math.abs

local avToRPM = 9.549296596425384

local soundConfiguration
local soundRPMSmoother
local soundLoadSmoother
local engineSoundID
local soundMaxLoadMix
local soundMinLoadMix
local fundamentalFrequencyRPMCoef
local engineVolumeCoef
local reverseOnlyCoef = 1

local engineNodeID = 0

local powertrainDeviceName
local soundRPMSettings = {}
local gear = 1

local function updateSounds(dt)
  local powertrainDevice = powertrain.getDevice(powertrainDeviceName)

  local deviceRPM = powertrainDevice.outputAV1 * avToRPM

  if #soundRPMSettings.gearRatios ~= 1 then
    deviceRPM = soundRPMSettings.idleRPM + deviceRPM * soundRPMSettings.gearRatios[gear]
    if deviceRPM > soundRPMSettings.idleRPM + soundRPMSettings.shiftUpRPM*math.max(electrics.values.throttle, 0.5) and gear ~= #soundRPMSettings.gearRatios then
      gear = gear + 1
      obj:playSFXOnceCT(soundRPMSettings.shiftSound or "event:>Vehicle>Afterfire>01_Shift_EQ1", engineNodeID, (soundRPMSettings.afterFireVolumeCoef or 0) * (soundRPMSettings.shiftVolumeCoef or 0), 1.0, 1 - (soundRPMSettings.afterFireMufflingCoef or 0), 0)
    elseif deviceRPM < soundRPMSettings.idleRPM + soundRPMSettings.shiftDownRPM and gear ~= 1 then
      gear = gear - 1
      obj:playSFXOnceCT(soundRPMSettings.shiftSound or "event:>Vehicle>Afterfire>01_Shift_EQ1", engineNodeID, (soundRPMSettings.afterFireVolumeCoef or 0) * (soundRPMSettings.shiftVolumeCoef or 0), 1.0, 1 - (soundRPMSettings.afterFireMufflingCoef or 0), 0)
    end
    gear = clamp(gear, 1, #soundRPMSettings.gearRatios)
  else
    deviceRPM = soundRPMSettings.idleRPM + deviceRPM * soundRPMSettings.gearRatios[1]
  end
  if soundRPMSettings.soundStandStillRPM then
    deviceRPM = math.min(soundRPMSettings.soundStandStillRPM, deviceRPM)
  end

  if not soundRPMSettings.fadeOutStartSpeedKMH then soundRPMSettings.fadeOutStartSpeedKMH = 9999999999 end
  if not soundRPMSettings.fadeOutStopSpeedKMH then soundRPMSettings.fadeOutStopSpeedKMH = 99999999999 end
  if not soundRPMSettings.fadeOutStopSpeedSecondaryKMH then soundRPMSettings.fadeOutStopSpeedSecondaryKMH = 999999999999 end

  deviceRPM = deviceRPM * (electrics.values.ignitionLevel == 2 and 1 or 0)
  local idle = (soundRPMSettings.idleRPM + math.random(0, soundRPMSettings.idleRPMMaxVariation or 0)) * (electrics.values.ignitionLevel == 2 and 1 or 0)
  local rpm = math.max(soundRPMSmoother:get(abs(math.max(deviceRPM)), dt), idle)
  local speedFadeout = (clamp(electrics.values.wheelspeed*3.6, soundRPMSettings.fadeOutStartSpeedKMH, soundRPMSettings.fadeOutStopSpeedKMH)-soundRPMSettings.fadeOutStartSpeedKMH)/(soundRPMSettings.fadeOutStopSpeedKMH-soundRPMSettings.fadeOutStartSpeedKMH)
  local speedFadeoutSecondary = (clamp(electrics.values.wheelspeed*3.6, soundRPMSettings.fadeOutStopSpeedKMH, soundRPMSettings.fadeOutStopSpeedSecondaryKMH)-soundRPMSettings.fadeOutStopSpeedKMH)/(soundRPMSettings.fadeOutStopSpeedSecondaryKMH-soundRPMSettings.fadeOutStopSpeedKMH)
  --local speedFaded = electrics.values.wheelspeed*3.6 > soundRPMSettings.fadeOutStopSpeedKMH and 0 or 1
  local engineLoad = clamp(soundLoadSmoother:get(abs(math.max(powertrainDevice.instantEngineLoad,0)), dt), soundMinLoadMix, soundMaxLoadMix)
  local fundamentalFreq = sounds.hzToFMODHz(rpm * fundamentalFrequencyRPMCoef)

  if soundRPMSettings.reverseOnly then
    reverseOnlyCoef = (electrics.values.gear == "R") and 1 or 0
  end

  obj:setEngineSound(engineSoundID, rpm*(1-speedFadeoutSecondary) * (reverseOnlyCoef), engineLoad*(1-speedFadeout), fundamentalFreq*(1-speedFadeout), speedFadeout)
end

local function updateGFX(dt)
  updateSounds(dt)
end

local function getSoundConfiguration()
  return soundConfiguration
end

local function resetSounds()
  soundRPMSmoother:reset()
  soundLoadSmoother:reset()
  engineVolumeCoef = 1
  gear = 1
  --dump(sounds)
  sounds.disableOldEngineSounds()
end

local function initEngineSound(soundID, samplePath, engineNodeIDs, offLoadGain, onLoadGain, reference)
  soundConfiguration[reference] = soundConfiguration[reference] or {}
  soundConfiguration[reference].blendFile = samplePath
  obj:queueGameEngineLua(string.format("core_sounds.initEngineSound(%d,%d,%q,%s,%f,%f)", objectId, soundID, samplePath, serialize(engineNodeIDs), offLoadGain, onLoadGain))
end

local function setEngineSoundParameterList(soundID, params, reference)
  soundConfiguration[reference] = soundConfiguration[reference] or {}
  soundConfiguration[reference].params = tableMergeRecursive(soundConfiguration[reference].params or {}, params)
  soundConfiguration[reference].soundID = soundID
  obj:queueGameEngineLua(string.format("core_sounds.setEngineSoundParameterList(%d,%d,%s)", objectId, soundID, serialize(params)))
end

local function initSounds(jbeamData)
  local soundConfig = v.data[jbeamData.soundConfig]
  if soundConfig and not sounds.usesOldCustomSounds then
    soundConfiguration = {}
    powertrainDeviceName = jbeamData.powertrainDeviceName
    soundRPMSettings = jbeamData.fakeSoundSettings
    for _, v in pairs(v.data.nodes) do
      if v.name == jbeamData.nodeName then
        engineNodeID = v.cid
      end
    end

    engineSoundID = powertrain.getEngineSoundID()
    local rpmInRate = soundConfig.rpmSmootherInRate or 15
    local rpmOutRate = soundConfig.rpmSmootherOutRate or 25
    soundRPMSmoother = newTemporalSmoothingNonLinear(rpmInRate, rpmOutRate)
    local loadInRate = soundConfig.loadSmootherInRate or 20
    local loadOutRate = soundConfig.loadSmootherOutRate or 20
    soundLoadSmoother = newTemporalSmoothingNonLinear(loadInRate, loadOutRate)
    soundMaxLoadMix = soundConfig.maxLoadMix or 1
    soundMinLoadMix = soundConfig.minLoadMix or 0
    local fundamentalFrequencyCylinderCount = soundConfig.fundamentalFrequencyCylinderCount or 6
    fundamentalFrequencyRPMCoef = fundamentalFrequencyCylinderCount / 120
    engineVolumeCoef = 1
    local onLoadGain = soundConfig.onLoadGain or 1
    local offLoadGain = soundConfig.offLoadGain or 1

    local sampleName = soundConfig.sampleName
    if sampleName then
    local sampleFolder = soundConfig.sampleFolder or "art/sound/blends/"
    local samplePath = sampleFolder .. sampleName .. ".sfxBlend2D.json"
    initEngineSound(engineSoundID, samplePath, {engineNodeID}, offLoadGain, onLoadGain, "motor")

    local main_gain = soundConfig.mainGain or 0

    local eq_a_freq = sounds.hzToFMODHz(soundConfig.lowCutFreq or 20)
    local eq_b_freq = sounds.hzToFMODHz(soundConfig.highCutFreq or 10000)
    local eq_c_freq = sounds.hzToFMODHz(soundConfig.eqLowFreq or 500)
    local eq_c_gain = soundConfig.eqLowGain or 0
    local eq_c_reso = soundConfig.eqLowWidth or 0
    local eq_d_freq = sounds.hzToFMODHz(soundConfig.eqHighFreq or 2000)
    local eq_d_gain = soundConfig.eqHighGain or 0
    local eq_d_reso = soundConfig.eqHighWidth or 0
    local eq_e_gain = soundConfig.eqFundamentalGain or 0
    local eq_e_reso = soundConfig.eqFundamentalWidth or 1

    local params = {
      main_gain = main_gain,
      eq_a_freq = eq_a_freq,
      eq_b_freq = eq_b_freq,
      eq_c_freq = eq_c_freq,
      eq_c_gain = eq_c_gain,
      eq_c_reso = eq_c_reso,
      eq_d_freq = eq_d_freq,
      eq_d_gain = eq_d_gain,
      eq_d_reso = eq_d_reso,
      eq_e_gain = eq_e_gain,
      eq_e_reso = eq_e_reso,
      onLoadGain = onLoadGain,
      offLoadGain = offLoadGain,
      muffled = 0.5
    }

    --dump(params)

    setEngineSoundParameterList(engineSoundID, params, "motor")

    updateSounds = updateSounds
    end
    --dump(sounds)
    sounds.disableOldEngineSounds()
  end
end

M.init = nop
M.updateGFX = updateGFX
M.initSounds = initSounds
M.resetSounds = resetSounds
M.initEngineSound = initEngineSound
M.setEngineSoundParameterList = setEngineSoundParameterList
M.getSoundConfiguration = getSoundConfiguration

return M
