'use strict';
angular.module('beamng.apps')
  .directive('sgrProCamSidebar', ['$interval', function ($interval) {
    return {
      template:
        '<div style="width: 100%; height: 100%; position: relative;">' +
          '<object style="width:100%; height:100%; position: absolute; top: 0; left: 0;" ' +
                  'type="image/svg+xml" data="/ui/modules/apps/sgrprocamsidebar/sgrprocamsidebar.svg" id="sidebarSvg"></object>' +
        '</div>',
      restrict: 'EA',
      scope: true,
      link: function (scope, element, attrs) {
        const svgObject = element.find('object')[0];
        svgObject.addEventListener('load', function () {
          const svgDoc = svgObject.contentDocument;
          // Time/date elements.
          const timeValue = svgDoc.getElementById('timeValue');
          const timeUnit = svgDoc.getElementById('timeUnit');
          const dateLabel = svgDoc.getElementById('dateLabel');
          // Trip meter element.
          const distanceValue = svgDoc.getElementById('distanceValue');
          // Altitude display element (current altitude in feet).
          const altitudeValue = svgDoc.getElementById('altitudeValue');
          // Elevation gain display element (cumulative change in feet).
          const elevationValueElement = svgDoc.getElementById('elevationValue');
          // Reset button element.
          const resetButton = svgDoc.getElementById('resetButton');

          // Array of month names.
          const monthNames = [
            "January", "February", "March", "April", "May", "June",
            "July", "August", "September", "October", "November", "December"
          ];

          // Variables for trip meter integration.
          let sgrProCamSidebartripDistance = parseFloat(localStorage.getItem("sgrProCamSidebartripDistance")) || 0;
          let lastUpdateTimestamp = Date.now();

          // Variables for elevation gain calculation.
          let sgrProCamSidebarcumulativeElevation = parseFloat(localStorage.getItem("sgrProCamSidebarcumulativeElevation")) || 0;
          let sgrProCamSidebarlastAltitude = localStorage.getItem("sgrProCamSidebarlastAltitude");
          if (sgrProCamSidebarlastAltitude !== null) {
            sgrProCamSidebarlastAltitude = parseFloat(sgrProCamSidebarlastAltitude);
          }

          // Update displayed values from saved stats.
          if (distanceValue) {
            distanceValue.textContent = sgrProCamSidebartripDistance.toFixed(1);
          }
          if (elevationValueElement) {
            elevationValueElement.textContent = Math.round(sgrProCamSidebarcumulativeElevation);
          }

          // Function to reset stats.
          function resetStats() {
            sgrProCamSidebartripDistance = 0;
            sgrProCamSidebarcumulativeElevation = 0;
            sgrProCamSidebarlastAltitude = undefined;
            lastUpdateTimestamp = Date.now();
            localStorage.removeItem("sgrProCamSidebartripDistance");
            localStorage.removeItem("sgrProCamSidebarcumulativeElevation");
            localStorage.removeItem("sgrProCamSidebarlastAltitude");
            if (distanceValue) {
              distanceValue.textContent = sgrProCamSidebartripDistance.toFixed(1);
            }
            if (elevationValueElement) {
              elevationValueElement.textContent = 0;
            }
          }

          // Attach click handler to the reset button.
          if (resetButton) {
            resetButton.style.cursor = 'pointer';
            resetButton.addEventListener('click', function () {
              resetStats();
            });
          }

          // Listen for stream updates (including airspeed and altitude).
          scope.$on('streamsUpdate', function (event, data) {
            // Update current altitude value (convert from meters to feet).
            if (altitudeValue && data.electrics.altitude !== undefined) {
              const currentAltitudeFeet = data.electrics.altitude * 3.28084;
              altitudeValue.textContent = Math.round(currentAltitudeFeet);
            }

            // Calculate cumulative elevation gain (in feet).
            if (data.electrics.altitude !== undefined) {
              const currentAltitude = data.electrics.altitude; // in meters
              if (typeof sgrProCamSidebarlastAltitude !== 'undefined' && sgrProCamSidebarlastAltitude !== null) {
                const deltaMeters = currentAltitude - sgrProCamSidebarlastAltitude;
                const deltaFeet = deltaMeters * 3.28084;
                sgrProCamSidebarcumulativeElevation += deltaFeet;
              }
              sgrProCamSidebarlastAltitude = currentAltitude;
              if (elevationValueElement) {
                elevationValueElement.textContent = Math.round(sgrProCamSidebarcumulativeElevation);
              }
            }

            // Get the airspeed value (in mph).
            const speed = UiUnits.speed(data.electrics.airspeed).val;
            // Calculate elapsed time in seconds.
            const currentTimestamp = Date.now();
            const dt = (currentTimestamp - lastUpdateTimestamp) / 1000;
            lastUpdateTimestamp = currentTimestamp;

            // Integrate distance traveled: (mph / 3600) gives miles per second.
            sgrProCamSidebartripDistance += (speed / 3600) * dt;
            if (distanceValue) {
              distanceValue.textContent = sgrProCamSidebartripDistance.toFixed(1);
            }

            // Save stats to localStorage.
            localStorage.setItem("sgrProCamSidebartripDistance", sgrProCamSidebartripDistance.toFixed(1));
            localStorage.setItem("sgrProCamSidebarcumulativeElevation", sgrProCamSidebarcumulativeElevation.toFixed(2));
            localStorage.setItem("sgrProCamSidebarlastAltitude", sgrProCamSidebarlastAltitude);
          });

          // Update the time/date elements every second.
          $interval(function () {
            const now = new Date();
            let hours = now.getHours();
            const minutes = ('0' + now.getMinutes()).slice(-2);
            let unit = 'AM';
            if (hours >= 12) {
              unit = 'PM';
              if (hours > 12) {
                hours -= 12;
              }
            }
            if (hours === 0) {
              hours = 12;
            }

            // Format time as "h:mm" (ignoring seconds)
            const timeStr = hours + ':' + minutes;
            // Format date as "MonthName day, year"
            const dateStr = monthNames[now.getMonth()] + ' ' +
              now.getDate() + ', ' + now.getFullYear();

            if (timeValue) {
              timeValue.textContent = timeStr;
            }
            if (timeUnit) {
              timeUnit.textContent = unit;
            }
            if (dateLabel) {
              dateLabel.textContent = dateStr;
            }
          }, 1000);
        });
      }
    };
  }]);
