local M = {}
M.dependencies = {'ui_imgui'}

local imgui = ui_imgui

local windowOpen = {false}
local windowSize = imgui.ImVec2(400, 500)

-- UI Colors
local colorButton = imgui.ImVec4(0.15, 0.15, 0.15, 1.0)
local colorButtonHover = imgui.ImVec4(0.25, 0.25, 0.25, 1.0)
local colorButtonActive = imgui.ImVec4(0.35, 0.35, 0.35, 1.0)
local colorText = imgui.ImVec4(0.8, 0.8, 0.8, 1.0)
local colorBackground = imgui.ImVec4(0.1, 0.1, 0.1, 1.0)

-- Button state tracking
local lastPressedButtons = {}

local function sendToVehicle(key, value)
    local vehicle = getPlayerVehicle(0)
    if vehicle then
        vehicle:queueLuaCommand(string.format("electrics.values.%s = %s", key, tostring(value)))
    end
end

local function handleButtonPress(navKey)
    if not lastPressedButtons[navKey] then
        sendToVehicle(navKey, 1)
        lastPressedButtons[navKey] = true
    end
end

local function handleButtonRelease(navKey)
    if lastPressedButtons[navKey] then
        sendToVehicle(navKey, 0)
        lastPressedButtons[navKey] = false
    end
end

local function drawNavButton(label, id, navKey, buttonWidth, buttonHeight)
    local cursorPos = imgui.GetCursorScreenPos()
    local drawList = imgui.GetWindowDrawList()
    
    -- Draw drop shadow
    local shadowOffsetY = 2
    local shadowExpand = 1
    local shadowColor = imgui.GetColorU322(imgui.ImVec4(0, 0, 0, 0.6))
    
    drawList:AddRectFilled(
        imgui.ImVec2(cursorPos.x - shadowExpand, cursorPos.y + shadowOffsetY),
        imgui.ImVec2(cursorPos.x + buttonWidth + shadowExpand, cursorPos.y + buttonHeight + shadowOffsetY),
        shadowColor,
        20,
        0
    )
    
    -- Button colors
    local btnColor = colorButton
    local txtColor = colorText
    
    -- Draw button
    imgui.PushStyleColor2(imgui.Col_Button, btnColor)
    imgui.PushStyleColor2(imgui.Col_ButtonHovered, colorButtonHover)
    imgui.PushStyleColor2(imgui.Col_ButtonActive, colorButtonActive)
    imgui.PushStyleColor2(imgui.Col_Text, txtColor)
    imgui.PushStyleVar1(imgui.StyleVar_FrameRounding, 20)
    imgui.PushStyleVar1(imgui.StyleVar_FrameBorderSize, 0)
    
    imgui.SetWindowFontScale(1.7)
    local clicked = imgui.Button(label .. "##" .. id, imgui.ImVec2(buttonWidth, buttonHeight))
    local isActive = imgui.IsItemActive()
    imgui.SetWindowFontScale(1.0)
    
    imgui.PopStyleVar(2)
    imgui.PopStyleColor(4)
    
    -- Handle button state
    if isActive then
        handleButtonPress(navKey)
    else
        handleButtonRelease(navKey)
    end
    
    -- Subtle bevel effect
    local bevelSize = 1
    local highlightColor = imgui.GetColorU322(imgui.ImVec4(0.3, 0.3, 0.3, 0.2))
    drawList:AddLine(
        imgui.ImVec2(cursorPos.x + 20, cursorPos.y + bevelSize),
        imgui.ImVec2(cursorPos.x + buttonWidth - 20, cursorPos.y + bevelSize),
        highlightColor,
        1
    )
    
    local shadowBevelColor = imgui.GetColorU322(imgui.ImVec4(0, 0, 0, 0.2))
    drawList:AddLine(
        imgui.ImVec2(cursorPos.x + 20, cursorPos.y + buttonHeight - bevelSize),
        imgui.ImVec2(cursorPos.x + buttonWidth - 20, cursorPos.y + buttonHeight - bevelSize),
        shadowBevelColor,
        1
    )
    
    return clicked
end

local function drawController()
    imgui.Dummy(imgui.ImVec2(0, 20))
    
    -- Title
    imgui.PushStyleColor2(imgui.Col_Text, colorText)
    imgui.SetWindowFontScale(2.0)
    local titleText = "CarPlay"
    local textWidth = imgui.CalcTextSize(titleText).x
    imgui.SetCursorPosX((imgui.GetContentRegionAvailWidth() - textWidth) / 2)
    imgui.Text(titleText)
    imgui.SetWindowFontScale(1.0)
    imgui.PopStyleColor(1)
    
    imgui.Dummy(imgui.ImVec2(0, 30))
    
    -- D-pad layout
    local buttonWidth = 70
    local buttonHeight = 70
    local centerX = imgui.GetContentRegionAvailWidth() / 2
    local spacing = 10
    
    -- UP button
    imgui.SetCursorPosX(centerX - buttonWidth / 2)
    drawNavButton("^", "up", "menuNavUp", buttonWidth, buttonHeight)
    
    imgui.Dummy(imgui.ImVec2(0, spacing))
    
    -- LEFT, SELECT, RIGHT row
    imgui.SetCursorPosX(centerX - buttonWidth * 1.5 - spacing)
    drawNavButton("<", "left", "menuNavLeft", buttonWidth, buttonHeight)
    
    imgui.SameLine()
    imgui.SetCursorPosX(centerX - buttonWidth / 2)
    drawNavButton("SEL", "select", "menuSelect", buttonWidth, buttonHeight)
    
    imgui.SameLine()
    imgui.SetCursorPosX(centerX + buttonWidth / 2 + spacing)
    drawNavButton(">", "right", "menuNavRight", buttonWidth, buttonHeight)
    
    imgui.Dummy(imgui.ImVec2(0, spacing))
    
    -- DOWN button
    imgui.SetCursorPosX(centerX - buttonWidth / 2)
    drawNavButton("v", "down", "menuNavDown", buttonWidth, buttonHeight)
    
    imgui.Dummy(imgui.ImVec2(0, 30))
    
    -- BACK button
    imgui.SetCursorPosX(30)
    drawNavButton("< BACK", "back", "menuBack", 120, 60)
end

local function drawWindow()
    if not windowOpen[0] then return end
    
    -- Make title bar same color as background
    local titleBarColor = imgui.ImVec4(0.08, 0.08, 0.08, 1.0)
    
    imgui.PushStyleColor2(imgui.Col_WindowBg, colorBackground)
    imgui.PushStyleColor2(imgui.Col_TitleBg, titleBarColor)
    imgui.PushStyleColor2(imgui.Col_TitleBgActive, titleBarColor)
    imgui.PushStyleColor2(imgui.Col_TitleBgCollapsed, titleBarColor)
    imgui.SetNextWindowSize(windowSize, imgui.Cond_Always)
    
    local beginResult = imgui.Begin("CarPlay Controller", windowOpen, imgui.WindowFlags_NoCollapse + imgui.WindowFlags_NoResize)
    
    if beginResult then
        drawController()
    end
    
    imgui.End()
    imgui.PopStyleColor(4)
end

local function toggle()
    windowOpen[0] = not windowOpen[0]
end

local function show()
    windowOpen[0] = true
end

local function hide()
    windowOpen[0] = false
end

local function onExtensionLoaded()
    log('I', 'CarPlayController', 'CarPlay Controller loaded')
end

local function onUpdate(dt)
    drawWindow()
end

M.onExtensionLoaded = onExtensionLoaded
M.onUpdate = onUpdate
M.toggle = toggle
M.show = show
M.hide = hide

return M