local M = {}

local logTag = "idleCinematicCamera"
local isEnabled = true

local config = {
  idleThreshold = 20.0,
  angleDuration = 8.0,
  showNotifications = true,
  playMusic = true  -- Re-enabled since sound effects should work properly
}

local state = {
  lastCameraMovementTime = 0,
  isActive = false,
  originalCamera = nil,
  initialized = false,
  lastAngleTime = 0,
  currentAngleIndex = 1,
  usedAngles = {},
  shotProgress = 0,
  refNodes = nil,
  lastMousePos = {x = 0, y = 0},
  musicPlaying = false
}

local audioHandle = nil
local musicStartTime = 0

local cinematicShots = {
  {
    name = "Front Bumper Detail",
    startPos = {x = 3, y = -6, z = 1.5},
    endPos = {x = -3, y = -6, z = 2},
    startFov = 40,
    endFov = 35,
    lookAt = {x = 0, y = -1.2, z = 0.7}
  },
  {
    name = "Left Side Glide",
    startPos = {x = -6, y = -2, z = 2.5},
    endPos = {x = -6, y = 3, z = 3},
    startFov = 42,
    endFov = 38,
    lookAt = {x = 0, y = 0, z = 0.9}
  },
  {
    name = "Right Side Track",
    startPos = {x = 6, y = -2, z = 2.5},
    endPos = {x = 6, y = 3, z = 3},
    startFov = 42,
    endFov = 38,
    lookAt = {x = 0, y = 0, z = 0.9}
  },
  {
    name = "Rear Sweep",
    startPos = {x = -4, y = 6, z = 2.5},
    endPos = {x = 4, y = 6, z = 2.5},
    startFov = 45,
    endFov = 40,
    lookAt = {x = 0, y = 1.2, z = 0.8}
  },
  {
    name = "Low Front Corner",
    startPos = {x = 4, y = -5, z = 0.8},
    endPos = {x = 4, y = -5, z = 3},
    startFov = 50,
    endFov = 35,
    lookAt = {x = 0, y = -0.5, z = 0.8}
  },
  {
    name = "Hood Line Slide",
    startPos = {x = 2.5, y = -4.5, z = 3},
    endPos = {x = -2.5, y = -4.5, z = 3},
    startFov = 38,
    endFov = 32,
    lookAt = {x = 0, y = -0.8, z = 0.9}
  },
  {
    name = "Door Detail Left",
    startPos = {x = -5.5, y = 0, z = 2.5},
    endPos = {x = -4.5, y = 0, z = 2},
    startFov = 32,
    endFov = 25,
    lookAt = {x = -1.5, y = 0, z = 1}
  },
  {
    name = "Door Detail Right",
    startPos = {x = 5.5, y = 0, z = 2.5},
    endPos = {x = 4.5, y = 0, z = 2},
    startFov = 32,
    endFov = 25,
    lookAt = {x = 1.5, y = 0, z = 1}
  },
  {
    name = "Rear Light Focus",
    startPos = {x = 4, y = 5.5, z = 2},
    endPos = {x = 3, y = 5.5, z = 2.8},
    startFov = 30,
    endFov = 25,
    lookAt = {x = 1, y = 1.8, z = 0.9}
  },
  {
    name = "Low Rear Corner",
    startPos = {x = -5, y = 5, z = 0.8},
    endPos = {x = -5, y = 5, z = 4},
    startFov = 45,
    endFov = 35,
    lookAt = {x = 0, y = 1, z = 0.8}
  },
  {
    name = "Front Corner Orbit",
    startPos = {x = 5, y = -5, z = 2.5},
    endPos = {x = 2.5, y = -6, z = 3.5},
    startFov = 40,
    endFov = 35,
    lookAt = {x = 0.5, y = -1, z = 0.8}
  },
  {
    name = "Rear Corner Orbit",
    startPos = {x = -5, y = 5, z = 2.5},
    endPos = {x = -2.5, y = 6, z = 3.5},
    startFov = 40,
    endFov = 35,
    lookAt = {x = -0.5, y = 1, z = 0.8}
  },
  {
    name = "Windshield Reflection",
    startPos = {x = 1.5, y = -4, z = 3.5},
    endPos = {x = -1.5, y = -4, z = 4},
    startFov = 35,
    endFov = 30,
    lookAt = {x = 0, y = -0.5, z = 1.2}
  },
  {
    name = "Side Mirror Detail",
    startPos = {x = -6, y = -1.5, z = 3},
    endPos = {x = -5, y = -1.5, z = 2.5},
    startFov = 28,
    endFov = 22,
    lookAt = {x = -1.8, y = -0.3, z = 1.2}
  },
  {
    name = "Grille Close-up",
    startPos = {x = 1, y = -5.5, z = 1.8},
    endPos = {x = -1, y = -5, z = 2.3},
    startFov = 30,
    endFov = 25,
    lookAt = {x = 0, y = -1.5, z = 0.8}
  }
}

local function startIdleMusic()
  if config.playMusic and not state.musicPlaying then
    if Engine and Engine.Audio and Engine.Audio.createSource then
      -- Use AudioGui channel instead of AudioChannelMusic (which doesn't exist)
      audioHandle = Engine.Audio.createSource('AudioGui', '/art/idleMusic/idleSong1.mp3')
      if audioHandle then
        -- Get the actual sound object and start playing
        local sound = scenetree.findObjectById(audioHandle)
        if sound then
          sound:play(0) -- Use 0 instead of -1, we'll handle looping manually
          state.musicPlaying = true
          musicStartTime = socket.gettime()
          log('I', logTag, "Started idle music using createSource method on AudioGui")
        else
          log('E', logTag, "Could not find sound object by ID")
        end
      else
        log('E', logTag, "Engine.Audio.createSource returned nil")
      end
    end
  end
end

local function restartMusicIfNeeded()
  if state.musicPlaying and audioHandle then
    local sound = scenetree.findObjectById(audioHandle)
    if sound then
      -- Restart music every 30 seconds (adjust based on your music file length)
      local currentTime = socket.gettime()
      if currentTime - musicStartTime >= 220 then
        sound:play(0) -- Restart the music
        musicStartTime = currentTime -- Reset the timer
        log('I', logTag, "Restarted idle music (timer-based looping)")
      end
    end
  end
end

local function stopIdleMusic()
  if state.musicPlaying and audioHandle then
    -- Get the sound object and stop it properly
    local sound = scenetree.findObjectById(audioHandle)
    if sound then
      sound:stop(-1)
      log('I', logTag, "Stopped idle music using sound:stop(-1)")
    end
    
    state.musicPlaying = false
    audioHandle = nil
    musicStartTime = 0
  end
end

local function isCameraMovementActive()
  if photoModeOpen == true then
    return true
  end
  
  if ui_imgui then
    local mousePos = ui_imgui.GetMousePos()
    if mousePos then
      if math.abs(mousePos.x - state.lastMousePos.x) > 2 or 
         math.abs(mousePos.y - state.lastMousePos.y) > 2 then
        state.lastMousePos.x = mousePos.x
        state.lastMousePos.y = mousePos.y
        return true
      end
    end
  end
  
  if MoveManager.yawRelative ~= 0 or MoveManager.pitchRelative ~= 0 then
    return true
  end
  
  if MoveManager.yawRight > 0.01 or MoveManager.yawLeft > 0.01 then
    return true
  end
  
  if MoveManager.pitchDown > 0.01 or MoveManager.pitchUp > 0.01 then
    return true
  end
  
  if MoveManager.zoomIn > 0.01 or MoveManager.zoomOut > 0.01 then
    return true
  end
  
  if MoveManager.rollRight > 0.01 or MoveManager.rollLeft > 0.01 then
    return true
  end
  
  return false
end

local function onExtensionLoaded()
  state.lastCameraMovementTime = socket.gettime()
  state.initialized = true
  
  if ui_imgui then
    local mousePos = ui_imgui.GetMousePos()
    if mousePos then
      state.lastMousePos.x = mousePos.x
      state.lastMousePos.y = mousePos.y
    end
  end
  
  if config.showNotifications then
    ui_message("Idle Camera loaded! (20s)", 3)
  end
end

local function deactivateIdleCamera()
  if not state.isActive then return end
  
  state.isActive = false
  state.usedAngles = {}
  state.refNodes = nil
  state.shotProgress = 0
  
  -- Stop the idle music when deactivating
  stopIdleMusic()
  
  local currentCam = core_camera.getActiveCamName(0)
  if currentCam == "free" then
    commands.setGameCamera()
    
    if state.originalCamera and state.originalCamera ~= "free" then
      core_camera.setByName(0, state.originalCamera, false)
    end
  end
  
  state.originalCamera = nil
  
  if config.showNotifications then
    ui_message("Camera Control Restored", 1)
  end
end

local function resetCameraMovementTimer()
  state.lastCameraMovementTime = socket.gettime()
  if state.isActive then
    deactivateIdleCamera()
  end
end

local function getNxyz(vehicle)
  if not state.refNodes then
    local vid = vehicle:getId()
    local vmvd = extensions.core_vehicle_manager.getVehicleData(vid)
    if vmvd and vmvd.vdata and vmvd.vdata.refNodes then
      state.refNodes = vmvd.vdata.refNodes[0]
    end
  end
  
  if not state.refNodes then
    return vec3(1,0,0), vec3(0,1,0), vec3(0,0,1)
  end
  
  local ref = vehicle:getNodePosition(state.refNodes.ref)
  local left = vehicle:getNodePosition(state.refNodes.left)
  local back = vehicle:getNodePosition(state.refNodes.back)
  
  local nx = (left - ref):normalized()
  local ny = (back - ref):normalized()
  local nz = nx:cross(ny):normalized()
  
  return nx, ny, nz
end

local function getNextShotIndex()
  local availableShots = {}
  
  if #state.usedAngles >= #cinematicShots - 1 then
    state.usedAngles = {}
  end
  
  for i = 1, #cinematicShots do
    local used = false
    for _, usedIndex in ipairs(state.usedAngles) do
      if i == usedIndex then
        used = true
        break
      end
    end
    if not used then
      table.insert(availableShots, i)
    end
  end
  
  if #availableShots > 0 then
    return availableShots[math.random(1, #availableShots)]
  else
    return 1
  end
end

local function lerp(a, b, t)
  return a + (b - a) * t
end

local function updateCameraMovement(vehicle)
  local shot = cinematicShots[state.currentAngleIndex]
  if not shot then return end
  
  local vehPos = vehicle:getPosition()
  local nx, ny, nz = getNxyz(vehicle)
  
  local currentPosX = lerp(shot.startPos.x, shot.endPos.x, state.shotProgress)
  local currentPosY = lerp(shot.startPos.y, shot.endPos.y, state.shotProgress)
  local currentPosZ = lerp(shot.startPos.z, shot.endPos.z, state.shotProgress)
  
  local currentFov = lerp(shot.startFov, shot.endFov, state.shotProgress)
  
  local offset = nx * currentPosX + ny * currentPosY + nz * currentPosZ
  local camPos = vehPos + offset
  
  local lookAtOffset = nx * shot.lookAt.x + ny * shot.lookAt.y + nz * shot.lookAt.z
  local lookAtPos = vehPos + lookAtOffset
  
  local lookDir = lookAtPos - camPos
  lookDir:normalize()
  local camRot = quatFromDir(lookDir)
  
  core_camera.setPosition(0, camPos)
  core_camera.setRotation(0, camRot)
  core_camera.setFOV(0, currentFov)
end

local function activateIdleCamera()
  if state.isActive or not isEnabled then return end
  
  local vehicle = getPlayerVehicle(0)
  if not vehicle then return end
  
  state.isActive = true
  state.lastAngleTime = socket.gettime()
  state.usedAngles = {}
  state.shotProgress = 0
  state.refNodes = nil
  
  state.originalCamera = core_camera.getActiveCamName(0)
  
  commands.setFreeCamera()
  
  state.currentAngleIndex = getNextShotIndex()
  table.insert(state.usedAngles, state.currentAngleIndex)
  updateCameraMovement(vehicle)
  
  startIdleMusic()
  
  if config.showNotifications then
    ui_message("Cinematic: " .. cinematicShots[state.currentAngleIndex].name, 2)
  end
end

local function onUpdate(dt)
  if not isEnabled or not state.initialized then return end
  
  local currentTime = socket.gettime()
  local vehicle = getPlayerVehicle(0)
  
  if not vehicle then
    if state.isActive then
      deactivateIdleCamera()
    end
    return
  end
  
  if isCameraMovementActive() then
    resetCameraMovementTimer()
    return
  end
  
  if state.isActive then
    -- Check if music needs to restart (for looping)
    restartMusicIfNeeded()
    
    local timeInShot = currentTime - state.lastAngleTime
    state.shotProgress = math.min(timeInShot / config.angleDuration, 1.0)
    
    updateCameraMovement(vehicle)
    
    if timeInShot >= config.angleDuration then
      state.currentAngleIndex = getNextShotIndex()
      table.insert(state.usedAngles, state.currentAngleIndex)
      state.lastAngleTime = currentTime
      state.shotProgress = 0
      
      if config.showNotifications then
        ui_message(cinematicShots[state.currentAngleIndex].name, 1.5)
      end
    end
    
    local velocity = vehicle:getVelocity()
    if velocity:length() > 1.0 then
      resetCameraMovementTimer()
      return
    end
  else
    local velocity = vehicle:getVelocity()
    local speed = velocity:length()
    
    if speed > 0.2 then
      state.lastCameraMovementTime = currentTime
      return
    end
    
    local timeSinceMovement = currentTime - state.lastCameraMovementTime
    if timeSinceMovement >= config.idleThreshold and speed <= 0.2 then
      activateIdleCamera()
    end
  end
end

local function onInputAction(action, value, filter)
  if math.abs(value) > 0.1 then 
    resetCameraMovementTimer()
  end
end

local function onKeyDown(key, modifiers)
  resetCameraMovementTimer()
end

local function onVehicleResetted(vehicleId)
  resetCameraMovementTimer()
end

local function onVehicleSwitched(oldVehId, newVehId)
  resetCameraMovementTimer()
  state.refNodes = nil
end

local function setIdleTime(seconds)
  config.idleThreshold = math.max(5, seconds or 60)
end

local function setShotDuration(seconds)
  config.angleDuration = math.max(3, seconds or 8)
end

local function enable()
  isEnabled = true
end

local function disable()
  isEnabled = false
  if state.isActive then deactivateIdleCamera() end
end

local function forceActivate()
  activateIdleCamera()
end

local function nextShot()
  if not state.isActive then return end
  
  state.currentAngleIndex = getNextShotIndex()
  table.insert(state.usedAngles, state.currentAngleIndex)
  state.lastAngleTime = socket.gettime()
  state.shotProgress = 0
end

local function setMusicEnabled(enabled)
  config.playMusic = enabled
  if not enabled and state.musicPlaying then
    stopIdleMusic()
  end
end

local function toggleMusic()
  setMusicEnabled(not config.playMusic)
  if config.showNotifications then
    ui_message("Music: " .. (config.playMusic and "ON" or "OFF"), 1)
  end
end

local function getStatus()
  local status = {
    enabled = isEnabled,
    active = state.isActive,
    idleThreshold = config.idleThreshold,
    shotDuration = config.angleDuration,
    timeSinceMovement = socket.gettime() - state.lastCameraMovementTime,
    hasRefNodes = state.refNodes ~= nil,
    cameraMovementActive = isCameraMovementActive()
  }
  
  if state.isActive then
    status.currentShot = cinematicShots[state.currentAngleIndex].name
    status.shotProgress = math.floor(state.shotProgress * 100) .. "%"
    status.usedShots = #state.usedAngles .. "/" .. #cinematicShots
  end
  
  return status
end

M.onExtensionLoaded = onExtensionLoaded
M.onUpdate = onUpdate
M.onInputAction = onInputAction
M.onKeyDown = onKeyDown
M.onVehicleResetted = onVehicleResetted
M.onVehicleSwitched = onVehicleSwitched

M.setIdleTime = setIdleTime
M.setShotDuration = setShotDuration
M.enable = enable
M.disable = disable
M.getStatus = getStatus
M.activate = forceActivate
M.nextShot = nextShot
M.setMusicEnabled = setMusicEnabled
M.toggleMusic = toggleMusic

return M