local M = {}

local settings = {
    configPath = "vehicles/RLA_A90/infotainment_settings.json",
    defaults = {
        saveChangesEnabled = 1,
        daytimeLights = 1,
        daytimeLightColorR = 255,
        daytimeLightColorG = 255,
        daytimeLightColorB = 255,
        footwellLights = 1,
        footwellLightColorR = 255,
        footwellLightColorG = 255,
        footwellLightColorB = 255,
        mirror_fold_enabled = 1,
        flashLockUnlock = 1,
        lockUnlockSoundEnabled = 1,
        doorHandleLighting = 1
    },
    current = {},
    lastKnownValues = {},
    initialized = false,
    startTime = 0,
    canSave = false
}

local function ensureDirectoryExists()
    local directoryPath = "vehicles/RLA_A90"
    if FS then
        if not FS:directoryExists(directoryPath) then
            FS:directoryCreate(directoryPath, true)
        end
    end
end

local function loadSettings()
    local success, settingsData = pcall(function()
        return jsonReadFile(settings.configPath)
    end)

    if success and settingsData then
        for key, defaultValue in pairs(settings.defaults) do
            settings.current[key] = settingsData[key] or defaultValue
        end
        return true
    else
        for key, value in pairs(settings.defaults) do
            settings.current[key] = value
        end
        return false
    end
end

local function saveSettings()
    if not settings.canSave then return end
    pcall(function()
        ensureDirectoryExists()
        jsonWriteFile(settings.configPath, settings.current, true)
    end)
end

local function saveSaveChangesState()
    if not settings.canSave then return end

    local existingData = {}
    local success, settingsData = pcall(function()
        return jsonReadFile(settings.configPath)
    end)
    if success and settingsData then
        existingData = settingsData
    end

    existingData.saveChangesEnabled = settings.current.saveChangesEnabled

    pcall(function()
        ensureDirectoryExists()
        jsonWriteFile(settings.configPath, existingData, true)
    end)
end

local function sendSettingsToJS()
    if not settings.initialized then return end

    local settingsJson = jsonEncode(settings.current)
    if electrics and electrics.values then
        electrics.values.infotainmentSettingsReady = 1
        electrics.values.infotainmentSettingsJson = settingsJson
        for key, value in pairs(settings.current) do
            if key ~= 'infotainmentSettingsReady' and key ~= 'infotainmentSettingsJson' then
                electrics.values[key] = value
            end
        end
    end
end

local function applySettingsToElectrics()
    if not electrics or not electrics.values then return end

    if settings.current.saveChangesEnabled == 1 then
        for key, value in pairs(settings.current) do
            electrics.values[key] = value
            settings.lastKnownValues[key] = value
        end
    else
        for key, value in pairs(settings.defaults) do
            if key == 'saveChangesEnabled' then
                electrics.values[key] = settings.current.saveChangesEnabled
                settings.lastKnownValues[key] = settings.current.saveChangesEnabled
            else
                electrics.values[key] = value
                settings.lastKnownValues[key] = value
            end
        end
    end

    sendSettingsToJS()
end

local function monitorElectricValues()
    if not electrics or not electrics.values then return end

    local currentSaveState = electrics.values.saveChangesEnabled or 1
    if settings.lastKnownValues.saveChangesEnabled ~= currentSaveState then
        settings.current.saveChangesEnabled = currentSaveState
        settings.lastKnownValues.saveChangesEnabled = currentSaveState
        saveSaveChangesState()
        return
    end

    if settings.current.saveChangesEnabled == 0 then return end

    local hasChanges = false
    for key, _ in pairs(settings.defaults) do
        if key ~= 'saveChangesEnabled' then
            local currentValue = electrics.values[key]
            if currentValue ~= nil and settings.lastKnownValues[key] ~= currentValue then
                local oldValue = settings.lastKnownValues[key]
                local actualChange = false
                if type(currentValue) == 'number' and type(oldValue) == 'number' then
                    actualChange = math.abs(currentValue - oldValue) > 0.001
                else
                    actualChange = (currentValue ~= oldValue)
                end

                if actualChange then
                    settings.current[key] = currentValue
                    settings.lastKnownValues[key] = currentValue
                    hasChanges = true
                end
            end
        end
    end

    if hasChanges then
        saveSettings()
    end
end

local function initialize()
    if settings.initialized then return end
    ensureDirectoryExists()
    loadSettings()
    if electrics and electrics.values then
        applySettingsToElectrics()
        settings.initialized = true
        settings.startTime = os.clock()
    end
end

local function onVehicleSpawned()
    if not settings.initialized then
        applySettingsToElectrics()
        settings.initialized = true
        settings.startTime = os.clock()
    end
end

local function updateGFX(dt)
    if not settings.canSave and settings.startTime > 0 then
        if os.clock() - settings.startTime > 1.0 then
            settings.canSave = true
        end
    end

    if settings.initialized then
        monitorElectricValues()
    elseif electrics and electrics.values then
        applySettingsToElectrics()
        settings.initialized = true
        settings.startTime = os.clock()
    end
end

local function onReset()
    if settings.initialized then
        applySettingsToElectrics()
    end
end

local function onExtensionLoaded()
    initialize()
end

local function onExtensionUnloaded()
    saveSaveChangesState()
    if settings.current.saveChangesEnabled == 1 then
        saveSettings()
    end
end

local function forceSave()
    if not settings.initialized then return end
    if settings.current.saveChangesEnabled == 0 then return end
    if electrics and electrics.values then
        local hasChanges = false
        for key, _ in pairs(settings.defaults) do
            if key ~= 'saveChangesEnabled' then
                local currentValue = electrics.values[key]
                if currentValue ~= nil and settings.current[key] ~= currentValue then
                    settings.current[key] = currentValue
                    settings.lastKnownValues[key] = currentValue
                    hasChanges = true
                end
            end
        end
        if hasChanges then
            saveSettings()
        end
    end
end

M.onExtensionLoaded = onExtensionLoaded
M.onExtensionUnloaded = onExtensionUnloaded
M.onVehicleSpawned = onVehicleSpawned
M.updateGFX = updateGFX
M.onReset = onReset
M.forceSave = forceSave

return M
