local M = {}

local current_sfx_sources = {}
local last_door_unlock_state = 0
local last_door_lock_state = 0
local last_door_toggle_lock_state = 0
local door_is_locked = false

local doorhandle_lights_timer = 0
local doorhandle_lights_duration = 30.0
local doorhandle_lights_active = false
local last_doorR_state = 0
local last_doorL_state = 0

local hazards_timer = 0
local hazards_active = false
local flash_count = 0
local flash_target = 0
local flash_on = false
local flash_interval = 0.5
local flash_speed_multiplier = 1

local CHIRP_NODE = "dsh3"
local LEFT_DOOR_NODE = "d15l"
local RIGHT_DOOR_NODE = "d15r"

local SOUNDS = {
  unlock = "/vehicles/RLA_A90/sounds/unlock.mp3",
  unlock_chirp = "/vehicles/RLA_A90/sounds/unlockchirp.mp3",
  lock = "/vehicles/RLA_A90/sounds/lock.mp3",
  lock_chirp = "/vehicles/RLA_A90/sounds/lockchirp.mp3"
}

local function resolveNodeIDs()
  local v = _G.v
  local nodeMap = {}
  if v and v.data and v.data.nodes then
    for _, node in pairs(v.data.nodes) do
      if node.name then
        nodeMap[node.name] = node.cid
      end
    end
  end
  return {
    chirp = nodeMap[CHIRP_NODE] or 0,
    left_door = nodeMap[LEFT_DOOR_NODE] or 0,
    right_door = nodeMap[RIGHT_DOOR_NODE] or 0
  }
end

local function sanitizeObjectName(path)
  return path:gsub("/", "_")
end

local function createAndPlaySound(sound_path, node_id, volume, pitch)
  if sound_path and obj then
    local safe_name = sanitizeObjectName(sound_path .. "_" .. node_id)
    local sfx_source = obj:createSFXSource(sound_path, "AudioDefault3D", safe_name, node_id)
    if sfx_source then
      obj:setVolumePitch(sfx_source, volume or 1.0, pitch or 1.0)
      obj:playSFX(sfx_source)
      table.insert(current_sfx_sources, sfx_source)
    end
  end
end

local function createAndPlayChirpSound(sound_path, node_id, volume, pitch)
  local lockUnlockSoundEnabled = electrics.values.lockUnlockSoundEnabled or 1
  if lockUnlockSoundEnabled == 0 then return end
  if sound_path and obj then
    local safe_name = sanitizeObjectName(sound_path .. "_" .. node_id)
    local sfx_source = obj:createSFXSource(sound_path, "AudioDefault3D", safe_name, node_id)
    if sfx_source then
      obj:setVolumePitch(sfx_source, volume or 1.0, pitch or 1.0)
      obj:playSFX(sfx_source)
      table.insert(current_sfx_sources, sfx_source)
    end
  end
end

local function cleanupSounds()
  for _, sfx in ipairs(current_sfx_sources) do
    if sfx then
      obj:stopSFX(sfx)
      obj:deleteSFXSource(sfx)
    end
  end
  current_sfx_sources = {}
end

local function activateLockSignals(flash_times, speed_multiplier)
  local flashLockUnlock = electrics.values.flashLockUnlock or 1
  if flashLockUnlock == 0 then return end
  hazards_active = true
  hazards_timer = 0
  flash_count = 0
  flash_target = flash_times
  flash_on = true
  flash_speed_multiplier = speed_multiplier or 1
  electrics.values.lock_signal_L = 1
  electrics.values.lock_signal_R = 1
end

local function deactivateLockSignals()
  hazards_active = false
  hazards_timer = 0
  flash_count = 0
  flash_on = false
  electrics.values.lock_signal_L = 0
  electrics.values.lock_signal_R = 0
end

local function activateDoorHandleLights()
  local doorHandleLightingEnabled = electrics.values.doorHandleLighting or 1
  if doorHandleLightingEnabled == 1 then
    doorhandle_lights_active = true
    doorhandle_lights_timer = 0
    electrics.values.doorhandle_lights = 1
  else
    doorhandle_lights_active = false
    electrics.values.doorhandle_lights = 0
  end
end

local function deactivateDoorHandleLights()
  doorhandle_lights_active = false
  doorhandle_lights_timer = 0
  electrics.values.doorhandle_lights = 0
end

local function playUnlockSounds()
  local nodes = resolveNodeIDs()
  createAndPlayChirpSound(SOUNDS.unlock_chirp, nodes.chirp, 1.0, 1.0)
  createAndPlaySound(SOUNDS.unlock, nodes.left_door, 1.0, 1.0)
  createAndPlaySound(SOUNDS.unlock, nodes.right_door, 1.0, 1.0)
  activateDoorHandleLights()
  activateLockSignals(2, 2)
end

local function playLockSounds()
  local nodes = resolveNodeIDs()
  createAndPlayChirpSound(SOUNDS.lock_chirp, nodes.chirp, 1.0, 1.0)
  createAndPlaySound(SOUNDS.lock, nodes.left_door, 1.0, 1.0)
  createAndPlaySound(SOUNDS.lock, nodes.right_door, 1.0, 1.0)
  activateLockSignals(1)
end

local function updateGFX(dt)
  if not electrics or not electrics.values then return end

  if electrics.values.door_unlock == nil then electrics.values.door_unlock = 0 end
  if electrics.values.door_lock == nil then electrics.values.door_lock = 0 end
  if electrics.values.door_toggle_lock == nil then electrics.values.door_toggle_lock = 0 end
  if electrics.values.doorhandle_lights == nil then electrics.values.doorhandle_lights = 0 end
  if electrics.values.lock_signal_L == nil then electrics.values.lock_signal_L = 0 end
  if electrics.values.lock_signal_R == nil then electrics.values.lock_signal_R = 0 end
  if electrics.values.doorRCoupler_notAttached == nil then electrics.values.doorRCoupler_notAttached = 0 end
  if electrics.values.doorLCoupler_notAttached == nil then electrics.values.doorLCoupler_notAttached = 0 end
  if electrics.values.doorHandleLighting == nil then electrics.values.doorHandleLighting = 1 end
  if electrics.values.lockUnlockSoundEnabled == nil then electrics.values.lockUnlockSoundEnabled = 1 end
  if electrics.values.flashLockUnlock == nil then electrics.values.flashLockUnlock = 1 end

  local current_unlock_state = electrics.values.door_unlock
  local current_lock_state = electrics.values.door_lock
  local current_toggle_lock_state = electrics.values.door_toggle_lock
  local current_doorR_state = electrics.values.doorRCoupler_notAttached
  local current_doorL_state = electrics.values.doorLCoupler_notAttached

  if last_door_unlock_state == 0 and current_unlock_state == 1 then
    if door_is_locked then
      playUnlockSounds()
      door_is_locked = false
    end
  end

  if last_door_lock_state == 0 and current_lock_state == 1 then
    if not door_is_locked and current_doorR_state == 0 and current_doorL_state == 0 then
      playLockSounds()
      door_is_locked = true
    end
  end

  if last_door_toggle_lock_state == 0 and current_toggle_lock_state == 1 then
    if door_is_locked then
      playUnlockSounds()
      door_is_locked = false
    elseif current_doorR_state == 0 and current_doorL_state == 0 then
      playLockSounds()
      door_is_locked = true
    end
  end

  if doorhandle_lights_active then
    doorhandle_lights_timer = doorhandle_lights_timer + dt
    local doorHandleLightingEnabled = electrics.values.doorHandleLighting or 1
    if doorHandleLightingEnabled == 0 then
      deactivateDoorHandleLights()
    elseif doorhandle_lights_timer >= doorhandle_lights_duration then
      deactivateDoorHandleLights()
    end
  end

  if hazards_active then
    hazards_timer = hazards_timer + dt
    local current_interval = flash_interval / flash_speed_multiplier
    if hazards_timer >= current_interval then
      hazards_timer = 0
      if flash_on then
        electrics.values.lock_signal_L = 0
        electrics.values.lock_signal_R = 0
        flash_on = false
      else
        flash_count = flash_count + 1
        if flash_count < flash_target then
          electrics.values.lock_signal_L = 1
          electrics.values.lock_signal_R = 1
          flash_on = true
        else
          deactivateLockSignals()
        end
      end
    end
  end

  if doorhandle_lights_active then
    if last_doorR_state == 1 and current_doorR_state == 0 then
      deactivateDoorHandleLights()
    end
    if last_doorL_state == 1 and current_doorL_state == 0 then
      deactivateDoorHandleLights()
    end
  end

  last_door_unlock_state = current_unlock_state
  last_door_lock_state = current_lock_state
  last_door_toggle_lock_state = current_toggle_lock_state
  last_doorR_state = current_doorR_state
  last_doorL_state = current_doorL_state

  electrics.values.door_is_locked = door_is_locked and 1 or 0
end

local function onExtensionLoaded() end

local function reset()
  cleanupSounds()
  door_is_locked = false
  last_door_unlock_state = 0
  last_door_lock_state = 0
  last_door_toggle_lock_state = 0
  last_doorR_state = 0
  last_doorL_state = 0
  deactivateDoorHandleLights()
  deactivateLockSignals()
  if electrics and electrics.values then
    electrics.values.door_unlock = 0
    electrics.values.door_lock = 0
    electrics.values.door_toggle_lock = 0
    electrics.values.doorhandle_lights = 0
    electrics.values.lock_signal_L = 0
    electrics.values.lock_signal_R = 0
  end
end

local function onVehicleActiveChanged(active)
  if active then
    electrics.registerHandler("updateGFX", updateGFX)
  else
    cleanupSounds()
  end
end

local function destroy()
  cleanupSounds()
end

M.onVehicleActiveChanged = onVehicleActiveChanged
M.onReset = reset
M.updateGFX = updateGFX
M.onExtensionLoaded = onExtensionLoaded
M.destroy = destroy

return M
