local M = {}

local htmlTexture = nil
pcall(function() htmlTexture = require("htmlTexture") end)

local lastVolume = 1.0
M.volume = 1.0
M.volumeStep = 0.4
M.volumeMin = 0.0
M.volumeMax = 2.0
M.volUpPressed = false
M.volDownPressed = false
M.volButtonTimer = 0
M.volInitialDelay = 1.0
M.volRepeatDelay = 0.1
M.alertDurations = {xBand=10.0,kBand=10.0,kaBand=10.0,laser=10.0}
M.alertTimers = {xBand=0,kBand=0,kaBand=0,laser=0}
M.soundPlayed = {xBand=false,kBand=false,kaBand=false,laser=false}
M.bandAlertPlayed = {xBand=false,kBand=false,kaBand=false}
M.blinkTimer = 0
M.blinkInterval = 0.2
M.blinkState = false
M.lastSightValue = 0
M.isInitialized = false
M.beepTimers = {xBand=0,kBand=0,kaBand=0}
M.beepIntervals = {xBand=1.0,kBand=1.0,kaBand=1.0}
M.lastBeepTime = {xBand=0,kBand=0,kaBand=0}
M.activeBand = nil
M.lastMuteState = 0

local screenMaterialName = nil
local htmlFilePath = nil
local textureWidth = 256
local textureHeight = 128
local textureFPS = 15
local updateTimer = 0
local invFPS = 1 / 15
local textureCreated = false

local lastSentData = nil
local lastUnitData = nil

local function areTablesEqual(t1, t2)
  if type(t1)~="table"or type(t2)~="table"then return false end
  for k,v in pairs(t1)do if t2[k]~=v then return false end end
  for k,v in pairs(t2)do if t1[k]~=v then return false end end
  return true
end

local function getRandomBand()
  local rand=math.random()
  if rand<0.70 then return"kaBand"
  elseif rand<0.85 then return"kBand"
  else return"xBand"end
end

local function calculateBeepInterval(sightValue)
  local minInterval=0.1
  local maxInterval=0.7
  local normalizedValue=math.min(sightValue,1.0)
  return maxInterval-(maxInterval-minInterval)*normalizedValue
end

local function resolveNodeID()
  local v=_G.v
  local nodeMap={}
  if v and v.data and v.data.nodes then
    for _,n in pairs(v.data.nodes)do
      if n.name then nodeMap[n.name]=n.cid end
    end
  end
  return nodeMap["rf1"]or-1
end

local function sanitizeObjectName(path)
  return path:gsub("/","_")
end

local function playBeepSound(band)
  if not obj then return end
  if electrics.values.radar_mute==1 then return end
  local soundPath=""
  if band=="xBand"then soundPath="vehicles/RLA_A90/sounds/xbandbeep.mp3"
  elseif band=="kBand"then soundPath="vehicles/RLA_A90/sounds/kbandbeep.mp3"
  elseif band=="kaBand"then soundPath="vehicles/RLA_A90/sounds/kabandbeep.mp3"end
  if soundPath~=""then
    pcall(function()
      local safe_name=sanitizeObjectName(soundPath)
      local node=resolveNodeID()
      obj:createSFXSource(soundPath,"AudioClosest3D",safe_name,node)
      obj:playSFXOnce(safe_name,node,M.volume,1)
    end)
  end
end

local function playBandAlertSound(band)
  if not obj then return end
  if electrics.values.radar_mute==1 then return end
  local soundPath=""
  if band=="xBand"then soundPath="vehicles/RLA_A90/sounds/xband_alert.mp3"
  elseif band=="kBand"then soundPath="vehicles/RLA_A90/sounds/kband_alert.mp3"
  elseif band=="kaBand"then soundPath="vehicles/RLA_A90/sounds/kaband_alert.mp3"end
  if soundPath~=""then
    pcall(function()
      local safe_name=sanitizeObjectName(soundPath)
      local node=resolveNodeID()
      obj:createSFXSource(soundPath,"AudioClosest3D",safe_name,node)
      obj:playSFXOnce(safe_name,node,M.volume,1)
    end)
  end
end

local function updateHTML(dt)
  if not htmlTexture or not screenMaterialName or not textureCreated then 
    return 
  end
  
  updateTimer = updateTimer + dt
  if updateTimer > invFPS then
    updateTimer = 0
    
    pcall(function()
      htmlTexture.call(screenMaterialName, "checkFunctions")
    end)
    
    local currentUnits = {
      uiUnitLength = settings.getValue('uiUnitLength') or 'metric'
    }
    
    local radarData = {
      electrics = {
        wheelspeed = electrics.values.wheelspeed or 0,
        xBand = electrics.values.xBand or 0,
        kBand = electrics.values.kBand or 0,
        kaBand = electrics.values.kaBand or 0,
        laser = electrics.values.laser or 0,
        radarSightValue = electrics.values.radarSightValue or 0,
        radar_mute = electrics.values.radar_mute or 0
      },
      units = currentUnits
    }
    
    local electricsChanged = not areTablesEqual(lastSentData, radarData.electrics)
    local unitsChanged = not areTablesEqual(lastUnitData, currentUnits)
    
    if electricsChanged or unitsChanged then
      local success = pcall(function()
        htmlTexture.call(screenMaterialName, "updateDisplay", radarData)
      end)
      
      if success then 
        lastSentData = radarData.electrics
        lastUnitData = currentUnits
      end
    end
  end
end

function M.init(jbeamData)
  electrics.values.radarSightValue = 0
  electrics.values.xBand = 0
  electrics.values.kBand = 0
  electrics.values.kaBand = 0
  electrics.values.laser = 0
  if electrics.values.radar_mute == nil then electrics.values.radar_mute = 0 end
  M.lastMuteState = electrics.values.radar_mute
  
  for band, _ in pairs(M.alertTimers) do
    M.alertTimers[band] = 0
    M.soundPlayed[band] = false
    M.bandAlertPlayed[band] = false
    M.beepTimers[band] = 0
    M.beepIntervals[band] = 1.0
    M.lastBeepTime[band] = 0
  end
  
  M.volume = 1.0
  M.volumeStep = 0.4
  M.volumeMin = 0.0
  M.volumeMax = 2.0
  M.volUpPressed = false
  M.volDownPressed = false
  M.volButtonTimer = 0
  M.volInitialDelay = 1.0
  M.volRepeatDelay = 0.1
  M.lastSightValue = 0
  M.blinkTimer = 0
  M.blinkState = false
  M.isInitialized = false
  M.activeBand = nil
  
  if not htmlTexture then 
    return 
  end
  
  screenMaterialName = jbeamData.materialName or "@RLA_A90_radarscreen"
  htmlFilePath = jbeamData.htmlPath or "local://local/vehicles/RLA_A90/Screen/radar_screen.html"
  textureWidth = jbeamData.textureWidth or 256
  textureHeight = jbeamData.textureHeight or 128
  textureFPS = jbeamData.textureFPS or 60
  
  if not screenMaterialName or not htmlFilePath then 
    return 
  end
  
  pcall(function() htmlTexture.destroy(screenMaterialName) end)
  
  local success = pcall(function() 
    htmlTexture.create(screenMaterialName, htmlFilePath, textureWidth, textureHeight, textureFPS, "automatic") 
  end)
  
  textureCreated = success
  
  if success then
    M.setupPending = true
    M.setupAttempts = 0
    M.maxSetupAttempts = 10
  end
  
  M.isInitialized = success
end

function M.updateGFX(dt)
  if not M.isInitialized then return end
  
  if M.setupPending and M.setupAttempts < M.maxSetupAttempts then
    M.setupAttempts = M.setupAttempts + 1
    local setupData = {
      uiUnitLength = settings.getValue('uiUnitLength') or 'metric'
    }
    local setupSuccess = pcall(function() 
      htmlTexture.call(screenMaterialName, "setup", setupData) 
    end)
    
    if setupSuccess then
      M.setupPending = false
    else
      if M.setupAttempts >= M.maxSetupAttempts then
        M.setupPending = false
      end
    end
  end
  
  M.updateVolumeControl(dt)
  
  local sightValue = electrics.values.radarSightValue or 0
  electrics.values.xBand = 0
  electrics.values.kBand = 0
  electrics.values.kaBand = 0
  electrics.values.laser = 0
  
  if sightValue > 0 then
    if not M.activeBand or M.lastSightValue == 0 then
      M.activeBand = getRandomBand()
      if not M.bandAlertPlayed[M.activeBand] then
        playBandAlertSound(M.activeBand)
        M.bandAlertPlayed[M.activeBand] = true
      end
    end
    
    if M.activeBand then
      electrics.values[M.activeBand] = 1
    end
    
    if sightValue >= 1.0 and M.lastSightValue < 1.0 then
      M.playAlertSound("laser")
      M.soundPlayed.laser = true
      M.alertTimers.laser = M.alertDurations.laser
      electrics.values.laser = 1
    elseif sightValue >= 1.0 then
      electrics.values.laser = 1
    end
    
    if M.activeBand then
      local beepInterval = calculateBeepInterval(sightValue)
      M.beepIntervals[M.activeBand] = beepInterval
      M.beepTimers[M.activeBand] = M.beepTimers[M.activeBand] + dt
      if M.beepTimers[M.activeBand] >= M.beepIntervals[M.activeBand] then
        M.beepTimers[M.activeBand] = 0
        playBeepSound(M.activeBand)
      end
    end
  else
    if M.activeBand then
      M.activeBand = nil
      for band, _ in pairs(M.beepTimers) do
        M.beepTimers[band] = 0
        M.bandAlertPlayed[band] = false
      end
    end
  end
  
  M.lastSightValue = sightValue
  
  if electrics.values.radar_mute ~= M.lastMuteState then
    lastSentData = nil
    M.lastMuteState = electrics.values.radar_mute
  end
  
  updateHTML(dt)
end

function M.updateVolumeControl(dt)
  local volUpState = electrics.values.radar_volup == 1
  if volUpState then
    if not M.volUpPressed then
      M.volUpPressed = true
      M.volButtonTimer = 0
      M.volume = math.min(M.volume + M.volumeStep, M.volumeMax)
      lastVolume = M.volume
    else
      M.volButtonTimer = M.volButtonTimer + dt
      if M.volButtonTimer >= M.volInitialDelay then
        M.volume = math.min(M.volume + M.volumeStep, M.volumeMax)
        lastVolume = M.volume
        M.volButtonTimer = M.volInitialDelay - M.volRepeatDelay
      end
    end
  else
    M.volUpPressed = false
  end
  
  local volDownState = electrics.values.radar_voldown == 1
  if volDownState then
    if not M.volDownPressed then
      M.volDownPressed = true
      M.volButtonTimer = 0
      M.volume = math.max(M.volume - M.volumeStep, M.volumeMin)
      lastVolume = M.volume
    else
      M.volButtonTimer = M.volButtonTimer + dt
      if M.volButtonTimer >= M.volInitialDelay then
        M.volume = math.max(M.volume - M.volumeStep, M.volumeMin)
        lastVolume = M.volume
        M.volButtonTimer = M.volInitialDelay - M.volRepeatDelay
      end
    end
  else
    M.volDownPressed = false
  end
end

function M.playAlertSound(band)
  if not obj then return end
  if electrics.values.radar_mute == 1 then return end
  
  local soundPath = ""
  if band == "laser" then soundPath = "vehicles/RLA_A90/sounds/laser_alert.mp3" end
  
  if soundPath ~= "" then
    pcall(function()
      local safe_name = sanitizeObjectName(soundPath)
      local node = resolveNodeID()
      obj:createSFXSource(soundPath, "AudioClosest3D", safe_name, node)
      obj:playSFXOnce(safe_name, node, M.volume, 1)
    end)
  end
end

function M.reset()
  electrics.values.radarSightValue = 0
  electrics.values.xBand = 0
  electrics.values.kBand = 0
  electrics.values.kaBand = 0
  electrics.values.laser = 0
  M.lastMuteState = electrics.values.radar_mute or 0
  
  for band, _ in pairs(M.alertTimers) do
    M.alertTimers[band] = 0
    M.soundPlayed[band] = false
    M.bandAlertPlayed[band] = false
    M.beepTimers[band] = 0
    M.beepIntervals[band] = 1.0
    M.lastBeepTime[band] = 0
  end
  
  M.blinkTimer = 0
  M.blinkState = false
  M.lastSightValue = 0
  M.activeBand = nil
  
  updateTimer = 0
  lastSentData = nil
  lastUnitData = nil
  
  if htmlTexture and screenMaterialName and textureCreated then
    pcall(function() 
      htmlTexture.call(screenMaterialName, "reset") 
    end)
  end
end

function M.destroy()
  if htmlTexture and type(htmlTexture.destroy) == "function" and screenMaterialName then
    pcall(function() 
      htmlTexture.destroy(screenMaterialName) 
    end)
  end
end

return M