local M = {}

local soundManager = require('baggedSuspension/soundManager')

local positions = {
    fl = {
        current = 0,
        default = 0.75
    },
    fr = {
        current = 0,
        default = 0.75
    },
    rl = {
        current = 0,
        default = 0.25
    },
    rr = {
        current = 0,
        default = 0.25
    }
}

local SPEEDS = {
    fl = { up = 0.3, down = 0.7 },
    fr = { up = 0.3, down = 0.7 },
    rl = { up = 0.3, down = 0.7 },
    rr = { up = 0.3, down = 0.7 }
}

local state = {
    defaultButtonPressed = false,
    movingToDefault = false
}

local function updateWheel(position, up, down, speeds, maxPosition, fillRateModifier, dt)
    if up > 0 or down > 0 then
        local baseSpeed = up > 0 and speeds.up or speeds.down
        local adjustedSpeed = up > 0 and (baseSpeed * fillRateModifier) or baseSpeed
        local delta = (up > 0 and adjustedSpeed or -adjustedSpeed) * dt
        
        if up > 0 then
            return math.min(maxPosition, math.max(-0.45, position + delta))
        else
            return math.min(1, math.max(-0.45, position + delta))
        end
    end
    return position
end

function M.update(dt, maxPosition, fillRateModifier)
    local fl_up = electrics.values.airbag_fl_up or 0
    local fl_down = electrics.values.airbag_fl_down or 0
    local fr_up = electrics.values.airbag_fr_up or 0
    local fr_down = electrics.values.airbag_fr_down or 0
    local rl_up = electrics.values.airbag_rl_up or 0
    local rl_down = electrics.values.airbag_rl_down or 0
    local rr_up = electrics.values.airbag_rr_up or 0
    local rr_down = electrics.values.airbag_rr_down or 0
    
    local front_up = electrics.values.airbag_F_up or 0
    local front_down = electrics.values.airbag_F_down or 0
    local rear_up = electrics.values.airbag_R_up or 0
    local rear_down = electrics.values.airbag_R_down or 0

    if (electrics.values.airbag_default_all or 0) > 0 then
        if not state.defaultButtonPressed then
            state.movingToDefault = not state.movingToDefault
            state.defaultButtonPressed = true
        end
    else
        state.defaultButtonPressed = false
    end

    if state.movingToDefault then
        local flAtDefault = math.abs(positions.fl.current - positions.fl.default) <= 0.01
        local frAtDefault = math.abs(positions.fr.current - positions.fr.default) <= 0.01
        local rlAtDefault = math.abs(positions.rl.current - positions.rl.default) <= 0.01
        local rrAtDefault = math.abs(positions.rr.current - positions.rr.default) <= 0.01
        
        if flAtDefault and frAtDefault and rlAtDefault and rrAtDefault then
            state.movingToDefault = false
        else
            if not flAtDefault then
                if positions.fl.current > positions.fl.default + 0.01 then
                    fl_down = 1
                    fl_up = 0
                elseif positions.fl.current < positions.fl.default - 0.01 then
                    fl_down = 0
                    fl_up = 1
                end
            end
            
            if not frAtDefault then
                if positions.fr.current > positions.fr.default + 0.01 then
                    fr_down = 1
                    fr_up = 0
                elseif positions.fr.current < positions.fr.default - 0.01 then
                    fr_down = 0
                    fr_up = 1
                end
            end
            
            if not rlAtDefault then
                if positions.rl.current > positions.rl.default + 0.01 then
                    rl_down = 1
                    rl_up = 0
                elseif positions.rl.current < positions.rl.default - 0.01 then
                    rl_down = 0
                    rl_up = 1
                end
            end
            
            if not rrAtDefault then
                if positions.rr.current > positions.rr.default + 0.01 then
                    rr_down = 1
                    rr_up = 0
                elseif positions.rr.current < positions.rr.default - 0.01 then
                    rr_down = 0
                    rr_up = 1
                end
            end
        end
    end

    local fl_at_max = positions.fl.current >= maxPosition
    local fr_at_max = positions.fr.current >= maxPosition
    local rl_at_max = positions.rl.current >= maxPosition
    local rr_at_max = positions.rr.current >= maxPosition

    local fl_at_min = positions.fl.current <= -0.45
    local fr_at_min = positions.fr.current <= -0.45
    local rl_at_min = positions.rl.current <= -0.3
    local rr_at_min = positions.rr.current <= -0.3

    if (fl_up > 0 and not fl_at_max) or (fr_up > 0 and not fr_at_max) or (front_up > 0 and not (fl_at_max and fr_at_max)) then
        soundManager.playFrontSound(soundManager.SOUNDS.up)
    elseif (fl_down > 0 and not fl_at_min) or (fr_down > 0 and not fr_at_min) or (front_down > 0 and not (fl_at_min and fr_at_min)) then
        soundManager.playFrontSound(soundManager.SOUNDS.down)
    else
        soundManager.stopFrontSound()
    end

    if (rl_up > 0 and not rl_at_max) or (rr_up > 0 and not rr_at_max) or (rear_up > 0 and not (rl_at_max and rr_at_max)) then
        soundManager.playRearSound(soundManager.SOUNDS.up)
    elseif (rl_down > 0 and not rl_at_min) or (rr_down > 0 and not rr_at_min) or (rear_down > 0 and not (rl_at_min and rr_at_min)) then
        soundManager.playRearSound(soundManager.SOUNDS.down)
    else
        soundManager.stopRearSound()
    end

    if fl_up > 0 or fl_down > 0 or front_up > 0 or front_down > 0 then
        local combined_up = math.max(fl_up, front_up)
        local combined_down = math.max(fl_down, front_down)
        positions.fl.current = updateWheel(positions.fl.current, combined_up, combined_down, SPEEDS.fl, maxPosition, fillRateModifier, dt)
    end

    if fr_up > 0 or fr_down > 0 or front_up > 0 or front_down > 0 then
        local combined_up = math.max(fr_up, front_up)
        local combined_down = math.max(fr_down, front_down)
        positions.fr.current = updateWheel(positions.fr.current, combined_up, combined_down, SPEEDS.fr, maxPosition, fillRateModifier, dt)
    end

    if rl_up > 0 or rl_down > 0 or rear_up > 0 or rear_down > 0 then
        local combined_up = math.max(rl_up, rear_up)
        local combined_down = math.max(rl_down, rear_down)
        positions.rl.current = updateWheel(positions.rl.current, combined_up, combined_down, SPEEDS.rl, maxPosition, fillRateModifier, dt)
    end

    if rr_up > 0 or rr_down > 0 or rear_up > 0 or rear_down > 0 then
        local combined_up = math.max(rr_up, rear_up)
        local combined_down = math.max(rr_down, rear_down)
        positions.rr.current = updateWheel(positions.rr.current, combined_up, combined_down, SPEEDS.rr, maxPosition, fillRateModifier, dt)
    end

    electrics.values.airbag_fl = positions.fl.current
    electrics.values.airbag_fr = positions.fr.current
    electrics.values.airbag_rl = positions.rl.current
    electrics.values.airbag_rr = positions.rr.current
    
    return front_up > 0 or rear_up > 0 or fl_up > 0 or fr_up > 0 or rl_up > 0 or rr_up > 0
end

function M.reset()
    for k, v in pairs(positions) do
        v.current = v.default
        electrics.values["airbag_" .. k] = v.default
    end
    state.defaultButtonPressed = false
    state.movingToDefault = false
end

function M.getPositions()
    return positions
end

return M