angular.module('infotainmentScreen', [])
.controller('InfotainmentController', function($scope, $window) {
  $scope.allOptions = ["Media/Radio", "Communication", "Navigation", "Connected Serv.", "My Vehicle", "Notifications"];

  $scope.selectedIndex = 0;
  $scope.currentPage = 0;
  $scope.currentMenu = 'main';
  $scope.isZoomedOut = false; // New property for zoom state

  var units = {
    uiUnitConsumptionRate: "metric",
    uiUnitDate: "ger",
    uiUnitEnergy: "metric",
    uiUnitLength: "metric",
    uiUnitPower: "hp",
    uiUnitPressure: "bar",
    uiUnitTemperature: "c",
    uiUnitTorque: "metric",
    uiUnitVolume: "l",
    uiUnitWeight: "kg"
  };

  $scope.currentDate = {};
  $scope.time = '';
  
  $scope.vehicleData = {
    fuelConsumption: '--.-',
    fuelUnit: 'l/100km',
    avgSpeed: '0',
    speedUnit: 'km/h'
  };

  let speedHistory = [];
  let speedHistoryLimit = 100;
  let speedSum = 0;
  let lastFuelReading = null;
  let lastSpeedTime = null;
  let accumulatedDistance = 0;
  let fuelConsumptionSamples = [];
  let sampleLimit = 10;
  let updateCounter = 0;

  // Continuous movement for color picker
  let continuousMovement = {
    up: false,
    down: false,
    left: false,
    right: false,
    interval: null
  };

  function startContinuousMovement() {
    if (continuousMovement.interval) return;
    
    continuousMovement.interval = setInterval(function() {
      $scope.$evalAsync(() => {
        if ($scope.currentMenu === 'daytimeRunningLights') {
          if (continuousMovement.up) $scope.$broadcast('daytime-lights-navigate-up');
          if (continuousMovement.down) $scope.$broadcast('daytime-lights-navigate-down');
          if (continuousMovement.left) $scope.$broadcast('daytime-lights-navigate-left');
          if (continuousMovement.right) $scope.$broadcast('daytime-lights-navigate-right');
        } else if ($scope.currentMenu === 'footwellLighting') {
          if (continuousMovement.up) $scope.$broadcast('footwell-lights-navigate-up');
          if (continuousMovement.down) $scope.$broadcast('footwell-lights-navigate-down');
          if (continuousMovement.left) $scope.$broadcast('footwell-lights-navigate-left');
          if (continuousMovement.right) $scope.$broadcast('footwell-lights-navigate-right');
        }
      });
    }, 50); // Movement every 50ms while held
  }

  function stopContinuousMovement() {
    if (continuousMovement.interval) {
      clearInterval(continuousMovement.interval);
      continuousMovement.interval = null;
    }
    continuousMovement.up = false;
    continuousMovement.down = false;
    continuousMovement.left = false;
    continuousMovement.right = false;
  }

  // Function to propagate electric data to all sub-controllers
  function propagateElectricDataToControllers(data) {
    // Get all controller scopes and call their updateElectricData function if it exists
    var allScopes = [];
    
    // Traverse the scope hierarchy to find child controllers
    function findChildScopes(scope) {
      if (scope.$$childHead) {
        var child = scope.$$childHead;
        do {
          allScopes.push(child);
          findChildScopes(child);
          child = child.$$nextSibling;
        } while (child);
      }
    }
    
    findChildScopes($scope);
    
    // Call updateElectricData on each scope that has it
    allScopes.forEach(function(childScope) {
      if (typeof childScope.updateElectricData === 'function') {
        childScope.updateElectricData(data);
      }
    });
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('Main controller received settings:', settings);
    // Settings are handled by individual sub-controllers
  });

  // Global function to receive settings from Lua
  window.loadInfotainmentSettings = function(settings) {
    console.log('Received settings from Lua:', settings);
    
    // Apply settings to all controllers via broadcast
    $scope.$evalAsync(function() {
      $scope.$broadcast('settings-loaded', settings);
    });
  };

  // Check for settings in electric values
  var lastSettingsCheck = '';
  function checkForSettingsInElectrics(data) {
    if (!data || !data.electrics) return;
    
    const e = data.electrics;
    
    // Debug logging
    if (e.infotainmentSettingsReady === 1) {
      console.log('Main: infotainmentSettingsReady detected!');
    }
    
    if (e.infotainmentSettingsJson) {
      console.log('Main: infotainmentSettingsJson found:', e.infotainmentSettingsJson.substring(0, 100) + '...');
    }
    
    if (e.infotainmentSettingsReady === 1 && e.infotainmentSettingsJson) {
      const settingsJson = e.infotainmentSettingsJson;
      if (settingsJson !== lastSettingsCheck) {
        lastSettingsCheck = settingsJson;
        try {
          const settings = JSON.parse(settingsJson);
          console.log('Main: Found settings in electrics, broadcasting:', settings);
          $scope.$broadcast('settings-loaded', settings);
        } catch (error) {
          console.error('Main: Failed to parse settings from electrics:', error);
        }
      }
    }
  }

  // Event listener for UI messages from Lua
  var originalOnUiMessage = window.onUiMessage;
  window.onUiMessage = function(type, data) {
    if (type === 'INFOTAINMENT_LOAD_SETTINGS') {
      try {
        var settings = JSON.parse(data);
        console.log('Received settings via ui_message:', settings);
        window.loadInfotainmentSettings(settings);
      } catch (e) {
        console.error('Failed to parse infotainment settings from ui_message:', e);
      }
    } else if (originalOnUiMessage) {
      originalOnUiMessage(type, data);
    }
  };

  $window.setup = (setupData) => {
    for(let dk in setupData){
      if(typeof dk == "string" && dk.startsWith("uiUnit")){
        units[dk] = setupData[dk];
      }
    }

    $scope.vehicleData.speedUnit = units.uiUnitLength == "metric" ? "km/h" : "mph";
    $scope.vehicleData.fuelUnit = units.uiUnitConsumptionRate == "metric" ? "l/100km" : "mpg";
  }
  
  function updateDate() {
    const now = new Date();
    const months = ['January', 'February', 'March', 'April', 'May', 'June',
                   'July', 'August', 'September', 'October', 'November', 'December'];
    
    $scope.currentDate = {
      day: now.getDate(),
      month: months[now.getMonth()],
      year: now.getFullYear()
    };
  }

  function updateTime() {
    const now = new Date();
    let hours = now.getHours();
    const minutes = now.getMinutes();
    const ampm = hours >= 12 ? 'pm' : 'am';
    
    hours = hours % 12;
    hours = hours ? hours : 12;
    
    const minutesStr = minutes < 10 ? '0' + minutes : minutes;
    
    $scope.time = hours + ':' + minutesStr + ' ' + ampm;
  }
  
  function updateVehicleData(data) {
    if (!data || !data.electrics) return;

    const e = data.electrics;
    const speed = e.airspeed || 0;
    const fuelVolume = e.fuelVolume || 0;
    const ignitionLevel = e.ignitionLevel || 0;

    const speedKmh = speed * 3.6;
    const speedMph = speed * 2.237;
    updateCounter++;

    const currentTime = Date.now() / 1000;
    if (lastSpeedTime !== null && ignitionLevel >= 1 && speedKmh > 5) {
      const timeDelta = currentTime - lastSpeedTime;
      const distanceThisUpdate = (speedKmh / 3600) * timeDelta;
      accumulatedDistance += distanceThisUpdate;
    }
    lastSpeedTime = currentTime;

    if (speedKmh > 1 && ignitionLevel >= 1) {
      speedHistory.push(speedKmh);
      speedSum += speedKmh;
      if (speedHistory.length > speedHistoryLimit) {
        speedSum -= speedHistory.shift();
      }
    }

    if (ignitionLevel >= 1 && speedKmh > 10) {
      if (lastFuelReading !== null && accumulatedDistance > 0.1) {
        const fuelDelta = lastFuelReading - fuelVolume;
        const distanceDelta = accumulatedDistance;
        
        if (fuelDelta > 0.005) {
          let consumption;
          if (units.uiUnitConsumptionRate == "metric") {
            consumption = (fuelDelta / distanceDelta) * 100;
          } else {
            const miles = distanceDelta / 1.609344;
            const gallons = fuelDelta / 3.78541;
            consumption = miles / gallons;
          }
          
          const minConsumption = units.uiUnitConsumptionRate == "metric" ? 1 : 8;
          const maxConsumption = units.uiUnitConsumptionRate == "metric" ? 30 : 60;
          
          if (consumption >= minConsumption && consumption <= maxConsumption) {
            fuelConsumptionSamples.push(consumption);
            if (fuelConsumptionSamples.length > sampleLimit) {
              fuelConsumptionSamples.shift();
            }
            const avgConsumption = fuelConsumptionSamples.reduce((a, b) => a + b, 0) / fuelConsumptionSamples.length;
            $scope.vehicleData.fuelConsumption = avgConsumption.toFixed(1);
          }
          
          lastFuelReading = fuelVolume;
          accumulatedDistance = 0;
        }
      } else if (lastFuelReading === null) {
        lastFuelReading = fuelVolume;
        accumulatedDistance = 0;
      }
    }

    const avgSpeedKmh = speedHistory.length > 0 ? speedSum / speedHistory.length : 0;
    const avgSpeedMph = avgSpeedKmh / 1.609;

    $scope.vehicleData.avgSpeed = (units.uiUnitLength == "metric") ?
      Math.round(avgSpeedKmh).toString() :
      Math.round(avgSpeedMph).toString();

    $scope.vehicleData.speedUnit = units.uiUnitLength == "metric" ? 'km/h' : 'mph';
    $scope.vehicleData.fuelUnit = units.uiUnitConsumptionRate == "metric" ? 'l/100km' : 'mpg';
  }

  updateDate();
  updateTime();
  setInterval(function() {
    $scope.$apply(function() {
      updateDate();
      updateTime();
    });
  }, 60000);

  let lastInput = {
    left: 0,
    right: 0,
    option: 0,
    menu: 0,
    back: 0,
    up: 0,
    down: 0
  };

  let buttonPressId = 0;
  let lastProcessedPressIds = {
    option: 0,
    menu: 0,
    back: 0,
    up: 0,
    down: 0,
    left: 0,
    right: 0
  };

  function shouldProcessButton(buttonName, currentState, lastState) {
    if (currentState === 1 && lastState === 0) {
      buttonPressId++;
      return buttonPressId;
    }
    return false;
  }

  $window.updateData = function(data) {
    if (!data || !data.electrics) return;
    
    if (data.units) {
      for(let key in data.units) {
        units[key] = data.units[key];
      }
      $scope.vehicleData.speedUnit = units.uiUnitLength == "metric" ? "km/h" : "mph";
      $scope.vehicleData.fuelUnit = units.uiUnitConsumptionRate == "metric" ? "l/100km" : "mpg";
    }
    
    const e = data.electrics;
    updateVehicleData(data);
    
    const currentMenuAtStart = $scope.currentMenu;

    $scope.$evalAsync(() => {
      // Check for settings from Lua and broadcast to all controllers
      checkForSettingsInElectrics(data);
      
      // Propagate electric data to all sub-controllers for real-time updates
      // THIS IS THE CRITICAL FIX - moved inside $evalAsync
      propagateElectricDataToControllers(data);

      if (currentMenuAtStart === 'main') {
        if (e.radio_left === 1 && lastInput.left === 0) {
          if ($scope.isZoomedOut) {
            // In zoomed out mode, navigate through all 6 items in a line
            if ($scope.selectedIndex > 0) {
              $scope.selectedIndex--;
            }
          } else {
            // Original paged navigation
            if ($scope.currentPage === 0) {
              if ($scope.selectedIndex > 0) {
                $scope.selectedIndex--;
              }
            } else {
              if ($scope.selectedIndex > 0) {
                $scope.selectedIndex--;
              } else {
                $scope.currentPage = 0;
                $scope.selectedIndex = 2;
              }
            }
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          if ($scope.isZoomedOut) {
            // In zoomed out mode, navigate through all 6 items in a line
            if ($scope.selectedIndex < 5) {
              $scope.selectedIndex++;
            }
          } else {
            // Original paged navigation
            if ($scope.currentPage === 0) {
              if ($scope.selectedIndex < 2) {
                $scope.selectedIndex++;
              } else {
                $scope.currentPage = 1;
                $scope.selectedIndex = 0;
              }
            } else {
              if ($scope.selectedIndex < 2) {
                $scope.selectedIndex++;
              }
            }
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            let actualIndex;
            
            if ($scope.isZoomedOut) {
              actualIndex = $scope.selectedIndex;
            } else {
              actualIndex = $scope.currentPage === 0 ? $scope.selectedIndex : $scope.selectedIndex + 3;
            }
            
            if (actualIndex === 4) {
              $scope.currentMenu = 'vehicle';
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'vehicle') {
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'vehicle') {
            $scope.$broadcast('vehicle-navigate-up');
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'vehicle') {
            $scope.$broadcast('vehicle-navigate-down');
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'vehicle') {
              $scope.$broadcast('vehicle-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'vehicleSettings') {
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'vehicleSettings') {
            $scope.$broadcast('settings-navigate-up');
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'vehicleSettings') {
            $scope.$broadcast('settings-navigate-down');
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'vehicleSettings') {
              $scope.$broadcast('settings-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'systemSettings') {
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'systemSettings') {
            $scope.$broadcast('system-settings-navigate-up');
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'systemSettings') {
            $scope.$broadcast('system-settings-navigate-down');
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'systemSettings') {
              $scope.$broadcast('system-settings-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'exteriorLighting') {
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'exteriorLighting') {
            $scope.$broadcast('lighting-navigate-up');
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'exteriorLighting') {
            $scope.$broadcast('lighting-navigate-down');
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'exteriorLighting') {
              $scope.$broadcast('lighting-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'daytimeRunningLights') {
        // Handle continuous movement for color picker
        if (e.radio_up === 1 && lastInput.up === 0) {
          $scope.$broadcast('daytime-lights-navigate-up');
          continuousMovement.up = true;
          startContinuousMovement();
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          $scope.$broadcast('daytime-lights-navigate-down');
          continuousMovement.down = true;
          startContinuousMovement();
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          $scope.$broadcast('daytime-lights-navigate-left');
          continuousMovement.left = true;
          startContinuousMovement();
        } else if (e.radio_left === 0 && lastInput.left === 1) {
          continuousMovement.left = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          $scope.$broadcast('daytime-lights-navigate-right');
          continuousMovement.right = true;
          startContinuousMovement();
        } else if (e.radio_right === 0 && lastInput.right === 1) {
          continuousMovement.right = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.left) {
            stopContinuousMovement();
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'daytimeRunningLights') {
              $scope.$broadcast('daytime-lights-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'footwellLighting') {
        // Handle continuous movement for color picker
        if (e.radio_up === 1 && lastInput.up === 0) {
          $scope.$broadcast('footwell-lights-navigate-up');
          continuousMovement.up = true;
          startContinuousMovement();
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          $scope.$broadcast('footwell-lights-navigate-down');
          continuousMovement.down = true;
          startContinuousMovement();
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          $scope.$broadcast('footwell-lights-navigate-left');
          continuousMovement.left = true;
          startContinuousMovement();
        } else if (e.radio_left === 0 && lastInput.left === 1) {
          continuousMovement.left = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          $scope.$broadcast('footwell-lights-navigate-right');
          continuousMovement.right = true;
          startContinuousMovement();
        } else if (e.radio_right === 0 && lastInput.right === 1) {
          continuousMovement.right = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.left) {
            stopContinuousMovement();
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'footwellLighting') {
              $scope.$broadcast('footwell-lights-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'doorsKey') {
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'doorsKey') {
            $scope.$broadcast('doors-key-navigate-up');
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'doorsKey') {
            $scope.$broadcast('doors-key-navigate-down');
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'doorsKey') {
              $scope.$broadcast('doors-key-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;
      }

      // Handle menu button press (only for zoom in main menu and direct navigation to main)
      if (e.radio_menu === 1 && lastInput.menu === 0) {
        const pressId = shouldProcessButton('menu', e.radio_menu, lastInput.menu);
        if (pressId && pressId !== lastProcessedPressIds.menu) {
          lastProcessedPressIds.menu = pressId;
          
          if ($scope.currentMenu === 'main') {
            // Toggle zoom state in main menu
            $scope.isZoomedOut = !$scope.isZoomedOut;
            
            // When transitioning to zoomed out, adjust selectedIndex based on current page/position
            if ($scope.isZoomedOut) {
              $scope.selectedIndex = $scope.currentPage === 0 ? $scope.selectedIndex : $scope.selectedIndex + 3;
            } else {
              // When zooming back in, set page and index based on selectedIndex
              if ($scope.selectedIndex < 3) {
                $scope.currentPage = 0;
              } else {
                $scope.currentPage = 1;
                $scope.selectedIndex = $scope.selectedIndex - 3;
              }
            }
          } else {
            // From any other menu, go directly to main menu
            $scope.currentMenu = 'main';
            $scope.isZoomedOut = false; // Reset zoom when returning to main
          }
        }
      }
      lastInput.menu = e.radio_menu;

      // Handle back button press (go back one level)
      if (e.radio_back === 1 && lastInput.back === 0) {
        const pressId = shouldProcessButton('back', e.radio_back, lastInput.back);
        if (pressId && pressId !== lastProcessedPressIds.back) {
          lastProcessedPressIds.back = pressId;
          
          if ($scope.currentMenu === 'doorsKey') {
            $scope.currentMenu = 'vehicleSettings';
          } else if ($scope.currentMenu === 'exteriorLighting') {
            $scope.currentMenu = 'vehicleSettings';
          } else if ($scope.currentMenu === 'daytimeRunningLights') {
            // Let the controller handle the back button completely
            $scope.$broadcast('daytime-lights-back');
          } else if ($scope.currentMenu === 'footwellLighting') {
            // Let the controller handle the back button completely
            $scope.$broadcast('footwell-lights-back');
          } else if ($scope.currentMenu === 'systemSettings') {
            $scope.currentMenu = 'vehicle';
          } else if ($scope.currentMenu === 'vehicleSettings') {
            $scope.currentMenu = 'vehicle';
          } else if ($scope.currentMenu === 'vehicle') {
            $scope.currentMenu = 'main';
          } else if ($scope.currentMenu === 'main' && $scope.isZoomedOut) {
            // If in main menu and zoomed out, zoom back in instead of going back
            $scope.isZoomedOut = false;
            // When zooming back in, set page and index based on selectedIndex
            if ($scope.selectedIndex < 3) {
              $scope.currentPage = 0;
            } else {
              $scope.currentPage = 1;
              $scope.selectedIndex = $scope.selectedIndex - 3;
            }
          }
        }
      }
      lastInput.back = e.radio_back;
    });
  };
});