angular.module('infotainmentScreen')
.controller('FootwellLightingController', function($scope) {
  // Options with electric keys - checked state will be determined by electric values
  var defaultOptions = [
    { name: "Enabled", disabled: false, hasCheckbox: true, electricKey: "footwellLights", checked: true },
    { name: "Color Selector", disabled: false}
  ];

  $scope.footwellLightingOptions = angular.copy(defaultOptions);
  $scope.selectedFootwellLightsIndex = 0;
  $scope.footwellLightsScrollOffset = 0;
  $scope.colorPickerMode = false;
  $scope.colorCursor = { x: 50, y: 0 }; // Start at top middle (white)

  // Current electric values cache - initialize with defaults
  var currentElectricValues = {
    footwellLights: 1, // Default to enabled
    footwellLightColorR: 255, // Default to white
    footwellLightColorG: 255,
    footwellLightColorB: 255
  };

  // Initialize default state only if values aren't already set
  function initializeDefaults() {
    // Check if values are already set by persistence system
    const hasPersistedValues = (
      typeof currentElectricValues.footwellLights !== 'undefined' ||
      typeof currentElectricValues.footwellLightColorR !== 'undefined'
    );
    
    if (hasPersistedValues) {
      log('FootwellLights: Values already loaded from persistence, skipping defaults');
      updateCheckboxStates();
      return;
    }
    
    log('FootwellLights: No persisted values found, applying defaults');
    
    // Set default checkbox state
    updateCheckboxStates();
    
    // Initialize default color values immediately
    setElectricValues(
      currentElectricValues.footwellLightColorR,
      currentElectricValues.footwellLightColorG, 
      currentElectricValues.footwellLightColorB
    );
    
    // Set default enabled state
    writeElectricValue('footwellLights', currentElectricValues.footwellLights);
    
    console.log('FootwellLights: Initialized with defaults - Enabled:', currentElectricValues.footwellLights, 
                'Color RGB:', currentElectricValues.footwellLightColorR, currentElectricValues.footwellLightColorG, currentElectricValues.footwellLightColorB);
  }

  // Don't call initialization immediately - wait for persistence
  // initializeDefaults();

  $scope.getCurrentFootwellLightsImage = function() {
    if ($scope.selectedFootwellLightsIndex >= 0 && $scope.selectedFootwellLightsIndex < $scope.footwellLightingOptions.length) {
      return $scope.footwellLightingOptions[$scope.selectedFootwellLightsIndex].image || 'default_lighting.png';
    }
    return 'default_lighting.png';
  };

  $scope.showColorPicker = function() {
    return $scope.selectedFootwellLightsIndex === 1;
  };

  $scope.shouldShowMenuSelection = function() {
    return !$scope.colorPickerMode;
  };

  $scope.getCursorClass = function() {
    return $scope.colorPickerMode ? 'active' : 'inactive';
  };

  // Function to trigger persistence save using electric trigger
  function triggerPersistenceSave() {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      // Use a timestamp to ensure the trigger value changes
      const timestamp = Date.now();
      window.beamng.sendActiveObjectLua(`
        if electrics and electrics.values then
          electrics.values.infotainmentSaveTrigger = ${timestamp}
        end
      `);
    }
  }

  // Simple function to write electric values
  function setElectricValues(r, g, b) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.footwellLightColorR = ${r}
          electrics.values.footwellLightColorG = ${g}
          electrics.values.footwellLightColorB = ${b}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
      
      // Also update our cache
      currentElectricValues.footwellLightColorR = r;
      currentElectricValues.footwellLightColorG = g;
      currentElectricValues.footwellLightColorB = b;
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    }
  }

  function writeElectricValue(key, value) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.${key} = ${value}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
      
      // Also update our cache
      currentElectricValues[key] = value;
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    } else if (window.bngApi && window.bngApi.engineLua) {
      const luaCommand = `
        local veh = be:getPlayerVehicle(0)
        if veh then
          veh:queueLuaCommand('if electrics and electrics.values then electrics.values.${key} = ${value} end')
        end
      `;
      
      window.bngApi.engineLua(luaCommand);
      
      // Also update our cache
      currentElectricValues[key] = value;
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    }
  }

  // Update checkbox states based on current electric values
  function updateCheckboxStates() {
    $scope.footwellLightingOptions.forEach(function(option) {
      if (option.electricKey && currentElectricValues.hasOwnProperty(option.electricKey)) {
        option.checked = currentElectricValues[option.electricKey] === 1;
      }
    });
  }

  // Process electric data and update checkboxes
  function processElectricData(data) {
    if (!data || !data.electrics) return;
    
    var hasChanges = false;
    const e = data.electrics;
    
    // Check for changes in electric values we care about
    $scope.footwellLightingOptions.forEach(function(option) {
      if (option.electricKey && e.hasOwnProperty(option.electricKey)) {
        var newValue = e[option.electricKey];
        if (currentElectricValues[option.electricKey] !== newValue) {
          currentElectricValues[option.electricKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Also check for color values
    ['footwellLightColorR', 'footwellLightColorG', 'footwellLightColorB'].forEach(function(colorKey) {
      if (e.hasOwnProperty(colorKey)) {
        var newValue = e[colorKey];
        if (currentElectricValues[colorKey] !== newValue) {
          currentElectricValues[colorKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Update checkbox states if any values changed
    if (hasChanges) {
      updateCheckboxStates();
    }
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('FootwellLights: Received settings:', settings);
    
    // Update our electric values cache, but preserve defaults if not set
    Object.keys(settings).forEach(function(key) {
      if (settings[key] !== undefined && settings[key] !== null) {
        currentElectricValues[key] = settings[key];
      }
    });
    
    // Update checkbox states
    updateCheckboxStates();
    
    // Apply color settings if available, otherwise use defaults
    const colorR = currentElectricValues.footwellLightColorR;
    const colorG = currentElectricValues.footwellLightColorG;
    const colorB = currentElectricValues.footwellLightColorB;
    
    setElectricValues(colorR, colorG, colorB);
    console.log('FootwellLights: Applied color RGB(' + colorR + ',' + colorG + ',' + colorB + ')');
    
    // Mark that we've loaded settings
    currentElectricValues._settingsLoaded = true;
    
    $scope.$apply();
  });

  // Listen for electric data updates (called from main controller)
  $scope.updateElectricData = function(data) {
    processElectricData(data);
  };

  $scope.getColorAtCursor = function() {
    if (!$scope.colorPickerMode) return;
    
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    const img = new Image();
    
    img.onload = function() {
      canvas.width = img.width;
      canvas.height = img.height;
      ctx.drawImage(img, 0, 0);
      
      const x = Math.floor((img.width * $scope.colorCursor.x) / 100);
      const y = Math.floor((img.height * $scope.colorCursor.y) / 100);
      
      const clampedX = Math.max(0, Math.min(img.width - 1, x));
      const clampedY = Math.max(0, Math.min(img.height - 1, y));
      
      const imageData = ctx.getImageData(clampedX, clampedY, 1, 1);
      const [r, g, b, a] = imageData.data;
      
      if (a > 0) {
        console.log(`Color at cursor: RGB(${r}, ${g}, ${b})`);
        setElectricValues(r, g, b);
      }
    };
    
    img.src = '/vehicles/RLA_A90/infotainment_screen/icons/color_wheel.png';
  };

  // Throttled color updates
  let colorUpdateTimeout = null;

  $scope.moveColorCursor = function(direction) {
    if (!$scope.colorPickerMode) return;
    
    const step = 2;
    
    switch(direction) {
      case 'up':
        $scope.colorCursor.y = Math.max(0, $scope.colorCursor.y - step);
        break;
      case 'down':
        $scope.colorCursor.y = Math.min(100, $scope.colorCursor.y + step);
        break;
      case 'left':
        $scope.colorCursor.x = Math.max(0, $scope.colorCursor.x - step);
        break;
      case 'right':
        $scope.colorCursor.x = Math.min(100, $scope.colorCursor.x + step);
        break;
    }
    
    if (colorUpdateTimeout) {
      clearTimeout(colorUpdateTimeout);
    }
    
    colorUpdateTimeout = setTimeout(function() {
      $scope.getColorAtCursor();
    }, 50);
  };

  // Navigation handlers
  $scope.getFootwellLightsScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedFootwellLightsIndex - $scope.footwellLightsScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getFootwellLightsScrollTransform = function() {
    return 'translateY(' + ($scope.footwellLightsScrollOffset * -40) + 'px)';
  };

  $scope.getFootwellLightsScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.footwellLightingOptions.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getFootwellLightsScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.footwellLightingOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getFootwellLightsScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.footwellLightsScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableFootwellLightsOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.footwellLightingOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.footwellLightingOptions.length; i++) {
        if (!$scope.footwellLightingOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateFootwellLightsMenu = function(direction) {
    if ($scope.colorPickerMode) return;
    
    if (direction === 'up' && $scope.selectedFootwellLightsIndex > 0) {
      const newIndex = $scope.findNextSelectableFootwellLightsOption('up', $scope.selectedFootwellLightsIndex);
      if (newIndex !== $scope.selectedFootwellLightsIndex) {
        $scope.selectedFootwellLightsIndex = newIndex;
        if ($scope.selectedFootwellLightsIndex < $scope.footwellLightsScrollOffset) {
          $scope.footwellLightsScrollOffset = $scope.selectedFootwellLightsIndex;
        }
      }
    } else if (direction === 'down' && $scope.selectedFootwellLightsIndex < $scope.footwellLightingOptions.length - 1) {
      const newIndex = $scope.findNextSelectableFootwellLightsOption('down', $scope.selectedFootwellLightsIndex);
      if (newIndex !== $scope.selectedFootwellLightsIndex) {
        $scope.selectedFootwellLightsIndex = newIndex;
        const visibleItems = 6;
        if ($scope.selectedFootwellLightsIndex >= $scope.footwellLightsScrollOffset + visibleItems) {
          $scope.footwellLightsScrollOffset = $scope.selectedFootwellLightsIndex - visibleItems + 1;
        }
      }
    }
  };

  // Event handlers
  $scope.$on('footwell-lights-navigate-up', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('up');
    } else {
      $scope.navigateFootwellLightsMenu('up');
    }
  });

  $scope.$on('footwell-lights-navigate-down', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('down');
    } else {
      $scope.navigateFootwellLightsMenu('down');
    }
  });

  $scope.$on('footwell-lights-navigate-left', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('left');
    }
  });

  $scope.$on('footwell-lights-navigate-right', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('right');
    }
  });

  $scope.$on('footwell-lights-select-option', function() {
    const selectedOption = $scope.footwellLightingOptions[$scope.selectedFootwellLightsIndex];
    if (selectedOption && selectedOption.hasCheckbox && !selectedOption.checkboxDisabled) {
      // Toggle the electric value directly
      const currentValue = currentElectricValues[selectedOption.electricKey] || 0;
      const newValue = currentValue === 1 ? 0 : 1;
      
      // Update our cache immediately for responsive UI
      currentElectricValues[selectedOption.electricKey] = newValue;
      selectedOption.checked = newValue === 1;
      
      // Send to Lua
      writeElectricValue(selectedOption.electricKey, newValue);
    } else if (selectedOption && selectedOption.name === "Color Selector") {
      $scope.colorPickerMode = true;
      $scope.getColorAtCursor();
    }
  });

  $scope.$on('footwell-lights-back', function() {
    if ($scope.colorPickerMode) {
      $scope.colorPickerMode = false;
    } else {
      $scope.$parent.currentMenu = 'exteriorLighting';
    }
  });

  $scope.handleBackButton = function() {
    if ($scope.colorPickerMode) {
      $scope.colorPickerMode = false;
      return false;
    } else {
      return true;
    }
  };

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'footwellLighting' && oldMenu === 'exteriorLighting') {
      $scope.selectedFootwellLightsIndex = 0;
      $scope.footwellLightsScrollOffset = 0;
      $scope.colorPickerMode = false;
      
      // Only apply defaults if settings haven't been loaded yet
      if (!currentElectricValues._settingsLoaded) {
        console.log('FootwellLights: Entering menu, applying defaults since no settings loaded');
        initializeDefaults();
      } else {
        console.log('FootwellLights: Entering menu, settings already loaded');
      }
    }
  });
});