local M = {}

local function resolveColorValue(value)
    if type(value) == "number" then
        return value
    elseif type(value) == "string" then
        local electricValue = electrics.values[value] or 0
        if type(electricValue) == "boolean" then
            electricValue = electricValue and 255 or 0
        end
        return math.max(0, math.min(255, electricValue))
    else
        return 0
    end
end

local function parseJBeamGlowmap(prop)
    if not prop.dynamicLightGlowmap then return nil end
    
    local glowmap = prop.dynamicLightGlowmap
    if not glowmap.simpleFunction or not glowmap.off or not glowmap.on then
        return nil
    end
    
    return {
        simpleFunction = glowmap.simpleFunction,
        offColor = glowmap.off,
        onColor = glowmap.on,
        onIntenseColor = glowmap.on_intense
    }
end

local function evaluateSimpleFunction(simpleFunction)
    local result = 0
    
    if type(simpleFunction) == "string" then
        local electricValue = electrics.values[simpleFunction] or 0
        if type(electricValue) == "boolean" then
            electricValue = electricValue and 1 or 0
        end
        result = electricValue
    elseif type(simpleFunction) == "table" then
        for electricName, multiplier in pairs(simpleFunction) do
            local electricValue = electrics.values[electricName] or 0
            if type(electricValue) == "boolean" then
                electricValue = electricValue and 1 or 0
            end
            result = result + (electricValue * multiplier)
        end
    end
    
    return result
end

local function selectGlowmapColor(offColor, onColor, onIntenseColor, factor)
    local resolvedOffColor = {
        r = resolveColorValue(offColor.r),
        g = resolveColorValue(offColor.g),
        b = resolveColorValue(offColor.b),
        a = resolveColorValue(offColor.a)
    }
    
    local resolvedOnColor = {
        r = resolveColorValue(onColor.r),
        g = resolveColorValue(onColor.g),
        b = resolveColorValue(onColor.b),
        a = resolveColorValue(onColor.a)
    }
    
    local resolvedOnIntenseColor = nil
    if onIntenseColor then
        resolvedOnIntenseColor = {
            r = resolveColorValue(onIntenseColor.r),
            g = resolveColorValue(onIntenseColor.g),
            b = resolveColorValue(onIntenseColor.b),
            a = resolveColorValue(onIntenseColor.a)
        }
    end
    
    if factor < 0.0001 then
        return resolvedOffColor
    elseif factor <= 0.5 then
        return resolvedOnColor
    else
        return resolvedOnIntenseColor or resolvedOnColor
    end
end

local function updateGFX(dt)
    electrics.values.dynamicLightsAlwaysOn = 1
    local propsData = v.data.props
    if not propsData then return end
    
    for i = 0, tableSizeC(propsData) - 1 do
        local prop = propsData[i]
        if prop and prop.pid and prop.lightColor then
            local glowmapSettings = parseJBeamGlowmap(prop)
            
            if glowmapSettings then
                local factor = evaluateSimpleFunction(glowmapSettings.simpleFunction)
                local selectedColor = selectGlowmapColor(glowmapSettings.offColor, glowmapSettings.onColor, glowmapSettings.onIntenseColor, factor)
                
                local lightColor = color(selectedColor.r, selectedColor.g, selectedColor.b, selectedColor.a)
                obj:setPropLight(prop.pid, prop.lightBrightness or 1, prop.flareScale or 1, lightColor)
            end
        end
    end
end

local function init()
end

local function reset()
    init()
end

M.init = init
M.reset = reset
M.updateGFX = updateGFX

return M