local M = {}

local CHIME_NODE = "dsh3"
local CHIME_PATH = "/vehicles/RLA_A90/sounds/doorChime.mp3"
local CHIME_INTERVAL = 2.5

local current_sfx_sources = {}
local chimeTimer = 0
local lastChimeCondition = false

local function resolveNodeIDs()
  local v = _G.v
  if not v or not v.data or not v.data.nodes then return { chime = 0 } end
  for _, node in ipairs(v.data.nodes) do
    if node.name == CHIME_NODE then
      return { chime = node.cid }
    end
  end
  return { chime = 0 }
end

local function sanitizeObjectName(path)
  return path:gsub("/", "_")
end

local function cleanupSounds()
  for _, sfx in ipairs(current_sfx_sources) do
    if sfx then
      obj:stopSFX(sfx)
      obj:deleteSFXSource(sfx)
    end
  end
  current_sfx_sources = {}
end

local function playChimeOnce()
  local node = resolveNodeIDs().chime
  if node == 0 then return end

  local safe = sanitizeObjectName(CHIME_PATH .. "_" .. node)
  local src = obj:createSFXSource(CHIME_PATH, "AudioDefault3D", safe, node)
  if src then
    obj:setVolumePitch(src, 1.0, 1.0)
    obj:playSFX(src)
    table.insert(current_sfx_sources, src)
  end
end

local function updateGFX(dt)
  if not electrics or not electrics.values then return end

  local leftDoorOpen = electrics.values.doorLCoupler_notAttached == 1
  local rightDoorOpen = electrics.values.doorRCoupler_notAttached == 1
  local ignition = electrics.values.ignitionLevel or 0
  local engineRunning = electrics.values.engineRunning or 0

  if ignition > 0 then
    electrics.values.door_indicator_left  = (leftDoorOpen and not rightDoorOpen) and 1 or 0
    electrics.values.door_indicator_right = (rightDoorOpen and not leftDoorOpen) and 1 or 0
    electrics.values.door_indicator_both  = (leftDoorOpen and rightDoorOpen) and 1 or 0
  else
    electrics.values.door_indicator_left = 0
    electrics.values.door_indicator_right = 0
    electrics.values.door_indicator_both = 0
  end

  local shouldChime = leftDoorOpen and ignition > 0 and engineRunning == 0

  if shouldChime then
    if not lastChimeCondition then
      playChimeOnce()
      chimeTimer = 0
    else
      chimeTimer = chimeTimer + dt
      if chimeTimer >= CHIME_INTERVAL then
        playChimeOnce()
        chimeTimer = 0
      end
    end
  else
    chimeTimer = 0
    cleanupSounds()
  end

  lastChimeCondition = shouldChime
end

local function reset()
  chimeTimer = 0
  cleanupSounds()
  if electrics and electrics.values then
    electrics.values.door_indicator_left = 0
    electrics.values.door_indicator_right = 0
    electrics.values.door_indicator_both = 0
  end
  lastChimeCondition = false
end

local function onVehicleActiveChanged(active)
  if not active then cleanupSounds() end
end

local function destroy()
  cleanupSounds()
end

M.updateGFX = updateGFX
M.onReset = reset
M.onInit = reset
M.onVehicleActiveChanged = onVehicleActiveChanged
M.destroy = destroy

return M
