local M = {}
M.type = "auxiliary"

local htmlTexture = require("htmlTexture")

local screenMaterialName = nil
local htmlFilePath = nil
local textureWidth = 0
local textureHeight = 0
local textureFPS = 0
local updateTimer = 0
local invFPS = 1 / 15
local textureCreated = false

-- Retry logic
local setupPending = false
local setupAttempts = 0
local maxSetupAttempts = 10
local initDelay = 2.0 -- Priority 4: 2.0s delay
local initTimer = 0
local isInitialized = false

-- Track starlight settings
local starlightSettings = {
  enabled = 1,
  mode = 0,
  speed = 1,
  brightness = 100,
  colorR = 102,
  colorG = 126,
  colorB = 234
}

local function updateGFX(dt)
  if not isInitialized then
    initTimer = initTimer + dt
    if initTimer >= initDelay and not textureCreated then
      -- Attempt texture creation
      local success = pcall(function()
        htmlTexture.create(screenMaterialName, htmlFilePath, textureWidth, textureHeight, textureFPS, "automatic")
      end)
      
      if success then
        textureCreated = true
        setupPending = true
        setupAttempts = 0
      end
    end
    
    -- Retry setup calls
    if setupPending and setupAttempts < maxSetupAttempts then
      setupAttempts = setupAttempts + 1
      
      local modeNames = {"Rainbow", "Pulse", "Static", "Twinkle", "Wave"}
      local initSettings = {
        enabled = electrics.values.starlightsEnabled == 1,
        mode = modeNames[electrics.values.starlightsMode + 1] or "Rainbow",
        speed = electrics.values.starlightsSpeed,
        brightness = electrics.values.starlightsBrightness,
        staticColorR = electrics.values.starlightsColorR,
        staticColorG = electrics.values.starlightsColorG,
        staticColorB = electrics.values.starlightsColorB
      }
      
      local setupSuccess = pcall(function()
        htmlTexture.call(screenMaterialName, "updateStarlightSettings", initSettings)
        
        local initData = {
          initialized = true,
          timestamp = os.time(),
          elapsedTime = os.clock()
        }
        htmlTexture.call(screenMaterialName, "updateStarlight", initData)
      end)
      
      if setupSuccess then
        setupPending = false
        isInitialized = true
      else
        if setupAttempts >= maxSetupAttempts then
          setupPending = false
          isInitialized = true
        end
      end
    end
    
    return
  end
  
  updateTimer = updateTimer + dt
  if updateTimer > invFPS and playerInfo.anyPlayerSeated then
    updateTimer = 0

    -- Check for changes in electric values
    local hasChanges = false
    
    if electrics.values.starlightsEnabled ~= nil and electrics.values.starlightsEnabled ~= starlightSettings.enabled then
      starlightSettings.enabled = electrics.values.starlightsEnabled
      hasChanges = true
    end
    
    if electrics.values.starlightsMode and electrics.values.starlightsMode ~= starlightSettings.mode then
      starlightSettings.mode = electrics.values.starlightsMode
      hasChanges = true
    end
    
    if electrics.values.starlightsSpeed and electrics.values.starlightsSpeed ~= starlightSettings.speed then
      starlightSettings.speed = electrics.values.starlightsSpeed
      hasChanges = true
    end
    
    if electrics.values.starlightsBrightness and electrics.values.starlightsBrightness ~= starlightSettings.brightness then
      starlightSettings.brightness = electrics.values.starlightsBrightness
      hasChanges = true
    end
    
    if electrics.values.starlightsColorR and electrics.values.starlightsColorR ~= starlightSettings.colorR then
      starlightSettings.colorR = electrics.values.starlightsColorR
      hasChanges = true
    end
    if electrics.values.starlightsColorG and electrics.values.starlightsColorG ~= starlightSettings.colorG then
      starlightSettings.colorG = electrics.values.starlightsColorG
      hasChanges = true
    end
    if electrics.values.starlightsColorB and electrics.values.starlightsColorB ~= starlightSettings.colorB then
      starlightSettings.colorB = electrics.values.starlightsColorB
      hasChanges = true
    end
    
    if hasChanges then
      local modeNames = {"Rainbow", "Pulse", "Static", "Twinkle", "Wave"}
      local modeName = modeNames[starlightSettings.mode + 1] or "Rainbow"
      
      local settingsData = {
        enabled = starlightSettings.enabled == 1,
        mode = modeName,
        speed = starlightSettings.speed,
        brightness = starlightSettings.brightness,
        staticColorR = starlightSettings.colorR,
        staticColorG = starlightSettings.colorG,
        staticColorB = starlightSettings.colorB
      }
      
      pcall(function()
        htmlTexture.call(screenMaterialName, "updateStarlightSettings", settingsData)
      end)
    end

    -- Regular update call
    local updateData = {
      timestamp = os.time(),
      elapsedTime = os.clock()
    }

    pcall(function()
      htmlTexture.call(screenMaterialName, "updateStarlight", updateData)
    end)
  end
end

local function init(jbeamData)
  if not jbeamData then 
    return 
  end
  
  if not jbeamData.configuration then
    return
  end
  
  local config = jbeamData.configuration
  
  screenMaterialName = config.screenMaterialName
  htmlFilePath = config.htmlFilePath
  textureWidth = config.textureWidth or 1024
  textureHeight = config.textureHeight or 512
  textureFPS = config.textureFPS or 15

  if not screenMaterialName or not htmlFilePath then
    return
  end
  
  -- Priority 4: 2.0s delay
  initDelay = 2.0
  initTimer = 0
  isInitialized = false
  textureCreated = false
  setupPending = false
  setupAttempts = 0
  
  -- Initialize electric values if not already set
  if not electrics.values.starlightsEnabled then
    electrics.values.starlightsEnabled = 1
  end
  if not electrics.values.starlightsMode then
    electrics.values.starlightsMode = 0
  end
  if not electrics.values.starlightsSpeed then
    electrics.values.starlightsSpeed = 1
  end
  if not electrics.values.starlightsBrightness then
    electrics.values.starlightsBrightness = 100
  end
  if not electrics.values.starlightsColorR then
    electrics.values.starlightsColorR = 102
  end
  if not electrics.values.starlightsColorG then
    electrics.values.starlightsColorG = 126
  end
  if not electrics.values.starlightsColorB then
    electrics.values.starlightsColorB = 234
  end
end

M.init = init
M.reset = nop
M.updateGFX = updateGFX

return M
