local M = {}
M.type = "auxiliary"

local htmlTexture = nil
pcall(function() htmlTexture = require("htmlTexture") end)

local screenMaterialName = nil
local htmlFilePath = nil
local textureWidth = 0
local textureHeight = 0
local textureFPS = 0
local updateTimer = 0
local invFPS = 1 / 60
local textureCreated = false

-- Retry logic
local setupPending = false
local setupAttempts = 0
local maxSetupAttempts = 10
local initDelay = 1.0 -- Priority 2: 1.0s delay
local initTimer = 0
local isInitialized = false

local function updateGFX(dt)
    if not htmlTexture then
        return
    end
    
    if not isInitialized then
        initTimer = initTimer + dt
        if initTimer >= initDelay and not textureCreated then
            -- Attempt texture creation
            local success = pcall(function()
                htmlTexture.create(screenMaterialName, htmlFilePath, textureWidth, textureHeight, textureFPS, "automatic")
            end)
            
            if success then
                textureCreated = true
                setupPending = true
                setupAttempts = 0
            end
        end
        
        -- Retry setup calls (if needed for this display)
        if setupPending and setupAttempts < maxSetupAttempts then
            setupAttempts = setupAttempts + 1
            
            -- Test if texture is responsive
            local setupSuccess = pcall(function()
                htmlTexture.call(screenMaterialName, "updateDisplay", {electrics = {rpm = 0}})
            end)
            
            if setupSuccess then
                setupPending = false
                isInitialized = true
            else
                if setupAttempts >= maxSetupAttempts then
                    setupPending = false
                    isInitialized = true
                end
            end
        end
        
        return
    end
    
    updateTimer = updateTimer + dt
    if updateTimer > invFPS and screenMaterialName and textureCreated then
        updateTimer = 0
        local shiftData = {
            electrics = {
                rpm = electrics.values.rpm or 0,
                wheelspeed = electrics.values.wheelspeed or 0,
                led0 = electrics.values.led0 or 0,
                led1 = electrics.values.led1 or 0,
                led2 = electrics.values.led2 or 0,
                led3 = electrics.values.led3 or 0,
                led4 = electrics.values.led4 or 0,
                led5 = electrics.values.led5 or 0,
                led6 = electrics.values.led6 or 0,
                led7 = electrics.values.led7 or 0,
                led8 = electrics.values.led8 or 0
            }
        }
        pcall(function()
            htmlTexture.call(screenMaterialName, "updateDisplay", shiftData)
        end)
    end
end

local function init(jbeamData)
    if not htmlTexture then
        return
    end

    screenMaterialName = jbeamData.materialName or "@RLA_A90_shiftscreen"
    htmlFilePath = jbeamData.htmlPath or "local://local/vehicles/RLA_A90/shift_screen/shiftscreen.html"
    textureWidth = jbeamData.textureWidth or 256
    textureHeight = jbeamData.textureHeight or 128
    textureFPS = jbeamData.textureFPS or 60
    
    -- Priority 2: 1.0s delay
    initDelay = 1.0
    initTimer = 0
    isInitialized = false
    textureCreated = false
    setupPending = false
    setupAttempts = 0
end

local function reset()
    updateTimer = 0
end

local function destroy()
    if htmlTexture and type(htmlTexture.destroy) == "function" and screenMaterialName then
        pcall(function()
            htmlTexture.destroy(screenMaterialName)
        end)
    end
end

M.init = init
M.updateGFX = updateGFX
M.reset = reset
M.destroy = destroy

return M
