local M = {}

local previousGear = "P"
local isInSpecialMode = false
local isInManualMode = false
local previousManualGear = ""

local forwardTimer = 0
local backwardTimer = 0
local manualForwardTimer = 0
local manualBackwardTimer = 0
local movementDuration = 0.15

local electricsData = {
    shifter_auto_rest = 1,
    shifter_auto_forward = 0,
    shifter_auto_backward = 0,
    shifter_auto_manual = 0,
    shifter_auto_manual_forward = 0,
    shifter_auto_manual_backward = 0
}

local function resetMovementElectrics()
    electricsData.shifter_auto_forward = 0
    electricsData.shifter_auto_backward = 0
    electricsData.shifter_auto_manual_forward = 0
    electricsData.shifter_auto_manual_backward = 0
end

local function startMovementTimer(movementType)
    resetMovementElectrics()
    electricsData.shifter_auto_rest = 0
    
    if movementType == "forward" then
        electricsData.shifter_auto_forward = 1
        forwardTimer = movementDuration
    elseif movementType == "backward" then
        electricsData.shifter_auto_backward = 1
        backwardTimer = movementDuration
    elseif movementType == "manual_forward" then
        electricsData.shifter_auto_manual = 0
        electricsData.shifter_auto_manual_forward = 1
        manualForwardTimer = movementDuration
    elseif movementType == "manual_backward" then
        electricsData.shifter_auto_manual = 0
        electricsData.shifter_auto_manual_backward = 1
        manualBackwardTimer = movementDuration
    end
end

local function checkManualMode(gear)
    return string.sub(gear, 1, 1) == "M"
end

local function checkSportMode(gear)
    return string.sub(gear, 1, 1) == "S"
end

local function checkSpecialMode(gear)
    return checkManualMode(gear) or checkSportMode(gear)
end

local function getAutoShiftDirection(fromGear, toGear)
    if fromGear == "R" and toGear == "P" then
        return nil
    end
    
    local gearOrder = {P = 1, R = 2, N = 3, D = 4}
    
    local fromPos = gearOrder[fromGear]
    local toPos = gearOrder[toGear]
    
    if fromPos and toPos then
        if toPos > fromPos then
            return "forward"
        elseif toPos < fromPos then
            return "backward"
        end
    end
    
    return nil
end

local function getManualShiftDirection(fromGear, toGear)
    local fromNum = tonumber(string.sub(fromGear, 2))
    local toNum = tonumber(string.sub(toGear, 2))
    
    if fromNum and toNum then
        if toNum > fromNum then
            return "up"
        elseif toNum < fromNum then
            return "down"
        end
    end
    
    return nil
end

local function updateTimers(dt)
    if forwardTimer > 0 then
        forwardTimer = forwardTimer - dt
        if forwardTimer <= 0 then
            electricsData.shifter_auto_forward = 0
            if not isInManualMode then
                electricsData.shifter_auto_rest = 1
            end
        end
    end
    
    if backwardTimer > 0 then
        backwardTimer = backwardTimer - dt
        if backwardTimer <= 0 then
            electricsData.shifter_auto_backward = 0
            if not isInManualMode then
                electricsData.shifter_auto_rest = 1
            end
        end
    end
    
    if manualForwardTimer > 0 then
        manualForwardTimer = manualForwardTimer - dt
        if manualForwardTimer <= 0 then
            electricsData.shifter_auto_manual_forward = 0
            if isInSpecialMode then
                electricsData.shifter_auto_manual = 1
            end
        end
    end
    
    if manualBackwardTimer > 0 then
        manualBackwardTimer = manualBackwardTimer - dt
        if manualBackwardTimer <= 0 then
            electricsData.shifter_auto_manual_backward = 0
            if isInSpecialMode then
                electricsData.shifter_auto_manual = 1
            end
        end
    end
end

local function init()
    previousGear = "P"
    isInSpecialMode = false
    isInManualMode = false
    previousManualGear = ""
    
    forwardTimer = 0
    backwardTimer = 0
    manualForwardTimer = 0
    manualBackwardTimer = 0
    
    electricsData.shifter_auto_rest = 1
    resetMovementElectrics()
    electricsData.shifter_auto_manual = 0
end

local function updateGFX(dt)
    updateTimers(dt)
    
    local currentGear = electrics.values.gear or "P"
    local currentManualMode = checkManualMode(currentGear)
    local currentSportMode = checkSportMode(currentGear)
    local currentSpecialMode = checkSpecialMode(currentGear)
    
    if currentSpecialMode ~= isInSpecialMode then
        if currentSpecialMode then
            electricsData.shifter_auto_manual = 1
            electricsData.shifter_auto_rest = 0
            resetMovementElectrics()
            isInSpecialMode = true
            isInManualMode = currentManualMode
            previousManualGear = currentGear
        else
            electricsData.shifter_auto_manual = 0
            resetMovementElectrics()
            electricsData.shifter_auto_rest = 1
            isInSpecialMode = false
            isInManualMode = false
            previousManualGear = ""
        end
    end
    
    if isInSpecialMode and currentManualMode ~= isInManualMode then
        if currentSportMode and isInManualMode then
            startMovementTimer("manual_forward")
        elseif currentManualMode and not isInManualMode then
            startMovementTimer("manual_backward")
        end
        isInManualMode = currentManualMode
    end
    
    if currentGear ~= previousGear then
        if isInSpecialMode and currentManualMode then
            local direction = getManualShiftDirection(previousGear, currentGear)
            if direction == "up" then
                startMovementTimer("manual_backward")
            elseif direction == "down" then
                startMovementTimer("manual_forward")
            end
            previousManualGear = currentGear
        elseif not isInSpecialMode then
            local direction = getAutoShiftDirection(previousGear, currentGear)
            if direction == "forward" then
                startMovementTimer("forward")
            elseif direction == "backward" then
                startMovementTimer("backward")
            end
        end
        
        previousGear = currentGear
    end
    
    for name, value in pairs(electricsData) do
        electrics.values[name] = value
    end
end

M.init = init
M.updateGFX = updateGFX

return M