local M = {}
M.type = "auxiliary"

local playlistConfig = require("vehicles/common/lua/sdd_carplay_playlist_config")

-- ========================================
-- AUDIO NODE CONFIGURATION
-- ========================================

local audioNodes = {
    { name = "dsh1", id = "front", volume = 1.5, type = "front_center" },
    { name = "d14r", id = "right", volume = 1.2, type = "side_right" },
    { name = "d14l", id = "left", volume = 1.8, type = "side_left" },
    { name = "f9r",  id = "rear_right", volume = 0.8, type = "rear_right" },
    { name = "f9l",  id = "rear_left", volume = 1.0, type = "rear_left" }
}

-- ========================================
-- VOLUME AND EQUALIZER CONTROL SYSTEM
-- ========================================

local volumeControl = {
    masterVolume = 0.5,
    minVolume = 0.0,
    maxVolume = 1.0,
    volumeStep = 0.005,
    lastVolumeUpState = false,
    lastVolumeDownState = false,
    volumeChangeAccumulator = 0,
    volumeChangeDelay = 0
}

local equalizerControl = {
    fade = 50,
    balance = 50,
    bass = 50,
    lastFade = 50,
    lastBalance = 50,
    lastBass = 50
}

-- ========================================
-- SAVED SONGS PERSISTENCE SYSTEM
-- ========================================

local savedSongs = {
    configPath = "vehicles/RLA_A90/saved_songs.json",
    songs = {},
    initialized = false,
    dirty = false,
    lastSaveTime = 0,
    saveInterval = 1.0
}

local playlistCache = {}
local cacheTimeout = 300
local cacheTimestamps = {}
local lastRequestedPlaylistId = ""
local lastRequestedPreviewPlaylistId = ""

-- ========================================
-- MUSIC SETTINGS PERSISTENCE SYSTEM
-- ========================================

local musicSettings = {
    configPath = "vehicles/RLA_A90/music_settings.json",
    defaults = {
        masterVolume = 0.50,
        equalizerFade = 50,
        equalizerBalance = 50,
        equalizerBass = 50
    },
    current = {},
    lastKnownValues = {},
    initialized = false,
    dirty = false,
    lastChangeTime = 0,
    debounceDelay = 0.5,
    startTime = 0,
    canSave = false
}

local function loadMusicSettings()
    local success, settingsData = pcall(function()
        return jsonReadFile(musicSettings.configPath)
    end)
    
    if success and settingsData then
        musicSettings.current = {
            masterVolume = settingsData.masterVolume or musicSettings.defaults.masterVolume,
            equalizerFade = settingsData.equalizerFade or musicSettings.defaults.equalizerFade,
            equalizerBalance = settingsData.equalizerBalance or musicSettings.defaults.equalizerBalance,
            equalizerBass = settingsData.equalizerBass or musicSettings.defaults.equalizerBass
        }
        return true
    else
        musicSettings.current = {
            masterVolume = musicSettings.defaults.masterVolume,
            equalizerFade = musicSettings.defaults.equalizerFade,
            equalizerBalance = musicSettings.defaults.equalizerBalance,
            equalizerBass = musicSettings.defaults.equalizerBass
        }
        musicSettings.dirty = true
        return false
    end
end

local function saveMusicSettings()
    if not musicSettings.canSave or not musicSettings.dirty then
        return
    end
    
    local saveSuccess, errorMsg = pcall(function()
        jsonWriteFile(musicSettings.configPath, musicSettings.current, true)
    end)
    
    if saveSuccess then
        musicSettings.dirty = false
    end
end

local function applyMusicSettings()
    volumeControl.masterVolume = musicSettings.current.masterVolume
    equalizerControl.fade = musicSettings.current.equalizerFade
    equalizerControl.balance = musicSettings.current.equalizerBalance
    equalizerControl.bass = musicSettings.current.equalizerBass
    
    musicSettings.lastKnownValues = {
        masterVolume = volumeControl.masterVolume,
        equalizerFade = equalizerControl.fade,
        equalizerBalance = equalizerControl.balance,
        equalizerBass = equalizerControl.bass
    }
end

local function monitorMusicSettings()
    if not musicSettings.canSave then 
        return 
    end
    
    local hasChanges = false
    
    if math.abs(volumeControl.masterVolume - musicSettings.lastKnownValues.masterVolume) > 0.001 then
        musicSettings.current.masterVolume = volumeControl.masterVolume
        musicSettings.lastKnownValues.masterVolume = volumeControl.masterVolume
        hasChanges = true
    end
    
    if equalizerControl.fade ~= musicSettings.lastKnownValues.equalizerFade then
        musicSettings.current.equalizerFade = equalizerControl.fade
        musicSettings.lastKnownValues.equalizerFade = equalizerControl.fade
        hasChanges = true
    end
    
    if equalizerControl.balance ~= musicSettings.lastKnownValues.equalizerBalance then
        musicSettings.current.equalizerBalance = equalizerControl.balance
        musicSettings.lastKnownValues.equalizerBalance = equalizerControl.balance
        hasChanges = true
    end
    
    if equalizerControl.bass ~= musicSettings.lastKnownValues.equalizerBass then
        musicSettings.current.equalizerBass = equalizerControl.bass
        musicSettings.lastKnownValues.equalizerBass = equalizerControl.bass
        hasChanges = true
    end
    
    if hasChanges then
        musicSettings.dirty = true
        musicSettings.lastChangeTime = os.clock()
    end
    
    if musicSettings.dirty then
        local timeSinceLastChange = os.clock() - musicSettings.lastChangeTime
        if timeSinceLastChange >= musicSettings.debounceDelay then
            saveMusicSettings()
        end
    end
end

-- ========================================
-- JSON ENCODING FUNCTIONS
-- ========================================

local function jsonEncodePlaylist(playlist)
    if not playlist or type(playlist) ~= "table" then
        return "{}"
    end
    
    local function escapeString(str)
        if type(str) == "string" then
            return tostring(str):gsub('"', '\\"'):gsub('\n', '\\n'):gsub('\r', '\\r')
        end
        return tostring(str)
    end
    
    local fields = {}
    
    local playlistName = playlist.name or "Unknown Playlist"
    local playlistId = playlist.id or ""
    local playlistCover = playlist.coverImage or playlist.cover or "default_playlist.png"
    
    table.insert(fields, '"name":"' .. escapeString(playlistName) .. '"')
    table.insert(fields, '"id":"' .. escapeString(playlistId) .. '"')
    table.insert(fields, '"coverImage":"' .. escapeString(playlistCover) .. '"')
    
    return "{" .. table.concat(fields, ",") .. "}"
end

local function jsonEncodePlaylists(data)
    if not data then return "[]" end
    
    if type(data) == "table" then
        if #data == 0 and next(data) == nil then
            return "[]"
        end
        
        local items = {}
        local itemCount = 0
        
        for i, playlist in ipairs(data) do
            if type(playlist) == "table" then
                itemCount = itemCount + 1
                items[itemCount] = jsonEncodePlaylist(playlist)
            end
        end
        return "[" .. table.concat(items, ",") .. "]"
    end
    
    return "[]"
end

local function jsonEncode(data)
    if not data then return "[]" end
    
    if type(data) == "table" then
        if #data == 0 and next(data) == nil then
            return "[]"
        end
        
        local items = {}
        local itemCount = 0
        
        for i, item in ipairs(data) do
            if type(item) == "table" then
                local songName = item.name or item.title or item.song or item.songName or "Unknown Song"
                local artistName = item.artist or item.artistName or ""
                local songId = item.id or item.songId or i
                local duration = item.duration or item.length or 0
                local albumArt = item.albumArt or item.album_art or item.cover or "music_note_album_default.png"
                
                local function escapeString(str)
                    if type(str) == "string" then
                        return tostring(str):gsub('"', '\\"'):gsub('\n', '\\n'):gsub('\r', '\\r')
                    end
                    return tostring(str)
                end
                
                itemCount = itemCount + 1
                items[itemCount] = string.format(
                    '{"name":"%s","artist":"%s","id":"%s","albumArt":"%s","duration":%d}',
                    escapeString(songName),
                    escapeString(artistName),
                    escapeString(songId),
                    escapeString(albumArt),
                    type(duration) == "number" and duration or 0
                )
            end
        end
        return "[" .. table.concat(items, ",") .. "]"
    end
    
    return "[]"
end

local function jsonEncodeSong(song)
    if not song or type(song) ~= "table" then
        return "{}"
    end
    
    local fields = {}
    
    local songName = song.name or song.title or song.song or song.songName or "Unknown Song"
    local artistName = song.artist or song.artistName or ""
    local songId = song.id or song.songId or ""
    local duration = song.duration or song.length or 0
    local albumArt = song.albumArt or song.album_art or song.cover or "music_note_album_default.png"
    local playlistName = song.playlistName or song.playlist or ""
    
    local function escapeString(str)
        if type(str) == "string" then
            return tostring(str):gsub('"', '\\"'):gsub('\n', '\\n'):gsub('\r', '\\r')
        end
        return tostring(str)
    end
    
    table.insert(fields, '"name":"' .. escapeString(songName) .. '"')
    table.insert(fields, '"artist":"' .. escapeString(artistName) .. '"')
    table.insert(fields, '"id":"' .. escapeString(songId) .. '"')
    table.insert(fields, '"albumArt":"' .. escapeString(albumArt) .. '"')
    table.insert(fields, '"playlistName":"' .. escapeString(playlistName) .. '"')
    
    if type(duration) == "number" then
        table.insert(fields, '"duration":' .. tostring(duration))
    else
        table.insert(fields, '"duration":0')
    end
    
    return "{" .. table.concat(fields, ",") .. "}"
end

-- ========================================
-- SAVED SONGS FUNCTIONS
-- ========================================

local function ensureDirectoryExists()
    local directoryPath = "vehicles/RLA_A90"
    if FS then
        if not FS:directoryExists(directoryPath) then
            FS:directoryCreate(directoryPath, true)
        end
    end
end

local function loadSavedSongs()
    local success, songsData = pcall(function()
        return jsonReadFile(savedSongs.configPath)
    end)
    
    if success and songsData then
        savedSongs.songs = songsData
    else
        savedSongs.songs = {}
    end
    
    savedSongs.initialized = true
end

local function saveSavedSongs()
    if not savedSongs.initialized or not savedSongs.dirty then
        return
    end
    
    local currentTime = os.clock()
    if currentTime - savedSongs.lastSaveTime < savedSongs.saveInterval then
        return
    end
    
    pcall(function()
        ensureDirectoryExists()
        jsonWriteFile(savedSongs.configPath, savedSongs.songs, true)
    end)
    
    savedSongs.dirty = false
    savedSongs.lastSaveTime = currentTime
end

local function isSongSaved(songId)
    if not songId or songId == "" then return false end
    
    for _, savedSong in ipairs(savedSongs.songs) do
        if savedSong.id == songId or savedSong.songId == songId then
            return true
        end
    end
    return false
end

local function addSavedSong(song)
    if not song or not song.id then return end
    
    if isSongSaved(song.id) then
        return
    end
    
    local savedSong = {
        id = song.id or song.songId,
        name = song.name or song.title or "Unknown Song",
        artist = song.artist or song.artistName or "",
        albumArt = song.albumArt or song.album_art or song.cover or "music_note_album_default.png",
        duration = song.duration or song.length or 0,
        songPath = song.songPath or song.file or "",
        playlistName = song.playlistName or song.playlist or "Unknown Playlist",
        savedDate = os.time()
    }
    
    table.insert(savedSongs.songs, savedSong)
    savedSongs.dirty = true
    
    saveSavedSongs()
    
    playlistCache["savedsongs"] = nil
    playlistCache["favorites"] = nil
    cacheTimestamps["savedsongs"] = nil
    cacheTimestamps["favorites"] = nil
    
    if lastRequestedPlaylistId == "savedsongs" or lastRequestedPlaylistId == "favorites" then
        lastRequestedPlaylistId = ""
    end
end

local function removeSavedSong(songId)
    if not songId or songId == "" then return end
    
    for i, savedSong in ipairs(savedSongs.songs) do
        if savedSong.id == songId or savedSong.songId == songId then
            table.remove(savedSongs.songs, i)
            savedSongs.dirty = true
            saveSavedSongs()
            
            playlistCache["savedsongs"] = nil
            playlistCache["favorites"] = nil
            cacheTimestamps["savedsongs"] = nil
            cacheTimestamps["favorites"] = nil
            
            if lastRequestedPlaylistId == "savedsongs" or lastRequestedPlaylistId == "favorites" then
                lastRequestedPlaylistId = ""
            end
            
            return
        end
    end
end

-- ========================================
-- MULTI-SOURCE AUDIO SYSTEM
-- ========================================

local sfx_sources = {}
local current_playing_sources = nil
local currentSongPath = ""
local songElapsedTime = 0
local actualSongDuration = 0

local pendingPlayback = nil
local syncFrameCounter = 0
local primingDelay = 1

local availablePlaylists = {}
local totalPlaylistCount = 0
local selectedPlaylistSongs = {}
local selectedPlaylistInfo = {}

local currentQueue = {}
local currentQueueIndex = 0
local currentQueuePlaylistName = ""
local isQueueInitialized = false
local isShuffleMode = false
local originalQueue = {}

local isInitialized = false
local updateTimer = 0
local invFPS = 1/20
local isPlaying = false
local currentTime = 0
local isFavorited = false
local lastMusicControlCommand = ""
local currentSongAlbumArt = ""
local isPausedByUser = false
local currentSongData = nil
local lastFavoriteButtonState = false
local musicSystemActive = false

-- ========================================
-- AUDIO NODE AND SPEAKER FUNCTIONS
-- ========================================

local function resolveNodeID(nodeName)
    if not v or not v.data or not v.data.nodes then 
        return 0 
    end
    
    for _, node in pairs(v.data.nodes) do
        if node.name == nodeName then 
            return node.cid 
        end
    end
    
    return 0
end

local function calculateSpeakerVolume(nodeType, baseVolume, fade, balance)
    local fadeValue = fade / 100.0
    local balanceValue = balance / 100.0
    
    local frontMultiplier, rearMultiplier
    
    if fadeValue <= 0.5 then
        rearMultiplier = 1.0
        frontMultiplier = fadeValue * 2.0
    else
        frontMultiplier = 1.0
        rearMultiplier = (1.0 - fadeValue) * 2.0
    end
    
    local leftMultiplier = math.min(1.0, math.max(0.0, (1.0 - balanceValue) * 2.0))
    local rightMultiplier = math.min(1.0, math.max(0.0, balanceValue * 2.0))
    
    local centerBalanceMultiplier = 1.0 - (math.abs(balanceValue - 0.5) * 2.0)
    
    local multiplier = 1.0
    
    if nodeType == "front_center" then
        multiplier = frontMultiplier * centerBalanceMultiplier
    elseif nodeType == "side_left" then
        local sideFadeMultiplier = (frontMultiplier + rearMultiplier) / 2.0
        multiplier = leftMultiplier * sideFadeMultiplier
    elseif nodeType == "side_right" then
        local sideFadeMultiplier = (frontMultiplier + rearMultiplier) / 2.0
        multiplier = rightMultiplier * sideFadeMultiplier
    elseif nodeType == "rear_left" then
        local rearBalanceMultiplier = leftMultiplier
        if balanceValue < 0.5 and leftMultiplier > 0 then
            rearBalanceMultiplier = leftMultiplier * 0.5 + 0.5
        end
        multiplier = rearBalanceMultiplier * rearMultiplier
    elseif nodeType == "rear_right" then
        local rearBalanceMultiplier = rightMultiplier
        if balanceValue > 0.5 and rightMultiplier > 0 then
            rearBalanceMultiplier = rightMultiplier * 0.5 + 0.5
        end
        multiplier = rearBalanceMultiplier * rearMultiplier
    end
    
    return baseVolume * multiplier
end

local function updateEqualizerVolumes()
    if not current_playing_sources then
        return
    end
    
    local fade = electrics.values.equalizerFade or equalizerControl.fade
    local balance = electrics.values.equalizerBalance or equalizerControl.balance
    local bass = electrics.values.equalizerBass or equalizerControl.bass
    
    equalizerControl.fade = fade
    equalizerControl.balance = balance
    equalizerControl.bass = bass
    
    if fade == equalizerControl.lastFade and 
       balance == equalizerControl.lastBalance and
       bass == equalizerControl.lastBass then
        return
    end
    
    local effectiveMasterVolume = volumeControl.masterVolume * volumeControl.masterVolume
    
    for _, nodeConfig in ipairs(audioNodes) do
        local sourceData = current_playing_sources[nodeConfig.id]
        if sourceData then
            local eqBaseVolume = calculateSpeakerVolume(nodeConfig.type, nodeConfig.volume, fade, balance)
            local bassMultiplier = 1.0 + (bass - 50) / 100.0
            eqBaseVolume = eqBaseVolume * bassMultiplier
            sourceData.eqBaseVolume = eqBaseVolume
            local finalVolume = eqBaseVolume * effectiveMasterVolume
            obj:setVolumePitch(sourceData.sfx, finalVolume, 1.0)
        end
    end
    
    equalizerControl.lastFade = fade
    equalizerControl.lastBalance = balance
    equalizerControl.lastBass = bass
end

-- ========================================
-- MULTI-SOURCE SYNC FUNCTIONS
-- ========================================

local function cleanupUnusedSources()
    for songPath, sources in pairs(sfx_sources) do
        if sources ~= current_playing_sources then
            for location, sourceData in pairs(sources) do
                obj:cutSFX(sourceData.sfx)
                obj:deleteSFXSource(sourceData.sfx)
            end
            sfx_sources[songPath] = nil
        end
    end
end

local function prepareSongMultiSource(songPath, shouldPrime)
    if not songPath or songPath == "" then
        return nil
    end
    
    cleanupUnusedSources()
    
    local newSources = {}
    local successCount = 0
    
    local fade = electrics.values.equalizerFade or equalizerControl.fade
    local balance = electrics.values.equalizerBalance or equalizerControl.balance
    local bass = electrics.values.equalizerBass or equalizerControl.bass
    
    local effectiveMasterVolume = volumeControl.masterVolume * volumeControl.masterVolume
    
    for _, nodeConfig in ipairs(audioNodes) do
        local nodeId = resolveNodeID(nodeConfig.name)
        local safeName = songPath:gsub("/", "_"):gsub("%.", "_") .. "_" .. nodeConfig.id .. "_" .. nodeId
        
        local sfx = obj:createSFXSource(songPath, "AudioDefault3D", safeName, nodeId)
        if sfx then
            local eqBaseVolume = calculateSpeakerVolume(nodeConfig.type, nodeConfig.volume, fade, balance)
            local bassMultiplier = 1.0 + (bass - 50) / 100.0
            eqBaseVolume = eqBaseVolume * bassMultiplier
            
            local effectiveVolume = eqBaseVolume * effectiveMasterVolume
            obj:setVolumePitch(sfx, effectiveVolume, 1.0)
            
            newSources[nodeConfig.id] = {
                sfx = sfx,
                baseVolume = nodeConfig.volume,
                eqBaseVolume = eqBaseVolume,
                nodeType = nodeConfig.type,
                nodeId = nodeId,
                nodeName = nodeConfig.name
            }
            successCount = successCount + 1
        end
    end
    
    if successCount > 0 then
        if shouldPrime then
            for location, sourceData in pairs(newSources) do
                obj:playSFX(sourceData.sfx)
            end
            
            for location, sourceData in pairs(newSources) do
                obj:cutSFX(sourceData.sfx)
            end
        end
        
        return newSources
    else
        for location, sourceData in pairs(newSources) do
            obj:deleteSFXSource(sourceData.sfx)
        end
        return nil
    end
end

local function executeSyncPlayback()
    if not pendingPlayback then
        return
    end
    
    local action = pendingPlayback.action
    local sources = pendingPlayback.sources
    local songPath = pendingPlayback.songPath
    
    print(string.format("DEBUG: executeSyncPlayback - action=%s, songPath=%s", action, songPath))
    
    if action == "play" then
        for location, sourceData in pairs(sources) do
            obj:playSFX(sourceData.sfx)
            print(string.format("DEBUG: Playing SFX for location %s", location))
        end
        
        current_playing_sources = sources
        currentSongPath = songPath
        songElapsedTime = 0
        currentTime = 0
        isPlaying = true
        isPausedByUser = false
        print("DEBUG: Set isPlaying=true, isPausedByUser=false")
        
    elseif action == "resume" then
        for location, sourceData in pairs(sources) do
            obj:playSFX(sourceData.sfx)
            print(string.format("DEBUG: Resuming SFX for location %s", location))
        end
        
        isPlaying = true
        isPausedByUser = false
        print("DEBUG: Set isPlaying=true, isPausedByUser=false")
    end
    
    pendingPlayback = nil
    syncFrameCounter = 0
    print("DEBUG: executeSyncPlayback complete - cleared pendingPlayback")
end

local function playSongMultiSource(songPath)
    if not songPath or songPath == "" then
        return false
    end
    
    if current_playing_sources then
        for location, sourceData in pairs(current_playing_sources) do
            obj:cutSFX(sourceData.sfx)
        end
        
        if currentSongPath ~= songPath and sfx_sources[currentSongPath] then
            for location, sourceData in pairs(sfx_sources[currentSongPath]) do
                obj:deleteSFXSource(sourceData.sfx)
            end
            sfx_sources[currentSongPath] = nil
        end
        
        current_playing_sources = nil
    end
    
    local sources = sfx_sources[songPath]
    
    if not sources then
        sources = prepareSongMultiSource(songPath, true)
        if not sources then
            return false
        end
        sfx_sources[songPath] = sources
    end
    
    pendingPlayback = {
        action = "play",
        songPath = songPath,
        sources = sources
    }
    syncFrameCounter = primingDelay
    
    return true
end

local function stopCurrentSongMultiSource()
    if current_playing_sources then
        for location, sourceData in pairs(current_playing_sources) do
            obj:cutSFX(sourceData.sfx)
        end
        
        current_playing_sources = nil
        currentSongPath = ""
        isPlaying = false
        isPausedByUser = false
        currentTime = 0
        songElapsedTime = 0
    end
end

local function pauseCurrentSongMultiSource()
    if current_playing_sources then
        print("DEBUG: Pausing music - current_playing_sources exists")
        local sourceCount = 0
        for location, sourceData in pairs(current_playing_sources) do
            sourceCount = sourceCount + 1
            print(string.format("DEBUG: Stopping and deleting SFX for location %s, handle: %s", location, tostring(sourceData.sfx)))
            if sourceData.sfx then
                obj:cutSFX(sourceData.sfx)
                obj:deleteSFXSource(sourceData.sfx)
                print(string.format("DEBUG: Deleted SFX source for %s", location))
            else
                print(string.format("DEBUG: WARNING - nil SFX handle for %s", location))
            end
        end
        print(string.format("DEBUG: Deleted %d sources total", sourceCount))
        
        if currentSongPath and sfx_sources[currentSongPath] then
            sfx_sources[currentSongPath] = nil
            print(string.format("DEBUG: Cleared cached sources for %s", currentSongPath))
        end
        
        isPlaying = false
        isPausedByUser = true
        currentTime = songElapsedTime
        print(string.format("DEBUG: Set isPlaying=false, isPausedByUser=true, currentTime=%.2f", currentTime))
    else
        print("DEBUG: WARNING - pauseCurrentSongMultiSource called but current_playing_sources is nil!")
    end
end

local function resumeCurrentSongMultiSource()
    if currentSongPath ~= "" and isPausedByUser and current_playing_sources then
        local sources = sfx_sources[currentSongPath]
        if not sources then
            sources = prepareSongMultiSource(currentSongPath, true)
            if not sources then
                return false
            end
            sfx_sources[currentSongPath] = sources
            current_playing_sources = sources
        else
            for location, sourceData in pairs(sources) do
                obj:playSFX(sourceData.sfx)
            end
            
            for location, sourceData in pairs(sources) do
                obj:cutSFX(sourceData.sfx)
            end
            
            updateEqualizerVolumes()
        end
        
        pendingPlayback = {
            action = "resume",
            songPath = currentSongPath,
            sources = current_playing_sources
        }
        syncFrameCounter = 1
        
        songElapsedTime = 0
        currentTime = 0
        electrics.values.currentSongTime = 0
        
        return true
    end
    return false
end

local function updateMasterVolume()
    if current_playing_sources then
        local effectiveMasterVolume = volumeControl.masterVolume * volumeControl.masterVolume
        
        for location, sourceData in pairs(current_playing_sources) do
            local baseVol = sourceData.eqBaseVolume or sourceData.baseVolume
            local effectiveVolume = baseVol * effectiveMasterVolume
            obj:setVolumePitch(sourceData.sfx, effectiveVolume, 1.0)
        end
    end
end

local function handleVolumeControl(dt)
    local volumeChanged = false
    local volumeUp = electrics.values.radio_vol_up or 0
    local volumeDown = electrics.values.radio_vol_down or 0
    
    if volumeUp == 1 then
        if volumeControl.masterVolume < volumeControl.maxVolume then
            volumeControl.masterVolume = math.min(
                volumeControl.masterVolume + volumeControl.volumeStep,
                volumeControl.maxVolume
            )
            volumeChanged = true
        end
    end
    
    if volumeDown == 1 then
        if volumeControl.masterVolume > volumeControl.minVolume then
            volumeControl.masterVolume = math.max(
                volumeControl.masterVolume - volumeControl.volumeStep,
                volumeControl.minVolume
            )
            volumeChanged = true
        end
    end
    
    if volumeChanged then
        updateMasterVolume()
        electrics.values.musicVolume = math.floor(volumeControl.masterVolume * 100)
    end
end

-- ========================================
-- QUEUE SYSTEM FUNCTIONS
-- ========================================

local function shuffleArray(array)
    local shuffled = {}
    local indices = {}
    
    for i = 1, #array do
        indices[i] = i
    end
    
    for i = #indices, 2, -1 do
        local j = math.random(1, i)
        indices[i], indices[j] = indices[j], indices[i]
    end
    
    for i = 1, #indices do
        shuffled[i] = array[indices[i]]
    end
    
    return shuffled
end

local function startShuffle()
    local playlistName = selectedPlaylistInfo.name or "Unknown Playlist"
    
    if not isQueueInitialized or #currentQueue == 0 or currentQueuePlaylistName ~= playlistName then
        if #selectedPlaylistSongs > 0 then
            currentQueue = {}
            originalQueue = {}
            
            for i, song in ipairs(selectedPlaylistSongs) do
                currentQueue[i] = {
                    name = song.name or song.title or "Unknown Song",
                    artist = song.artist or song.artistName or "",
                    id = song.id or song.songId or tostring(i),
                    albumArt = song.albumArt or song.album_art or song.cover or "music_note_album_default.png",
                    duration = song.duration or song.length or 0,
                    playlistName = playlistName,
                    songPath = song.songPath or song.file or ""
                }
            end
            
            currentQueueIndex = 0
            currentQueuePlaylistName = playlistName
            isQueueInitialized = true
            isShuffleMode = false
        else
            return false
        end
    end
    
    isShuffleMode = true
    
    math.randomseed(os.time())
    
    if #originalQueue == 0 then
        for i, song in ipairs(currentQueue) do
            originalQueue[i] = song
        end
    end
    
    local shuffledQueue = shuffleArray(currentQueue)
    currentQueue = shuffledQueue
    
    local startIndex = 1
    
    if startIndex >= 1 and startIndex <= #currentQueue then
        currentQueueIndex = startIndex
        local song = currentQueue[startIndex]
        
        if song then
            currentSongData = song
            isFavorited = isSongSaved(song.id)
            
            electrics.values.currentSongJson = jsonEncodeSong(song)
            electrics.values.currentSongReady = 1
            electrics.values.currentSongId = song.id
            electrics.values.currentSongName = song.name
            electrics.values.currentSongArtist = song.artist
            electrics.values.currentSongAlbumArt = song.albumArt
            electrics.values.currentSongDuration = song.duration
            electrics.values.currentSongTime = 0
            electrics.values.currentSongIsPlaying = 0
            electrics.values.currentSongIsFavorited = isFavorited and 1 or 0
            
            currentSongAlbumArt = song.albumArt
            actualSongDuration = song.duration
            currentTime = 0
            songElapsedTime = 0
            
            if song.songPath then
                if playSongMultiSource(song.songPath) then
                    electrics.values.currentSongIsPlaying = 1
                    return true
                else
                    return false
                end
            else
                return false
            end
        else
            return false
        end
    else
        return false
    end
end

local function initializeQueue(songs, playlistName)
    currentQueue = {}
    originalQueue = {}
    
    for i, song in ipairs(songs) do
        currentQueue[i] = {
            name = song.name or song.title or "Unknown Song",
            artist = song.artist or song.artistName or "",
            id = song.id or song.songId or tostring(i),
            albumArt = song.albumArt or song.album_art or song.cover or "music_note_album_default.png",
            duration = song.duration or song.length or 0,
            playlistName = playlistName,
            songPath = song.songPath or song.file or ""
        }
    end
    
    currentQueueIndex = 0
    currentQueuePlaylistName = playlistName
    isQueueInitialized = true
    isShuffleMode = false
end

local function setCurrentSongFromQueue(index)
    if not isQueueInitialized or #currentQueue == 0 then
        return false
    end
    
    if index < 1 or index > #currentQueue then
        return false
    end
    
    currentQueueIndex = index
    local song = currentQueue[index]
    
    if song then
        currentSongData = song
        isFavorited = isSongSaved(song.id)
        
        electrics.values.currentSongJson = jsonEncodeSong(song)
        electrics.values.currentSongReady = 1
        electrics.values.currentSongId = song.id
        electrics.values.currentSongName = song.name
        electrics.values.currentSongArtist = song.artist
        electrics.values.currentSongAlbumArt = song.albumArt
        electrics.values.currentSongDuration = song.duration
        electrics.values.currentSongTime = currentTime
        electrics.values.currentSongIsPlaying = isPlaying and 1 or 0
        electrics.values.currentSongIsFavorited = isFavorited and 1 or 0
        
        currentSongAlbumArt = song.albumArt
        actualSongDuration = song.duration
        
        return true
    end
    
    return false
end

local function queueNext()
    if not isQueueInitialized or #currentQueue == 0 then
        return false
    end
    
    if currentQueueIndex < #currentQueue then
        local newIndex = currentQueueIndex + 1
        
        stopCurrentSongMultiSource()
        
        if setCurrentSongFromQueue(newIndex) then
            currentTime = 0
            songElapsedTime = 0
            electrics.values.currentSongTime = 0
            
            local song = currentQueue[newIndex]
            if song and song.songPath then
                if playSongMultiSource(song.songPath) then
                    electrics.values.currentSongIsPlaying = 1
                    return true
                end
            end
        end
    end
    
    return false
end

local function queuePrevious()
    if not isQueueInitialized or #currentQueue == 0 then
        return false
    end
    
    if currentTime >= 5.0 then
        local song = currentQueue[currentQueueIndex]
        if song and song.songPath then
            stopCurrentSongMultiSource()
            
            currentTime = 0
            songElapsedTime = 0
            electrics.values.currentSongTime = 0
            
            if playSongMultiSource(song.songPath) then
                electrics.values.currentSongIsPlaying = 1
                return true
            end
        end
        return false
    end
    
    if currentQueueIndex > 1 then
        local newIndex = currentQueueIndex - 1
        
        stopCurrentSongMultiSource()
        
        if setCurrentSongFromQueue(newIndex) then
            currentTime = 0
            songElapsedTime = 0
            electrics.values.currentSongTime = 0
            
            local song = currentQueue[newIndex]
            if song and song.songPath then
                if playSongMultiSource(song.songPath) then
                    electrics.values.currentSongIsPlaying = 1
                    return true
                end
            end
        end
    end
    
    return false
end

local function queueSelectSongById(songId)
    local songPlaylistName = nil
    local foundSong = nil
    local foundIndex = nil
    
    for i, song in ipairs(selectedPlaylistSongs) do
        if song.id == songId then
            foundSong = song
            foundIndex = i
            songPlaylistName = selectedPlaylistInfo.name or "Unknown Playlist"
            break
        end
    end
    
    if not foundSong then
        return false
    end
    
    if not isQueueInitialized or currentQueuePlaylistName ~= songPlaylistName then
        initializeQueue(selectedPlaylistSongs, songPlaylistName)
    end
    
    if isShuffleMode and #originalQueue > 0 then
        currentQueue = {}
        for i, song in ipairs(originalQueue) do
            currentQueue[i] = song
        end
        isShuffleMode = false
    end
    
    for i, song in ipairs(currentQueue) do
        if song.id == songId then
            stopCurrentSongMultiSource()
            
            if setCurrentSongFromQueue(i) then
                currentTime = 0
                songElapsedTime = 0
                electrics.values.currentSongTime = 0
                
                if song.songPath then
                    if playSongMultiSource(song.songPath) then
                        electrics.values.currentSongIsPlaying = 1
                        return true
                    end
                end
            end
        end
    end
    
    return false
end

-- ========================================
-- MUSIC CONTROL COMMAND HANDLER
-- ========================================

local function handleMusicControlCommand(command)
    if command == "selectSong" then
        local selectedSongId = electrics.values.selectedSongId or ""
        
        if selectedSongId ~= "" then
            if selectedSongId == "shuffle" or selectedSongId == "shuffleAll" then
                startShuffle()
            else
                queueSelectSongById(selectedSongId)
            end
        end
        
    elseif command == "togglePlayPause" then
        print(string.format("DEBUG: togglePlayPause command - isPlaying=%s, isPausedByUser=%s", tostring(isPlaying), tostring(isPausedByUser)))
        if isPlaying then
            print("DEBUG: Calling pauseCurrentSongMultiSource()")
            pauseCurrentSongMultiSource()
            electrics.values.currentSongIsPlaying = 0
            print("DEBUG: Set currentSongIsPlaying = 0")
        else
            print("DEBUG: Calling resumeCurrentSongMultiSource()")
            resumeCurrentSongMultiSource()
            electrics.values.currentSongIsPlaying = 1
            print("DEBUG: Set currentSongIsPlaying = 1")
        end
        
    elseif command == "previousSong" then
        queuePrevious()
        
    elseif command == "nextSong" then
        queueNext()
        
    elseif command == "toggleFavorite" then
        if currentSongData and currentSongData.id then
            if isFavorited then
                removeSavedSong(currentSongData.id)
                isFavorited = false
            else
                local songToSave = {}
                for k, v in pairs(currentSongData) do
                    songToSave[k] = v
                end
                addSavedSong(songToSave)
                isFavorited = true
            end
            electrics.values.currentSongIsFavorited = isFavorited and 1 or 0
        end
        
    elseif command == "seekTo" then
        local seekTime = electrics.values.seekToTime or 0
        if current_playing_sources and actualSongDuration > 0 then
            if seekTime >= 0 and seekTime <= actualSongDuration then
                songElapsedTime = seekTime
                currentTime = seekTime
                electrics.values.currentSongTime = currentTime
            end
        end
    end
end

-- ========================================
-- PLAYLIST SYSTEM FUNCTIONS
-- ========================================

local function loadPlaylistSongs(playlistId)
    if playlistId == lastRequestedPlaylistId then
        return
    end
    
    lastRequestedPlaylistId = playlistId
    
    local now = os.time()
    if playlistCache[playlistId] and cacheTimestamps[playlistId] and 
       (now - cacheTimestamps[playlistId]) < cacheTimeout then
        selectedPlaylistSongs = playlistCache[playlistId]
        
        local playlistName = "Songs"
        if playlistId == "favorites" or playlistId == "savedsongs" then
            playlistName = "Saved Songs"
        elseif playlistId == "allsongs" then
            playlistName = "All Songs"
        else
            for _, playlist in ipairs(availablePlaylists) do
                if playlist.id == playlistId then
                    playlistName = playlist.name
                    break
                end
            end
        end
        
        selectedPlaylistInfo = {name = playlistName}
        
        local songsJson = jsonEncode(selectedPlaylistSongs)
        
        electrics.values.playlistSongsJson = songsJson
        electrics.values.playlistSongsReady = 1
        electrics.values.selectedPlaylistId = playlistId
        electrics.values.selectedPlaylistName = playlistName
        electrics.values.playlistSongsCount = #selectedPlaylistSongs
        
        if #selectedPlaylistSongs > 0 then
            local firstSong = selectedPlaylistSongs[1]
            currentSongAlbumArt = firstSong.albumArt or firstSong.album_art or firstSong.cover or "music_note_album_default.png"
        else
            currentSongAlbumArt = (playlistId == "favorites" or playlistId == "savedsongs") and "favorite_songs.png" or "default_playlist.png"
        end
        electrics.values.currentSongAlbumArt = currentSongAlbumArt
        
        return
    end
    
    if playlistId == "favorites" or playlistId == "savedsongs" then
        local enhancedSavedSongs = {}
        for i, savedSong in ipairs(savedSongs.songs) do
            enhancedSavedSongs[i] = {
                id = savedSong.id,
                name = savedSong.name,
                artist = savedSong.artist,
                albumArt = savedSong.albumArt,
                duration = savedSong.duration,
                songPath = savedSong.songPath,
                playlistName = savedSong.playlistName
            }
        end
        
        selectedPlaylistSongs = enhancedSavedSongs
        selectedPlaylistInfo = {name = "Saved Songs"}
        
        playlistCache[playlistId] = enhancedSavedSongs
        cacheTimestamps[playlistId] = now
        
        local songsJson = jsonEncode(selectedPlaylistSongs)
        
        electrics.values.playlistSongsJson = songsJson
        electrics.values.playlistSongsReady = 1
        electrics.values.selectedPlaylistId = playlistId
        electrics.values.selectedPlaylistName = "Saved Songs"
        electrics.values.playlistSongsCount = #selectedPlaylistSongs
        
        if #selectedPlaylistSongs > 0 then
            local firstSong = selectedPlaylistSongs[1]
            currentSongAlbumArt = firstSong.albumArt or firstSong.album_art or firstSong.cover or "music_note_album_default.png"
        else
            currentSongAlbumArt = "favorite_songs.png"
        end
        electrics.values.currentSongAlbumArt = currentSongAlbumArt
        
        return
    end
    
    local success, songs = pcall(function()
        return playlistConfig.getPlaylistSongs(playlistId)
    end)
    
    if success and songs then
        local processedSongs = {}
        for i, song in ipairs(songs) do
            processedSongs[i] = {
                name = song.name or song.title or "Unknown Song",
                artist = song.artist or song.artistName or "",
                id = song.id or song.songId or tostring(i),
                albumArt = song.albumArt or song.album_art or song.cover or "music_note_album_default.png",
                duration = song.duration or song.length or 0,
                songPath = song.songPath or song.file or "",
                playlistName = playlistName
            }
        end
        
        selectedPlaylistSongs = processedSongs
        
        playlistCache[playlistId] = processedSongs
        cacheTimestamps[playlistId] = now
        
        local playlistInfo = nil
        for _, playlist in ipairs(availablePlaylists) do
            if playlist.id == playlistId then
                playlistInfo = playlist
                break
            end
        end
        selectedPlaylistInfo = playlistInfo or {name = "Unknown Playlist"}
        
        local songsJson = jsonEncode(selectedPlaylistSongs)
        
        electrics.values.playlistSongsJson = songsJson
        electrics.values.playlistSongsReady = 1
        electrics.values.selectedPlaylistId = playlistId
        electrics.values.selectedPlaylistName = selectedPlaylistInfo.name
        electrics.values.playlistSongsCount = #selectedPlaylistSongs
        
        if #selectedPlaylistSongs > 0 then
            local firstSong = selectedPlaylistSongs[1]
            currentSongAlbumArt = firstSong.albumArt or firstSong.album_art or firstSong.cover or "music_note_album_default.png"
        else
            currentSongAlbumArt = "default_playlist.png"
        end
        electrics.values.currentSongAlbumArt = currentSongAlbumArt
        
    else
        selectedPlaylistSongs = {}
        selectedPlaylistInfo = {}
    end
end

local function loadPlaylists()
    local success, playlists = pcall(function()
        return playlistConfig.getAllPlaylists()
    end)
    
    if success and playlists then
        local filteredPlaylists = {}
        for _, playlist in ipairs(playlists) do
            if playlist.id ~= "allsongs" then
                local playlistData = {
                    id = playlist.id,
                    name = playlist.name,
                    coverImage = playlist.coverImage or "default_playlist.png",
                    songCount = playlist.songCount or 0
                }
                table.insert(filteredPlaylists, playlistData)
            end
        end
        
        availablePlaylists = filteredPlaylists
        totalPlaylistCount = #availablePlaylists
        
        local playlistsJson = jsonEncodePlaylists(availablePlaylists)
        electrics.values.playlistsJson = playlistsJson
        electrics.values.playlistsReady = 1
        electrics.values.playlistCount = totalPlaylistCount
        
    else
        availablePlaylists = {}
        totalPlaylistCount = 0
        
        electrics.values.playlistsJson = "[]"
        electrics.values.playlistsReady = 0
        electrics.values.playlistCount = 0
    end
end

-- ========================================
-- INITIALIZATION AND MAIN UPDATE
-- ========================================

local function init(jbeamData)
    if not isInitialized then
        isInitialized = true
        
        loadSavedSongs()
        
        loadMusicSettings()
        applyMusicSettings()
        musicSettings.initialized = true
        musicSettings.startTime = os.clock()
        
        electrics.values.playlistsJson = "[]"
        electrics.values.playlistsReady = 0
        electrics.values.playlistCount = 0
        electrics.values.playlistSongsJson = "[]"
        electrics.values.playlistSongsReady = 0
        electrics.values.selectedPlaylistId = ""
        electrics.values.selectedPlaylistName = ""
        electrics.values.playlistSongsCount = 0
        electrics.values.currentSongAlbumArt = "default_playlist.png"
        electrics.values.currentSongJson = "{}"
        electrics.values.currentSongReady = 0
        electrics.values.currentSongId = ""
        electrics.values.currentSongName = ""
        electrics.values.currentSongArtist = ""
        electrics.values.currentSongDuration = 0
        electrics.values.currentSongTime = 0
        electrics.values.currentSongIsPlaying = 0
        electrics.values.currentSongIsFavorited = 0
        electrics.values.musicControlCommand = ""
        electrics.values.selectedSongId = ""
        electrics.values.selectedSongJson = ""
        electrics.values.favoriteSongId = ""
        electrics.values.seekToTime = 0
        electrics.values.requestPlaylistSongs = ""
        
        electrics.values.musicVolume = math.floor(volumeControl.masterVolume * 100)
        
        electrics.values.equalizerFade = equalizerControl.fade
        electrics.values.equalizerBalance = equalizerControl.balance
        electrics.values.equalizerBass = equalizerControl.bass
        
        electrics.values.requestPreviewPlaylistSongs = ""
        electrics.values.previewPlaylistSongsJson = "[]"
        electrics.values.previewPlaylistSongsReady = 0
        electrics.values.previewPlaylistId = ""
        electrics.values.previewPlaylistName = ""
        electrics.values.previewPlaylistCover = "default_playlist.png"
        
        sfx_sources = {}
        current_playing_sources = nil
        currentSongPath = ""
        songElapsedTime = 0
        musicSystemActive = false
        
        loadPlaylists()
    end
end

local function updateGFX(dt)
    local systemActiveInput = electrics.values.musicSystemActiveInput or 0
    if systemActiveInput == 1 and not musicSystemActive then
        musicSystemActive = true
    elseif systemActiveInput == 0 and musicSystemActive then
        musicSystemActive = false
        stopCurrentSongMultiSource()
    end
    
    if not musicSettings.canSave and musicSettings.startTime > 0 then
        if os.clock() - musicSettings.startTime > 1.0 then
            musicSettings.canSave = true
            if musicSettings.dirty then
                saveMusicSettings()
            end
        end
    end
    
    if pendingPlayback and syncFrameCounter > 0 then
        syncFrameCounter = syncFrameCounter - 1
        print(string.format("DEBUG: syncFrameCounter=%d, pendingAction=%s", syncFrameCounter, pendingPlayback.action))
        if syncFrameCounter == 0 then
            executeSyncPlayback()
        end
    end
    
    handleVolumeControl(dt)
    
    local currentFade = electrics.values.equalizerFade
    local currentBalance = electrics.values.equalizerBalance
    local currentBass = electrics.values.equalizerBass
    
    if currentFade and currentBalance then
        if currentFade ~= equalizerControl.fade or 
           currentBalance ~= equalizerControl.balance or
           (currentBass and currentBass ~= equalizerControl.bass) then
            
            equalizerControl.fade = currentFade
            equalizerControl.balance = currentBalance
            equalizerControl.bass = currentBass or 50
            
            updateEqualizerVolumes()
        end
    end
    
    if musicSettings.canSave then
        monitorMusicSettings()
    end
    
    updateTimer = updateTimer + dt
    
    if updateTimer > invFPS then
        updateTimer = 0
        
        local requestedPlaylistId = electrics.values.requestPlaylistSongs or ""
        if requestedPlaylistId ~= "" and requestedPlaylistId ~= lastRequestedPlaylistId then
            loadPlaylistSongs(requestedPlaylistId)
            electrics.values.requestPlaylistSongs = ""
        end
        
        local requestedPreviewId = electrics.values.requestPreviewPlaylistSongs or ""
        if requestedPreviewId ~= "" and requestedPreviewId ~= lastRequestedPreviewPlaylistId then
            lastRequestedPreviewPlaylistId = requestedPreviewId
            
            local playlistCover = "default_playlist.png"
            
            local success, playlistInfo = pcall(function()
                return playlistConfig.getPlaylistInfo(requestedPreviewId)
            end)
            
            if success and playlistInfo and playlistInfo.coverImage then
                playlistCover = playlistInfo.coverImage
            else
                for _, playlist in ipairs(availablePlaylists) do
                    if playlist.id == requestedPreviewId then
                        playlistCover = playlist.coverImage or "default_playlist.png"
                        break
                    end
                end
            end
            
            electrics.values.previewPlaylistId = requestedPreviewId
            electrics.values.previewPlaylistCover = playlistCover
            electrics.values.previewPlaylistSongsReady = 1
            
            electrics.values.requestPreviewPlaylistSongs = ""
        end
        
        local musicControlCommand = electrics.values.musicControlCommand or ""
        if musicControlCommand ~= "" then
            handleMusicControlCommand(musicControlCommand)
            electrics.values.musicControlCommand = ""
            lastMusicControlCommand = musicControlCommand
        end
        
        if electrics.values.radio_up == 1 and electrics.values.nowPlayingActive == 1 then
            if not lastFavoriteButtonState then
                lastFavoriteButtonState = true
                if currentSongData and currentSongData.id then
                    if isFavorited then
                        removeSavedSong(currentSongData.id)
                        isFavorited = false
                    else
                        addSavedSong(currentSongData)
                        isFavorited = true
                    end
                    electrics.values.currentSongIsFavorited = isFavorited and 1 or 0
                end
            end
        else
            lastFavoriteButtonState = false
        end
        
        if savedSongs.dirty then
            saveSavedSongs()
        end
    end
    
    if isPlaying and current_playing_sources and not isPausedByUser then
        songElapsedTime = songElapsedTime + dt
        currentTime = songElapsedTime
        
        if actualSongDuration > 0 and currentTime >= actualSongDuration then
            currentTime = actualSongDuration
            stopCurrentSongMultiSource()
            queueNext()
        end
        
        electrics.values.currentSongTime = currentTime
    end
    
    electrics.values.currentSongAlbumArt = currentSongAlbumArt
end

local function onReset()
    if pendingPlayback then
        pendingPlayback = nil
        syncFrameCounter = 0
    end
    
    if musicSettings.initialized then
        applyMusicSettings()
        if current_playing_sources then
            updateMasterVolume()
            updateEqualizerVolumes()
        end
    end
    
    for songPath, sources in pairs(sfx_sources) do
        if sources ~= current_playing_sources then
            for location, sourceData in pairs(sources) do
                obj:cutSFX(sourceData.sfx)
                obj:deleteSFXSource(sourceData.sfx)
            end
            sfx_sources[songPath] = nil
        end
    end
end

local function onExtensionUnloaded()
    if savedSongs.dirty then
        saveSavedSongs()
    end
    
    if musicSettings.dirty then
        musicSettings.canSave = true
        saveMusicSettings()
    end
end

M.init = init
M.updateGFX = updateGFX
M.onReset = onReset
M.onExtensionUnloaded = onExtensionUnloaded

return M