angular.module('infotainmentScreen')
.controller('SystemSettingsController', function($scope) {
  // Options with electric keys - checked state will be determined by electric values
  var defaultOptions = [
    { name: "Save Changes", disabled: false, hasCheckbox: true, electricKey: "saveChangesEnabled" }
  ];

  $scope.systemSettingsOptions = angular.copy(defaultOptions);
  $scope.selectedSystemSettingsIndex = 0;
  $scope.systemSettingsScrollOffset = 0;

  // Current electric values cache
  var currentElectricValues = {};

  function writeElectricValue(key, value) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.${key} = ${value}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
    } else if (window.bngApi && window.bngApi.engineLua) {
      const luaCommand = `
        local veh = be:getPlayerVehicle(0)
        if veh then
          veh:queueLuaCommand('if electrics and electrics.values then electrics.values.${key} = ${value} end')
        end
      `;
      
      window.bngApi.engineLua(luaCommand);
    }
  }

  // Update checkbox states based on current electric values
  function updateCheckboxStates() {
    $scope.systemSettingsOptions.forEach(function(option) {
      if (option.electricKey && currentElectricValues.hasOwnProperty(option.electricKey)) {
        option.checked = currentElectricValues[option.electricKey] === 1;
      }
    });
  }

  // Process electric data and update checkboxes
  function processElectricData(data) {
    if (!data || !data.electrics) return;
    
    var hasChanges = false;
    const e = data.electrics;
    
    // Check for changes in electric values we care about
    $scope.systemSettingsOptions.forEach(function(option) {
      if (option.electricKey && e.hasOwnProperty(option.electricKey)) {
        var newValue = e[option.electricKey];
        if (currentElectricValues[option.electricKey] !== newValue) {
          currentElectricValues[option.electricKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Update checkbox states if any values changed
    if (hasChanges) {
      updateCheckboxStates();
    }
  }

  // Listen for settings loaded from Lua (for initial load)
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('SystemSettings: Received settings:', settings);
    
    // Update our electric values cache
    Object.keys(settings).forEach(function(key) {
      currentElectricValues[key] = settings[key];
    });
    
    // Update checkbox states
    updateCheckboxStates();
  });

  // Listen for electric data updates (called from main controller)
  $scope.updateElectricData = function(data) {
    processElectricData(data);
  };

  $scope.getSystemSettingsScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedSystemSettingsIndex - $scope.systemSettingsScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getSystemSettingsScrollTransform = function() {
    return 'translateY(' + ($scope.systemSettingsScrollOffset * -40) + 'px)';
  };

  $scope.getSystemSettingsScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.systemSettingsOptions.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getSystemSettingsScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.systemSettingsOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getSystemSettingsScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.systemSettingsScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableSystemSettingsOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.systemSettingsOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.systemSettingsOptions.length; i++) {
        if (!$scope.systemSettingsOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateSystemSettingsMenu = function(direction) {
    if (direction === 'up' && $scope.selectedSystemSettingsIndex > 0) {
      const newIndex = $scope.findNextSelectableSystemSettingsOption('up', $scope.selectedSystemSettingsIndex);
      if (newIndex !== $scope.selectedSystemSettingsIndex) {
        $scope.selectedSystemSettingsIndex = newIndex;
        if ($scope.selectedSystemSettingsIndex < $scope.systemSettingsScrollOffset) {
          $scope.systemSettingsScrollOffset = $scope.selectedSystemSettingsIndex;
        }
      }
    } else if (direction === 'down' && $scope.selectedSystemSettingsIndex < $scope.systemSettingsOptions.length - 1) {
      const newIndex = $scope.findNextSelectableSystemSettingsOption('down', $scope.selectedSystemSettingsIndex);
      if (newIndex !== $scope.selectedSystemSettingsIndex) {
        $scope.selectedSystemSettingsIndex = newIndex;
        const visibleItems = 6;
        if ($scope.selectedSystemSettingsIndex >= $scope.systemSettingsScrollOffset + visibleItems) {
          $scope.systemSettingsScrollOffset = $scope.selectedSystemSettingsIndex - visibleItems + 1;
        }
      }
    }
  };

  $scope.$on('system-settings-navigate-up', function() {
    $scope.navigateSystemSettingsMenu('up');
  });

  $scope.$on('system-settings-navigate-down', function() {
    $scope.navigateSystemSettingsMenu('down');
  });

  $scope.$on('system-settings-select-option', function() {
    const selectedOption = $scope.systemSettingsOptions[$scope.selectedSystemSettingsIndex];
    if (selectedOption && selectedOption.hasCheckbox && !selectedOption.checkboxDisabled) {
      // Toggle the electric value directly
      const currentValue = currentElectricValues[selectedOption.electricKey] || 0;
      const newValue = currentValue === 1 ? 0 : 1;
      
      // Update our cache immediately for responsive UI
      currentElectricValues[selectedOption.electricKey] = newValue;
      selectedOption.checked = newValue === 1;
      
      // Send to Lua
      writeElectricValue(selectedOption.electricKey, newValue);
    }
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'systemSettings' && oldMenu === 'vehicle') {
      $scope.selectedSystemSettingsIndex = 0;
      $scope.systemSettingsScrollOffset = 0;
    }
  });
});
