angular.module('infotainmentScreen')
.controller('StarlightsController', function($scope, $interval) {
  // Options for starlights control
  var defaultOptions = [
    { name: "Enabled", disabled: false, hasCheckbox: true, electricKey: "starlightsEnabled", checked: true },
    { name: "Animation Mode", disabled: false, hasDropdown: true, dropdownValue: "Rainbow" },
    { name: "Speed", disabled: false, hasDropdown: true, dropdownValue: "Normal" },
    { name: "Brightness", disabled: false, hasDropdown: true, dropdownValue: "100%" },
    { name: "Color Selection", disabled: true, hidden: true }
  ];

  $scope.starlightsOptions = angular.copy(defaultOptions);
  $scope.selectedStarlightsIndex = 0;
  $scope.starlightsScrollOffset = 0;
  
  // Color picker state
  $scope.colorPickerMode = false;
  $scope.colorCursor = { x: 50, y: 50 }; // Start at center
  
  // Animation modes
  $scope.animationModes = ["Rainbow", "Pulse", "Static", "Twinkle", "Wave"];
  $scope.currentModeIndex = 0;
  
  // Modes that support color selection
  $scope.colorSupportedModes = [1, 2, 3, 4]; // Pulse, Static, Twinkle, Wave (not Rainbow)
  
  // Speed settings
  $scope.speedSettings = ["Slow", "Normal", "Fast"];
  $scope.currentSpeedIndex = 1;
  
  // Brightness levels
  $scope.brightnessLevels = ["25%", "50%", "75%", "100%"];
  $scope.currentBrightnessIndex = 3;

  // Current electric values cache
  var currentElectricValues = {
    starlightsEnabled: 1,
    starlightsMode: 0,
    starlightsSpeed: 1,
    starlightsBrightness: 100,
    starlightsColorR: 102,
    starlightsColorG: 126,
    starlightsColorB: 234,
    starlightsCursorX: 50,
    starlightsCursorY: 50
  };

  // Continuous movement for color picker
  let continuousMovement = {
    up: false,
    down: false,
    left: false,
    right: false,
    interval: null
  };

  function startContinuousMovement() {
    if (continuousMovement.interval) return;
    
    continuousMovement.interval = setInterval(function() {
      $scope.$evalAsync(() => {
        if ($scope.colorPickerMode) {
          if (continuousMovement.up) $scope.moveColorCursor('up');
          if (continuousMovement.down) $scope.moveColorCursor('down');
          if (continuousMovement.left) $scope.moveColorCursor('left');
          if (continuousMovement.right) $scope.moveColorCursor('right');
        }
      });
    }, 50); // Movement every 50ms while held
  }

  function stopContinuousMovement() {
    if (continuousMovement.interval) {
      clearInterval(continuousMovement.interval);
      continuousMovement.interval = null;
    }
    continuousMovement.up = false;
    continuousMovement.down = false;
    continuousMovement.left = false;
    continuousMovement.right = false;
  }

  // Throttled color updates
  let colorUpdateTimeout = null;

  // Function to trigger persistence save
  function triggerPersistenceSave() {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const timestamp = Date.now();
      window.beamng.sendActiveObjectLua(`
        if electrics and electrics.values then
          electrics.values.infotainmentSaveTrigger = ${timestamp}
        end
      `);
    }
  }

  // Write electric values
  function writeElectricValue(key, value) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.${key} = ${value}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
      currentElectricValues[key] = value;
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    }
  }

  // Write multiple electric values for color
  function setColorValues(r, g, b) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.starlightsColorR = ${r}
          electrics.values.starlightsColorG = ${g}
          electrics.values.starlightsColorB = ${b}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
      
      currentElectricValues.starlightsColorR = r;
      currentElectricValues.starlightsColorG = g;
      currentElectricValues.starlightsColorB = b;
      
      setTimeout(triggerPersistenceSave, 100);
    }
  }

  // Update starlight HTML with current settings
  function updateStarlightHTML() {
    // The starlight_headliner.lua will pick up the electric values directly
    console.log('Starlight settings updated - Mode:', $scope.animationModes[$scope.currentModeIndex], 
                'Speed:', $scope.currentSpeedIndex, 
                'Brightness:', $scope.brightnessLevels[$scope.currentBrightnessIndex],
                'Enabled:', currentElectricValues.starlightsEnabled);
  }

  // Update Color Selection option visibility
  function updateColorSelectionVisibility() {
    const colorOption = $scope.starlightsOptions[4]; // Color Selection is at index 4
    
    // Show/hide and enable/disable based on mode
    if ($scope.colorSupportedModes.indexOf($scope.currentModeIndex) !== -1) {
      colorOption.hidden = false;
      colorOption.disabled = false;
    } else {
      colorOption.hidden = true;
      colorOption.disabled = true;
      // If we're on the color selection option and it becomes hidden, move up
      if ($scope.selectedStarlightsIndex === 4) {
        $scope.selectedStarlightsIndex = 3;
      }
    }
  }

  // Update checkbox states based on current electric values
  function updateCheckboxStates() {
    $scope.starlightsOptions.forEach(function(option) {
      if (option.electricKey && currentElectricValues.hasOwnProperty(option.electricKey)) {
        option.checked = currentElectricValues[option.electricKey] === 1;
      }
    });
  }

  // Update dropdown displays based on current indices
  function updateDropdownDisplays() {
    // Update Animation Mode dropdown
    $scope.starlightsOptions[1].dropdownValue = $scope.animationModes[$scope.currentModeIndex];
    
    // Update Speed dropdown
    $scope.starlightsOptions[2].dropdownValue = $scope.speedSettings[$scope.currentSpeedIndex];
    
    // Update Brightness dropdown
    $scope.starlightsOptions[3].dropdownValue = $scope.brightnessLevels[$scope.currentBrightnessIndex];
  }

  // Process electric data
  function processElectricData(data) {
    if (!data || !data.electrics) return;
    
    var hasChanges = false;
    const e = data.electrics;
    
    // Check for enabled state changes
    if (e.hasOwnProperty('starlightsEnabled')) {
      var newValue = e.starlightsEnabled;
      if (currentElectricValues.starlightsEnabled !== newValue) {
        currentElectricValues.starlightsEnabled = newValue;
        hasChanges = true;
      }
    }
    
    // Check for mode changes
    if (e.hasOwnProperty('starlightsMode')) {
      var newMode = e.starlightsMode;
      if (currentElectricValues.starlightsMode !== newMode) {
        currentElectricValues.starlightsMode = newMode;
        $scope.currentModeIndex = Math.min(newMode, $scope.animationModes.length - 1);
        hasChanges = true;
      }
    }
    
    // Check for speed changes
    if (e.hasOwnProperty('starlightsSpeed')) {
      var newSpeed = e.starlightsSpeed;
      if (currentElectricValues.starlightsSpeed !== newSpeed) {
        currentElectricValues.starlightsSpeed = newSpeed;
        $scope.currentSpeedIndex = Math.min(newSpeed, $scope.speedSettings.length - 1);
        hasChanges = true;
      }
    }
    
    // Check for brightness changes
    if (e.hasOwnProperty('starlightsBrightness')) {
      var newBrightness = e.starlightsBrightness;
      if (currentElectricValues.starlightsBrightness !== newBrightness) {
        currentElectricValues.starlightsBrightness = newBrightness;
        // Convert brightness value to index
        if (newBrightness <= 25) $scope.currentBrightnessIndex = 0;
        else if (newBrightness <= 50) $scope.currentBrightnessIndex = 1;
        else if (newBrightness <= 75) $scope.currentBrightnessIndex = 2;
        else $scope.currentBrightnessIndex = 3;
        hasChanges = true;
      }
    }
    
    // Check for cursor position changes
    if (e.hasOwnProperty('starlightsCursorX')) {
      var newX = e.starlightsCursorX;
      if (currentElectricValues.starlightsCursorX !== newX) {
        currentElectricValues.starlightsCursorX = newX;
        $scope.colorCursor.x = newX;
        hasChanges = true;
      }
    }
    
    if (e.hasOwnProperty('starlightsCursorY')) {
      var newY = e.starlightsCursorY;
      if (currentElectricValues.starlightsCursorY !== newY) {
        currentElectricValues.starlightsCursorY = newY;
        $scope.colorCursor.y = newY;
        hasChanges = true;
      }
    }
    
    // Check for color value changes
    ['starlightsColorR', 'starlightsColorG', 'starlightsColorB'].forEach(function(colorKey) {
      if (e.hasOwnProperty(colorKey)) {
        var newValue = e[colorKey];
        if (currentElectricValues[colorKey] !== newValue) {
          currentElectricValues[colorKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    if (hasChanges) {
      updateCheckboxStates();
      updateDropdownDisplays();
      updateColorSelectionVisibility();
      updateStarlightHTML();
    }
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('Starlights: Received settings:', settings);
    
    // Update all values from settings
    Object.keys(settings).forEach(function(key) {
      if (settings[key] !== undefined && settings[key] !== null) {
        currentElectricValues[key] = settings[key];
      }
    });
    
    // Update mode index and dropdown
    if (settings.starlightsMode !== undefined) {
      $scope.currentModeIndex = Math.min(settings.starlightsMode, $scope.animationModes.length - 1);
      currentElectricValues.starlightsMode = settings.starlightsMode;
    }
    
    // Update speed index and dropdown
    if (settings.starlightsSpeed !== undefined) {
      $scope.currentSpeedIndex = Math.min(settings.starlightsSpeed, $scope.speedSettings.length - 1);
      currentElectricValues.starlightsSpeed = settings.starlightsSpeed;
    }
    
    // Update brightness index and dropdown
    if (settings.starlightsBrightness !== undefined) {
      const brightnessPercent = settings.starlightsBrightness;
      if (brightnessPercent <= 25) $scope.currentBrightnessIndex = 0;
      else if (brightnessPercent <= 50) $scope.currentBrightnessIndex = 1;
      else if (brightnessPercent <= 75) $scope.currentBrightnessIndex = 2;
      else $scope.currentBrightnessIndex = 3;
      currentElectricValues.starlightsBrightness = brightnessPercent;
    }
    
    // Restore cursor position if saved
    if (settings.starlightsCursorX !== undefined && settings.starlightsCursorY !== undefined) {
      $scope.colorCursor.x = settings.starlightsCursorX;
      $scope.colorCursor.y = settings.starlightsCursorY;
      currentElectricValues.starlightsCursorX = settings.starlightsCursorX;
      currentElectricValues.starlightsCursorY = settings.starlightsCursorY;
    }
    
    // Update the UI elements
    updateCheckboxStates();
    updateDropdownDisplays();
    updateColorSelectionVisibility();
    updateStarlightHTML();
    
    // Write values to electrics to ensure synchronization
    writeElectricValue('starlightsEnabled', currentElectricValues.starlightsEnabled);
    writeElectricValue('starlightsMode', currentElectricValues.starlightsMode);
    writeElectricValue('starlightsSpeed', currentElectricValues.starlightsSpeed);
    writeElectricValue('starlightsBrightness', currentElectricValues.starlightsBrightness);
    writeElectricValue('starlightsCursorX', currentElectricValues.starlightsCursorX);
    writeElectricValue('starlightsCursorY', currentElectricValues.starlightsCursorY);
    setColorValues(
      currentElectricValues.starlightsColorR,
      currentElectricValues.starlightsColorG,
      currentElectricValues.starlightsColorB
    );
    
  });

  // Listen for electric data updates
  $scope.updateElectricData = function(data) {
    processElectricData(data);
  };

  $scope.getCurrentStarlightsImage = function() {
    // Return default image - color picker is shown separately
    return 'starlight_preview.png';
  };

  $scope.showColorPicker = function() {
    // Show color picker if in color picker mode OR if hovering over Color Selection option
    return $scope.colorPickerMode || 
           ($scope.selectedStarlightsIndex === 4 && 
            !$scope.starlightsOptions[4].hidden && 
            !$scope.starlightsOptions[4].disabled);
  };

  $scope.shouldShowMenuSelection = function() {
    return !$scope.colorPickerMode;
  };

  $scope.getCursorClass = function() {
    return $scope.colorPickerMode ? 'active' : 'inactive';
  };



  // Color picker functions
  $scope.getColorAtCursor = function() {
    if (!$scope.colorPickerMode && $scope.selectedStarlightsIndex !== 4) return;
    
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    const img = new Image();
    
    img.onload = function() {
      canvas.width = img.width;
      canvas.height = img.height;
      ctx.drawImage(img, 0, 0);
      
      const x = Math.floor((img.width * $scope.colorCursor.x) / 100);
      const y = Math.floor((img.height * $scope.colorCursor.y) / 100);
      
      const clampedX = Math.max(0, Math.min(img.width - 1, x));
      const clampedY = Math.max(0, Math.min(img.height - 1, y));
      
      const imageData = ctx.getImageData(clampedX, clampedY, 1, 1);
      const [r, g, b, a] = imageData.data;
      
      if (a > 0) {
        console.log(`Starlight color selected: RGB(${r}, ${g}, ${b})`);
        setColorValues(r, g, b);
      }
    };
    
    img.src = '/vehicles/RLA_A90/infotainment_screen/icons/color_wheel.png';
  };

  $scope.moveColorCursor = function(direction) {
    if (!$scope.colorPickerMode) return;
    
    const step = 2;
    
    switch(direction) {
      case 'up':
        $scope.colorCursor.y = Math.max(0, $scope.colorCursor.y - step);
        break;
      case 'down':
        $scope.colorCursor.y = Math.min(100, $scope.colorCursor.y + step);
        break;
      case 'left':
        $scope.colorCursor.x = Math.max(0, $scope.colorCursor.x - step);
        break;
      case 'right':
        $scope.colorCursor.x = Math.min(100, $scope.colorCursor.x + step);
        break;
    }
    
    // Save cursor position to electric values
    currentElectricValues.starlightsCursorX = $scope.colorCursor.x;
    currentElectricValues.starlightsCursorY = $scope.colorCursor.y;
    writeElectricValue('starlightsCursorX', $scope.colorCursor.x);
    writeElectricValue('starlightsCursorY', $scope.colorCursor.y);
    
    if (colorUpdateTimeout) {
      clearTimeout(colorUpdateTimeout);
    }
    
    colorUpdateTimeout = setTimeout(function() {
      $scope.getColorAtCursor();
    }, 50);
  };

  // Navigation functions
  $scope.getStarlightsScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedStarlightsIndex - $scope.starlightsScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getStarlightsScrollTransform = function() {
    return 'translateY(' + ($scope.starlightsScrollOffset * -40) + 'px)';
  };

  $scope.getStarlightsScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.starlightsOptions.filter(opt => !opt.hidden).length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getStarlightsScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.starlightsOptions.filter(opt => !opt.hidden).length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getStarlightsScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.starlightsScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableStarlightsOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.starlightsOptions[i].disabled && !$scope.starlightsOptions[i].hidden) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.starlightsOptions.length; i++) {
        if (!$scope.starlightsOptions[i].disabled && !$scope.starlightsOptions[i].hidden) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateStarlightsMenu = function(direction) {
    if ($scope.colorPickerMode) return;
    
    if (direction === 'up' && $scope.selectedStarlightsIndex > 0) {
      const newIndex = $scope.findNextSelectableStarlightsOption('up', $scope.selectedStarlightsIndex);
      if (newIndex !== $scope.selectedStarlightsIndex) {
        $scope.selectedStarlightsIndex = newIndex;
        if ($scope.selectedStarlightsIndex < $scope.starlightsScrollOffset) {
          $scope.starlightsScrollOffset = $scope.selectedStarlightsIndex;
        }
        
        // Don't try to position cursor when hovering - color is already saved
        if ($scope.selectedStarlightsIndex === 4 && !$scope.starlightsOptions[4].hidden) {
          $scope.getColorAtCursor(); // Preview the current saved color
        }
      }
    } else if (direction === 'down' && $scope.selectedStarlightsIndex < $scope.starlightsOptions.length - 1) {
      const newIndex = $scope.findNextSelectableStarlightsOption('down', $scope.selectedStarlightsIndex);
      if (newIndex !== $scope.selectedStarlightsIndex) {
        $scope.selectedStarlightsIndex = newIndex;
        const visibleItems = 6;
        if ($scope.selectedStarlightsIndex >= $scope.starlightsScrollOffset + visibleItems) {
          $scope.starlightsScrollOffset = $scope.selectedStarlightsIndex - visibleItems + 1;
        }
        
        // Don't try to position cursor when hovering - color is already saved
        if ($scope.selectedStarlightsIndex === 4 && !$scope.starlightsOptions[4].hidden) {
          $scope.getColorAtCursor(); // Preview the current saved color
        }
      }
    }
  };

  // Cycle through dropdown values
  function cycleDropdownValue(optionIndex, direction) {
    const option = $scope.starlightsOptions[optionIndex];
    
    if (optionIndex === 1) { // Animation Mode
      if (direction === 'right') {
        $scope.currentModeIndex = ($scope.currentModeIndex + 1) % $scope.animationModes.length;
      } else {
        $scope.currentModeIndex = ($scope.currentModeIndex - 1 + $scope.animationModes.length) % $scope.animationModes.length;
      }
      option.dropdownValue = $scope.animationModes[$scope.currentModeIndex];
      writeElectricValue('starlightsMode', $scope.currentModeIndex);
      
      // Update Color Selection option visibility
      updateColorSelectionVisibility();
    } else if (optionIndex === 2) { // Speed
      if (direction === 'right') {
        $scope.currentSpeedIndex = ($scope.currentSpeedIndex + 1) % $scope.speedSettings.length;
      } else {
        $scope.currentSpeedIndex = ($scope.currentSpeedIndex - 1 + $scope.speedSettings.length) % $scope.speedSettings.length;
      }
      option.dropdownValue = $scope.speedSettings[$scope.currentSpeedIndex];
      writeElectricValue('starlightsSpeed', $scope.currentSpeedIndex);
    } else if (optionIndex === 3) { // Brightness
      if (direction === 'right') {
        $scope.currentBrightnessIndex = ($scope.currentBrightnessIndex + 1) % $scope.brightnessLevels.length;
      } else {
        $scope.currentBrightnessIndex = ($scope.currentBrightnessIndex - 1 + $scope.brightnessLevels.length) % $scope.brightnessLevels.length;
      }
      option.dropdownValue = $scope.brightnessLevels[$scope.currentBrightnessIndex];
      const brightnessValue = parseInt($scope.brightnessLevels[$scope.currentBrightnessIndex]);
      writeElectricValue('starlightsBrightness', brightnessValue);
    }
    
    updateStarlightHTML();
  }

  // Event handlers
  $scope.$on('starlights-navigate-up', function() {
    if ($scope.colorPickerMode) {
      continuousMovement.up = true;
      startContinuousMovement();
    } else {
      $scope.navigateStarlightsMenu('up');
    }
  });

  $scope.$on('starlights-navigate-down', function() {
    if ($scope.colorPickerMode) {
      continuousMovement.down = true;
      startContinuousMovement();
    } else {
      $scope.navigateStarlightsMenu('down');
    }
  });

  $scope.$on('starlights-navigate-left', function() {
    if ($scope.colorPickerMode) {
      continuousMovement.left = true;
      startContinuousMovement();
    } else {
      const selectedOption = $scope.starlightsOptions[$scope.selectedStarlightsIndex];
      if (selectedOption && selectedOption.hasDropdown) {
        cycleDropdownValue($scope.selectedStarlightsIndex, 'left');
      }
    }
  });

  $scope.$on('starlights-navigate-right', function() {
    if ($scope.colorPickerMode) {
      continuousMovement.right = true;
      startContinuousMovement();
    } else {
      const selectedOption = $scope.starlightsOptions[$scope.selectedStarlightsIndex];
      if (selectedOption && selectedOption.hasDropdown) {
        cycleDropdownValue($scope.selectedStarlightsIndex, 'right');
      }
    }
  });

  // Stop movement when key is released
  $scope.$on('starlights-stop-up', function() {
    continuousMovement.up = false;
    if (!continuousMovement.down && !continuousMovement.left && !continuousMovement.right) {
      stopContinuousMovement();
    }
  });

  $scope.$on('starlights-stop-down', function() {
    continuousMovement.down = false;
    if (!continuousMovement.up && !continuousMovement.left && !continuousMovement.right) {
      stopContinuousMovement();
    }
  });

  $scope.$on('starlights-stop-left', function() {
    continuousMovement.left = false;
    if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.right) {
      stopContinuousMovement();
    }
  });

  $scope.$on('starlights-stop-right', function() {
    continuousMovement.right = false;
    if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.left) {
      stopContinuousMovement();
    }
  });

  $scope.$on('starlights-select-option', function() {
    const selectedOption = $scope.starlightsOptions[$scope.selectedStarlightsIndex];
    if (selectedOption && selectedOption.hasCheckbox && !selectedOption.checkboxDisabled) {
      // Toggle the electric value
      const currentValue = currentElectricValues[selectedOption.electricKey] || 0;
      const newValue = currentValue === 1 ? 0 : 1;
      
      currentElectricValues[selectedOption.electricKey] = newValue;
      selectedOption.checked = newValue === 1;
      
      writeElectricValue(selectedOption.electricKey, newValue);
      updateStarlightHTML();
    } else if (selectedOption && selectedOption.name === "Color Selection" && !selectedOption.disabled && !selectedOption.hidden) {
      // Enter color picker mode
      $scope.colorPickerMode = true;
      $scope.getColorAtCursor();
    }
  });

  $scope.$on('starlights-back', function() {
    if ($scope.colorPickerMode) {
      $scope.colorPickerMode = false;
    } else {
      $scope.$parent.currentMenu = 'exteriorLighting';
    }
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'starlights' && oldMenu === 'exteriorLighting') {
      $scope.selectedStarlightsIndex = 0;
      $scope.starlightsScrollOffset = 0;
      $scope.colorPickerMode = false;
      
      // Update Color Selection option visibility based on current mode
      updateColorSelectionVisibility();
      
      // Update dropdown displays to ensure they show correct values
      updateDropdownDisplays();
      
      // Initialize starlight display
      updateStarlightHTML();
    }
  });
});