angular.module('infotainmentScreen')
.controller('SongsController', function($scope, $timeout) {
  // Initialize with Shuffle All as the first option
  $scope.songOptions = [
    { name: "Shuffle All", icon_image: "shuffle.png", isShuffle: true }
  ];

  $scope.selectedSongIndex = 0;
  $scope.songScrollOffset = 0;
  $scope.hasSongs = false;
  $scope.noSongsMessage = "No songs found";
  $scope.playlistName = "Songs";

  // Function to process song data - Clean version with All Songs and Saved Songs support
  $scope.processSongData = function(songs, playlistName) {
    // Reset to just shuffle
    $scope.songOptions = [
      { name: "Shuffle All", icon_image: "shuffle.png", isShuffle: true }
    ];
    
    // Set playlist name with special handling for "All Songs" and "Saved Songs"
    if (playlistName === "All Songs") {
      $scope.playlistName = "All Songs";
    } else if (playlistName === "Saved Songs") {
      $scope.playlistName = "Saved Songs";
      // Update no songs message for saved songs
      $scope.noSongsMessage = "No saved songs yet";
    } else if (playlistName === "My Favorites") {
      $scope.playlistName = "My Favorites";
      $scope.noSongsMessage = "No favorite songs yet";
    } else {
      $scope.playlistName = playlistName || "Songs";
      $scope.noSongsMessage = "No songs found";
    }
    
    // Add dynamic songs
    if (songs && songs.length > 0) {
      songs.forEach(function(song) {
        if (song.name) {
          $scope.songOptions.push({
            name: song.name,
            artist: song.artist || "",
            duration: song.duration || 0,
            albumArt: song.albumArt || "default_playlist.png",
            icon_image: "all_music.png",
            songId: song.id,
            disabled: false,
            songData: song // Store the full song data for later use
          });
        }
      });
    }
    
    // If we only have shuffle (no other songs loaded), show no songs message
    $scope.hasSongs = $scope.songOptions.length > 1;
  };

  // Function to receive electric data updates
  $scope.updateElectricData = function(data) {
    if (data && data.electrics) {
      // Check for song data in the electric fields
      if (data.electrics.playlistSongsReady === 1 && data.electrics.playlistSongsJson) {
        try {
          const songs = JSON.parse(data.electrics.playlistSongsJson);
          const playlistName = data.electrics.selectedPlaylistName || "Songs";
          $scope.processSongData(songs, playlistName);
        } catch (error) {
          console.error('Songs Controller: Error parsing song data from electric fields:', error);
        }
      } else {
        // If no song data, ensure we're showing just shuffle
        if ($scope.songOptions.length <= 1) {
          $scope.songOptions = [
            { name: "Shuffle All", icon_image: "shuffle.png", isShuffle: true }
          ];
          $scope.hasSongs = false;
        }
      }
    }
  };

  // Helper function to determine if an image source is a full URL
  $scope.isFullUrl = function(imageSrc) {
    if (!imageSrc) return false;
    return imageSrc.startsWith('http://') || 
           imageSrc.startsWith('https://') || 
           imageSrc.startsWith('local://') ||
           imageSrc.includes('://');
  };

  // Get current song image for the album art display
  $scope.getCurrentSongImage = function() {
    if ($scope.songOptions && $scope.songOptions[$scope.selectedSongIndex]) {
      const selectedSong = $scope.songOptions[$scope.selectedSongIndex];
      if (selectedSong.isShuffle) {
        // Special handling for shuffle - use different images based on context
        if ($scope.playlistName === "All Songs") {
          return "shuffle.png"; // Use shuffle icon for All Songs shuffle
        } else if ($scope.playlistName === "Saved Songs" || $scope.playlistName === "My Favorites") {
          return "favorite_songs.png"; // Use favorites icon for saved songs shuffle
        } else {
          return "shuffle.png"; // Default shuffle icon
        }
      } else if (selectedSong.albumArt) {
        return selectedSong.albumArt;
      }
    }
    return "default_playlist.png";
  };

  // Get the full background image URL, handling both local icons and full URLs
  $scope.getCurrentSongImageUrl = function() {
    const imageSrc = $scope.getCurrentSongImage();
    
    if ($scope.isFullUrl(imageSrc)) {
      // If it's already a full URL, return as-is
      return imageSrc;
    } else {
      // If it's just a filename, prepend the local icons path
      return '/vehicles/RLA_A90/infotainment_screen/icons/' + imageSrc;
    }
  };

  $scope.getSongScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedSongIndex - $scope.songScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getSongScrollTransform = function() {
    const itemHeight = 48;
    return 'translateY(' + ($scope.songScrollOffset * -itemHeight) + 'px)';
  };

  $scope.getSongScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.songOptions.length;
    const scrollbarHeight = 300;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getSongScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.songOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getSongScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.songScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableSongOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.songOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.songOptions.length; i++) {
        if (!$scope.songOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateSongMenu = function(direction) {
    const visibleItems = 6;
    const totalItems = $scope.songOptions.length;
    
    if (direction === 'up' && $scope.selectedSongIndex > 0) {
      const newIndex = $scope.findNextSelectableSongOption('up', $scope.selectedSongIndex);
      if (newIndex !== $scope.selectedSongIndex) {
        $scope.selectedSongIndex = newIndex;
        // Only adjust scroll if the selected item is above the visible area
        if ($scope.selectedSongIndex < $scope.songScrollOffset) {
          $scope.songScrollOffset = $scope.selectedSongIndex;
        }
      }
    } else if (direction === 'down' && $scope.selectedSongIndex < totalItems - 1) {
      const newIndex = $scope.findNextSelectableSongOption('down', $scope.selectedSongIndex);
      if (newIndex !== $scope.selectedSongIndex) {
        $scope.selectedSongIndex = newIndex;
        // Only adjust scroll if the selected item is below the visible area
        if ($scope.selectedSongIndex >= $scope.songScrollOffset + visibleItems) {
          $scope.songScrollOffset = $scope.selectedSongIndex - (visibleItems - 1);
        }
      }
    }
  };

  $scope.$on('songs-navigate-up', function() {
    $scope.navigateSongMenu('up');
  });

  $scope.$on('songs-navigate-down', function() {
    $scope.navigateSongMenu('down');
  });

  $scope.$on('songs-select-option', function() {
    const selectedOption = $scope.songOptions[$scope.selectedSongIndex];
    if (selectedOption && !selectedOption.disabled) {
      if (selectedOption.isShuffle) {
        console.log("Songs Controller: Shuffle selected for playlist: " + $scope.playlistName);
        
        // Send shuffle command to Lua
        if (window.beamng && window.beamng.sendActiveObjectLua) {
          const luaCommand = `
            if electrics and electrics.values then
              electrics.values.selectedSongId = "shuffle"
              electrics.values.musicControlCommand = "selectSong"
            end
          `;
          
          console.log('Songs Controller: Sending shuffle command to Lua for playlist:', $scope.playlistName);
          window.beamng.sendActiveObjectLua(luaCommand);
        } else {
          console.error('Songs Controller: window.beamng.sendActiveObjectLua not available');
        }
        
        // Navigate to now playing page
        $scope.$parent.currentMenu = 'nowPlaying';
        
      } else if (selectedOption.songId) {
        console.log("Songs Controller: Song selected - " + selectedOption.name + 
                   (selectedOption.artist ? " by " + selectedOption.artist : "") +
                   " from " + $scope.playlistName);
        
        // Prepare song data for now playing page
        const songData = {
          id: selectedOption.songId,
          name: selectedOption.name,
          artist: selectedOption.artist,
          albumArt: selectedOption.albumArt,
          duration: selectedOption.duration,
          playlistName: $scope.playlistName,
          songData: selectedOption.songData // Full song data
        };
        
        // Send current song selection to Lua
        if (window.beamng && window.beamng.sendActiveObjectLua) {
          const luaCommand = `
            if electrics and electrics.values then
              electrics.values.selectedSongId = "${selectedOption.songId}"
              electrics.values.selectedSongJson = '${JSON.stringify(songData).replace(/'/g, "\\'")}'
              electrics.values.musicControlCommand = "selectSong"
            end
          `;
          
          console.log('Songs Controller: Sending song selection to Lua:', selectedOption.songId);
          window.beamng.sendActiveObjectLua(luaCommand);
        } else {
          console.error('Songs Controller: window.beamng.sendActiveObjectLua not available');
        }
        
        // Broadcast song data to now playing controller
        $scope.$parent.$broadcast('current-song-updated', songData);
        
        // Navigate to now playing page
        $scope.$parent.currentMenu = 'nowPlaying';
      }
    }
  });

  // FIXED: Listen for immediate playlist clearing to prevent flicker
  $scope.$on('clear-songs-for-new-playlist', function(event, playlistId) {
    console.log('Songs Controller: Clearing songs for new playlist:', playlistId);
    
    // Immediately reset to just shuffle to prevent showing old songs
    $scope.songOptions = [
      { name: "Shuffle All", icon_image: "shuffle.png", isShuffle: true }
    ];
    $scope.hasSongs = false;
    $scope.selectedSongIndex = 0;
    $scope.songScrollOffset = 0;
    
    // Set appropriate loading message based on playlist type
    if (playlistId === "allsongs") {
      $scope.playlistName = "Loading All Songs...";
      $scope.noSongsMessage = "Loading...";
    } else if (playlistId === "savedsongs") {
      $scope.playlistName = "Loading Saved Songs...";
      $scope.noSongsMessage = "Loading...";
    } else if (playlistId === "favorites") {
      $scope.playlistName = "Loading Favorites...";
      $scope.noSongsMessage = "Loading...";
    } else {
      $scope.playlistName = "Loading...";
      $scope.noSongsMessage = "Loading...";
    }
    
    // Force UI update immediately
    $scope.$apply();
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'songs' && (oldMenu === 'playlists' || oldMenu === 'media')) {
      $scope.selectedSongIndex = 0;
      $scope.songScrollOffset = 0;
    }
  });

  // Initialize with just shuffle
  $scope.songOptions = [
    { name: "Shuffle All", icon_image: "shuffle.png", isShuffle: true }
  ];
  $scope.hasSongs = false;
});