angular.module('infotainmentScreen')
.controller('PlaylistController', function($scope, $timeout) {
  // Initialize with no default playlists
  $scope.playlistOptions = [];

  $scope.selectedPlaylistIndex = 0;
  $scope.playlistScrollOffset = 0;
  $scope.hasPlaylists = false;
  $scope.noPlaylistsMessage = "No playlists found";
  
  // Preview panel state
  $scope.showPlaylistPreview = false;
  $scope.previewPlaylistName = "";
  $scope.previewPlaylistCover = "default_playlist.png";
  $scope.previewSongs = [];
  $scope.previewLoading = false;
  $scope.hoveredPlaylistId = null;
  $scope.previewTimer = null;

  // Function to process playlist data
  $scope.processPlaylistData = function(playlists) {
    // Reset to empty
    $scope.playlistOptions = [];
    
    // Add dynamic playlists
    if (playlists && playlists.length > 0) {
      playlists.forEach(function(playlist) {
        if (playlist.name) {
          $scope.playlistOptions.push({
            name: playlist.name,
            icon_image: "playlists.png",
            playlistId: playlist.id,
            coverImage: playlist.coverImage || "default_playlist.png"
          });
        }
      });
    }
    
    $scope.hasPlaylists = $scope.playlistOptions.length > 0;
  };

  // Helper function to determine if an image source is a full URL
  $scope.isFullUrl = function(imageSrc) {
    if (!imageSrc) return false;
    return imageSrc.startsWith('http://') || 
           imageSrc.startsWith('https://') || 
           imageSrc.startsWith('local://') ||
           imageSrc.includes('://');
  };

  // Get the full URL for the playlist cover
  $scope.getPlaylistCoverUrl = function() {
    const imageSrc = $scope.previewPlaylistCover;
    
    // If it's already a full URL (like local://), return as-is
    if ($scope.isFullUrl(imageSrc)) {
      return imageSrc;
    } else {
      // Otherwise, prepend the path
      return '/vehicles/RLA_A90/infotainment_screen/icons/' + imageSrc;
    }
  };

  // Function to receive electric data updates
  $scope.updateElectricData = function(data) {
    if (data && data.electrics) {
      if (data.electrics.playlistsReady === 1 && data.electrics.playlistsJson) {
        try {
          const playlists = JSON.parse(data.electrics.playlistsJson);
          $scope.processPlaylistData(playlists);
        } catch (error) {
          console.error('Playlists: Error parsing playlist data from electric fields:', error);
        }
      } else {
        if ($scope.playlistOptions.length === 0) {
          $scope.playlistOptions = [];
          $scope.hasPlaylists = false;
        }
      }
      
      // Check for preview playlist cover data
      if (data.electrics.previewPlaylistSongsReady === 1) {
        const playlistId = data.electrics.previewPlaylistId || "";
        const playlistCover = data.electrics.previewPlaylistCover || "default_playlist.png";
        
        console.log('Received preview data:', { playlistId, playlistCover });
        
        if ($scope.hoveredPlaylistId === playlistId) {
          $scope.previewPlaylistCover = playlistCover;
          $scope.previewLoading = false;
          console.log('Set preview cover to:', playlistCover);
        }
        
        // Reset the ready flag
        data.electrics.previewPlaylistSongsReady = 0;
      }
    }
  };

  // Handle playlist hover
  $scope.onPlaylistHover = function(index) {
    if ($scope.previewTimer) {
      $timeout.cancel($scope.previewTimer);
    }
    
    $scope.previewTimer = $timeout(function() {
      const playlist = $scope.playlistOptions[index];
      if (playlist) {
        const playlistId = playlist.playlistId || "";
        $scope.hoveredPlaylistId = playlistId;
        $scope.previewPlaylistName = playlist.name;
        $scope.previewPlaylistCover = playlist.coverImage || "default_playlist.png";
        
        $scope.showPlaylistPreview = true;
        $scope.previewLoading = true;
        
        console.log('Playlist Preview: Requesting cover for:', playlistId, 'Initial cover:', playlist.coverImage);
        
        if (window.beamng && window.beamng.sendActiveObjectLua) {
          const luaCommand = `
            if electrics and electrics.values then
              electrics.values.requestPreviewPlaylistSongs = "${playlistId}"
            end
          `;
          
          window.beamng.sendActiveObjectLua(luaCommand);
        }
      }
    }, 100);
  };

  $scope.onPlaylistLeave = function() {
    if ($scope.previewTimer) {
      $timeout.cancel($scope.previewTimer);
      $scope.previewTimer = null;
    }
    
    $timeout(function() {
      $scope.showPlaylistPreview = false;
      $scope.hoveredPlaylistId = null;
      $scope.previewPlaylistCover = "default_playlist.png";
      $scope.previewLoading = false;
    }, 100);
  };

  $scope.getPlaylistScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedPlaylistIndex - $scope.playlistScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getPlaylistScrollTransform = function() {
    const itemHeight = 48;
    return 'translateY(' + ($scope.playlistScrollOffset * -itemHeight) + 'px)';
  };

  $scope.getPlaylistScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.playlistOptions.length;
    const scrollbarHeight = 300;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    return Math.max(thumbHeight, 80);
  };

  $scope.getPlaylistScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.playlistOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getPlaylistScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.playlistScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectablePlaylistOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.playlistOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.playlistOptions.length; i++) {
        if (!$scope.playlistOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigatePlaylistMenu = function(direction) {
    const visibleItems = 6;
    const totalItems = $scope.playlistOptions.length;
    
    if (direction === 'up' && $scope.selectedPlaylistIndex > 0) {
      const newIndex = $scope.findNextSelectablePlaylistOption('up', $scope.selectedPlaylistIndex);
      if (newIndex !== $scope.selectedPlaylistIndex) {
        $scope.selectedPlaylistIndex = newIndex;
        if ($scope.selectedPlaylistIndex < $scope.playlistScrollOffset) {
          $scope.playlistScrollOffset = $scope.selectedPlaylistIndex;
        }
        $scope.onPlaylistHover($scope.selectedPlaylistIndex);
      }
    } else if (direction === 'down' && $scope.selectedPlaylistIndex < totalItems - 1) {
      const newIndex = $scope.findNextSelectablePlaylistOption('down', $scope.selectedPlaylistIndex);
      if (newIndex !== $scope.selectedPlaylistIndex) {
        $scope.selectedPlaylistIndex = newIndex;
        if ($scope.selectedPlaylistIndex >= $scope.playlistScrollOffset + visibleItems) {
          $scope.playlistScrollOffset = $scope.selectedPlaylistIndex - (visibleItems - 1);
        }
        $scope.onPlaylistHover($scope.selectedPlaylistIndex);
      }
    }
  };

  $scope.$on('playlist-navigate-up', function() {
    $scope.navigatePlaylistMenu('up');
  });

  $scope.$on('playlist-navigate-down', function() {
    $scope.navigatePlaylistMenu('down');
  });

  $scope.$on('playlist-select-option', function() {
    const selectedOption = $scope.playlistOptions[$scope.selectedPlaylistIndex];
    if (selectedOption && !selectedOption.disabled) {
      const playlistId = selectedOption.playlistId || "";
      
      $scope.showPlaylistPreview = false;
      $scope.hoveredPlaylistId = null;
      
      $scope.$parent.$broadcast('clear-songs-for-new-playlist', playlistId);
      
      console.log('Playlist Controller: Requesting songs for playlist:', playlistId);
      
      if (window.beamng && window.beamng.sendActiveObjectLua) {
        const luaCommand = `
          if electrics and electrics.values then
            electrics.values.requestPlaylistSongs = "${playlistId}"
          end
        `;
        
        window.beamng.sendActiveObjectLua(luaCommand);
      } else {
        console.error('Playlist Controller: window.beamng.sendActiveObjectLua not available');
      }
      
      $scope.$parent.selectedPlaylistId = playlistId;
      $scope.$parent.currentMenu = 'songs';
    }
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'playlists') {
      // Coming back to playlists from any menu
      if (oldMenu === 'media') {
        // Reset to first item when coming from media menu
        $scope.selectedPlaylistIndex = 0;
        $scope.playlistScrollOffset = 0;
      }
      
      // Always refresh the preview when entering playlists menu
      $scope.showPlaylistPreview = false;
      $scope.previewPlaylistCover = "default_playlist.png";
      
      $timeout(function() {
        if ($scope.playlistOptions.length > 0) {
          // Trigger hover for the currently selected playlist
          $scope.onPlaylistHover($scope.selectedPlaylistIndex);
        }
      }, 100);
    } else if (newMenu !== 'playlists') {
      // Leaving playlists menu - hide preview
      $scope.showPlaylistPreview = false;
      $scope.hoveredPlaylistId = null;
      $scope.previewPlaylistCover = "default_playlist.png";
    }
  });

  // Initialize empty
  $scope.playlistOptions = [];
  $scope.hasPlaylists = false;
});