angular.module('infotainmentScreen')
.controller('NowPlayingController', function($scope, $timeout) {
  // Current song data
  $scope.currentSong = {
    title: "Loading...",
    artist: "",
    albumArt: "default_playlist.png",
    duration: 0,
    currentTime: 0,
    playlistName: "",
    isPlaying: false,
    isFavorited: false
  };

  // Progress bar state
  $scope.progressPercentage = 0;

  // Playback control states
  $scope.isPlaying = false;
  $scope.isShuffleMode = false;

  // Function to receive electric data updates for current song
  $scope.updateElectricData = function(data) {
    if (data && data.electrics) {
      // Check for current song data from Lua
      if (data.electrics.currentSongReady === 1 && data.electrics.currentSongJson) {
        try {
          const songData = JSON.parse(data.electrics.currentSongJson);
          $scope.updateCurrentSong(songData);
        } catch (error) {
          console.error('Now Playing: Error parsing current song data:', error);
        }
      }

      // Update playback state if available
      if (data.electrics.currentSongIsPlaying !== undefined) {
        $scope.isPlaying = data.electrics.currentSongIsPlaying === 1;
        $scope.currentSong.isPlaying = $scope.isPlaying;
      }

      // Update current time and progress if available
      if (data.electrics.currentSongTime !== undefined) {
        $scope.currentSong.currentTime = data.electrics.currentSongTime || 0;
        if ($scope.currentSong.duration > 0) {
          $scope.progressPercentage = ($scope.currentSong.currentTime / $scope.currentSong.duration) * 100;
        }
      }

      // Update favorite status if available - This is the important part!
      if (data.electrics.currentSongIsFavorited !== undefined) {
        const wasFavorited = $scope.currentSong.isFavorited;
        $scope.currentSong.isFavorited = data.electrics.currentSongIsFavorited === 1;
        
        // Log when favorite status changes
        if (wasFavorited !== $scope.currentSong.isFavorited) {
          console.log('Now Playing: Favorite status changed to:', $scope.currentSong.isFavorited);
        }
      }
    }
  };

  // Function to update current song data
  $scope.updateCurrentSong = function(songData) {
    if (songData) {
      $scope.currentSong.title = songData.name || songData.title || "Unknown Song";
      $scope.currentSong.artist = songData.artist || songData.artistName || "";
      $scope.currentSong.albumArt = songData.albumArt || songData.album_art || songData.cover || "default_playlist.png";
      $scope.currentSong.duration = songData.duration || 0;
      $scope.currentSong.playlistName = songData.playlistName || "";
      $scope.currentSong.songId = songData.id || songData.songId;
      
      // Don't override the favorite status from the songData if it doesn't have it
      // The electric value currentSongIsFavorited is the authoritative source
      if (songData.isFavorited !== undefined) {
        $scope.currentSong.isFavorited = songData.isFavorited;
      }
    }
  };

  // Helper function to determine if an image source is a full URL
  $scope.isFullUrl = function(imageSrc) {
    if (!imageSrc) return false;
    return imageSrc.startsWith('http://') || 
           imageSrc.startsWith('https://') || 
           imageSrc.startsWith('local://') ||
           imageSrc.includes('://');
  };

  // Get the full background image URL for album art
  $scope.getAlbumArtUrl = function() {
    const imageSrc = $scope.currentSong.albumArt;
    
    if ($scope.isFullUrl(imageSrc)) {
      return imageSrc;
    } else {
      return '/vehicles/RLA_A90/infotainment_screen/icons/' + imageSrc;
    }
  };

  // Format time in mm:ss format
  $scope.formatTime = function(seconds) {
    if (!seconds || seconds < 0) return "0:00";
    
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = Math.floor(seconds % 60);
    return minutes + ":" + (remainingSeconds < 10 ? "0" : "") + remainingSeconds;
  };

  // Playback control functions
  $scope.togglePlayPause = function() {
    console.log('Now Playing: Toggle play/pause');
    
    // Send command to Lua to toggle playback
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "togglePlayPause"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
    
    // Toggle local state (will be updated by electric data)
    $scope.isPlaying = !$scope.isPlaying;
    $scope.currentSong.isPlaying = $scope.isPlaying;
  };

  $scope.previousSong = function() {
    console.log('Now Playing: Previous song');
    
    // Send command to Lua to go to previous song
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "previousSong"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
  };

  $scope.nextSong = function() {
    console.log('Now Playing: Next song');
    
    // Send command to Lua to go to next song
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "nextSong"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
  };

  $scope.toggleFavorite = function() {
    console.log('Now Playing: Toggle favorite button clicked');
    
    // Send command to Lua to toggle favorite status
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "toggleFavorite"
          electrics.values.favoriteSongId = "${$scope.currentSong.songId || ''}"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
    
    // Toggle local state immediately for responsive UI
    $scope.currentSong.isFavorited = !$scope.currentSong.isFavorited;
    console.log('Now Playing: Favorite toggled to:', $scope.currentSong.isFavorited);
  };

  // Progress bar interaction
  $scope.onProgressBarClick = function(event) {
    // Calculate position clicked on progress bar
    const progressBar = event.currentTarget;
    const rect = progressBar.getBoundingClientRect();
    const clickX = event.clientX - rect.left;
    const percentage = (clickX / rect.width) * 100;
    
    // Calculate new time position
    const newTime = ($scope.currentSong.duration * percentage) / 100;
    
    console.log('Now Playing: Seek to', $scope.formatTime(newTime));
    
    // Send seek command to Lua
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "seekTo"
          electrics.values.seekToTime = ${newTime}
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
  };

  // Navigation functions
  $scope.goBack = function() {
    // Determine where to go back to based on how we got here
    if ($scope.$parent.cameFromAllSongs || $scope.$parent.cameFromSavedSongs) {
      $scope.$parent.currentMenu = 'media';
      $scope.$parent.cameFromAllSongs = false;
      $scope.$parent.cameFromSavedSongs = false;
    } else {
      $scope.$parent.currentMenu = 'songs';
    }
  };

  // Listen for menu changes to reset state
  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'nowPlaying' && oldMenu !== 'nowPlaying') {
      // Reset progress when entering now playing
      $scope.progressPercentage = 0;
      console.log('Now Playing: Entered now playing menu');
    }
  });

  // Listen for current song updates from parent
  $scope.$on('current-song-updated', function(event, songData) {
    console.log('Now Playing: Received song update:', songData);
    $scope.updateCurrentSong(songData);
  });

  // Listen for navigation events from the main controller
  $scope.$on('now-playing-previous', function() {
    console.log('Now Playing: Previous button pressed');
    $scope.previousSong();
  });

  $scope.$on('now-playing-next', function() {
    console.log('Now Playing: Next button pressed');
    $scope.nextSong();
  });

  $scope.$on('now-playing-toggle-play', function() {
    console.log('Now Playing: Play/pause button pressed');
    $scope.togglePlayPause();
  });

  // Listen for UP button press for favoriting (handled by Lua, but we can show visual feedback)
  $scope.$on('now-playing-toggle-favorite', function() {
    console.log('Now Playing: Favorite toggle via UP button');
    // The actual toggle is handled by Lua, but we can show immediate visual feedback
    $scope.currentSong.isFavorited = !$scope.currentSong.isFavorited;
  });

  // Initialize with default values
  $scope.currentSong = {
    title: "No Song Selected",
    artist: "",
    albumArt: "default_playlist.png",
    duration: 0,
    currentTime: 0,
    playlistName: "",
    isPlaying: false,
    isFavorited: false
  };
});