angular.module('infotainmentScreen')
.controller('MediaController', function($scope, $timeout) {
  // Initialize media menu options
  $scope.mediaMenuOptions = [
    { name: "Saved Songs", icon_image: "favorite_songs.png" },
    { name: "All Songs", icon_image: "all_music.png" },
    { name: "Playlists", icon_image: "playlists.png" },
    { name: "Equalizer", icon_image: "eq.png" },
    { name: "How To Add Music", icon_image: "all_music.png" },
    { name: "Info", icon_image: "info.png" },
  ];

  $scope.selectedMediaIndex = 0;
  $scope.mediaScrollOffset = 0;
  
  // Current song preview data
  $scope.previewSong = {
    title: "No Song Playing",
    artist: "",
    albumArt: "default_playlist.png",
    duration: 0,
    currentTime: 0,
    isPlaying: false,
    hasData: false
  };
  
  $scope.previewProgressPercentage = 0;
  $scope.showPreview = false; // Hide preview by default when no song is playing

  // Function to receive electric data updates for current song preview
  $scope.updateElectricData = function(data) {
    if (data && data.electrics) {
      // Check for current song data from Lua
      if (data.electrics.currentSongReady === 1 && data.electrics.currentSongJson) {
        try {
          const songData = JSON.parse(data.electrics.currentSongJson);
          $scope.updatePreviewSong(songData);
          $scope.previewSong.hasData = true;
          // Show preview if we have valid song data
          $scope.showPreview = data.electrics.currentSongName && data.electrics.currentSongName !== "";
        } catch (error) {
          console.error('Media Controller: Error parsing current song data:', error);
        }
      } else if (data.electrics.currentSongName) {
        // Fallback to individual fields if JSON not available
        $scope.previewSong.title = data.electrics.currentSongName || "No Song Playing";
        $scope.previewSong.artist = data.electrics.currentSongArtist || "";
        $scope.previewSong.albumArt = data.electrics.currentSongAlbumArt || "default_playlist.png";
        $scope.previewSong.duration = data.electrics.currentSongDuration || 0;
        $scope.previewSong.hasData = data.electrics.currentSongName ? true : false;
        // Show preview only if we have a song name
        $scope.showPreview = data.electrics.currentSongName && data.electrics.currentSongName !== "";
      } else {
        // No song data available, hide the preview
        $scope.showPreview = false;
        $scope.previewSong.hasData = false;
      }

      // Update playback state
      if (data.electrics.currentSongIsPlaying !== undefined) {
        $scope.previewSong.isPlaying = data.electrics.currentSongIsPlaying === 1;
      }

      // Update current time and progress
      if (data.electrics.currentSongTime !== undefined) {
        $scope.previewSong.currentTime = data.electrics.currentSongTime || 0;
        if ($scope.previewSong.duration > 0) {
          $scope.previewProgressPercentage = ($scope.previewSong.currentTime / $scope.previewSong.duration) * 100;
        } else {
          $scope.previewProgressPercentage = 0;
        }
      }
    }
  };

  // Function to update preview song data
  $scope.updatePreviewSong = function(songData) {
    if (songData) {
      $scope.previewSong.title = songData.name || songData.title || "Unknown Song";
      $scope.previewSong.artist = songData.artist || songData.artistName || "";
      $scope.previewSong.albumArt = songData.albumArt || songData.album_art || songData.cover || "default_playlist.png";
      $scope.previewSong.duration = songData.duration || 0;
    }
  };

  // Helper function to determine if an image source is a full URL
  $scope.isFullUrl = function(imageSrc) {
    if (!imageSrc) return false;
    return imageSrc.startsWith('http://') || 
           imageSrc.startsWith('https://') || 
           imageSrc.startsWith('local://') ||
           imageSrc.includes('://');
  };

  // Get the full background image URL for album art
  $scope.getPreviewAlbumArtUrl = function() {
    const imageSrc = $scope.previewSong.albumArt;
    
    if ($scope.isFullUrl(imageSrc)) {
      return imageSrc;
    } else {
      return '/vehicles/RLA_A90/infotainment_screen/icons/' + imageSrc;
    }
  };

  // Playback control functions for preview
  $scope.previewTogglePlayPause = function() {
    console.log('Media Preview: Toggle play/pause');
    
    // Send command to Lua to toggle playback
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "togglePlayPause"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
    
    // Toggle local state (will be updated by electric data)
    $scope.previewSong.isPlaying = !$scope.previewSong.isPlaying;
  };

  $scope.previewPreviousSong = function() {
    console.log('Media Preview: Previous song');
    
    // Send command to Lua to go to previous song
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "previousSong"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
  };

  $scope.previewNextSong = function() {
    console.log('Media Preview: Next song');
    
    // Send command to Lua to go to next song
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.musicControlCommand = "nextSong"
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
  };

  $scope.getMediaScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedMediaIndex - $scope.mediaScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getMediaScrollTransform = function() {
    // Adjust transform to start at 0 and account for item height (32px per item)
    const itemHeight = 40; // Matches the min-height + padding of .media-option
    return 'translateY(' + ($scope.mediaScrollOffset * -itemHeight) + 'px)';
  };

  $scope.getMediaScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.mediaMenuOptions.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getMediaScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.mediaMenuOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getMediaScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.mediaScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableMediaOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.mediaMenuOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.mediaMenuOptions.length; i++) {
        if (!$scope.mediaMenuOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateMediaMenu = function(direction) {
    const visibleItems = 6;
    const totalItems = $scope.mediaMenuOptions.length;
    if (direction === 'up' && $scope.selectedMediaIndex > 0) {
      const newIndex = $scope.findNextSelectableMediaOption('up', $scope.selectedMediaIndex);
      if (newIndex !== $scope.selectedMediaIndex) {
        $scope.selectedMediaIndex = newIndex;
        if ($scope.selectedMediaIndex < $scope.mediaScrollOffset) {
          $scope.mediaScrollOffset = $scope.selectedMediaIndex;
        } else if ($scope.mediaScrollOffset > 0 && $scope.selectedMediaIndex >= $scope.mediaScrollOffset) {
          $scope.mediaScrollOffset = Math.max(0, $scope.selectedMediaIndex);
        }
      }
    } else if (direction === 'down' && $scope.selectedMediaIndex < totalItems - 1) {
      const newIndex = $scope.findNextSelectableMediaOption('down', $scope.selectedMediaIndex);
      if (newIndex !== $scope.selectedMediaIndex) {
        $scope.selectedMediaIndex = newIndex;
        if ($scope.selectedMediaIndex >= $scope.mediaScrollOffset + visibleItems) {
          $scope.mediaScrollOffset = $scope.selectedMediaIndex - (visibleItems - 1);
        }
      }
    }
  };

  $scope.$on('media-navigate-up', function() {
    $scope.navigateMediaMenu('up');
  });

  $scope.$on('media-navigate-down', function() {
    $scope.navigateMediaMenu('down');
  });

  $scope.$on('media-select-option', function() {
    const selectedOption = $scope.mediaMenuOptions[$scope.selectedMediaIndex];
    if (selectedOption && !selectedOption.disabled) {
      console.log('Selected media option:', selectedOption.name);
      
	  // Navigate to saved songs menu if "Saved Songs" is selected
      if (selectedOption.name === "Saved Songs") {
        console.log('Media Controller: Saved Songs selected, requesting saved songs');
        
        // Clear songs immediately before requesting new ones
        $scope.$parent.$broadcast('clear-songs-for-new-playlist', 'savedsongs');
        
        // Use the same Lua command pattern for saved songs
        if (window.beamng && window.beamng.sendActiveObjectLua) {
          const luaCommand = `
            if electrics and electrics.values then
              electrics.values.requestPlaylistSongs = "savedsongs"
            end
          `;
          
          console.log('Media Controller: Using sendActiveObjectLua for saved songs');
          window.beamng.sendActiveObjectLua(luaCommand);
          console.log('Media Controller: Lua command sent successfully for saved songs');
        } else {
          console.error('Media Controller: window.beamng.sendActiveObjectLua not available');
          console.log('Available window.beamng methods:', window.beamng ? Object.keys(window.beamng) : 'window.beamng not found');
        }
        
        // Set the special "savedsongs" ID and switch to songs menu
        $scope.$parent.selectedPlaylistId = 'savedsongs';
        $scope.$parent.currentMenu = 'songs';
      }
      // Navigate to Equalizer if "Equalizer" is selected
      else if (selectedOption.name === "Equalizer") {
        console.log('Media Controller: Equalizer selected, navigating to equalizer menu');
        $scope.$parent.currentMenu = 'equalizer';
      }
      else if (selectedOption.name === "How To Add Music") {
        console.log('Media Controller: How To Add Music selected, navigating to howToAddMusic menu');
        if ($scope.$parent) {
          $scope.$parent.currentMenu = 'howToAddMusic';
          console.log('Media Controller: currentMenu set to howToAddMusic');
        }
      }
      // Navigate based on selected option
      else if (selectedOption.name === "Playlists") {
        console.log('Media Controller: Navigating to playlists menu');
        $scope.$parent.currentMenu = 'playlists';
      }
      // Navigate to info menu if "Info" is selected
      else if (selectedOption.name === "Info") {
        console.log('Media Controller: Info selected, navigating to mediaInfo menu');
        // Make sure we're setting the parent scope's currentMenu
        if ($scope.$parent) {
          $scope.$parent.currentMenu = 'mediaInfo';
          console.log('Media Controller: currentMenu set to mediaInfo');
        } else {
          console.error('Media Controller: Parent scope not found!');
        }
      }
      // Navigate directly to songs menu with all songs if "All Songs" is selected
      else if (selectedOption.name === "All Songs") {
        console.log('Media Controller: All Songs selected, requesting all songs');
        
        // Clear songs immediately before requesting new ones
        $scope.$parent.$broadcast('clear-songs-for-new-playlist', 'allsongs');
        
        // Use the same Lua command pattern as playlist_controller.js
        if (window.beamng && window.beamng.sendActiveObjectLua) {
          const luaCommand = `
            if electrics and electrics.values then
              electrics.values.requestPlaylistSongs = "allsongs"
            end
          `;
          
          console.log('Media Controller: Using sendActiveObjectLua for all songs');
          window.beamng.sendActiveObjectLua(luaCommand);
          console.log('Media Controller: Lua command sent successfully for all songs');
        } else {
          console.error('Media Controller: window.beamng.sendActiveObjectLua not available');
          console.log('Available window.beamng methods:', window.beamng ? Object.keys(window.beamng) : 'window.beamng not found');
        }
        
        // Set the special "allsongs" ID and switch to songs menu
        $scope.$parent.selectedPlaylistId = 'allsongs';
        $scope.$parent.currentMenu = 'songs';
      }
    }
  });

  // Listen for current song updates from other controllers
  $scope.$on('current-song-updated', function(event, songData) {
    console.log('Media Controller: Received song update:', songData);
    $scope.updatePreviewSong(songData);
    $scope.previewSong.hasData = true;
    // Show preview when we get a song update with valid data
    $scope.showPreview = songData && songData.name && songData.name !== "";
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'media' && oldMenu === 'main') {
      $scope.selectedMediaIndex = 0;
      $scope.mediaScrollOffset = 0;
    }
  });
  
  // Initialize preview with default values
  $scope.previewSong = {
    title: "No Song Playing",
    artist: "",
    albumArt: "default_playlist.png",
    duration: 0,
    currentTime: 0,
    isPlaying: false,
    hasData: false
  };
});