angular.module('infotainmentScreen', [])
.controller('InfotainmentController', function($scope, $window) {
  $scope.allOptions = ["Media/Radio", "Communication", "Navigation", "Connected Serv.", "My Vehicle", "Notifications"];

  $scope.selectedIndex = 0;
  $scope.currentPage = 0;
  $scope.currentMenu = 'main';
  $scope.isZoomedOut = false; // New property for zoom state
  $scope.selectedPlaylistId = null;
  $scope.playlistRequestPending = false; // Track if we have a pending request
  $scope.cameFromAllSongs = false; // Track if we came from "All Songs" in media menu
  $scope.cameFromSavedSongs = false; // Track if we came from "Saved Songs" in media menu

  // Current song data for main menu display
  $scope.mainMenuCurrentSong = {
    title: "",
    artist: "",
    albumArt: "default_playlist.png",
    isPlaying: false
  };

  // Volume overlay state
  $scope.volumeOverlay = {
    visible: false,
    percentage: 50,
    hideTimeout: null
  };
  
  let lastVolumeValue = 50;

  // Show volume overlay
  function showVolumeOverlay(percentage) {
    $scope.volumeOverlay.percentage = Math.round(percentage);
    $scope.volumeOverlay.visible = true;
    
    if ($scope.volumeOverlay.hideTimeout) {
      clearTimeout($scope.volumeOverlay.hideTimeout);
    }
    
    $scope.volumeOverlay.hideTimeout = setTimeout(function() {
      $scope.$evalAsync(function() {
        $scope.volumeOverlay.visible = false;
      });
    }, 2000);
  }
  
  var units = {
    uiUnitConsumptionRate: "metric",
    uiUnitDate: "ger",
    uiUnitEnergy: "metric",
    uiUnitLength: "metric",
    uiUnitPower: "hp",
    uiUnitPressure: "bar",
    uiUnitTemperature: "c",
    uiUnitTorque: "metric",
    uiUnitVolume: "l",
    uiUnitWeight: "kg"
  };

  $scope.currentDate = {};
  $scope.time = '';
  
  $scope.vehicleData = {
    fuelConsumption: '--.-',
    fuelUnit: 'l/100km',
    avgSpeed: '0',
    speedUnit: 'km/h'
  };

  let speedHistory = [];
  let speedHistoryLimit = 100;
  let speedSum = 0;
  let lastFuelReading = null;
  let lastSpeedTime = null;
  let accumulatedDistance = 0;
  let fuelConsumptionSamples = [];
  let sampleLimit = 10;
  let updateCounter = 0;

  // Continuous movement for all menus with delay
  let continuousMovement = {
    up: false,
    down: false,
    left: false,
    right: false,
    interval: null,
    delayTimeout: null
  };

  // Function to update current song data for main menu
  $scope.updateMainMenuSong = function(data) {
    if (!data || !data.electrics) return;
    
    const e = data.electrics;
    
    // Check if a song is playing and has valid data
    if (e.currentSongIsPlaying === 1 && e.currentSongName && e.currentSongName !== "") {
      $scope.mainMenuCurrentSong.title = e.currentSongName || "";
      $scope.mainMenuCurrentSong.artist = e.currentSongArtist || "";
      $scope.mainMenuCurrentSong.albumArt = e.currentSongAlbumArt || "default_playlist.png";
      $scope.mainMenuCurrentSong.isPlaying = true;
    } else {
      // No song playing, reset to default
      $scope.mainMenuCurrentSong.title = "";
      $scope.mainMenuCurrentSong.artist = "";
      $scope.mainMenuCurrentSong.albumArt = "default_playlist.png";
      $scope.mainMenuCurrentSong.isPlaying = false;
    }
    
    // Also check JSON data if available
    if (e.currentSongReady === 1 && e.currentSongJson) {
      try {
        const songData = JSON.parse(e.currentSongJson);
        if (songData && e.currentSongIsPlaying === 1) {
          $scope.mainMenuCurrentSong.title = songData.name || songData.title || "";
          $scope.mainMenuCurrentSong.artist = songData.artist || songData.artistName || "";
          $scope.mainMenuCurrentSong.albumArt = songData.albumArt || songData.album_art || songData.cover || "default_playlist.png";
          $scope.mainMenuCurrentSong.isPlaying = true;
        }
      } catch (error) {
        console.error('Main Menu: Error parsing current song data:', error);
      }
    }
  };

  // Helper function to determine if an image source is a full URL
  $scope.isFullUrl = function(imageSrc) {
    if (!imageSrc) return false;
    return imageSrc.startsWith('http://') || 
           imageSrc.startsWith('https://') || 
           imageSrc.startsWith('local://') ||
           imageSrc.includes('://');
  };

  // Get the full background image URL for album art on main menu
  $scope.getMainMenuAlbumArtUrl = function() {
    const imageSrc = $scope.mainMenuCurrentSong.albumArt;
    
    if ($scope.isFullUrl(imageSrc)) {
      return imageSrc;
    } else {
      return '/vehicles/RLA_A90/infotainment_screen/icons/' + imageSrc;
    }
  };

  function startContinuousMovement() {
    if (continuousMovement.interval || continuousMovement.delayTimeout) return;
    
    // Start continuous movement after 1 second delay
    continuousMovement.delayTimeout = setTimeout(function() {
      continuousMovement.delayTimeout = null;
      
      continuousMovement.interval = setInterval(function() {
        $scope.$evalAsync(() => {
          if ($scope.currentMenu === 'daytimeRunningLights') {
            if (continuousMovement.up) $scope.$broadcast('daytime-lights-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('daytime-lights-navigate-down');
            if (continuousMovement.left) $scope.$broadcast('daytime-lights-navigate-left');
            if (continuousMovement.right) $scope.$broadcast('daytime-lights-navigate-right');
          } else if ($scope.currentMenu === 'footwellLighting') {
            if (continuousMovement.up) $scope.$broadcast('footwell-lights-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('footwell-lights-navigate-down');
            if (continuousMovement.left) $scope.$broadcast('footwell-lights-navigate-left');
            if (continuousMovement.right) $scope.$broadcast('footwell-lights-navigate-right');
          } else if ($scope.currentMenu === 'media') {
            if (continuousMovement.up) $scope.$broadcast('media-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('media-navigate-down');
          } else if ($scope.currentMenu === 'mediaInfo') {
            if (continuousMovement.up) $scope.$broadcast('media-info-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('media-info-navigate-down');
          } else if ($scope.currentMenu === 'howToAddMusic') {
            if (continuousMovement.up) $scope.$broadcast('how-to-add-music-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('how-to-add-music-navigate-down');
          } else if ($scope.currentMenu === 'playlists') {
            if (continuousMovement.up) $scope.$broadcast('playlist-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('playlist-navigate-down');
          } else if ($scope.currentMenu === 'songs') {
            if (continuousMovement.up) $scope.$broadcast('songs-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('songs-navigate-down');
          } else if ($scope.currentMenu === 'vehicle') {
            if (continuousMovement.up) $scope.$broadcast('vehicle-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('vehicle-navigate-down');
          } else if ($scope.currentMenu === 'vehicleSettings') {
            if (continuousMovement.up) $scope.$broadcast('settings-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('settings-navigate-down');
          } else if ($scope.currentMenu === 'systemSettings') {
            if (continuousMovement.up) $scope.$broadcast('system-settings-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('system-settings-navigate-down');
          } else if ($scope.currentMenu === 'exteriorLighting') {
            if (continuousMovement.up) $scope.$broadcast('lighting-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('lighting-navigate-down');
          } else if ($scope.currentMenu === 'doorsKey') {
            if (continuousMovement.up) $scope.$broadcast('doors-key-navigate-up');
            if (continuousMovement.down) $scope.$broadcast('doors-key-navigate-down');
          }
        });
      }, 200); // Slower movement every 200ms for regular scrolling
    }, 1000); // 1 second delay before starting continuous movement
  }

  function stopContinuousMovement() {
    // Clear both the delay timeout and the interval
    if (continuousMovement.delayTimeout) {
      clearTimeout(continuousMovement.delayTimeout);
      continuousMovement.delayTimeout = null;
    }
    if (continuousMovement.interval) {
      clearInterval(continuousMovement.interval);
      continuousMovement.interval = null;
    }
    continuousMovement.up = false;
    continuousMovement.down = false;
    continuousMovement.left = false;
    continuousMovement.right = false;
  }

  // Function to propagate electric data to all sub-controllers
  function propagateElectricDataToControllers(data) {
    // Get all controller scopes and call their updateElectricData function if it exists
    var allScopes = [];
    
    // Traverse the scope hierarchy to find child controllers
    function findChildScopes(scope) {
      if (scope.$$childHead) {
        var child = scope.$$childHead;
        do {
          allScopes.push(child);
          findChildScopes(child);
          child = child.$$nextSibling;
        } while (child);
      }
    }
    
    findChildScopes($scope);
    
    // Call updateElectricData on each scope that has it
    allScopes.forEach(function(childScope) {
      if (typeof childScope.updateElectricData === 'function') {
        childScope.updateElectricData(data);
      }
    });
  }

  // Function to set now playing active state for Lua
  function setNowPlayingActive(active) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.nowPlayingActive = ${active ? 1 : 0}
        end
      `;
      window.beamng.sendActiveObjectLua(luaCommand);
    }
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('Main controller received settings:', settings);
    // Settings are handled by individual sub-controllers
  });

  // Global function to receive settings from Lua
  window.loadInfotainmentSettings = function(settings) {
    console.log('Received settings from Lua:', settings);
    
    // Apply settings to all controllers via broadcast
    $scope.$evalAsync(function() {
      $scope.$broadcast('settings-loaded', settings);
    });
  };

  // Check for settings in electric values
  var lastSettingsCheck = '';
  function checkForSettingsInElectrics(data) {
    if (!data || !data.electrics) return;
    
    const e = data.electrics;
        
    if (e.infotainmentSettingsReady === 1 && e.infotainmentSettingsJson) {
      const settingsJson = e.infotainmentSettingsJson;
      if (settingsJson !== lastSettingsCheck) {
        lastSettingsCheck = settingsJson;
        try {
          const settings = JSON.parse(settingsJson);
          console.log('Main: Found settings in electrics, broadcasting:', settings);
          $scope.$broadcast('settings-loaded', settings);
        } catch (error) {
          console.error('Main: Failed to parse settings from electrics:', error);
        }
      }
    }
  }

  // Event listener for UI messages from Lua
  var originalOnUiMessage = window.onUiMessage;
  window.onUiMessage = function(type, data) {
    if (type === 'INFOTAINMENT_LOAD_SETTINGS') {
      try {
        var settings = JSON.parse(data);
        console.log('Received settings via ui_message:', settings);
        window.loadInfotainmentSettings(settings);
      } catch (e) {
        console.error('Failed to parse infotainment settings from ui_message:', e);
      }
    } else if (originalOnUiMessage) {
      originalOnUiMessage(type, data);
    }
  };

  window.setup = function(setupData) {
    for(let dk in setupData){
      if(typeof dk == "string" && dk.startsWith("uiUnit")){
        units[dk] = setupData[dk];
      }
    }

    $scope.vehicleData.speedUnit = units.uiUnitLength == "metric" ? "km/h" : "mph";
    $scope.vehicleData.fuelUnit = units.uiUnitConsumptionRate == "metric" ? "l/100km" : "mpg";
  }
  
  function updateDate() {
    const now = new Date();
    const months = ['January', 'February', 'March', 'April', 'May', 'June',
                   'July', 'August', 'September', 'October', 'November', 'December'];
    
    $scope.currentDate = {
      day: now.getDate(),
      month: months[now.getMonth()],
      year: now.getFullYear()
    };
  }

  function updateTime() {
    const now = new Date();
    let hours = now.getHours();
    const minutes = now.getMinutes();
    const ampm = hours >= 12 ? 'pm' : 'am';
    
    hours = hours % 12;
    hours = hours ? hours : 12;
    
    const minutesStr = minutes < 10 ? '0' + minutes : minutes;
    
    $scope.time = hours + ':' + minutesStr + ' ' + ampm;
  }
  
  function updateVehicleData(data) {
    if (!data || !data.electrics) return;

    const e = data.electrics;
    const speed = e.airspeed || 0;
    const fuelVolume = e.fuelVolume || 0;
    const ignitionLevel = e.ignitionLevel || 0;

    const speedKmh = speed * 3.6;
    const speedMph = speed * 2.237;
    updateCounter++;

    const currentTime = Date.now() / 1000;
    if (lastSpeedTime !== null && ignitionLevel >= 1 && speedKmh > 5) {
      const timeDelta = currentTime - lastSpeedTime;
      const distanceThisUpdate = (speedKmh / 3600) * timeDelta;
      accumulatedDistance += distanceThisUpdate;
    }
    lastSpeedTime = currentTime;

    if (speedKmh > 1 && ignitionLevel >= 1) {
      speedHistory.push(speedKmh);
      speedSum += speedKmh;
      if (speedHistory.length > speedHistoryLimit) {
        speedSum -= speedHistory.shift();
      }
    }

    if (ignitionLevel >= 1 && speedKmh > 10) {
      if (lastFuelReading !== null && accumulatedDistance > 0.1) {
        const fuelDelta = lastFuelReading - fuelVolume;
        const distanceDelta = accumulatedDistance;
        
        if (fuelDelta > 0.005) {
          let consumption;
          if (units.uiUnitConsumptionRate == "metric") {
            consumption = (fuelDelta / distanceDelta) * 100;
          } else {
            const miles = distanceDelta / 1.609344;
            const gallons = fuelDelta / 3.78541;
            consumption = miles / gallons;
          }
          
          const minConsumption = units.uiUnitConsumptionRate == "metric" ? 1 : 8;
          const maxConsumption = units.uiUnitConsumptionRate == "metric" ? 30 : 60;
          
          if (consumption >= minConsumption && consumption <= maxConsumption) {
            fuelConsumptionSamples.push(consumption);
            if (fuelConsumptionSamples.length > sampleLimit) {
              fuelConsumptionSamples.shift();
            }
            const avgConsumption = fuelConsumptionSamples.reduce((a, b) => a + b, 0) / fuelConsumptionSamples.length;
            $scope.vehicleData.fuelConsumption = avgConsumption.toFixed(1);
          }
          
          lastFuelReading = fuelVolume;
          accumulatedDistance = 0;
        }
      } else if (lastFuelReading === null) {
        lastFuelReading = fuelVolume;
        accumulatedDistance = 0;
      }
    }

    const avgSpeedKmh = speedHistory.length > 0 ? speedSum / speedHistory.length : 0;
    const avgSpeedMph = avgSpeedKmh / 1.609;

    $scope.vehicleData.avgSpeed = (units.uiUnitLength == "metric") ?
      Math.round(avgSpeedKmh).toString() :
      Math.round(avgSpeedMph).toString();

    $scope.vehicleData.speedUnit = units.uiUnitLength == "metric" ? 'km/h' : 'mph';
    $scope.vehicleData.fuelUnit = units.uiUnitConsumptionRate == "metric" ? 'l/100km' : 'mpg';
  }

  updateDate();
  updateTime();
  setInterval(function() {
    $scope.$apply(function() {
      updateDate();
      updateTime();
    });
  }, 60000);

  let lastInput = {
    left: 0,
    right: 0,
    option: 0,
    menu: 0,
    media: 0,  // NEW: Add tracking for media button
    back: 0,
    up: 0,
    down: 0
  };

  let buttonPressId = 0;
  let lastProcessedPressIds = {
	volUp: 0,
    volDown: 0,
    option: 0,
    menu: 0,
    media: 0,  // NEW: Add tracking for media button
    back: 0,
    up: 0,
    down: 0,
    left: 0,
    right: 0
  };

  function shouldProcessButton(buttonName, currentState, lastState) {
    if (currentState === 1 && lastState === 0) {
      buttonPressId++;
      return buttonPressId;
    }
    return false;
  }

  // Make it accessible to BeamNG
  window.updateData = function(data) {
    $scope.$evalAsync(function() {
      $scope.processUpdateData(data);
    });
  };
  
  $scope.processUpdateData = function(data) {
    if (!data || !data.electrics) return;
    
    if (data.units) {
      for(let key in data.units) {
        units[key] = data.units[key];
      }
      $scope.vehicleData.speedUnit = units.uiUnitLength == "metric" ? "km/h" : "mph";
      $scope.vehicleData.fuelUnit = units.uiUnitConsumptionRate == "metric" ? "l/100km" : "mpg";
    }
    
    // FIXED: Better handling of playlist song requests
    if ($scope.selectedPlaylistId && !$scope.playlistRequestPending) {
      console.log('Main: Requesting songs for playlist:', $scope.selectedPlaylistId);
      data.electrics.requestPlaylistSongs = $scope.selectedPlaylistId;
      $scope.playlistRequestPending = true;
      // Don't clear selectedPlaylistId immediately - let it persist until we confirm the request was processed
    }
    
    // Check if the request has been processed by checking for response
    if ($scope.playlistRequestPending && data.electrics.playlistSongsReady === 1) {
      console.log('Main: Playlist songs request completed, clearing pending request');
      $scope.selectedPlaylistId = null;
      $scope.playlistRequestPending = false;
    }

    const e = data.electrics;
    updateVehicleData(data);
    
    const currentMenuAtStart = $scope.currentMenu;

    $scope.$evalAsync(() => {
      // Update main menu song display
      $scope.updateMainMenuSong(data);
      
      // Check for settings from Lua and broadcast to all controllers
      checkForSettingsInElectrics(data);
      
      // Propagate electric data to all sub-controllers for real-time updates
      propagateElectricDataToControllers(data);

	  // Volume control - track musicVolume from Lua
      const currentVolume = e.musicVolume !== undefined ? e.musicVolume : 75;
      const volumeChanged = Math.abs(currentVolume - lastVolumeValue) > 0.5;
      
      if ((e.radio_vol_up === 1) || (e.radio_vol_down === 1) || volumeChanged) {
        showVolumeOverlay(currentVolume);
        lastVolumeValue = currentVolume;
      }
      
      lastInput.volUp = e.radio_vol_up || 0;
      lastInput.volDown = e.radio_vol_down || 0;
	  
      // NEW: Handle radio_media button press
      if (e.radio_media === 1 && lastInput.media === 0) {
        const pressId = shouldProcessButton('media', e.radio_media, lastInput.media);
        if (pressId && pressId !== lastProcessedPressIds.media) {
          lastProcessedPressIds.media = pressId;
          
          // Check if there's a current song (playing OR paused)
          const hasCurrentSong = (e.currentSongName && e.currentSongName !== "") || 
                                 (e.currentSongId && e.currentSongId !== "");
          
          if (hasCurrentSong) {
            // There's a current song (playing or paused) - go to now playing menu
            console.log('Media button pressed: Current song exists (playing or paused), going to Now Playing');
            $scope.currentMenu = 'nowPlaying';
            setNowPlayingActive(true);
          } else {
            // No current song loaded - go to media menu
            console.log('Media button pressed: No current song loaded, going to Media menu');
            $scope.currentMenu = 'media';
            setNowPlayingActive(false);
          }
        }
      }
      lastInput.media = e.radio_media;

      if (currentMenuAtStart === 'main') {
        if (e.radio_left === 1 && lastInput.left === 0) {
          if ($scope.isZoomedOut) {
            // In zoomed out mode, navigate through all 6 items in a line
            if ($scope.selectedIndex > 0) {
              $scope.selectedIndex--;
            }
          } else {
            // Original paged navigation
            if ($scope.currentPage === 0) {
              if ($scope.selectedIndex > 0) {
                $scope.selectedIndex--;
              }
            } else {
              if ($scope.selectedIndex > 0) {
                $scope.selectedIndex--;
              } else {
                $scope.currentPage = 0;
                $scope.selectedIndex = 2;
              }
            }
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          if ($scope.isZoomedOut) {
            // In zoomed out mode, navigate through all 6 items in a line
            if ($scope.selectedIndex < 5) {
              $scope.selectedIndex++;
            }
          } else {
            // Original paged navigation
            if ($scope.currentPage === 0) {
              if ($scope.selectedIndex < 2) {
                $scope.selectedIndex++;
              } else {
                $scope.currentPage = 1;
                $scope.selectedIndex = 0;
              }
            } else {
              if ($scope.selectedIndex < 2) {
                $scope.selectedIndex++;
              }
            }
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            let actualIndex;
            
            if ($scope.isZoomedOut) {
              actualIndex = $scope.selectedIndex;
            } else {
              actualIndex = $scope.currentPage === 0 ? $scope.selectedIndex : $scope.selectedIndex + 3;
            }
            
            if (actualIndex === 0) {
              // Media selected
              $scope.currentMenu = 'media';
              setNowPlayingActive(false);
            } else if (actualIndex === 4) {
              // My Vehicle selected
              $scope.currentMenu = 'vehicle';
              setNowPlayingActive(false);
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'media') {
        // Handle continuous scrolling for media menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'media') {
            $scope.$broadcast('media-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'media') {
            $scope.$broadcast('media-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        // NEW: Handle right button to go to now playing if there's a current song
        if (e.radio_right === 1 && lastInput.right === 0) {
          // Check if there's a current song (playing or paused)
          const hasCurrentSong = (e.currentSongName && e.currentSongName !== "") || 
                                 (e.currentSongId && e.currentSongId !== "");
          if (hasCurrentSong) {
            const pressId = shouldProcessButton('right', e.radio_right, lastInput.right);
            if (pressId && pressId !== lastProcessedPressIds.right) {
              lastProcessedPressIds.right = pressId;
              console.log('Media: Right button pressed, going to now playing (current song exists)');
              $scope.currentMenu = 'nowPlaying';
              setNowPlayingActive(true);
            }
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'media') {
              $scope.$broadcast('media-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

	  } else if (currentMenuAtStart === 'mediaInfo') {
        // Handle continuous scrolling for media info content
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'mediaInfo') {
            $scope.$broadcast('media-info-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;
        
        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'mediaInfo') {
            $scope.$broadcast('media-info-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

      } else if (currentMenuAtStart === 'howToAddMusic') {
        // Handle continuous scrolling for how to add music content
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'howToAddMusic') {
            $scope.$broadcast('how-to-add-music-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;
        
        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'howToAddMusic') {
            $scope.$broadcast('how-to-add-music-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

      } else if (currentMenuAtStart === 'equalizer') {
        // Handle input for equalizer menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'equalizer') {
            $scope.$broadcast('equalizer-navigate-up');
          }
        }
        lastInput.up = e.radio_up;
        
        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'equalizer') {
            $scope.$broadcast('equalizer-navigate-down');
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          if ($scope.currentMenu === 'equalizer') {
            $scope.$broadcast('equalizer-navigate-left');
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          if ($scope.currentMenu === 'equalizer') {
            $scope.$broadcast('equalizer-navigate-right');
          }
        }
        lastInput.right = e.radio_right;

      } else if (currentMenuAtStart === 'playlists') {
        // Handle continuous scrolling for playlists menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'playlists') {
            $scope.$broadcast('playlist-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'playlists') {
            $scope.$broadcast('playlist-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        // NEW: Handle right button to go to now playing if there's a current song
        if (e.radio_right === 1 && lastInput.right === 0) {
          // Check if there's a current song (playing or paused)
          const hasCurrentSong = (e.currentSongName && e.currentSongName !== "") || 
                                 (e.currentSongId && e.currentSongId !== "");
          if (hasCurrentSong) {
            const pressId = shouldProcessButton('right', e.radio_right, lastInput.right);
            if (pressId && pressId !== lastProcessedPressIds.right) {
              lastProcessedPressIds.right = pressId;
              console.log('Playlists: Right button pressed, going to now playing (current song exists)');
              $scope.currentMenu = 'nowPlaying';
              setNowPlayingActive(true);
            }
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'playlists') {
              $scope.$broadcast('playlist-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

	  } else if (currentMenuAtStart === 'songs') {
        // Handle continuous scrolling for songs menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'songs') {
            $scope.$broadcast('songs-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;
      
        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'songs') {
            $scope.$broadcast('songs-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        // NEW: Handle right button to go to now playing if there's a current song
        if (e.radio_right === 1 && lastInput.right === 0) {
          // Check if there's a current song (playing or paused)
          const hasCurrentSong = (e.currentSongName && e.currentSongName !== "") || 
                                 (e.currentSongId && e.currentSongId !== "");
          if (hasCurrentSong) {
            const pressId = shouldProcessButton('right', e.radio_right, lastInput.right);
            if (pressId && pressId !== lastProcessedPressIds.right) {
              lastProcessedPressIds.right = pressId;
              console.log('Songs: Right button pressed, going to now playing (current song exists)');
              $scope.currentMenu = 'nowPlaying';
              setNowPlayingActive(true);
            }
          }
        }
        lastInput.right = e.radio_right;
      
        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'songs') {
              $scope.$broadcast('songs-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'nowPlaying') {
        // Handle now playing controls
        // Note: radio_up is now handled in Lua for favoriting when nowPlayingActive is 1
        
        if (e.radio_up === 1 && lastInput.up === 0) {
          // The favorite functionality is now handled by Lua directly
          // when nowPlayingActive electric value is 1
          console.log('Now Playing: Up button pressed - handled by Lua for favorite toggle');
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          // Currently no down navigation in now playing
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          // Previous song
          if ($scope.currentMenu === 'nowPlaying') {
            $scope.$broadcast('now-playing-previous');
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          // Next song
          if ($scope.currentMenu === 'nowPlaying') {
            $scope.$broadcast('now-playing-next');
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'nowPlaying') {
              $scope.$broadcast('now-playing-toggle-play');
            }
          }
        }
        lastInput.option = e.radio_option;
  
      } else if (currentMenuAtStart === 'vehicle') {
        // Handle continuous scrolling for vehicle menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'vehicle') {
            $scope.$broadcast('vehicle-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'vehicle') {
            $scope.$broadcast('vehicle-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'vehicle') {
              $scope.$broadcast('vehicle-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'vehicleSettings') {
        // Handle continuous scrolling for vehicle settings menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'vehicleSettings') {
            $scope.$broadcast('settings-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'vehicleSettings') {
            $scope.$broadcast('settings-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'vehicleSettings') {
              $scope.$broadcast('settings-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'systemSettings') {
        // Handle continuous scrolling for system settings menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'systemSettings') {
            $scope.$broadcast('system-settings-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'systemSettings') {
            $scope.$broadcast('system-settings-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'systemSettings') {
              $scope.$broadcast('system-settings-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'exteriorLighting') {
        // Handle continuous scrolling for exterior lighting menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'exteriorLighting') {
            $scope.$broadcast('lighting-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'exteriorLighting') {
            $scope.$broadcast('lighting-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'exteriorLighting') {
              $scope.$broadcast('lighting-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'starlights') {
        // Handle continuous movement for color picker
        if (e.radio_up === 1 && lastInput.up === 0) {
          $scope.$broadcast('starlights-navigate-up');
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          $scope.$broadcast('starlights-stop-up');
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          $scope.$broadcast('starlights-navigate-down');
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          $scope.$broadcast('starlights-stop-down');
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          $scope.$broadcast('starlights-navigate-left');
        } else if (e.radio_left === 0 && lastInput.left === 1) {
          $scope.$broadcast('starlights-stop-left');
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          $scope.$broadcast('starlights-navigate-right');
        } else if (e.radio_right === 0 && lastInput.right === 1) {
          $scope.$broadcast('starlights-stop-right');
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'starlights') {
              $scope.$broadcast('starlights-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'daytimeRunningLights') {
        // Handle continuous movement for color picker
        if (e.radio_up === 1 && lastInput.up === 0) {
          $scope.$broadcast('daytime-lights-navigate-up');
          continuousMovement.up = true;
          startContinuousMovement();
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          $scope.$broadcast('daytime-lights-navigate-down');
          continuousMovement.down = true;
          startContinuousMovement();
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          $scope.$broadcast('daytime-lights-navigate-left');
          continuousMovement.left = true;
          startContinuousMovement();
        } else if (e.radio_left === 0 && lastInput.left === 1) {
          continuousMovement.left = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          $scope.$broadcast('daytime-lights-navigate-right');
          continuousMovement.right = true;
          startContinuousMovement();
        } else if (e.radio_right === 0 && lastInput.right === 1) {
          continuousMovement.right = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.left) {
            stopContinuousMovement();
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'daytimeRunningLights') {
              $scope.$broadcast('daytime-lights-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'footwellLighting') {
        // Handle continuous movement for color picker
        if (e.radio_up === 1 && lastInput.up === 0) {
          $scope.$broadcast('footwell-lights-navigate-up');
          continuousMovement.up = true;
          startContinuousMovement();
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          $scope.$broadcast('footwell-lights-navigate-down');
          continuousMovement.down = true;
          startContinuousMovement();
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up && !continuousMovement.left && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_left === 1 && lastInput.left === 0) {
          $scope.$broadcast('footwell-lights-navigate-left');
          continuousMovement.left = true;
          startContinuousMovement();
        } else if (e.radio_left === 0 && lastInput.left === 1) {
          continuousMovement.left = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.right) {
            stopContinuousMovement();
          }
        }
        lastInput.left = e.radio_left;

        if (e.radio_right === 1 && lastInput.right === 0) {
          $scope.$broadcast('footwell-lights-navigate-right');
          continuousMovement.right = true;
          startContinuousMovement();
        } else if (e.radio_right === 0 && lastInput.right === 1) {
          continuousMovement.right = false;
          if (!continuousMovement.up && !continuousMovement.down && !continuousMovement.left) {
            stopContinuousMovement();
          }
        }
        lastInput.right = e.radio_right;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'footwellLighting') {
              $scope.$broadcast('footwell-lights-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;

      } else if (currentMenuAtStart === 'doorsKey') {
        // Handle continuous scrolling for doors/key menu
        if (e.radio_up === 1 && lastInput.up === 0) {
          if ($scope.currentMenu === 'doorsKey') {
            $scope.$broadcast('doors-key-navigate-up');
            continuousMovement.up = true;
            startContinuousMovement();
          }
        } else if (e.radio_up === 0 && lastInput.up === 1) {
          continuousMovement.up = false;
          if (!continuousMovement.down) {
            stopContinuousMovement();
          }
        }
        lastInput.up = e.radio_up;

        if (e.radio_down === 1 && lastInput.down === 0) {
          if ($scope.currentMenu === 'doorsKey') {
            $scope.$broadcast('doors-key-navigate-down');
            continuousMovement.down = true;
            startContinuousMovement();
          }
        } else if (e.radio_down === 0 && lastInput.down === 1) {
          continuousMovement.down = false;
          if (!continuousMovement.up) {
            stopContinuousMovement();
          }
        }
        lastInput.down = e.radio_down;

        if (e.radio_option === 1 && lastInput.option === 0) {
          const pressId = shouldProcessButton('option', e.radio_option, lastInput.option);
          if (pressId && pressId !== lastProcessedPressIds.option) {
            lastProcessedPressIds.option = pressId;
            if ($scope.currentMenu === 'doorsKey') {
              $scope.$broadcast('doors-key-select-option');
            }
          }
        }
        lastInput.option = e.radio_option;
      }

      // Handle menu button press (only for zoom in main menu and direct navigation to main)
      if (e.radio_menu === 1 && lastInput.menu === 0) {
        const pressId = shouldProcessButton('menu', e.radio_menu, lastInput.menu);
        if (pressId && pressId !== lastProcessedPressIds.menu) {
          lastProcessedPressIds.menu = pressId;
          
          if ($scope.currentMenu === 'main') {
            // Toggle zoom state in main menu
            $scope.isZoomedOut = !$scope.isZoomedOut;
            
            // When transitioning to zoomed out, adjust selectedIndex based on current page/position
            if ($scope.isZoomedOut) {
              $scope.selectedIndex = $scope.currentPage === 0 ? $scope.selectedIndex : $scope.selectedIndex + 3;
            } else {
              // When zooming back in, set page and index based on selectedIndex
              if ($scope.selectedIndex < 3) {
                $scope.currentPage = 0;
              } else {
                $scope.currentPage = 1;
                $scope.selectedIndex = $scope.selectedIndex - 3;
              }
            }
          } else {
            // From any other menu (including Media), go directly to main menu
            $scope.currentMenu = 'main';
            $scope.isZoomedOut = false; // Reset zoom when returning to main
            $scope.cameFromAllSongs = false; // Reset flag when returning to main
            $scope.cameFromSavedSongs = false; // Reset saved songs flag
            setNowPlayingActive(false);
          }
        }
      }
      lastInput.menu = e.radio_menu;

      // Handle back button press (go back one level)
      if (e.radio_back === 1 && lastInput.back === 0) {
        const pressId = shouldProcessButton('back', e.radio_back, lastInput.back);
        if (pressId && pressId !== lastProcessedPressIds.back) {
          lastProcessedPressIds.back = pressId;
          
          if ($scope.currentMenu === 'nowPlaying') {
            // Always go back to songs menu when leaving now playing
            $scope.currentMenu = 'songs';
            setNowPlayingActive(false);
		  } else if ($scope.currentMenu === 'equalizer') {
		  $scope.currentMenu = 'media';
          } else if ($scope.currentMenu === 'doorsKey') {
            $scope.currentMenu = 'vehicleSettings';
          } else if ($scope.currentMenu === 'exteriorLighting') {
            $scope.currentMenu = 'vehicleSettings';
          } else if ($scope.currentMenu === 'daytimeRunningLights') {
            // Let the controller handle the back button completely
            $scope.$broadcast('daytime-lights-back');
          } else if ($scope.currentMenu === 'footwellLighting') {
            // Let the controller handle the back button completely
            $scope.$broadcast('footwell-lights-back');
          } else if ($scope.currentMenu === 'starlights') {
            // Let the controller handle the back button completely
            $scope.$broadcast('starlights-back');
		  } else if ($scope.currentMenu === 'playlists') {
            $scope.currentMenu = 'media';
          } else if ($scope.currentMenu === 'systemSettings') {
            $scope.currentMenu = 'vehicle';
          } else if ($scope.currentMenu === 'vehicleSettings') {
            $scope.currentMenu = 'vehicle';
          } else if ($scope.currentMenu === 'vehicle') {
            $scope.currentMenu = 'main';
          } else if ($scope.currentMenu === 'media') {
            $scope.currentMenu = 'main';
		  } else if ($scope.currentMenu === 'mediaInfo') {
			$scope.currentMenu = 'media';
		  } else if ($scope.currentMenu === 'howToAddMusic') {
			$scope.currentMenu = 'media';
		  } else if ($scope.currentMenu === 'songs') {
            // Check if we came from "All Songs" or "Saved Songs" in media menu or from playlists
            if ($scope.cameFromAllSongs || $scope.cameFromSavedSongs) {
              $scope.currentMenu = 'media';
              $scope.cameFromAllSongs = false;
              $scope.cameFromSavedSongs = false;
            } else {
              $scope.currentMenu = 'playlists';
            }
            // Clear any pending playlist requests when going back
            $scope.selectedPlaylistId = null;
            $scope.playlistRequestPending = false;
          } else if ($scope.currentMenu === 'main' && $scope.isZoomedOut) {
            // If in main menu and zoomed out, zoom back in instead of going back
            $scope.isZoomedOut = false;
            // When zooming back in, set page and index based on selectedIndex
            if ($scope.selectedIndex < 3) {
              $scope.currentPage = 0;
            } else {
              $scope.currentPage = 1;
              $scope.selectedIndex = $scope.selectedIndex - 3;
            }
          }
        }
      }
      lastInput.back = e.radio_back;
    });
  };

  // Listen for when songs menu is entered from media menu (All Songs or Saved Songs)
  $scope.$watch('currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'songs' && oldMenu === 'media') {
      if ($scope.selectedPlaylistId === 'allsongs') {
        $scope.cameFromAllSongs = true;
        $scope.cameFromSavedSongs = false;
      } else if ($scope.selectedPlaylistId === 'savedsongs') {
        $scope.cameFromSavedSongs = true;
        $scope.cameFromAllSongs = false;
      }
    }
    if (newMenu === 'nowPlaying' && (oldMenu === 'media' || oldMenu === 'songs')) {
      // Track how we got to now playing for proper back navigation
      if (oldMenu === 'media') {
        if ($scope.selectedPlaylistId === 'savedsongs') {
          $scope.cameFromSavedSongs = true;
          $scope.cameFromAllSongs = false;
        } else {
          $scope.cameFromAllSongs = true;
          $scope.cameFromSavedSongs = false;
        }
      }
      setNowPlayingActive(true);
    } else if (newMenu !== 'nowPlaying' && oldMenu === 'nowPlaying') {
      setNowPlayingActive(false);
    }
  });
});