angular.module('infotainmentScreen')
.controller('ExteriorLightingController', function($scope) {
  // Options with electric keys - checked state will be determined by electric values
  var defaultOptions = [
    { name: "Interior", disabled: true},
    { name: "Footwell Lighting", disabled: false},
    { name: "Starlights", disabled: false},
    { name: "Exterior", disabled: true},
    { name: "Daytime Running Lights", disabled: false,  image: "Supra_blue_drls.png"},
    { name: "Door handle lighting", disabled: false, hasCheckbox: true, electricKey: "doorHandleLighting", image: "Supra_door_handle_lights.png" },
  ];

  $scope.exteriorLightingOptions = angular.copy(defaultOptions);
  $scope.selectedLightingIndex = 1;
  $scope.lightingScrollOffset = 0;

  // Current electric values cache
  var currentElectricValues = {};

  $scope.getCurrentLightingImage = function() {
    if ($scope.selectedLightingIndex >= 0 && $scope.selectedLightingIndex < $scope.exteriorLightingOptions.length) {
      return $scope.exteriorLightingOptions[$scope.selectedLightingIndex].image;
    }
    return 'default_lighting.png';
  };

  function writeElectricValue(key, value) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.${key} = ${value}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
    } else if (window.bngApi && window.bngApi.engineLua) {
      const luaCommand = `
        local veh = be:getPlayerVehicle(0)
        if veh then
          veh:queueLuaCommand('if electrics and electrics.values then electrics.values.${key} = ${value} end')
        end
      `;
      
      window.bngApi.engineLua(luaCommand);
    }
  }

  // Update checkbox states based on current electric values
  function updateCheckboxStates() {
    $scope.exteriorLightingOptions.forEach(function(option) {
      if (option.electricKey && currentElectricValues.hasOwnProperty(option.electricKey)) {
        option.checked = currentElectricValues[option.electricKey] === 1;
      }
    });
  }

  // Process electric data and update checkboxes
  function processElectricData(data) {
    if (!data || !data.electrics) return;
    
    var hasChanges = false;
    const e = data.electrics;
    
    // Check for changes in electric values we care about
    $scope.exteriorLightingOptions.forEach(function(option) {
      if (option.electricKey && e.hasOwnProperty(option.electricKey)) {
        var newValue = e[option.electricKey];
        if (currentElectricValues[option.electricKey] !== newValue) {
          currentElectricValues[option.electricKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Update checkbox states if any values changed
    if (hasChanges) {
      updateCheckboxStates();
    }
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('ExteriorLighting: Received settings:', settings);
    
    // Update our electric values cache
    Object.keys(settings).forEach(function(key) {
      currentElectricValues[key] = settings[key];
    });
    
    // Update checkbox states
    updateCheckboxStates();
  });

  // Listen for electric data updates (called from main controller)
  $scope.updateElectricData = function(data) {
    processElectricData(data);
  };

  $scope.getLightingScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedLightingIndex - $scope.lightingScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getLightingScrollTransform = function() {
    return 'translateY(' + ($scope.lightingScrollOffset * -40) + 'px)';
  };

  $scope.getLightingScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.exteriorLightingOptions.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getLightingScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.exteriorLightingOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getLightingScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.lightingScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.exteriorLightingOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.exteriorLightingOptions.length; i++) {
        if (!$scope.exteriorLightingOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateLightingMenu = function(direction) {
    if (direction === 'up' && $scope.selectedLightingIndex > 0) {
      const newIndex = $scope.findNextSelectableOption('up', $scope.selectedLightingIndex);
      if (newIndex !== $scope.selectedLightingIndex) {
        $scope.selectedLightingIndex = newIndex;
        if ($scope.selectedLightingIndex < $scope.lightingScrollOffset) {
          $scope.lightingScrollOffset = $scope.selectedLightingIndex;
        }
      }
    } else if (direction === 'down' && $scope.selectedLightingIndex < $scope.exteriorLightingOptions.length - 1) {
      const newIndex = $scope.findNextSelectableOption('down', $scope.selectedLightingIndex);
      if (newIndex !== $scope.selectedLightingIndex) {
        $scope.selectedLightingIndex = newIndex;
        const visibleItems = 6;
        if ($scope.selectedLightingIndex >= $scope.lightingScrollOffset + visibleItems) {
          $scope.lightingScrollOffset = $scope.selectedLightingIndex - visibleItems + 1;
        }
      }
    }
  };

  $scope.$on('lighting-navigate-up', function() {
    $scope.navigateLightingMenu('up');
  });

  $scope.$on('lighting-navigate-down', function() {
    $scope.navigateLightingMenu('down');
  });

  $scope.$on('lighting-select-option', function() {
    const selectedOption = $scope.exteriorLightingOptions[$scope.selectedLightingIndex];
    if (selectedOption && selectedOption.hasCheckbox && !selectedOption.checkboxDisabled) {
      // Toggle the electric value directly
      const currentValue = currentElectricValues[selectedOption.electricKey] || 0;
      const newValue = currentValue === 1 ? 0 : 1;
      
      // Update our cache immediately for responsive UI
      currentElectricValues[selectedOption.electricKey] = newValue;
      selectedOption.checked = newValue === 1;
      
      // Send to Lua
      writeElectricValue(selectedOption.electricKey, newValue);
    } else if (selectedOption && selectedOption.name === "Daytime Running Lights") {
      $scope.$parent.currentMenu = 'daytimeRunningLights';
    } else if (selectedOption && selectedOption.name === "Footwell Lighting") {
      $scope.$parent.currentMenu = 'footwellLighting';
    } else if (selectedOption && selectedOption.name === "Starlights") {
      $scope.$parent.currentMenu = 'starlights';
    }
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'exteriorLighting' && oldMenu === 'vehicleSettings') {
      $scope.selectedLightingIndex = 1;
      $scope.lightingScrollOffset = 0;
    }
  });
});