angular.module('infotainmentScreen')
.controller('EqualizerController', function($scope) {
  // Initialize equalizer settings
  $scope.equalizerSettings = [
    { name: "Fade", value: 50, min: 0, max: 100, unit: "F/R", disabled: false },
    { name: "Balance", value: 50, min: 0, max: 100, unit: "L/R", disabled: false },
    //{ name: "Bass", value: 50, min: 0, max: 100, unit: "%", disabled: true }  // Default to disabled
  ];

  $scope.selectedEqualizerIndex = 0;
  $scope.equalizerScrollOffset = 0;
  $scope.subwooferEquipped = false;
  
  // Get scroll wheel rotation based on selected index
  $scope.getEqualizerScrollWheelRotation = function() {
    const visibleItems = 4;
    const relativeIndex = $scope.selectedEqualizerIndex - $scope.equalizerScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 45;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 45;
    }
  };

  // Get transform for menu scrolling
  $scope.getEqualizerScrollTransform = function() {
    const itemHeight = 65; // Height of each equalizer item
    return 'translateY(' + ($scope.equalizerScrollOffset * -itemHeight) + 'px)';
  };

  // Scrollbar calculations
  $scope.getEqualizerScrollbarThumbHeight = function() {
    const visibleItems = 4;
    const totalItems = $scope.equalizerSettings.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    return Math.max(thumbHeight, 80);
  };

  $scope.getEqualizerScrollbarThumbPosition = function() {
    const visibleItems = 4;
    const totalItems = $scope.equalizerSettings.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getEqualizerScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.equalizerScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  // Navigate up/down through settings
  $scope.navigateEqualizerMenu = function(direction) {
    const visibleItems = 4;
    const totalItems = $scope.equalizerSettings.length;
    
    if (direction === 'up') {
      let nextIndex = $scope.selectedEqualizerIndex;
      
      // Find the next non-disabled item upward
      do {
        nextIndex--;
        if (nextIndex < 0) {
          return; // Stop if we hit the top
        }
      } while ($scope.equalizerSettings[nextIndex] && $scope.equalizerSettings[nextIndex].disabled);
      
      if (nextIndex >= 0) {
        $scope.selectedEqualizerIndex = nextIndex;
        if ($scope.selectedEqualizerIndex < $scope.equalizerScrollOffset) {
          $scope.equalizerScrollOffset = $scope.selectedEqualizerIndex;
        }
      }
    } else if (direction === 'down') {
      let nextIndex = $scope.selectedEqualizerIndex;
      
      // Find the next non-disabled item downward
      do {
        nextIndex++;
        if (nextIndex >= totalItems) {
          return; // Stop if we hit the bottom
        }
      } while ($scope.equalizerSettings[nextIndex] && $scope.equalizerSettings[nextIndex].disabled);
      
      if (nextIndex < totalItems) {
        $scope.selectedEqualizerIndex = nextIndex;
        if ($scope.selectedEqualizerIndex >= $scope.equalizerScrollOffset + visibleItems) {
          $scope.equalizerScrollOffset = $scope.selectedEqualizerIndex - (visibleItems - 1);
        }
      }
    }
  };

  // Send equalizer values to Lua
  function sendEqualizerToLua() {
    if (!window.beamng || !window.beamng.sendActiveObjectLua) return;
    
    // Find fade, balance, and bass settings
    let fadeValue = 50;
    let balanceValue = 50;
    let bassValue = 50;
    
    for (let setting of $scope.equalizerSettings) {
      if (setting.name === 'Fade') {
        fadeValue = setting.value;
      } else if (setting.name === 'Balance') {
        balanceValue = setting.value;
      } else if (setting.name === 'Bass') {
        bassValue = setting.value;
      }
    }
    
    // Send values to Lua electrics system
    const luaCommand = `
      if electrics and electrics.values then
        electrics.values.equalizerFade = ${fadeValue}
        electrics.values.equalizerBalance = ${balanceValue}
        electrics.values.equalizerBass = ${bassValue}
      end
    `;
    window.beamng.sendActiveObjectLua(luaCommand);
    
    console.log(`Equalizer: Sent to Lua - Fade: ${fadeValue}, Balance: ${balanceValue}, Bass: ${bassValue} (Subwoofer equipped: ${$scope.subwooferEquipped})`);
  }

  // Adjust the selected setting value
  $scope.adjustEqualizerValue = function(direction) {
    const setting = $scope.equalizerSettings[$scope.selectedEqualizerIndex];
    if (!setting || setting.disabled) return; // Don't adjust if disabled
    
    const increment = 10; // 10% increment
    
    if (direction === 'left') {
      setting.value = Math.max(setting.min, setting.value - increment);
    } else if (direction === 'right') {
      setting.value = Math.min(setting.max, setting.value + increment);
    }
    
    console.log(`Equalizer: ${setting.name} adjusted to ${setting.value}`);
    
    // Send updated values to Lua
    sendEqualizerToLua();
  };

  // Get the display value for fade/balance/bass
  $scope.getDisplayValue = function(setting) {
    if (setting.disabled && setting.name === 'Bass') {
      return '50';
    }
    
    if (setting.name === 'Fade') {
      if (setting.value < 50) {
        return 'R' + (50 - setting.value);
      } else if (setting.value > 50) {
        return 'F' + (setting.value - 50);
      } else {
        return 'Center';
      }
    } else if (setting.name === 'Balance') {
      if (setting.value < 50) {
        return 'L' + (50 - setting.value);
      } else if (setting.value > 50) {
        return 'R' + (setting.value - 50);
      } else {
        return 'Center';
      }
    } else {
      return setting.value + '%';
    }
  };

  // Update subwoofer status and Bass setting availability
  function updateSubwooferStatus(equipped) {
    $scope.subwooferEquipped = equipped;
    
    // Find the Bass setting and update its disabled state
    for (let setting of $scope.equalizerSettings) {
      if (setting.name === 'Bass') {
        setting.disabled = !equipped;
        
        // If currently selected Bass and it becomes disabled, move to previous available option
        if (!equipped && $scope.selectedEqualizerIndex === 2) {
          // Move to Balance
          $scope.selectedEqualizerIndex = 1;
        }
        break;
      }
    }
    
    console.log('Equalizer: Subwoofer equipped:', equipped, '- Bass control', equipped ? 'enabled' : 'disabled');
  }

  // Load equalizer settings from Lua
  function loadEqualizerSettings(settings) {
    if (settings && settings.equalizer) {
      console.log('Equalizer: Loading settings from Lua:', settings.equalizer);
      
      for (let setting of $scope.equalizerSettings) {
        if (setting.name === 'Fade' && settings.equalizer.fade !== undefined) {
          setting.value = settings.equalizer.fade;
        } else if (setting.name === 'Balance' && settings.equalizer.balance !== undefined) {
          setting.value = settings.equalizer.balance;
        } else if (setting.name === 'Bass' && settings.equalizer.bass !== undefined) {
          setting.value = settings.equalizer.bass;
        }
      }
    }
  }

  // Event listeners
  $scope.$on('equalizer-navigate-up', function() {
    $scope.navigateEqualizerMenu('up');
  });

  $scope.$on('equalizer-navigate-down', function() {
    $scope.navigateEqualizerMenu('down');
  });

  $scope.$on('equalizer-navigate-left', function() {
    $scope.adjustEqualizerValue('left');
  });

  $scope.$on('equalizer-navigate-right', function() {
    $scope.adjustEqualizerValue('right');
  });

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    loadEqualizerSettings(settings);
  });

  // Reset on menu entry
  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'equalizer' && oldMenu === 'media') {
      // When entering equalizer, skip Bass if it's disabled
      if ($scope.equalizerSettings[2].disabled) {
        $scope.selectedEqualizerIndex = 0; // Start at Fade
      } else {
        $scope.selectedEqualizerIndex = 0; // Still start at Fade
      }
      $scope.equalizerScrollOffset = 0;
    }
  });

  // Function to receive electric data updates
  $scope.updateElectricData = function(data) {
    // Update from Lua if values have changed externally
    if (data && data.electrics) {
      // Check for subwoofer equipped status
      if (data.electrics.subwooferEquipped !== undefined) {
        const isEquipped = data.electrics.subwooferEquipped === 1;
        if (isEquipped !== $scope.subwooferEquipped) {
          updateSubwooferStatus(isEquipped);
        }
      }
      
      // Update equalizer values
      if (data.electrics.equalizerFade !== undefined) {
        for (let setting of $scope.equalizerSettings) {
          if (setting.name === 'Fade') {
            setting.value = data.electrics.equalizerFade;
            break;
          }
        }
      }
      if (data.electrics.equalizerBalance !== undefined) {
        for (let setting of $scope.equalizerSettings) {
          if (setting.name === 'Balance') {
            setting.value = data.electrics.equalizerBalance;
            break;
          }
        }
      }
      if (data.electrics.equalizerBass !== undefined) {
        for (let setting of $scope.equalizerSettings) {
          if (setting.name === 'Bass' && !setting.disabled) {
            setting.value = data.electrics.equalizerBass;
            break;
          }
        }
      }
    }
  };

  // Initialize by sending default values to Lua
  sendEqualizerToLua();
});