angular.module('infotainmentScreen')
.controller('DoorsKeyController', function($scope) {
  // Options with electric keys - checked state will be determined by electric values
  var defaultOptions = [
    { name: "Fold mirrors when locked", disabled: false, hasCheckbox: true, electricKey: "mirror_fold_enabled", image: "doors_comfort.png" },
    { name: "Flash for lock/unlock", disabled: false, hasCheckbox: true, electricKey: "flashLockUnlock", image: "flash_lock.png" },
    { name: "Acoustic Lock/Unlock Beep", disabled: false, hasCheckbox: true, electricKey: "lockUnlockSoundEnabled", image: "flash_lock.png" },
  ];

  $scope.doorsKeyOptions = angular.copy(defaultOptions);
  $scope.selectedDoorsKeyIndex = 1;
  $scope.doorsKeyScrollOffset = 0;

  // Current electric values cache
  var currentElectricValues = {};

  $scope.getCurrentDoorsKeyImage = function() {
    if ($scope.selectedDoorsKeyIndex >= 0 && $scope.selectedDoorsKeyIndex < $scope.doorsKeyOptions.length) {
      return $scope.doorsKeyOptions[$scope.selectedDoorsKeyIndex].image;
    }
    return 'default_doors_key.png';
  };

  function writeElectricValue(key, value) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.${key} = ${value}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
    } else if (window.bngApi && window.bngApi.engineLua) {
      const luaCommand = `
        local veh = be:getPlayerVehicle(0)
        if veh then
          veh:queueLuaCommand('if electrics and electrics.values then electrics.values.${key} = ${value} end')
        end
      `;
      
      window.bngApi.engineLua(luaCommand);
    }
  }

  // Update checkbox states based on current electric values
  function updateCheckboxStates() {
    $scope.doorsKeyOptions.forEach(function(option) {
      if (option.electricKey && currentElectricValues.hasOwnProperty(option.electricKey)) {
        option.checked = currentElectricValues[option.electricKey] === 1;
      }
    });
  }

  // Process electric data and update checkboxes
  function processElectricData(data) {
    if (!data || !data.electrics) return;
    
    var hasChanges = false;
    const e = data.electrics;
    
    // Check for changes in electric values we care about
    $scope.doorsKeyOptions.forEach(function(option) {
      if (option.electricKey && e.hasOwnProperty(option.electricKey)) {
        var newValue = e[option.electricKey];
        if (currentElectricValues[option.electricKey] !== newValue) {
          currentElectricValues[option.electricKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Update checkbox states if any values changed
    if (hasChanges) {
      updateCheckboxStates();
    }
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('DoorsKey: Received settings:', settings);
    
    // Update our electric values cache
    Object.keys(settings).forEach(function(key) {
      currentElectricValues[key] = settings[key];
    });
    
    // Update checkbox states
    updateCheckboxStates();
  });

  // Listen for electric data updates (called from main controller)
  $scope.updateElectricData = function(data) {
    processElectricData(data);
  };

  $scope.getDoorsKeyScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedDoorsKeyIndex - $scope.doorsKeyScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getDoorsKeyScrollTransform = function() {
    return 'translateY(' + ($scope.doorsKeyScrollOffset * -40) + 'px)';
  };

  $scope.getDoorsKeyScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.doorsKeyOptions.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getDoorsKeyScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.doorsKeyOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getDoorsKeyScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.doorsKeyScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableDoorsKeyOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.doorsKeyOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.doorsKeyOptions.length; i++) {
        if (!$scope.doorsKeyOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateDoorsKeyMenu = function(direction) {
    if (direction === 'up' && $scope.selectedDoorsKeyIndex > 0) {
      const newIndex = $scope.findNextSelectableDoorsKeyOption('up', $scope.selectedDoorsKeyIndex);
      if (newIndex !== $scope.selectedDoorsKeyIndex) {
        $scope.selectedDoorsKeyIndex = newIndex;
        if ($scope.selectedDoorsKeyIndex < $scope.doorsKeyScrollOffset) {
          $scope.doorsKeyScrollOffset = $scope.selectedDoorsKeyIndex;
        }
      }
    } else if (direction === 'down' && $scope.selectedDoorsKeyIndex < $scope.doorsKeyOptions.length - 1) {
      const newIndex = $scope.findNextSelectableDoorsKeyOption('down', $scope.selectedDoorsKeyIndex);
      if (newIndex !== $scope.selectedDoorsKeyIndex) {
        $scope.selectedDoorsKeyIndex = newIndex;
        const visibleItems = 6;
        if ($scope.selectedDoorsKeyIndex >= $scope.doorsKeyScrollOffset + visibleItems) {
          $scope.doorsKeyScrollOffset = $scope.selectedDoorsKeyIndex - visibleItems + 1;
        }
      }
    }
  };

  $scope.$on('doors-key-navigate-up', function() {
    $scope.navigateDoorsKeyMenu('up');
  });

  $scope.$on('doors-key-navigate-down', function() {
    $scope.navigateDoorsKeyMenu('down');
  });

  $scope.$on('doors-key-select-option', function() {
    const selectedOption = $scope.doorsKeyOptions[$scope.selectedDoorsKeyIndex];
    if (selectedOption && selectedOption.hasCheckbox && !selectedOption.checkboxDisabled) {
      // Toggle the electric value directly
      const currentValue = currentElectricValues[selectedOption.electricKey] || 0;
      const newValue = currentValue === 1 ? 0 : 1;
      
      // Update our cache immediately for responsive UI
      currentElectricValues[selectedOption.electricKey] = newValue;
      selectedOption.checked = newValue === 1;
      
      // Send to Lua
      writeElectricValue(selectedOption.electricKey, newValue);
    } else if (selectedOption && selectedOption.hasBorder) {
      // Handle tailgate or other border options if needed
    }
  });

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'doorsKey' && oldMenu === 'vehicleSettings') {
      $scope.selectedDoorsKeyIndex = 1;
      $scope.doorsKeyScrollOffset = 0;
    }
  });
});