angular.module('infotainmentScreen')
.controller('DaytimeRunningLightsController', function($scope) {
  // Options with electric keys - checked state will be determined by electric values
  var defaultOptions = [
    { name: "Enabled", disabled: false, hasCheckbox: true, electricKey: "daytimeLights" },
    { name: "Color Selector", disabled: false}
  ];

  $scope.daytimeRunningLightsOptions = angular.copy(defaultOptions);
  $scope.selectedDaytimeLightsIndex = 0;
  $scope.daytimeLightsScrollOffset = 0;
  $scope.colorPickerMode = false;
  $scope.colorCursor = { x: 50, y: 0 }; // Start at top middle (white)

  // Current electric values cache
  var currentElectricValues = {};

  $scope.getCurrentDaytimeLightsImage = function() {
    if ($scope.selectedDaytimeLightsIndex >= 0 && $scope.selectedDaytimeLightsIndex < $scope.daytimeRunningLightsOptions.length) {
      return $scope.daytimeRunningLightsOptions[$scope.selectedDaytimeLightsIndex].image || 'default_lighting.png';
    }
    return 'default_lighting.png';
  };

  $scope.showColorPicker = function() {
    return $scope.selectedDaytimeLightsIndex === 1;
  };

  $scope.shouldShowMenuSelection = function() {
    return !$scope.colorPickerMode;
  };

  $scope.getCursorClass = function() {
    return $scope.colorPickerMode ? 'active' : 'inactive';
  };

  // Function to trigger persistence save using electric trigger
  function triggerPersistenceSave() {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      // Use a timestamp to ensure the trigger value changes
      const timestamp = Date.now();
      window.beamng.sendActiveObjectLua(`
        if electrics and electrics.values then
          electrics.values.infotainmentSaveTrigger = ${timestamp}
        end
      `);
    }
  }

  // Simple function to write electric values
  function setElectricValues(r, g, b) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.daytimeLightColorR = ${r}
          electrics.values.daytimeLightColorG = ${g}
          electrics.values.daytimeLightColorB = ${b}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    }
  }

  function writeElectricValue(key, value) {
    if (window.beamng && window.beamng.sendActiveObjectLua) {
      const luaCommand = `
        if electrics and electrics.values then
          electrics.values.${key} = ${value}
        end
      `;
      
      window.beamng.sendActiveObjectLua(luaCommand);
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    } else if (window.bngApi && window.bngApi.engineLua) {
      const luaCommand = `
        local veh = be:getPlayerVehicle(0)
        if veh then
          veh:queueLuaCommand('if electrics and electrics.values then electrics.values.${key} = ${value} end')
        end
      `;
      
      window.bngApi.engineLua(luaCommand);
      
      // Trigger persistence save
      setTimeout(triggerPersistenceSave, 100);
    }
  }

  // Update checkbox states based on current electric values
  function updateCheckboxStates() {
    $scope.daytimeRunningLightsOptions.forEach(function(option) {
      if (option.electricKey && currentElectricValues.hasOwnProperty(option.electricKey)) {
        option.checked = currentElectricValues[option.electricKey] === 1;
      }
    });
  }

  // Process electric data and update checkboxes
  function processElectricData(data) {
    if (!data || !data.electrics) return;
    
    var hasChanges = false;
    const e = data.electrics;
    
    // Check for changes in electric values we care about
    $scope.daytimeRunningLightsOptions.forEach(function(option) {
      if (option.electricKey && e.hasOwnProperty(option.electricKey)) {
        var newValue = e[option.electricKey];
        if (currentElectricValues[option.electricKey] !== newValue) {
          currentElectricValues[option.electricKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Also check for color values
    ['daytimeLightColorR', 'daytimeLightColorG', 'daytimeLightColorB'].forEach(function(colorKey) {
      if (e.hasOwnProperty(colorKey)) {
        var newValue = e[colorKey];
        if (currentElectricValues[colorKey] !== newValue) {
          currentElectricValues[colorKey] = newValue;
          hasChanges = true;
        }
      }
    });
    
    // Update checkbox states if any values changed
    if (hasChanges) {
      updateCheckboxStates();
    }
  }

  // Listen for settings loaded from Lua
  $scope.$on('settings-loaded', function(event, settings) {
    console.log('DaytimeLights: Received settings:', settings);
    
    // Update our electric values cache
    Object.keys(settings).forEach(function(key) {
      currentElectricValues[key] = settings[key];
    });
    
    // Update checkbox states
    updateCheckboxStates();
    
    // Apply color settings if available
    if (settings.daytimeLightColorR !== undefined && 
        settings.daytimeLightColorG !== undefined && 
        settings.daytimeLightColorB !== undefined) {
      setElectricValues(settings.daytimeLightColorR, settings.daytimeLightColorG, settings.daytimeLightColorB);
      console.log('DaytimeLights: Applied color RGB(' + settings.daytimeLightColorR + ',' + settings.daytimeLightColorG + ',' + settings.daytimeLightColorB + ')');
    }
    
  });

  // Listen for electric data updates (called from main controller)
  $scope.updateElectricData = function(data) {
    processElectricData(data);
  };

  $scope.getColorAtCursor = function() {
    if (!$scope.colorPickerMode) return;
    
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    const img = new Image();
    
    img.onload = function() {
      canvas.width = img.width;
      canvas.height = img.height;
      ctx.drawImage(img, 0, 0);
      
      const x = Math.floor((img.width * $scope.colorCursor.x) / 100);
      const y = Math.floor((img.height * $scope.colorCursor.y) / 100);
      
      const clampedX = Math.max(0, Math.min(img.width - 1, x));
      const clampedY = Math.max(0, Math.min(img.height - 1, y));
      
      const imageData = ctx.getImageData(clampedX, clampedY, 1, 1);
      const [r, g, b, a] = imageData.data;
      
      if (a > 0) {
        console.log(`Color at cursor: RGB(${r}, ${g}, ${b})`);
        setElectricValues(r, g, b);
      }
    };
    
    img.src = '/vehicles/RLA_A90/infotainment_screen/icons/color_wheel.png';
  };

  // Throttled color updates
  let colorUpdateTimeout = null;

  $scope.moveColorCursor = function(direction) {
    if (!$scope.colorPickerMode) return;
    
    const step = 2;
    
    switch(direction) {
      case 'up':
        $scope.colorCursor.y = Math.max(0, $scope.colorCursor.y - step);
        break;
      case 'down':
        $scope.colorCursor.y = Math.min(100, $scope.colorCursor.y + step);
        break;
      case 'left':
        $scope.colorCursor.x = Math.max(0, $scope.colorCursor.x - step);
        break;
      case 'right':
        $scope.colorCursor.x = Math.min(100, $scope.colorCursor.x + step);
        break;
    }
    
    if (colorUpdateTimeout) {
      clearTimeout(colorUpdateTimeout);
    }
    
    colorUpdateTimeout = setTimeout(function() {
      $scope.getColorAtCursor();
    }, 50);
  };

  // Navigation handlers
  $scope.getDaytimeLightsScrollWheelRotation = function() {
    const visibleItems = 6;
    const relativeIndex = $scope.selectedDaytimeLightsIndex - $scope.daytimeLightsScrollOffset;
    
    if (relativeIndex >= 0 && relativeIndex < visibleItems) {
      return (relativeIndex + 1) * 25;
    } else {
      return Math.min(visibleItems, Math.max(1, relativeIndex + 1)) * 25;
    }
  };

  $scope.getDaytimeLightsScrollTransform = function() {
    return 'translateY(' + ($scope.daytimeLightsScrollOffset * -40) + 'px)';
  };

  $scope.getDaytimeLightsScrollbarThumbHeight = function() {
    const visibleItems = 6;
    const totalItems = $scope.daytimeRunningLightsOptions.length;
    const scrollbarHeight = 260;
    
    const thumbHeight = (visibleItems / totalItems) * scrollbarHeight;
    
    return Math.max(thumbHeight, 80);
  };

  $scope.getDaytimeLightsScrollbarThumbPosition = function() {
    const visibleItems = 6;
    const totalItems = $scope.daytimeRunningLightsOptions.length;
    const maxScroll = Math.max(0, totalItems - visibleItems);
    
    if (maxScroll === 0) return 0;
    
    const scrollbarHeight = 260;
    const thumbHeight = $scope.getDaytimeLightsScrollbarThumbHeight();
    const availableSpace = scrollbarHeight - thumbHeight;
    
    const scrollPercentage = $scope.daytimeLightsScrollOffset / maxScroll;
    return scrollPercentage * availableSpace;
  };

  $scope.findNextSelectableDaytimeLightsOption = function(direction, currentIndex) {
    let nextIndex = currentIndex;
    
    if (direction === 'up') {
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (!$scope.daytimeRunningLightsOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    } else if (direction === 'down') {
      for (let i = currentIndex + 1; i < $scope.daytimeRunningLightsOptions.length; i++) {
        if (!$scope.daytimeRunningLightsOptions[i].disabled) {
          nextIndex = i;
          break;
        }
      }
    }
    
    return nextIndex;
  };

  $scope.navigateDaytimeLightsMenu = function(direction) {
    if ($scope.colorPickerMode) return;
    
    if (direction === 'up' && $scope.selectedDaytimeLightsIndex > 0) {
      const newIndex = $scope.findNextSelectableDaytimeLightsOption('up', $scope.selectedDaytimeLightsIndex);
      if (newIndex !== $scope.selectedDaytimeLightsIndex) {
        $scope.selectedDaytimeLightsIndex = newIndex;
        if ($scope.selectedDaytimeLightsIndex < $scope.daytimeLightsScrollOffset) {
          $scope.daytimeLightsScrollOffset = $scope.selectedDaytimeLightsIndex;
        }
      }
    } else if (direction === 'down' && $scope.selectedDaytimeLightsIndex < $scope.daytimeRunningLightsOptions.length - 1) {
      const newIndex = $scope.findNextSelectableDaytimeLightsOption('down', $scope.selectedDaytimeLightsIndex);
      if (newIndex !== $scope.selectedDaytimeLightsIndex) {
        $scope.selectedDaytimeLightsIndex = newIndex;
        const visibleItems = 6;
        if ($scope.selectedDaytimeLightsIndex >= $scope.daytimeLightsScrollOffset + visibleItems) {
          $scope.daytimeLightsScrollOffset = $scope.selectedDaytimeLightsIndex - visibleItems + 1;
        }
      }
    }
  };

  // Event handlers
  $scope.$on('daytime-lights-navigate-up', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('up');
    } else {
      $scope.navigateDaytimeLightsMenu('up');
    }
  });

  $scope.$on('daytime-lights-navigate-down', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('down');
    } else {
      $scope.navigateDaytimeLightsMenu('down');
    }
  });

  $scope.$on('daytime-lights-navigate-left', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('left');
    }
  });

  $scope.$on('daytime-lights-navigate-right', function() {
    if ($scope.colorPickerMode) {
      $scope.moveColorCursor('right');
    }
  });

  $scope.$on('daytime-lights-select-option', function() {
    const selectedOption = $scope.daytimeRunningLightsOptions[$scope.selectedDaytimeLightsIndex];
    if (selectedOption && selectedOption.hasCheckbox && !selectedOption.checkboxDisabled) {
      // Toggle the electric value directly
      const currentValue = currentElectricValues[selectedOption.electricKey] || 0;
      const newValue = currentValue === 1 ? 0 : 1;
      
      // Update our cache immediately for responsive UI
      currentElectricValues[selectedOption.electricKey] = newValue;
      selectedOption.checked = newValue === 1;
      
      // Send to Lua
      writeElectricValue(selectedOption.electricKey, newValue);
    } else if (selectedOption && selectedOption.name === "Color Selector") {
      $scope.colorPickerMode = true;
      $scope.getColorAtCursor();
    }
  });

  $scope.$on('daytime-lights-back', function() {
    if ($scope.colorPickerMode) {
      $scope.colorPickerMode = false;
    } else {
      $scope.$parent.currentMenu = 'exteriorLighting';
    }
  });

  $scope.handleBackButton = function() {
    if ($scope.colorPickerMode) {
      $scope.colorPickerMode = false;
      return false;
    } else {
      return true;
    }
  };

  $scope.$watch('$parent.currentMenu', function(newMenu, oldMenu) {
    if (newMenu === 'daytimeRunningLights' && oldMenu === 'exteriorLighting') {
      $scope.selectedDaytimeLightsIndex = 0;
      $scope.daytimeLightsScrollOffset = 0;
      $scope.colorPickerMode = false;
    }
  });
});