angular.module('gaugeInfo', [])
.controller('GaugeInfoController', function($scope, $window) {
  var units = {
    uiUnitConsumptionRate: "metric",
    uiUnitDate: "ger",
    uiUnitEnergy: "metric",
    uiUnitLength: "metric",
    uiUnitPower: "hp",
    uiUnitPressure: "bar",
    uiUnitTemperature: "c",
    uiUnitTorque: "metric",
    uiUnitVolume: "l",
    uiUnitWeight: "kg"
  };

  // Initialize values
  $scope.speed = 0;
  $scope.speedUnit = 'km/h';
  $scope.gear = 'P';
  $scope.osTime = '12:00 am';
  $scope.envTemp = '+ 20';
  $scope.tempUnit = '°C';
  $scope.tripOdometer = '0000.0 km';
  $scope.odometer = '00000 km';
  $scope.distanceUnit = 'km';

  // Setup function to handle unit settings (like in your working example)
  $window.setup = (setupData) => {
    for(let dk in setupData){
      if(typeof dk == "string" && dk.startsWith("uiUnit")){
        units[dk] = setupData[dk];
      }
    }

    // Set units based on user settings
    $scope.speedUnit = units.uiUnitLength == "metric" ? "km/h" : "mph";
    $scope.distanceUnit = units.uiUnitLength == "metric" ? "km" : "mi";
    $scope.tempUnit = units.uiUnitTemperature == "c" ? "°C" : "°F";
  }

  $window.updateData = function(data) {
    $scope.$evalAsync(function() {
      if (data && data.electrics) {
        // Ensure distance unit is set (fallback if setup wasn't called)
        $scope.distanceUnit = units.uiUnitLength == "metric" ? "km" : "mi";
        
        // Use the same speed conversion as your working example
        if (typeof UiUnits !== 'undefined' && UiUnits.speed) {
          let speedData = UiUnits.speed(data.electrics.wheelspeed || data.electrics.airspeed || 0);
          $scope.speed = speedData.val.toFixed(0);
          $scope.speedUnit = speedData.unit;
        } else {
          // Fallback manual conversion
          let isImperial = (units.uiUnitLength !== "metric");
          if (data.electrics.airspeed !== undefined) {
            if (isImperial) {
              $scope.speed = Math.round(data.electrics.airspeed * 2.237) || 0; // m/s to mph
            } else {
              $scope.speed = Math.round(data.electrics.airspeed * 3.6) || 0; // m/s to km/h
            }
          } else if (data.electrics.wheelspeed !== undefined) {
            if (isImperial) {
              $scope.speed = Math.round(data.electrics.wheelspeed * 2.237) || 0;
            } else {
              $scope.speed = Math.round(data.electrics.wheelspeed * 3.6) || 0;
            }
          }
        }
        
        // Gear
        if (data.electrics.gear_A !== undefined) {
          let gearValue = data.electrics.gear_A;
          if (gearValue === 0) {
            $scope.gear = 'N';
          } else if (gearValue < 0) {
            $scope.gear = 'R';
          } else {
            $scope.gear = gearValue.toString();
          }
        } else if (data.electrics.gear !== undefined) {
          if (data.electrics.gear === -1) {
            $scope.gear = 'R';
          } else if (data.electrics.gear === 0) {
            $scope.gear = 'N';
          } else {
            $scope.gear = data.electrics.gear.toString() || 'P';
          }
        }
        
        // Temperature - using UiUnits like your working example
        if (data.customModules && data.customModules.environmentData && data.customModules.environmentData.temperatureEnv !== undefined) {
          if (typeof UiUnits !== 'undefined' && UiUnits.temperature) {
            let tempData = UiUnits.temperature(data.customModules.environmentData.temperatureEnv);
            if (tempData.val > 99.9 || tempData.val < -99.9) {
              $scope.envTemp = '---';
            } else {
              $scope.envTemp = '+' + tempData.val.toFixed(0);
            }
            $scope.tempUnit = tempData.unit;
          } else {
            // Fallback manual conversion
            let temp = data.customModules.environmentData.temperatureEnv;
            if (units.uiUnitTemperature !== "c") {
              temp = (temp * 9/5) + 32;
              $scope.tempUnit = '°F';
            } else {
              $scope.tempUnit = '°C';
            }
            $scope.envTemp = '+' + Math.round(temp);
          }
        }
        
        // OS Time
        let timeValue = (data.customModules && data.customModules.environmentData && data.customModules.environmentData.time) || data.electrics.time;
        if (timeValue !== undefined && timeValue !== null && !isNaN(timeValue)) {
          if (typeof timeValue === 'string') {
            $scope.osTime = timeValue;
          } else {
            let timeSeconds = parseFloat(timeValue);
            if (!isNaN(timeSeconds)) {
              let hours = Math.floor(timeSeconds / 3600) % 24;
              let minutes = Math.floor((timeSeconds % 3600) / 60);
              let period = hours >= 12 ? 'pm' : 'am';
              let displayHours = hours % 12;
              if (displayHours === 0) displayHours = 12;
              
              $scope.osTime = displayHours + ':' + String(minutes).padStart(2, '0') + ' ' + period;
            } else {
              let now = new Date();
              $scope.osTime = now.toLocaleTimeString('en-US', {
                hour: 'numeric',
                minute: '2-digit',
                hour12: true
              }).toLowerCase();
            }
          }
        } else {
          let now = new Date();
          $scope.osTime = now.toLocaleTimeString('en-US', {
            hour: 'numeric',
            minute: '2-digit',
            hour12: true
          }).toLowerCase();
        }
        
        // Trip odometer
        if (data.electrics.tripOdometer !== undefined) {
          let val = data.electrics.tripOdometer;
          val *= (units.uiUnitLength == "metric") ? 0.001 : 0.000621371; // Convert to km or miles
          
          if (val > 9999.9) {
            val = 0;
          }
          
          $scope.tripOdometer = String(val.toFixed(1)).padStart(6, '0') + " " + ((units.uiUnitLength == "metric") ? "km" : "mi");
        } else if (data.electrics.trip !== undefined) {
          let val = data.electrics.trip;
          val *= (units.uiUnitLength == "metric") ? 0.001 : 0.000621371;
          
          if (val > 9999.9) {
            val = 0;
          }
          
          $scope.tripOdometer = String(val.toFixed(1)).padStart(6, '0') + " " + ((units.uiUnitLength == "metric") ? "km" : "mi");
        } else {
          $scope.tripOdometer = '0000.0' + " " + ((units.uiUnitLength == "metric") ? "km" : "mi");
        }
        
        // Main odometer - using the same method as your working example
        if (data.electrics.odometer) {
          let val = data.electrics.odometer;
          val *= (units.uiUnitLength == "metric") ? 0.001 : 0.0006215;
          val = Math.min(999999, val);
          
          let odometerValue = Math.floor(val);
          if (odometerValue < 100000) {
            $scope.odometer = String(odometerValue).padStart(5, '0') + " " + ((units.uiUnitLength == "metric") ? "km" : "mi");
          } else {
            $scope.odometer = String(odometerValue) + " " + ((units.uiUnitLength == "metric") ? "km" : "mi");
          }
        } else {
          $scope.odometer = '00000' + " " + ((units.uiUnitLength == "metric") ? "km" : "mi");
        }
      }
    });
  };
});