-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
M.type = "auxiliary"
M.relevantDevice = nil

local abs = math.abs
local max = math.max

local idleFrontPosition = -0.066
local idleRearPosition = -0.07
local idleMidPosition = 0

local mediumSpeedFrontPosition = 0
local mediumSpeedRearPosition = 0
local mediumSpeedMidPosition = 0

local highSpeedFrontPosition = 0
local highSpeedRearPosition = 0
local highSpeedMidPosition = 0

local highSpeedCorneringFrontPosition = 0
local highSpeedCorneringRearPosition = 0
local highSpeedCorneringMidPosition = 0


local brakingMidPosition = 0

local transitionTimeIdle = 4.0
local transitionTimeBraking = 0.25

local lastEngineRunning = 0
local lastAirBrakeActive = false
local lastMediumSpeedPositionActive
local lastHighSpeedPositionActive
local lastHighSpeedCorneringPositionActive

local frontPositionSmoother = newTemporalSmoothing(1000, 1000)
local rearPositionSmoother = newTemporalSmoothing(1000, 1000)
local midPositionSmoother = newTemporalSmoothing(1000, 1000)
local highSpeedCorneringInputSmoother = newTemporalSmoothing(10, 1000)

local speedThresholdMedium = 20
local speedThresholdHigh = 50
local brakeThresholdHigh = 0.4
local speedThresholdAirBrake = 5

local targetFrontPosition
local targetRearPosition
local targetMidPosition
local currentFrontPosition
local currentRearPosition
local currentMidPosition
local positionTransitionTimer

local sensorHub

local function updateGFX(dt)
  targetFrontPosition = idleFrontPosition
  targetRearPosition = idleRearPosition
  targetMidPosition = idleMidPosition

  currentFrontPosition = electrics.values.spoilerF
  currentRearPosition = electrics.values.spoilerR
  currentMidPosition = electrics.values.spoilerM

  local speed = electrics.values.wheelspeed

  local mediumSpeedPositionActive = speed > speedThresholdMedium
  if mediumSpeedPositionActive ~= lastMediumSpeedPositionActive then
    positionTransitionTimer = transitionTimeIdle
  end
  if mediumSpeedPositionActive then
    targetFrontPosition = mediumSpeedFrontPosition
    targetRearPosition = mediumSpeedRearPosition
    targetMidPosition = mediumSpeedMidPosition
  end

  local highSpeedPositionActive = speed > speedThresholdHigh
  if highSpeedPositionActive ~= lastHighSpeedPositionActive then
    positionTransitionTimer = transitionTimeBraking
  end
  if highSpeedPositionActive then
    targetFrontPosition = highSpeedFrontPosition
    targetRearPosition = highSpeedRearPosition
    targetMidPosition = highSpeedMidPosition
  end

  local speedHighEnough = speed > speedThresholdHigh
  local accHighEnough = abs(sensors.gx2) > 3
  local yawHighEnough = sensorHub and abs(sensorHub.yawAV) > 0.2
  local steeringHighEnough = abs(electrics.values.steering_input or 0) > 0.1
  local highSpeedCorneringInput = speedHighEnough and (accHighEnough or yawHighEnough or steeringHighEnough)

  local highSpeedCorneringPositionActive = highSpeedCorneringInputSmoother:getUncapped(highSpeedCorneringInput and 1 or 0, dt) > 0
  if highSpeedCorneringPositionActive ~= lastHighSpeedCorneringPositionActive then
    positionTransitionTimer = transitionTimeBraking
  end
  if highSpeedCorneringPositionActive then
    targetFrontPosition = highSpeedCorneringFrontPosition
    targetRearPosition = highSpeedCorneringRearPosition
    targetMidPosition = highSpeedCorneringMidPosition
  end

  local yawControlOverrideAirBrake = electrics.values.yawControlRequestReduceOversteer or 0
  local activateAirBrake = (electrics.values.brake > brakeThresholdHigh or yawControlOverrideAirBrake > 0) and speed >= speedThresholdAirBrake
  if activateAirBrake ~= lastAirBrakeActive then
    positionTransitionTimer = transitionTimeBraking
  end
  if activateAirBrake then

    targetMidPosition = brakingMidPosition
  end

  if electrics.values.engineRunning ~= lastEngineRunning then
    positionTransitionTimer = transitionTimeIdle
  end

  local frontRate = abs(currentFrontPosition - targetFrontPosition) / positionTransitionTimer
  local rearRate = abs(currentRearPosition - targetRearPosition) / positionTransitionTimer
  local midRate = abs(currentMidPosition - targetMidPosition) / positionTransitionTimer

  currentFrontPosition = frontPositionSmoother:getWithRateUncapped(targetFrontPosition, dt, frontRate)
  currentRearPosition = rearPositionSmoother:getWithRateUncapped(targetRearPosition, dt, rearRate)
  currentMidPosition = midPositionSmoother:getWithRateUncapped(targetMidPosition, dt, midRate)

  positionTransitionTimer = max(positionTransitionTimer - dt, 0)

  electrics.values.spoilerF = currentFrontPosition
  electrics.values.spoilerR = currentRearPosition
  electrics.values.spoilerM = currentMidPosition

  lastEngineRunning = electrics.values.engineRunning
  lastAirBrakeActive = activateAirBrake
  lastMediumSpeedPositionActive = mediumSpeedPositionActive
  lastHighSpeedPositionActive = highSpeedPositionActive
  lastHighSpeedCorneringPositionActive = highSpeedCorneringPositionActive
end

local function reset(jbeamData)
  idleFrontPosition = 0
  idleRearPosition = 0
  idleMidPosition = 0
  brakingMidPosition = 0

  electrics.values.spoilerF = idleFrontPosition
  electrics.values.spoilerR = idleRearPosition
  electrics.values.spoilerM = idleMidPosition

  frontPositionSmoother:set(idleFrontPosition)
  rearPositionSmoother:set(idleRearPosition)
  midPositionSmoother:set(idleMidPosition)
  highSpeedCorneringInputSmoother:reset()
end

local function init(jbeamData)
  idleFrontPosition = 0
  idleRearPosition = 0
  idleMidPosition = 0

  brakingMidPosition = 0

  electrics.values.spoilerF = idleFrontPosition
  electrics.values.spoilerR = idleRearPosition
  electrics.values.spoilerM = idleMidPosition

  frontPositionSmoother:set(idleFrontPosition)
  rearPositionSmoother:set(idleRearPosition)
  midPositionSmoother:set(idleMidPosition)
end

local function initLastStage(jbeamData)
  local CMU = controller.getController("CMU")
  if CMU then
    sensorHub = CMU.sensorHub
  end
end

local function setParameters(parameters)
  if parameters.speedThresholdMedium then
    speedThresholdMedium = parameters.speedThresholdMedium
  end
  if parameters.speedThresholdHigh then
    speedThresholdHigh = parameters.speedThresholdHigh
  end
  if parameters.speedThresholdAirBrake then
    speedThresholdAirBrake = parameters.speedThresholdAirBrake
  end

  if parameters.idleFrontPosition then
    idleFrontPosition = parameters.idleFrontPosition
  end
  if parameters.idleRearPosition then
    idleRearPosition = parameters.idleRearPosition
  end
  if parameters.idleMidPosition then
    idleMidPosition = parameters.idleMidPosition
  end

  if parameters.mediumSpeedFrontPosition then
    mediumSpeedFrontPosition = parameters.mediumSpeedFrontPosition
  end
  if parameters.mediumSpeedRearPosition then
    mediumSpeedRearPosition = parameters.mediumSpeedRearPosition
  end
  if parameters.mediumSpeedMidPosition then
    mediumSpeedMidPosition = parameters.mediumSpeedMidPosition
  end

  if parameters.highSpeedFrontPosition then
    highSpeedFrontPosition = parameters.highSpeedFrontPosition
  end
  if parameters.highSpeedRearPosition then
    highSpeedRearPosition = parameters.highSpeedRearPosition
  end
  if parameters.highSpeedMidPosition then
    highSpeedMidPosition = parameters.highSpeedMidPosition
  end

  if parameters.highSpeedCorneringFrontPosition then
    highSpeedCorneringFrontPosition = parameters.highSpeedCorneringFrontPosition
  end
  if parameters.highSpeedCorneringRearPosition then
    highSpeedCorneringRearPosition = parameters.highSpeedCorneringRearPosition
  end
  if parameters.highSpeedCorneringMidPosition then
    highSpeedCorneringMidPosition = parameters.highSpeedCorneringMidPosition
  end


  if parameters.brakingMidPosition then
    brakingMidPosition = parameters.brakingMidPosition
  end

  positionTransitionTimer = transitionTimeIdle
end

M.init = init
M.initLastStage = initLastStage
M.reset = reset
M.updateGFX = updateGFX
M.setParameters = setParameters

return M
