-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}

-- Automation state
local autoRampMode = false
local autoRampState = "idle"
local autoRampTimer = 0
local rampsDeployed = false

-- Tunable timing for each phase (in seconds)
local time_openDoors   = 5.0
local time_extendRamps = 6.0
local time_lowerDoors  = 1.5  -- partial door close for deploy
local time_closeDoors  = 5.0  -- full door close for retract

-- Reset the electric values for all wheels and actions
local function onReset()
  electrics.values['lifter_FL'] = 0
  electrics.values['lifter_FL_input'] = 0
  electrics.values['lifter_FR'] = 0
  electrics.values['lifter_FR_input'] = 0
  electrics.values['shoot'] = 0

  autoRampMode = false
  autoRampState = "idle"
  autoRampTimer = 0
  rampsDeployed = false
end

-- Helper function to update the lifter values for a specific wheel
local function updateLifterValue(lifterKey, inputKey, dt)
  electrics.values[lifterKey] = math.min(1, math.max(0, electrics.values[lifterKey] + electrics.values[inputKey] * dt * 0.2))
end

-- Update the graphics and lifter values for all wheels
local function updateGFX(dt)
  -- Manual override
  updateLifterValue('lifter_FL', 'lifter_FL_input', dt)
  updateLifterValue('lifter_FR', 'lifter_FR_input', dt)

  if autoRampMode then
    autoRampTimer = autoRampTimer + dt

    -- DEPLOY SEQUENCE
    if not rampsDeployed then
      if autoRampState == "idle" then
        autoRampState = "openDoors"
        autoRampTimer = 0

      elseif autoRampState == "openDoors" then
        electrics.values.lifter_FR_input = 1
        electrics.values.lifter_FL_input = 0
        if autoRampTimer >= time_openDoors then
          autoRampState = "extendRamps"
          autoRampTimer = 0
        end

      elseif autoRampState == "extendRamps" then
        electrics.values.lifter_FR_input = 0
        electrics.values.lifter_FL_input = 1
        if autoRampTimer >= time_extendRamps then
          autoRampState = "lowerDoors"
          autoRampTimer = 0
        end

      elseif autoRampState == "lowerDoors" then
        electrics.values.lifter_FR_input = -1  -- lower slightly
        electrics.values.lifter_FL_input = 0
        if autoRampTimer >= time_lowerDoors then
          autoRampState = "done"
          autoRampTimer = 0
        end

      elseif autoRampState == "done" then
        electrics.values.lifter_FR_input = 0
        electrics.values.lifter_FL_input = 0
        autoRampMode = false
        rampsDeployed = true
      end

    -- RETRACT SEQUENCE
    else
      if autoRampState == "idle" then
        autoRampState = "openDoors"
        autoRampTimer = 0

      elseif autoRampState == "openDoors" then
        electrics.values.lifter_FR_input = 1
        electrics.values.lifter_FL_input = 0
        if autoRampTimer >= time_openDoors then
          autoRampState = "retractRamps"
          autoRampTimer = 0
        end

      elseif autoRampState == "retractRamps" then
        electrics.values.lifter_FR_input = 0
        electrics.values.lifter_FL_input = -1
        if autoRampTimer >= time_extendRamps then
          autoRampState = "closeDoors"
          autoRampTimer = 0
        end

      elseif autoRampState == "closeDoors" then
        electrics.values.lifter_FR_input = -1  -- FULL close
        electrics.values.lifter_FL_input = 0
        if autoRampTimer >= time_closeDoors then
          autoRampState = "done"
          autoRampTimer = 0
        end

      elseif autoRampState == "done" then
        electrics.values.lifter_FR_input = 0
        electrics.values.lifter_FL_input = 0
        autoRampMode = false
        rampsDeployed = false
      end
    end
  end
end

-- Manual lifter control
local function liftWheel(wheel, value)
  if wheel == "FL" then
    electrics.values.lifter_FL_input = value
  elseif wheel == "FR" then
    electrics.values.lifter_FR_input = value
  end
end

-- Toggle automation
local function toggleAutoRamp()
  if not autoRampMode then
    autoRampMode = true
    autoRampState = "idle"
    autoRampTimer = 0
  end
end

-- Public interface
M.onInit          = onReset
M.onReset         = onReset
M.updateGFX       = updateGFX
M.liftWheel       = liftWheel
M.toggleAutoRamp  = toggleAutoRamp

return M
