local M = {}

local registeredVehicles, mailboxData, lightPropCache, zoneObbTable = {}, {}, {}, {}
local playerId, playerVeh = be:getPlayerVehicleID(0), be:getObjectByID(be:getPlayerVehicleID(0))
local curSlot = 0

local function getLaneOfPoint(_map, n1, n2, pos, vehFwd)
    local nodeA, nodeB = _map.nodes[n1], _map.nodes[n2]
    local invert = true
    if invert then nodeA, nodeB = _map.nodes[n1], _map.nodes[n2] else nodeA, nodeB = _map.nodes[n2], _map.nodes[n1] end

    local radius = nodeA.radius
    local roadWidth, laneWidth = radius * 2, (radius * 2 >= 6.1) and 3.05 or 2.4
    local link = nodeA.links[n2] or nodeB.links[n2]

    if link and link.oneWay then
        local forward = vec3(nodeB.pos:z0() - nodeA.pos:z0()):normalized()
        local rightVec = vec3(forward.y, -forward.x, 0):normalized()
        local dist = pos:distanceToLine(nodeA.pos + rightVec * radius, nodeB.pos + rightVec * nodeB.radius)
        return math.ceil(dist / laneWidth)
    else
        local midDist = pos:distanceToLine(nodeA.pos, nodeB.pos)
        local side = ((nodeA.pos.x - nodeB.pos.x) * (pos.y - nodeB.pos.y) - (nodeA.pos.y - nodeB.pos.y) * (pos.x - nodeB.pos.x)) > 0 and 1 or -1
        return math.ceil(midDist / laneWidth) * side
    end
end

local function getLane(center, halfX, axisX, checkCorners)
    local p1, p2 = map.findClosestRoad(center)
    if not p1 or not p2 then return 1 end
    local _map = map.getMap()

    if not checkCorners then return getLaneOfPoint(_map, p1, p2, center) end

    local offsets = {vec3(0, 0, 0), halfX * axisX, -halfX * axisX}
    for i = 1, 3 do offsets[i] = getLaneOfPoint(_map, p1, p2, center + offsets[i]) end

    local playerData = mailboxData[playerId]
    local playerLane = playerData and playerData.lane
    if playerLane and (playerLane == offsets[1] or playerLane == offsets[2] or playerLane == offsets[3]) then
        return playerLane
    end

    return math.floor((offsets[1] + offsets[2] + offsets[3]) + 0.5)
end

local function getCombinedVec3(v)
    return (v[1] + v[2] + v[3]) / 3
end

local function getAmbColor(veh)
    for _, data in pairs(zoneObbTable) do
        if data[1]:isContained(veh:getPosition()) then
            return getCombinedVec3(data[2]:toTable())
        end
    end

    local sky = scenetree.sunsky
    if not sky then
        local ids = scenetree.findClassObjects("ScatterSky")
        if ids[1] then sky = scenetree.findObject(ids[1]) end
    end

    if sky and sky.ambientScale then
        return getCombinedVec3(sky.ambientScale:toTable())
    end
    return 1
end

local function getLightsActive(object, id)
    local cache = lightPropCache[id]
    if not cache then
        local vdata = (core_vehicle_manager.getVehicleData(id) or {}).vdata or {}
        for _, prop in pairs(vdata.props or {}) do
            if prop.func and (prop.func:match("lowbeam") or prop.func:match("lowhighbeam")) then
                lightPropCache[id] = prop.pid
                break
            end
        end
        return false
    end
    return object.getProp and object:getProp(cache) and object:getProp(cache):getDataValue() > 0
end

local function buildCarTable(veh, vid)
    if not (veh and veh:getActive()) then return nil end

    local vehOBB = veh:getSpawnWorldOOBB()
    local c = vehOBB:getCenter()
    local a0, a1, a2 = vehOBB:getAxis(0), vehOBB:getAxis(1), vehOBB:getAxis(2)
    local hx, hy, hz = vehOBB:getHalfExtents().x, vehOBB:getHalfExtents().y, vehOBB:getHalfExtents().z

    local playerData = mailboxData[playerId]
    local playerCenter = playerData and playerData.center or vec3(0, 0, 0)
    local forward = playerVeh:getForwardVector()
    local dist = c:distance(playerCenter)
    local sameDir = dist < 70 and forward:dot((c - playerCenter):normalized()) > 0.5

    local tbl = {
        center = c,
        lane = getLane(c, hx, a0, sameDir),
        halfExtentsX = hx,
        x = hx * a0,
        y = hy * a1,
        z = hz * a2,
        lightsActive = getLightsActive(veh, vid),
        ambColor = vid == playerId and getAmbColor(playerVeh) or nil
    }

    tbl.x09, tbl.y2 = tbl.x * 0.9, tbl.y * 2
    return tbl
end

local function onUpdate(dt)
    local veh = be:getObject(curSlot - 1)
    if veh then mailboxData[veh:getId()] = buildCarTable(veh, veh:getId()) end

    curSlot = curSlot - 1
    if curSlot <= 0 then curSlot = be:getObjectCount() end

    be:sendToMailbox("systemelectricsvectoringcctsm", lpack.encode(mailboxData))
end

local function updateColTriMailbox()
    local trisData = {}
    for i = 0, be:getObjectCount() - 1 do
        local veh = be:getObject(i)
        local id = veh:getId()
        local vdata = core_vehicle_manager.getVehicleData(id) or {}
        trisData[id] = (vdata.vdata and vdata.vdata.triangles) or {}
    end
    be:sendToMailbox("systemelectricsvectoringVehTris", lpack.encode(trisData))
end

local function updateAmbientZones()
    zoneObbTable = {}
    for _, id in pairs(scenetree.findClassObjects("Zone")) do
        local obj = scenetree.findObject(id)
        if obj.useAmbientLightColor then
            local box = OrientedBox3F()
            box:set2(obj:getTransform(), obj:getScale())
            zoneObbTable[obj:getId()] = {box, obj.ambientLightColor}
        end
    end
end

local function registerVehicle(id)
    registeredVehicles[id] = true
end

local function onVehicleDestroyed(id)
    mailboxData[id] = nil
    updateColTriMailbox()
    registeredVehicles[id] = nil
    if not next(registeredVehicles) then extensions.unload("systemelectrics_nmg_stingerraycastmonitoring") end
end

local function onVehicleSpawned()
    updateColTriMailbox()
end

local function onVehicleSwitched(_, newId)
    playerId = newId
    playerVeh = be:getObjectByID(newId)
    curSlot = be:getObjectCount()
end

local function onExtensionLoaded()
    if FS:fileExists("/lua/ge/extensions/systemelectrics/systemelectricsvectoringGetVehData.lua") then
        extensions.load("systemelectrics_systemelectricsvectoringGetVehData")
        extensions.unload("systemelectrics_nmg_stingerraycastmonitoring")
        rawset(_G, "systemelectrics_nmg_stingerraycastmonitoring", systemelectrics_systemelectricsvectoringGetVehData)
        return
    end
    updateAmbientZones()
    updateColTriMailbox()
end

M.onExtensionLoaded = onExtensionLoaded
M.onVehicleSwitched = onVehicleSwitched
M.onVehicleDestroyed = onVehicleDestroyed
M.onVehicleSpawned = onVehicleSpawned
M.onClientPostStartMission = updateAmbientZones
M.registerVehicle = registerVehicle
M.onUpdate = onUpdate

return M
