--[[
MIT License

Copyright (c) 2025 DaddelZeit

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
]]

local M = {}
M.type = "auxiliary"

local modeOrder = {}
local currentMode = "default"
local modes = {}
local sendModeChangeMessage = false

local engine = nil
local psToWatt = 735.499

local manualGearUpdate = false
local limitsByGearActive = false

local powerLimitsByGear = nil
local powerLimit = nil
local boostLimitsByGear = nil
local boostLimit = nil

local powerLimitPID

local baseWastegate = 0
local lastGear = 0

local function setTurboOffset(wastegateOffset)
  if engine then
    engine.turbocharger.setWastegateOffset(-baseWastegate + wastegateOffset)
  end
end

local function updateGFX(dt)
  local power = engine.outputTorque1 * engine.outputAV1

  if not limitsByGearActive then
    if boostLimit then
      setTurboOffset(boostLimit or baseWastegate)
    elseif powerLimit then
      setTurboOffset(baseWastegate*powerLimitPID:get(power, powerLimit, dt))
    end
  else
    if boostLimitsByGear then
      if lastGear ~= electrics.values.gearIndex or manualGearUpdate then
        lastGear = electrics.values.gearIndex
        setTurboOffset(boostLimitsByGear[lastGear] or boostLimitsByGear[#boostLimitsByGear] or baseWastegate)
        manualGearUpdate = false
      end
    elseif powerLimitsByGear then
      local currentPowerLimit = powerLimitsByGear[electrics.values.gearIndex] or powerLimitsByGear[#powerLimitsByGear]
      setTurboOffset(currentPowerLimit and baseWastegate*powerLimitPID:get(power, currentPowerLimit, dt) or baseWastegate)
    end
  end
end

local function setParameters(parameters)
  if parameters.modeName then
    if modes[parameters.modeName] then
      currentMode = parameters.modeName
      parameters = modes[currentMode]
      if sendModeChangeMessage then
        guihooks.message("Boost Control: Set mode to "..(parameters.uiName or currentMode), 3, "powertrain.boostcontrol.setMode")
      end
    else
      return -- invalid config
    end
  end

  boostLimit = parameters.boostLimit or baseWastegate

  limitsByGearActive = false
  if not parameters.boostLimit and parameters.boostLimitsByGear then
    limitsByGearActive = true

    powerLimitsByGear = nil
    boostLimitsByGear = {}
    for k,v in ipairs(tableFromHeaderTable(parameters.boostLimitsByGear)) do
      boostLimitsByGear[v.gear] = v.boost
    end
    manualGearUpdate = true
  elseif not parameters.powerLimit and parameters.powerLimitsByGear then
    limitsByGearActive = true

    powerLimitsByGear = {}
    boostLimitsByGear = nil
    for k,v in ipairs(tableFromHeaderTable(parameters.powerLimitsByGear)) do
      powerLimitsByGear[v.gear] = v.power * psToWatt
    end
  end
end

local function changeMode(direction)
  direction = direction or 1
  local currentModeIndex = arrayFindValueIndex(modeOrder, currentMode) - 1
  local newMode = ((currentModeIndex+direction) % #modeOrder) + 1

  setParameters({modeName = modeOrder[newMode]})
end

local function getModeKeys(tbl)
  local keys = {}
  local keysidx = 0
  for k in pairs(tbl) do
    if v.data.enabledBoostModes then
      if not v.data.enabledBoostModes[k] then
        goto next
      end
    end

    keysidx = keysidx + 1
    keys[keysidx] = k

    ::next::
  end
  return keys
end

local function init(jbeamData)
  local engineName = jbeamData.engineName or "mainEngine"
  engine = powertrain.getDevice(engineName)
  if engine and v.data[engineName].turbocharger then
    local turbochargerName = v.data[engineName].turbocharger
    if not turbochargerName then return end
    local turbocharger = v.data[turbochargerName]
    if not turbocharger then return end

    if turbocharger.wastegateLimit then
      if type(turbocharger.wastegateLimit) == "number" then
        baseWastegate = turbocharger.wastegateLimit
      else
        log("E", "boostControl.init", "wastegateLimit is not a number, boostControl will not work.")
      end
    elseif turbocharger.wastegateStart then
      if type(turbocharger.wastegateStart) == "number" then
        baseWastegate = turbocharger.wastegateStart + 0.01
      else
        log("E", "boostControl.init", "wastegateStart is not a number, boostControl will not work.")
      end
    else
      return
    end

    modes = jbeamData.modes
    modeOrder = getModeKeys(modes)
    table.sort(modeOrder, function(a,b)
      return modes[a].order < modes[b].order
    end)

    currentMode = jbeamData.defaultMode or "default"
    sendModeChangeMessage = jbeamData.sendModeChangeMessage or false

    setParameters({modeName = currentMode})

    powerLimitPID = newPIDStandard(0.0001, 0.0015, 0.00, 0, 1, integralInCoef, integralOutCoef, minIntegral, maxIntegral)
    M.updateGFX = updateGFX
  end
end

M.init = init
M.reset = nop
M.updateGFX = nop

M.setParameters = setParameters
M.changeMode = changeMode

return M
