-- This Source Code Form is subject to the terms of the bCDDL, v. 1.1.
-- If a copy of the bCDDL was not distributed with this
-- file, You can obtain one at http://beamng.com/bCDDL-1.1.txt

local M = {}
M.type = "auxilliary"
M.relevantDevice = nil

local abs = math.abs
local max = math.max

local idleFrontPosition = 0
local idleMidPosition = 0
local idleRearPosition = 0
local brakingFrontPosition = 0
local brakingMidPosition = 0
local brakingRearPosition = 0

local transitionTimeIdle = 4
local transitionTimeBraking = 0.35

local lastEngineRunning = 0
local lastAirBrakeActive = false

local frontPositionSmoother = newTemporalSmoothing(1, 1)
local midPositionSmoother = newTemporalSmoothing(1, 1)
local rearPositionSmoother = newTemporalSmoothing(1, 1)

local speedThresholdAirBrake = 5
local brakeThresholdHigh = 0.4

local targetFrontPosition
local targetMidPosition
local targetRearPosition
local currentFrontPosition
local currentMidPosition
local currentRearPosition
local positionTransitionTimer

local function updateGFX(dt)
  targetFrontPosition = idleFrontPosition
  targetMidPosition = idleMidPosition
  targetRearPosition = idleRearPosition
  currentFrontPosition = electrics.values.spoilerF
  currentMidPosition = electrics.values.spoilerM
  currentRearPosition = electrics.values.spoilerR
  local speed = electrics.values.wheelspeed

  local yawControlOverrideAirBrake = electrics.values.yawControlRequestReduceOversteer or 0
  local activateAirBrake = (electrics.values.brake > brakeThresholdHigh or yawControlOverrideAirBrake > 0) and speed >= speedThresholdAirBrake
  if activateAirBrake ~= lastAirBrakeActive then
    positionTransitionTimer = transitionTimeBraking
  end
  if activateAirBrake then
    targetFrontPosition = brakingFrontPosition
    targetMidPosition = brakingMidPosition
    targetRearPosition = brakingRearPosition
  end

  if electrics.values.engineRunning ~= lastEngineRunning then
    positionTransitionTimer = transitionTimeIdle
  end
  if electrics.values.engineRunning < 1 then
    targetFrontPosition = 0
    targetMidPosition = 0
    targetRearPosition = 0
  end

  local frontRate = abs(currentFrontPosition - targetFrontPosition) / positionTransitionTimer
  local midRate = abs(currentMidPosition - targetMidPosition) / positionTransitionTimer
  local rearRate = abs(currentRearPosition - targetRearPosition) / positionTransitionTimer

  currentFrontPosition = frontPositionSmoother:getWithRateUncapped(targetFrontPosition, dt, frontRate)
  currentMidPosition = midPositionSmoother:getWithRateUncapped(targetMidPosition, dt, midRate)
  currentRearPosition = rearPositionSmoother:getWithRateUncapped(targetRearPosition, dt, rearRate)

  positionTransitionTimer = max(positionTransitionTimer - dt, 0)

  electrics.values.spoilerF = currentFrontPosition
  electrics.values.spoilerM = currentMidPosition
  electrics.values.spoilerR = currentRearPosition

  lastEngineRunning = electrics.values.engineRunning
  lastAirBrakeActive = activateAirBrake
end

local function reset(jbeamData)
  idleFrontPosition = 0
  idleMidPosition = 0
  idleRearPosition = 0
  brakingFrontPosition = 0
  brakingMidPosition = 0
  brakingRearPosition = 0
  electrics.values.spoilerF = idleFrontPosition
  electrics.values.spoilerM = idleMidPosition
  electrics.values.spoilerR = idleRearPosition
  frontPositionSmoother:set(idleFrontPosition)
  midPositionSmoother:set(idleMidPosition)
  rearPositionSmoother:set(idleRearPosition)
end

local function init(jbeamData)
  idleFrontPosition = 0
  idleMidPosition = 0
  idleRearPosition = 0
  brakingFrontPosition = 0
  brakingMidPosition = 0
  brakingRearPosition = 0
  electrics.values.spoilerF = idleFrontPosition
  electrics.values.spoilerM = idleMidPosition
  electrics.values.spoilerR = idleRearPosition
  frontPositionSmoother:set(idleFrontPosition)
  midPositionSmoother:set(idleMidPosition)
  rearPositionSmoother:set(idleRearPosition)
end

local function setParameters(parameters)

  if parameters.speedThresholdAirBrake then
    speedThresholdAirBrake = parameters.speedThresholdAirBrake
  end

  if parameters.idleFrontPosition then
    idleFrontPosition = parameters.idleFrontPosition
  end
  if parameters.idleMidPosition then
    idleMidPosition = parameters.idleMidPosition
  end
  if parameters.idleRearPosition then
    idleRearPosition = parameters.idleRearPosition
  end

  if parameters.brakingFrontPosition then
    brakingFrontPosition = parameters.brakingFrontPosition
  end
  if parameters.brakingMidPosition then
    brakingMidPosition = parameters.brakingMidPosition
  end
  if parameters.brakingRearPosition then
    brakingRearPosition = parameters.brakingRearPosition
  end

  positionTransitionTimer = transitionTimeIdle
end

M.init = init
M.reset = reset
M.updateGFX = updateGFX
M.setParameters = setParameters

return M
